pro HISTO1, shadow_in, col, shadow_out, GROUP=group, $
XRANGE=xrange, NBINS=nbins,NOLOST=nolost, REFLECTIVITY=ref, $
XTITLE=xtitle,YTITLE=ytitle, YRANGE=yrange, $
TITLE=title, WRITE=write,  OVER=over, SAVE=save, $
GAUSSFIT=gauss, fwhm_fit=fwhm_fit, $
X0=x0,PUBLISH=publish, YLOG=YLOG, MULFAC=mulfac, CM=CM, $
XSTYLE=xstyle,YSTYLE=ystyle, $
nsplit=nsplit, int_val=int_val, int_error=int_error, $
CALFWHM=CALFWHM, FWHM_VAL=fwhm_val,FWHM_MEAN=fwhm_mean,FWHM_ERROR=fwhm_error,$
CALERROR=CALERROR , _extra=extra
;+
; NAME:
;	HISTO1 
; PURPOSE:
;	an IDL implementation of the HISTO1 SHADOW's utility
; CATEGORY:
;	SHADOW's utilities.
; CALLING SEQUENCE:
;	histo1, shadow_in, col, shadow_out, keyword_parameters
; INPUTS:
;	shadow_in   IDL structure with SHADOW data
;	col   column to analyze (see doc in PLOTXY for a column description)
; OPTIONAL INPUTS:
;	shadow_out  name of the IDL-SHADOW structure where store the file
; KEYWORD PARAMETERS:
;	XRANGE = [min,max], limits of the histogram. 
;	YRANGE = [min,max], heigth limits of the histogram. 
;	NBINS = number of bins (def = 25)
;	NOLOST = consider losses
;		0 all rays (default option)
;		1 exclude losses 
;		2 consider only losses 
;	REFLECTIVITY = include reflectivity
;		0 no (default)
;		1 yes (A**2 as weight=column 23)
;               n (21<n<34, n=column to weight (see PLOTXY for col description))
;	GAUSSFIT = makes a gaussian fit of the histo
;		0  No (default)
;		1  Gauss+2nd degree polynomial (using gauss_fit routine)
;		2  Gaussian: using (fit_voigt(...,/Gaussian) routine)
;		3  PseudoVoigt: using (fit_voigt() routine)
;		4  Lorentzian: using (fit_voigt(...,/Lorentzian) routine)
;       TITLE = top title
;       XTITLE = title in X column
;       YTITLE = title in Y column
;	[XY]STYLE = Same effect as in plot. 
;		Default: 1 if [XY]RANGE is set, otherwise 0.
;	MULFAC = A multiplicative factor to change the X scales.
;       OVER   = overplots the previous histogram when set to 1
;       WRITE  = set this keyword to write the histogram in the file HISTO1
;               which also contains information of the calculation.
;
;		If WRITE=1, the starting position of each bin 
;		is in the bin's center. This is the default, and 
;               the resulting file can be plotted using xplot or
;               most other programs. 
;
;		If WRITE=2 the abscissas points correspond to the 
;		starting position of each bin (as in IDL histogram()),
;		and not to the central position of the bin. 
;
;	CALFWHM = if set to 1 then calculates the fwhm of the histogram
;	FWHM_VAL = fwhm_val stores the FWHM value of the histogram in
;	          an IDL varianle
;	FWHM_FIT = fwhm_fit stores the FWHM value of the gauss fit 
;	           in an IDL varianle
;	IDL_VAR = idl_var srores the histogram in an IDL variable
;	SAVE = a named variable where to save the histogram (if
;		defined as a single element) or to accumulate the 
;		histogram in sucessive runs. (see note in /WRITE).
;	CM = set this keyword to center the histogram abscissas at the
;               CenterOfMass of the data.
;	X0 = a numerical value to be deduced from the X array (useful 
;		when one wants to plot vs. E-Eo, for example).
;	PUBLISH = flag to supress graphic output not suitable for publication, 
;		i.e. username, date, time, filepath, etc.
;		When flag is present and non-zero, output is supressed.
;       GROUP The parent id for the caller widget (used to positioning
;               error and information windows).
;       NSPLIT: an integer to be used for statistical calculations of 
;               fwhm, etc. by splitting the rays in NSPLIT groups
;       FWHM_MEAN: mean value of fwhm, when NSPLIT is used. 
;       FWHM_ERROR: RMS error of the fwhm, when NSPLIT is used. 
;	CALERROR = if set to 1 add errorbars
;
;	OUTPUTS:
;	an histogram on the screen.
; OPTIONAL OUTPUT PARAMETERS:
;	shadow_out  name of the IDL-SHADOW structure where store the file
; COMMON BLOCKS:
;	Calls histogramw
; SIDE EFFECTS:
; 	None.
; RESTRICTIONS:
; 	None.
; KNOWN BUGS:
;	It has been found that the routine (or idl) gives bad results
;	when plotting vs col 11 using very small interval (very high
;	resolution). A temporal solution of this problem is to use the X0
;	keyword.
; PROCEDURE:
;	Similar to SHADOW's original.
; EXAMPLES:
;	1) Simplest run:
;		histo1,'begin.dat',1
;	2) Save variable:
;		histoall = 1
;		histo1,'begin.dat',1,xrange=[-1,1],save=histoall
;		plotfile,histoall,psym=10 ; to check if it contains data
;		histo1,'begin2.dat',1,xrange=[-1,1],save=histoall ;add
;							; a new histogram
;		(xrange must be defined to force histo1 to produce the
;		same gridding for the X values)
;
; MODOFICATION HISTORY:
;	M. Sanchez del Rio. ESRF. Grenoble May 1991
;       11-12-91 MSR output for Sigma in fit option
;       01-09-92 MSR  change input type.
;       03-09-92 MSR  introduce new columns. Few changes
;       02-10-92 MSR  added GETPHASE to correct phases for stokes parameters
;       06-10-92 MSR  added YRANGE and WRITE OPTIONS
;       19-04-93 MSR  changes GAUSSFIT by GAUSS_FIT
;       14-05-93 MSR  allows using either gaussfit or gauss_fit
;       29-07-93 MSR  adds CALFWHM,FWHM_VAL and FWHM_FIT
;       13-06-95 MSR  used getshcol, and allows external ref array.
;		      Use the histosh function.
;	26-06-96 MSR Adds save option. Speeds-up printing of time, date...
;	02-04-97 MSR Adds X0 keyword. Cosmetics.
;	03-03-98 MSR Adds PUBLISH keyword. Adapts to xsh_histo1 interface.
;	07-04-98 MSR uses column_name to display it. 
;	13-06-01 MSR sets the minimum for the plot different from zero
;		for the default case. Useful for negative values, like S3
;	16-10-06 MSR changes fitting routines. Added lorentzian
;	14-06-06 srio@esrf.eu adapted it to ne histogramw *some problems
;		affecting a shift of one bin removed*  Added WRITE=2
;	08-09-08 srio@esrf.eu added CM and MULFAC keywords.
;	08-09-12 srio@esrf.eu added [XY]STYLE keywords.
;	08-10-24 srio@esrf.eu bug fixed when save with different dimensions
;	2009-01-12 srio@esrf.eu added NSPLIT and intensity info.
;	2009-06-12 srio@esrf.eu documented NSPLIT etc. Fixed bug when Ref=0
;	2010-01-29 srio@esrf.eu now HISTO1 file also contains info and fit
;	2013-10-14 srio@esrf.eu adds max kw in histogramw call. 
;                   Calculation of COUNTS
;
;-
;on_error,2

forward_function histoerror
catch, error_status
 if error_status ne 0 then begin
   catch, /cancel
   message,/info,'Error caught: '+!err_string
   if sdep(/w) then itmp = dialog_message(/Error,Dialog_Parent=group,$
	'HISTO1: Error caught: '+!err_string)
   goto,out
endif

p_position_old=!p.position
p_charsize_old=!p.charsize


!p.position = [.1,.1,.9,.9]
bfy = float(!d.y_ch_size)/!d.y_size
bfx = float(!d.x_ch_size)/!d.x_size
tfact = 2.0
xsize = tfact*0.0065/bfx
ysize = tfact*0.018/bfy
!p.charsize = min([xsize,ysize]) 
;
; defaults
;
if not(keyword_set(nbins)) then  nbins = 25
if not(keyword_set(title)) then  title = ' '

if type(shadow_in) LT 6 then begin
  message,/info,'Using arrays...'
  arr1 =  shadow_in
  input_type = 1
  publish = 1
endif else input_type = 0

if keyword_set(ref) then begin
  if input_type eq 0 then begin
       if ref eq 1 then ref = 23
  endif
endif else begin
  ref=0
endelse

if not(keyword_set(ytitle)) and not(keyword_set(publish)) then  begin
  if keyword_set(ref) then begin
    if n_elements(ref) EQ 1 then $
       ytitle = 'Histogram weight column = '+column_name(ref) else $
       ytitle = 'Histogram weight column: EXTERNAL' 
  endif else begin
    ytitle = 'Intensity [arb. units]'
  endelse
endif
IF input_type EQ 0 THEN BEGIN
  if not(keyword_set(xtitle)) and not(keyword_set(publish)) then  begin
    xtitle = 'Column '+column_name(col)
    if keyword_set(x0) then xtitle=xtitle+' - '+strcompress(x0,/rem)
  endif
ENDIF ELSE BEGIN
   IF N_Elements(xtitle) EQ 0 THEN xtitle='X' ELSE xtitle=StrCompress(xtitle)
   IF N_Elements(ytitle) EQ 0 THEN ytitle='Y' ELSE ytitle=StrCompress(ytitle)
ENDELSE
;
; load shadow-idl structure and define arrays and constants
;
IF input_type EQ 0 THEN BEGIN
  shadow_out = readsh(shadow_in,GROUP=group,verbose=0)
  if type(shadow_out) NE 8 then message,'bad input'
  nelem = shadow_out.npoint
  flg = getshcol(shadow_out,10,verbose=0)
  arr = getshcol(shadow_out,col,nolost=nolost,verbose=0)
ENDIF ELSE BEGIN
  arr = shadow_in
  flg = arr*0.0+1
ENDELSE


if Keyword_Set(mulfac) then begin
  arr = arr*double(mulfac)
endif else mulfac=1D0


if keyword_set(x0) then arr = arr-x0
;
if keyword_set(ref) then begin
  if n_elements(ref) EQ 1 then $
  cp =  getshcol(shadow_out,ref,nolost=nolost) else $
  cp = ref( where(flg > 0) )
endif else begin
  cp= arr*0.+1.
endelse
if keyword_set(CM) then begin
  centerOfMass=Total(arr*cp)/Total(cp)
  arr = arr-centerOfmass
  cm_text='Zero at CenterOfMass: '+strcompress(centerOfMass,/rem)
  message,/info,cm_text
endif

if keyword_set(xrange) then begin
   IF N_Elements(xstyle) EQ 0 THEN xstyle=1
endif else begin
   xrange=[min(arr),max(arr)]
   IF N_Elements(xstyle) EQ 0 THEN xstyle=0
endelse
IF abs(xrange[0] -xrange[1]) LT 1D-10 THEN xrange=xrange[0]+[-1,1]

hh = histosh(arr,cp,nbins=nbins,xrange=xrange,binsize=binsize,/noshift)
;hh = histosh(arr,cp,nbins=nbins,xrange=xrange,binsize=binsize,/plot,/erase,/noshift)

if keyword_set(yrange) then begin
  IF N_Elements(ystyle) EQ 0 THEN ystyle=1
endif else begin
  yrange=[min(hh(1,*))<0,max(hh(1,*))]
  IF N_Elements(ystyle) EQ 0 THEN ystyle=0
endelse
IF abs(yrange[0]-yrange[1]) LT 1D-10 THEN yrange[0]=0


nEvents=N_Elements(arr)
IF keyword_set(calError) THEN BEGIN
  plothisto,hh(0,*),hh(1,*),hh(2,*),xrange=xrange,yrange=yrange, $
     xtitle=xtitle,ytitle=ytitle,ylog=ylog, xstyle=xstyle, ystyle=ystyle, $
     over=over,_Extra=extra,nEvents=nEvents
ENDIF ELSE BEGIN
  plothisto,hh(0,*),hh(1,*),xrange=xrange,yrange=yrange, $
     xtitle=xtitle,ytitle=ytitle,ylog=ylog, xstyle=xstyle, ystyle=ystyle, $
     over=over,_Extra=extra
ENDELSE


;
; for fitting and FWHM calculation, place the abscissas of the histogram
; in the center of the bin
;
hh0 = hh
hh[0,*]=hh[0,*]+binsize/2

; intensity
IF input_type NE 1 THEN BEGIN
  intens,shadow_out,int_val,nolost=nolost,nsplit=nsplit,splitError=int_error,verbose=0
  int_text = 'INTENS:'+StrCompress(int_val,/Remove_all)
  IF keyword_set(nsplit) THEN int_text = int_text+STRING("261B)+StrCompress(int_error,/Rem)
ENDIF
int_text = int_text 
counts_text = 'COUNTS: '+StrCompress(total(hh[1,*]),/remove_all)


if keyword_set(calfwhm) then begin
 fwhm_val = getfwhm(hh,/zeroBase,/plot,xxl=xxl,xxr=xxr)
 fwhm_text='FWHM: '+strcompress(fwhm_val,/rem)
 ; counts inside FWHM 
 igood = where( (hh[0,*] GE xxl) and (hh[0,*] le xxr) ) 
 if igood[0] NE -1 then begin
   counts_in_fwhm = total(hh[1,igood] )
   counts_text = counts_text +' ('+strcompress(counts_in_fwhm,/remove_all)+' in FWHM)'
 endif

 message,/info,int_text
 message,/info,fwhm_text
 message,/info,counts_text
 IF keyword_Set(nsplit) THEN BEGIN
   tmpAll=DblArr(nsplit)
   FOR jj=0L,NSplit-1 DO BEGIN
     tmpy = GetShCol(shadow_out,col,noLost=noLost,Block=[jj,nSplit],verbose=0)
     IF ref EQ 0 THEN tmpw = tmpy*0+1 ELSE $
       tmpw=GetShCol(shadow_out,ref,noLost=noLost,Block=[jj,nSplit],verbose=0)
     tmph = histosh(tmpy*mulFac,tmpw,nbins=nbins,xrange=xrange)
     tmpFwhm = getfwhm(tmph,/zeroBase)
     tmpAll[jj]=tmpFwhm
   ENDFOR
   tmpM = moment(tmpAll)
   fwhm_mean = tmpM[0]
   fwhm_error = Sqrt(tmpM[1])
   print,'HISTO1:        nSplit: ',nSplit
   print,'HISTO1:        FWHM (mean): ',fwhm_mean[0]
   print,'HISTO1:        FWHM (stDev): ',fwhm_error[0]
   fwhm_text=StrCompress(fwhm_text+STRING("261B)+String(fwhm_error),/Rem)
 ENDIF 
endif
;
; saves file
;
if N_Elements(save) GT 0 then begin
  if n_elements(save) eq 1 then save = hh else begin
    nhh = N_Elements(hh[1,*])
    nsave = N_Elements(save[1,*])
    if nhh NE nsave then $
	message,/Info,'Warning: Different dimensions for saving data'
    imax = Min([nhh,nsave])-1
    FOR i=0L,imax DO BEGIN
      save[1,i] = save[1,i] + hh[1,i]
    ENDFOR
  endelse
endif
;
; gauss fit
;
if keyword_set(gauss) then begin
 hx = reform(hh(0,*))  &  hy = reform(hh(1,*))
 CASE gauss OF
 1: begin
  hfit2 = gauss_fit (hx,hy,afit)
  oplot, hx,hfit2
  END
 2: begin
  yfit = fit_voigt (hx,hy,afit,/gauss)
  hfit2 = voigt1(hx,afit)
  oplot, hx,hfit2
  afit[2] = afit[2]/2.35 ; for compatibility with other fits
  END
 3: begin
  yfit = fit_voigt (hx,hy,afit)
  hfit2 = voigt1(hx,afit)
  oplot, hx,hfit2
  afit[2] = afit[2]/2.35 ; for compatibility with other fits
  END
 4: begin
  yfit = fit_voigt (hx,hy,afit,/lorent)
  hfit2 = voigt1(hx,afit)
  oplot, hx,hfit2
  afit[2] = afit[2]/2.35 ; for compatibility with other fits
  END
 else:
 ENDCASE
 message,/info,'SIGMA of fit is '+strcompress(afit(2),/rem)
 text_fit = 'FWHM  of fit is: '+StrCompress( 2.35*afit(2),/Rem)
 message,/info, text_fit
 fwhm_fit= 2.35*afit(2)
 endif
;
; titles
;
date = systime()
cd,current=pwd
if sdep() EQ 'UNIX' then begin
    user_host = getenv('USER')+'@'+ getenv('HOST')
    host = getenv('HOST')
endif else user_host=''
subtitle=''
IF input_type EQ 0 THEN subtitle=pwd+sdep(/ds)+shadow_out.name
subtitle=subtitle+'  '+date+'  '+user_host
ttext=''

if not(keyword_set(over)) and not(keyword_set(publish)) then begin
  xyouts,.1,.94,/norm,subtitle
  tsize = !p.charsize * 1.43
  xyouts,.1,.97,/norm,siz=tsize,title
  ttext=ttext+int_text
  if keyword_set(gauss) or keyword_set(calfwhm) or keyword_set(cm) then begin
    if keyword_set(gauss) then ttext=text_fit
    if keyword_set(calfwhm) then ttext=ttext+' '+fwhm_text
    if keyword_set(cm) then ttext=ttext+'    '+cm_text
    ttext=ttext+' '+counts_text
  endif
  xyouts,.1,.91,/norm,ttext
endif

;
; writes file
;
if keyword_set(write) then begin
openw,Unit,'HISTO1',/get_lun
  printf,Unit,'#F HISTO1'
  printf,Unit,'#C This file has been created using histo1 (XOP/SHADOWVUI)'
  printf,Unit,'#D '+date
  printf,Unit,'#UTITLE '+title
  printf,Unit,'#USUBTITLE '+subtitle
  printf,Unit,'#UTTEXT '+ttext
  printf,Unit,' '
  printf,Unit,'#S 1 histogram'
  CASE write OF 
    1:    BEGIN
          hherror = histoerror(hh[1,*],hh[2,*],nEvents,verbose=0)
          IF keyword_set(gauss) THEN BEGIN
            printf,Unit,'#N 4'
            printf,Unit,'#L '+xtitle+'  '+ytitle+'  StDev  GaussFit='+StrCompress(gauss,/Remove_All)
            printf,Unit,'#C NOTE THAT ABSCISSAS CORRESPOND TO THE CENTER OF EACH BIN'
            FOR i=0L,N_Elements(hh[0,*])-1 DO printf,Unit, $
                 hh[0,i],hh[1,i],hherror[i],hfit2[i],Format='(4G22.9)'
          ENDIF ELSE BEGIN 
            printf,Unit,'#N 3'
            printf,Unit,'#L '+xtitle+'  '+ytitle+'  StDev'
            printf,Unit,'#C NOTE THAT ABSCISSAS CORRESPOND TO THE CENTER OF EACH BIN'
            FOR i=0L,N_Elements(hh[0,*])-1 DO printf,Unit,hh[0,i],hh[1,i],hherror[i],Format='(3G22.9)'
          ENDELSE
          END
    2: BEGIN
          hh0error = histoerror(hh0[1,*],hh0[2,*],nEvents,verbose=0)
          printf,Unit,'#N 3'
          printf,Unit,'#L '+xtitle+'  '+ytitle+'  StDev'
          printf,Unit,'#C NOTE THAT ABSCISSAS CORRESPOND TO THE LEFT CORNER OF EACH BIN'
          FOR i=0L,N_Elements(hh0[0,*])-1 DO printf,Unit,hh0[0,i],hh0[1,i],hh0error[i],Format='(3G22.9)'
          END
    else: printf,Unit,'#U ERROR SELECTING WRITE KEYWORD IN HISTO1;'
  ENDCASE
free_lun,Unit
message,/info,'File HISTO1 written to disk.'
endif

OUT:
!p.position = p_position_old
!p.charsize = p_charsize_old
end
