PRO xsh_spectrum,parent=iparent,sourceType=sourceType, group=group

;+
; NAME:
;	XSH_SPECTRUM
;
; PURPOSE:
;	Computes and visualizes the source spectrum for the current 
;	selected source in the ShadowVui interface
;
; CATEGORY:
;	ShadowVUI  application.
;
; CALLING SEQUENCE:
;
;	XSH_SPECTRUM 
;
; INPUTS:
;	
;
; KEYWORD PARAMETERS:
;	PARENT = the widget if of the parent ShadowVUI window
;		(default: it uses the id of the last opened ShadowVUI window)
;	SOURCE_TYPE = type of source (0=geometrical, 1=bm, 2=wiggler,
;		3=undulator)
;		(default: it uses the selected source in ShadowVui)
;
; OUTPUTS:
;	An xplot window with the plot
;
; COMMON BLOCKS:
;       COMMON shadow_ifc,parent1
;
; SIDE EFFECTS:
;	creates temporary files for wiggler and undulators. 
;	These files could interfere with the source utilities
;
; RESTRICTIONS: 
;
; PROCEDURE:
;	runs sync_*.pro, shadow3 wiggler_spectrum and urgent, depending on the 
;	synchrotron source selected.
;
;	
; MODIFICATION HISTORY:
;	Idea: Bernd Meyer (LNLS) bmeyer@lnls.br
; 	Written by: M. Sanchez del Rio using some code from B.Meyer
;	2009/03/06 srio@esrf.eu first test version
;	2013/04/26 srio@esrf.eu adapted for shadow3
;
;-
COMMON shadow_ifc,parent1
common shadow3,shadow3_mode, shadow3_binary

catch, error_status
 if error_status ne 0 then begin
   catch, /cancel
   message,/info,'Error caught: '+!err_string
   if sdep(/w) then itmp = dialog_message('XSH_SPECTRUM: Error caught: '+$
        !err_string,/error)
   return
endif

;
; inputs: define defaults
;
IF N_Elements(iparent) EQ 0 THEN iparent=parent1

IF N_Elements(sourceType) EQ 0 THEN BEGIN
  widget_control,iparent,get_uvalue=state
  widget_control,state.wids.src,get_val=val
  sourceType=val
  print,'XSH_SPECTRUM: Calculating source index: ',sourceType
ENDIF



CASE sourceType OF
  0: BEGIN  ; geometrical
     rcurrent=100.0 ; use by now 100mA
     ; 
     ; delete old spectrum file
     ;
     ;
     shadow_SetIfcPar,Parent=iParent,'F_COLOR',f_color,source=sourceType,/Get

     ; 
     ; calculate flux for the different geometric sources
     ;
     case F_COLOR of
        1: begin ; single energy 
                shadow_SetIfcPar,Parent=iParent,'PH1',ph1,source=sourceType,/Get
                ;spectrum=dblarr(2,1)
                ; now use two points to avoid plot crashes
                spectrum=dblarr(2,2)
                spectrum[0,0]=PH1
                spectrum[0,1]=PH1+1
                ;spectrum(1,0)=str.PH1/1.602189E-10*tmp1.rcurrent
                spectrum[1,0]=PH1/physical_constants('ec')*1d9*rcurrent
                spectrum[1,1]=(PH1+1)/physical_constants('ec')*1d9*rcurrent
             end
        2: begin ; mutliple discrete energies, up to 10 energies
                shadow_SetIfcPar,Parent=iParent,'N_COLOR',n_color,source=sourceType,/Get
                spectrum=dblarr(2,n_color)
                for ii=0,n_color-1 do begin
                   shadow_SetIfcPar,Parent=iParent, $
                    'PH'+StrCompress(ii+1,/Remove_All),ph,source=sourceType,/Get
                   spectrum[0,ii]=ph
                   spectrum[1,ii]=ph/physical_constants('ec')*1d9*rcurrent
                endfor
             end
        3: begin ; uniform energy distribution
                spectrum=dblarr(2,2)
                for ii=0,1 do begin
                   shadow_SetIfcPar,Parent=iParent, $
                    'PH'+StrCompress(ii+1,/Remove_All),ph,source=sourceType,/Get
                   spectrum[0,ii]=ph
                   spectrum[1,ii]=ph/physical_constants('ec')*1d9*rcurrent
                endfor
             end
        4: begin ; relative intensities
                shadow_SetIfcPar,Parent=iParent,'N_COLOR',n_color,source=sourceType,/Get
                spectrum=dblarr(2,n_color)
                for ii=0,n_color-1 do begin
                   shadow_SetIfcPar,Parent=iParent, $
                    'PH'+StrCompress(ii+1,/Remove_All),ph,source=sourceType,/Get
                   shadow_SetIfcPar,Parent=iParent, $
                    'RL'+StrCompress(ii+1,/Remove_All),rl,source=sourceType,/Get
                   spectrum[0,ii]=ph
                   spectrum[1,ii]=ph/physical_constants('ec')*1d9*rcurrent*rl
                endfor
             end
        else:
     endcase  
     END
  1: BEGIN  ; bending magnet
     tmp1 = {ecurrent:200D0, npoints:101}
     XscrMenu,tmp1,/NOTYPE,ACTION=ACTION,DIALOG_PARENT=group, Wtitle='xsh_spectrum', $
        TITLES=['Electron current intensity [mA]','Number of points in the spectrum']
     if ACTION EQ 'DONT' then return
     ecurrent = tmp1.ecurrent
     npoints  = tmp1.npoints ; number of spectral points
     if npoints LE 0 then npoints=2
     ; 
     ; get energy limits
     ;
     npointsPsi = 300 ; number of points for Psi integration 
     shadow_SetIfcPar,Parent=iParent,'PH1',ph1,source=sourceType,/Get
     shadow_SetIfcPar,Parent=iParent,'PH2',ph2,source=sourceType,/Get

     ; 
     ; if source is monochromatic srcomp fails in the calculation
     ; To avoid that, separate limits by 1eV
     ;
     IF abs(ph1-ph2) LE 1d-6 THEN BEGIN
       ph2=ph1+1.0
       ;npoints = 2
     ENDIF

     ; 
     ; retrieve useful parameters
     ;
     shadow_SetIfcPar,Parent=iParent,'R_MAGNET',machine_r_m,source=sourceType,/Get
     shadow_SetIfcPar,Parent=iParent,'BENER',bener,source=sourceType,/Get
     ;retrieve the electron current. Not yet available: use 0.1A
     shadow_SetIfcPar,Parent=iParent,'HDIV1',hdiv1,source=sourceType,/Get
     shadow_SetIfcPar,Parent=iParent,'HDIV2',hdiv2,source=sourceType,/Get
     shadow_SetIfcPar,Parent=iParent,'VDIV1',vdiv1,source=sourceType,/Get
     shadow_SetIfcPar,Parent=iParent,'VDIV2',vdiv2,source=sourceType,/Get
     ;
     ; method 1: srcomp (for shadow2)
     ;
     IF shadow3_mode EQ 0 THEN BEGIN 
         delete_files,'SRCOMP'
         OPENW,unit,'xsh_spectrum_tmp.inp',/GET_LUN
         PRINTF,unit,'Bending Magnet'
         PRINTF,unit,'Bending Magnet'
    
         PRINTF,unit,STRCOMPRESS(machine_r_m,/REMOVE_ALL)
         PRINTF,unit,STRCOMPRESS(bener,/REMOVE_ALL)
         PRINTF,unit,STRCOMPRESS(ecurrent,/REMOVE_ALL)
         PRINTF,unit,STRCOMPRESS((hdiv1+hdiv2)*1000.0,/REMOVE_ALL)
         PRINTF,unit,STRCOMPRESS(ph1,/REMOVE_ALL)
         PRINTF,unit,STRCOMPRESS(ph2,/REMOVE_ALL)
         PRINTF,unit,STRCOMPRESS((ph2-ph1)/(npoints-1),/REMOVE_ALL) ; E step
         PRINTF,unit,'3'  ; [ Photons/sec/%bandwidth ]
         PRINTF,unit,'0.1'; 0.1%bw
         PRINTF,unit,'1'  ; use vertical acceptance
         PRINTF,unit,STRCOMPRESS( (vdiv1+vdiv2)*1e3,/REMOVE_ALL)
         FREE_LUN,unit
         ;
         ; run srcomp
         ;
         xsh_run,'srcomp < xsh_spectrum_tmp.inp'
         IF checkFile('SRCOMP') NE 1 THEN BEGIN
            itmp = Dialog_Message('Error calculating source spectrum with srcomp')
            RETURN
         ENDIF 
         spectrum = rascii('SRCOMP')
     ENDIF ELSE BEGIN ; shadow3 
         ; 
         ; method 2: using IDL functions sync_*
         ; 
         energy = makearray1(npoints,ph1,ph2)
         ; critical energy
         gammaSync = BENER*1d3/Physical_Constants('MEE') ; gamma
         lambdaCritical = 4.0*!dpi*machine_r_m/3.0/gammaSync^3*1D10 ; critical wavelength in A
         energyCritical =  Physical_Constants('HC')/lambdaCritical ; critical energy in eV
         flag = 0 
         if ( (vdiv1+vdiv2) LE 0.1) then flag=2 ; consider vdiv for small apaertures only
    
         print,"Critical wavelength [A] = ",lambdaCritical
         print,"Critical energy [eV] = ",energyCritical
         print,"machine_r_m [m]: ",machine_r_m
         print,"electron energy [GeV]: ",bener
         print,"electron current [A]: ",ecurrent
         print,"from hdiv1 [mrad]: ",-hdiv1*1e3
         print,"  to hdiv2 [mrad]: ",hdiv2*1e3
         print,"from vdiv1 [mrad]: ",-vdiv1*1e3
         print,"  to vdiv2 [mrad]: ",vdiv2*1e3
         ;print,"vdiv1+vdiv2: ",vdiv1+vdiv2
         if (flag EQ 2) then print,"Flux fully integrated in angle (Psi)"
    
         a5 = Sync_Ene(flag,energy,E_GeV=bener, I_A=ecurrent, HDIV_MRAD=(hdiv1+hdiv2)*1e3, $
             EC_eV=energyCritical, Psi_Min=-vdiv1*1d3,Psi_Max=vdiv2*1d3,Psi_NPoints=npointsPsi)
         spectrum = make_set(energy,a5)
     ENDELSE
     END
  2: BEGIN  ; wiggler

     tmp1 = {ecurrent:200D0, npoints:101}
     XscrMenu,tmp1,/NOTYPE,ACTION=ACTION,DIALOG_PARENT=group, Wtitle='xsh_spectrum', $
        TITLES=['Electron current intensity [mA]','Number of points in the spectrum']
     if ACTION EQ 'DONT' then return
     ecurrent = tmp1.ecurrent
     npoints  = tmp1.npoints ; number of spectral points
     if npoints LE 0 then npoints=2

     IF shadow3_mode EQ 0 THEN BEGIN ; shadow2
         ; 
         ; delete old spectrum file
         ;
         delete_files,[$
            'xsh_wiggler_spectrum_tmp.inp',$
            'xsh_wiggler_spectrum_tmp.out',$
            'xsh_ws_epath_tmp.inp',$
            'xsh_ws_epath_tmp.par',$
            'xsh_ws_epath_tmp.traj']
    
         ; 
         ; write input files for epath and wiggler_spectrum
         ;
    
         OPENW,unit,'xsh_ws_epath_tmp.inp',/GET_LUN
         ;
         ; write xsh_ws_epath_tmp.inp in the case of 
         ; standard wiggler (sinusoidal)
         ; Not yet implemented: 
         ;    - Magnetic field in external file
         ;    - Magnetic field from file with harmonics
         ;
         PRINTF,Unit,'1'
         shadow_SetIfcPar,Parent=iParent,'PERIODS',tmp,source=sourceType,/Get
         PRINTF,unit,tmp
         PRINTF,Unit,'1'
         shadow_SetIfcPar,Parent=iParent,'WAVLEN',tmp,source=sourceType,/Get
         PRINTF,unit,tmp
         shadow_SetIfcPar,Parent=iParent,'K',tmp,source=sourceType,/Get
         PRINTF,unit,tmp
         shadow_SetIfcPar,Parent=iParent,'BENER',tmp,source=sourceType,/Get
         PRINTF,unit,STRCOMPRESS(tmp,/REMOVE_ALL)
         PRINTF,unit,'101'
         PRINTF,unit,'1'
         PRINTF,unit,'xsh_ws_epath_tmp.par'
         PRINTF,unit,'xsh_ws_epath_tmp.traj'
         FREE_LUN,unit
         ;
         OPENW,unit,'xsh_wiggler_spectrum_tmp.inp',/GET_LUN
         PRINTF,unit,'xsh_ws_epath_tmp.traj'
         PRINTF,unit,'1'
         shadow_SetIfcPar,Parent=iParent,'PH1',ph1,source=sourceType,/Get
         PRINTF,unit,STRCOMPRESS(ph1,/REMOVE_ALL)
         shadow_SetIfcPar,Parent=iParent,'PH2',ph2,source=sourceType,/Get
         PRINTF,unit,STRCOMPRESS(ph2,/REMOVE_ALL)
    
         PRINTF,unit,STRCOMPRESS((ph2-ph1)/(npoints-1),/REMOVE_ALL)
         PRINTF,unit,'xsh_wiggler_spectrum_tmp.out'
         PRINTF,unit,'2'
         ; By now, fixed current to 100mA
         ; PRINTF,unit,STRCOMPRESS(tmp1.rcurrent,/REMOVE_ALL)
         PRINTF,unit,ecurrent
         FREE_LUN,unit
         ; 
         ; run epath and wiggler_spectrum
         ;
         widget_control,/hourglass
         xsh_run,'epath < xsh_ws_epath_tmp.inp'
         ;xsh_run,'btraj < xsh_ws_epath_tmp.inp'
         widget_control,/hourglass
         xsh_run,'wiggler_spectrum < xsh_wiggler_spectrum_tmp.inp'
         spectrum=rascii('xsh_wiggler_spectrum_tmp.out')
     ENDIF ELSE BEGIN  ; shadow3
         delete_files,'xsh_spectrum.'+['inp','out','traj','par']
         ; 
         ; write input files for shadow3
         ;
         OPENW,unit,'xsh_spectrum.inp',/GET_LUN
         ;
         ; standard wiggler only (sinusoidal)
         ; Not yet implemented: 
         ;    - Magnetic field in external file
         ;    - Magnetic field from file with harmonics
         ;
         PRINTF,Unit,'epath'
         PRINTF,Unit,'1'
         shadow_SetIfcPar,Parent=iParent,'PERIODS',tmp,source=sourceType,/Get
         PRINTF,unit,tmp
         PRINTF,Unit,'1'
         shadow_SetIfcPar,Parent=iParent,'WAVLEN',tmp,source=sourceType,/Get
         PRINTF,unit,tmp
         shadow_SetIfcPar,Parent=iParent,'K',tmp,source=sourceType,/Get
         PRINTF,unit,tmp
         shadow_SetIfcPar,Parent=iParent,'BENER',tmp,source=sourceType,/Get
         PRINTF,unit,STRCOMPRESS(tmp,/REMOVE_ALL)
         PRINTF,unit,'101'
         PRINTF,unit,'1'
         PRINTF,unit,'xsh_spectrum.par'
         PRINTF,unit,'xsh_spectrum.traj'
         ;
         PRINTF,unit,' '
         PRINTF,unit,'wiggler_spectrum'
         PRINTF,unit,'xsh_spectrum.traj'
         PRINTF,unit,'1'
         shadow_SetIfcPar,Parent=iParent,'PH1',ph1,source=sourceType,/Get
         PRINTF,unit,STRCOMPRESS(ph1,/REMOVE_ALL)
         shadow_SetIfcPar,Parent=iParent,'PH2',ph2,source=sourceType,/Get
         PRINTF,unit,STRCOMPRESS(ph2,/REMOVE_ALL)
    
         PRINTF,unit,STRCOMPRESS((ph2-ph1)/(npoints-1),/REMOVE_ALL)
         PRINTF,unit,'xsh_spectrum.out'
         PRINTF,unit,'2'
         ; By now, fixed current to 100mA
         ; PRINTF,unit,STRCOMPRESS(tmp1.rcurrent,/REMOVE_ALL)
         PRINTF,unit,ecurrent
         PRINTF,unit,'exit'
         FREE_LUN,unit
         ; 
         ; run epath and wiggler_spectrum
         ;
         widget_control,/hourglass
         xsh_run,shadow3_binary()+' < xsh_spectrum.inp'
         widget_control,/hourglass
         spectrum=rascii('xsh_spectrum.out')
     ENDELSE
     END
  3: BEGIN  ; undulator
     ;
     ; write the contents of the structure with parameters for running URGENT
     ;
     tmp1 = {ecurrent:200D0, npoints:201}
     XscrMenu,tmp1,/NOTYPE,ACTION=ACTION,DIALOG_PARENT=group, Wtitle='xsh_spectrum', $
        TITLES=['Electron current intensity [mA]','Number of points in the spectrum']
     if ACTION EQ 'DONT' then return
     ecurrent = tmp1.ecurrent
     npoints  = tmp1.npoints ; number of spectral points
     if npoints LE 0 then npoints=2

     name='urgent.inp'
     
     OPENW,unit,name,/GET_LUN

     PrintF,unit,STRCOMPRESS(1,/REMOVE_ALL)
     shadow_SetIfcPar,Parent=iParent,'LAMBDAU',tmp,source=sourceType,/Get
     PrintF,unit,STRCOMPRESS(tmp,/REMOVE_ALL)
     PrintF,unit,STRCOMPRESS(0.0,/REMOVE_ALL)    ;kx !!!!!!!!!!

     shadow_SetIfcPar,Parent=iParent,'K',tmp,source=sourceType,/Get
     PrintF,unit,STRCOMPRESS(tmp,/REMOVE_ALL)
     PrintF,unit,STRCOMPRESS(0.0,/REMOVE_ALL)

     shadow_SetIfcPar,Parent=iParent,'NPERIODS',tmp,source=sourceType,/Get
     PrintF,unit,STRCOMPRESS(tmp,/REMOVE_ALL)
   
     shadow_SetIfcPar,Parent=iParent,'EMIN',tmp,source=sourceType,/Get
     PrintF,unit,STRCOMPRESS(tmp,/REMOVE_ALL)
     shadow_SetIfcPar,Parent=iParent,'EMAX',tmp,source=sourceType,/Get
     PrintF,unit,STRCOMPRESS(tmp,/REMOVE_ALL)
     PrintF,unit,STRCOMPRESS(npoints,/REMOVE_ALL)
   
     shadow_SetIfcPar,Parent=iParent,'E_ENERGY',tmp,source=sourceType,/Get
     PrintF,unit,STRCOMPRESS(tmp,/REMOVE_ALL)

     PrintF,unit,ecurrent*1e-3
     shadow_SetIfcPar,Parent=iParent,'SX',sigmax,source=sourceType,/Get
     PrintF,unit,STRCOMPRESS(sigmax*10.0,/REMOVE_ALL)
     shadow_SetIfcPar,Parent=iParent,'SZ',sigmaz,source=sourceType,/Get
     PrintF,unit,STRCOMPRESS(sigmaz*10.0,/REMOVE_ALL)
     shadow_SetIfcPar,Parent=iParent,'EX',epsi_x,source=sourceType,/Get
     shadow_SetIfcPar,Parent=iParent,'EZ',epsi_z,source=sourceType,/Get
     PrintF,unit,STRCOMPRESS(epsi_x/sigmax*1000.0,/REMOVE_ALL)
     PrintF,unit,STRCOMPRESS(epsi_z/sigmaz*1000.0,/REMOVE_ALL)
   
     PrintF,unit,STRCOMPRESS(0.0,/REMOVE_ALL) ; dist
     PrintF,unit,STRCOMPRESS(0.0,/REMOVE_ALL) ; x center
     PrintF,unit,STRCOMPRESS(0.0,/REMOVE_ALL) ; z center
     shadow_SetIfcPar,Parent=iParent,'MAXANGLE',maxangle,source=sourceType,/Get
help,maxangle
     PrintF,unit,STRCOMPRESS(maxangle*2,/REMOVE_ALL) ; x mrad acceptance
     PrintF,unit,STRCOMPRESS(maxangle*2,/REMOVE_ALL) ; z mrad acceptance
     PrintF,unit,STRCOMPRESS(50,/REMOVE_ALL)
     PrintF,unit,STRCOMPRESS(50,/REMOVE_ALL)
   
     PrintF,unit,STRCOMPRESS(4,/REMOVE_ALL)   ; in angular aperture
     PrintF,unit,STRCOMPRESS(2,/REMOVE_ALL)
     PrintF,unit,STRCOMPRESS(-1,/REMOVE_ALL)
   
     PrintF,unit,STRCOMPRESS(0,/REMOVE_ALL)
     PrintF,unit,STRCOMPRESS(0,/REMOVE_ALL)
     PrintF,unit,STRCOMPRESS(0,/REMOVE_ALL)
     PrintF,unit,STRCOMPRESS(0.0,/REMOVE_ALL)
     PrintF,unit,STRCOMPRESS(0,/REMOVE_ALL)
     PrintF,unit,STRCOMPRESS(0.0,/REMOVE_ALL)
     FREE_LUN,unit

     ; 
     ; run urgent
     ; 
     widget_control,/hourglass
     xop_spawn,'urgent < urgent.inp'
     a = rascii('urgent.out')
     spectrum=a[[0,2],*]
     xdisplayfile1,"urgent.out",group=group
     END
  ELSE: 
ENDCASE

;
; visualization
;
spec2=dblarr(5,n_elements(spectrum[0,*]))
spec2[0,*]=spectrum[0,*]
spec2[1,*]=spectrum[1,*]
spec2[2,*]=spectrum[1,*]/spectrum[0,*]*1000.0
spec2[3,*]=spectrum[1,*]*physical_constants('ec')/(0.1D-2)
spec2[4,*]=spectrum[1,*]*spectrum[0,*]*physical_constants('ec')
titles=['Photon Energy [eV]', $
        'Flux[Phot/sec/0.1%bw]', $
        'Flux[Phot/sec/eV]',$
        'Power[Watts/0.1%bw]', $
        'Power[Watts/eV]']
Xplot,spec2,coltitles=titles,xtitle='-1',ytitle='-1',  $
        wtitle='Spectrum for current SHADOW source',$
        xcol=1,ycol=2,XLOG=0,YLOG=0

END

