;+
; NAME:
;	nrbEval
;
; PURPOSE:
;	This function returns the evaluated value of a NURBS for the
;	parametric values u and v
;
;       (Evaluation of NURBS curves or surfaces at parametric points along the
;       U and V directions. Either homogeneous coordinates are returned if the
;       weights are requested in the lhs arguments, or as cartesian coordinates)
;
; CATEGORY:
;	NURBS library
;
; CALLING SEQUENCE:
;    p = nrbeval(n1, upar, [vpar])
;
; INPUTS:
;    n1:	The NURBS structure (as created by nrbStruct() )
;    u: 	The u value (or array of values)
;    v: 	The v value (or array of values) (not needed for curves)
;
; OUTPUTS:
;   P = Evaluated points on the NURBS curve or surface as cartesian
;    coordinates (x,y,z). If w is included on the lhs XXX argument list
;    the points are returned as homogeneous coordinates (wx,wy,wz).
;
;
; OUTPUT KEYWORD PARAMETERS:
;    w: Array of weights 
;
; INPUT KEYWORD PARAMETERS:
;    space: set this keyword for returning the non-weighted points
;    draw: set this keyword for generate the plot of the curve or the surface
;
; PROCEDURE:
;	
; EXAMPLE:
;	
; MODIFICATION HISTORY:
;	written by Davide Bianchi. ESRF. 2006
;
;-	
FUNCTION nrbeval, srf, upar, vpar, w=w, space=space, draw=draw

;catch, error_status
;if error_status ne 0 then begin
;   message,/info,'error caught: '+!err_string
;   if sdep(/w) then itmp = Dialog_Message(/Error,$
;	'NURBS: nrbEval: error caught: '+!err_string)
;   catch, /cancel
;   on_error,2
;   RETURN,0
;endif


IF n_elements(srf.dim) EQ 2 THEN BEGIN ; SURFACE

  IF (min(upar,max=mupar) LT min(srf.uknots,max=muknots)) OR (mupar GT muknots) THEN BEGIN
    message,'wrong set of parametric U value'
  ENDIF

  IF (min(vpar,max=mvpar) LT min(srf.vknots,max=mvknots)) OR (mvpar GT mvknots) THEN BEGIN
    message,'wrong set of parametric V value'
  ENDIF

  Xgrid = srf.Xgrid    ;X grid coordinates
  Ygrid = srf.Ygrid    ;Y grid coordinates
  Zgrid = srf.Zgrid    ;Z grid coordinates
  Wgrid = srf.Wgrid    ;W grid values

  nx=srf.dim[0]
  ny=srf.dim[1]

  n = nx - 1       ;n is the number of the last index of X (in accordance with theory notation)
  m = ny - 1       ;m is the number of the last index of Y (in accordance with theory notation)

  ;== KNOTS RENAME==;
  uknots = srf.uknots
  vknots = srf.vknots

  ;== CONTROL MATRIX OF BASIS FUNCTION ==;

  ;the propouse is to use matrix calculation for fasting the operations
  ; RU an RV ar square matrices, coming by the thoery /*(ref)*/
  ;given a grid of 3D point we have that:
  ;S(u,v)=RU##Pij##RV'   (1)
  ;where RU and RV are the control basis matrices
  ;and ## is the IDL operator for row-by-column matrix product
  ;the main idea, is that given the parametric point (UU && VV) and the knots (u_knots, && v_knots)
  ;it's easly to generate the matrix of control point.
  
  p=srf.p
  q=srf.q

  ;== GENERATION OF THE CONTROL POLYNOM MATRICES ==;

  ;RU is the basis matrix along the U direction
  ;RV is the basis matrix along the V direction
  RU=__nrbctrlpolymat(upar,uknots, p, nx)  
  RV=transpose(__nrbctrlpolymat(vpar,vknots, q, ny))  



  det=RU##Wgrid##RV

  ;== EVALUATION OF THE SURFACE BY EACH PARAMETRIC POINTS ==;

  Xeval = RU##(Xgrid)##RV
  Yeval = RU##(Ygrid)##RV
  Zeval = RU##(Zgrid)##RV

  IF keyword_set(space) EQ 0 THEN BEGIN
    Xeval = Xeval/det
    Yeval = Yeval/det
    Zeval = Zeval/det
  ENDIF

  nup = n_elements(upar)
  nvp = n_elements(vpar)
  ;surf = dblarr(nup,4,nvp)

  surf=nrbconvsurf(weights=det,xx=xeval,yy=yeval,zz=zeval)
  ;surf=nrbconvsurf(weights=Wgrid,xx=xeval,yy=yeval,zz=zeval)

  ;== OUTPUT ==;
  eval=surf(*,0:2,*)
  w=surf(*,3,*)
  ;surface,Zeval, Xeval, Yeval
  IF keyword_set(draw) THEN isurface,Zeval, Xeval, Yeval

ENDIF ELSE BEGIN  ; CURVE

  IF (min(upar,max=mupar) LT min(srf.uknots,max=muknots)) OR (mupar GT muknots) THEN BEGIN
    message,'wrong set of parametric U value'
  ENDIF

  Xgrid = srf.Xgrid    ;X grid coordinates
  Ygrid = srf.Ygrid    ;Y grid coordinates
  Zgrid = srf.Zgrid    ;Z grid coordinates
  Wgrid = srf.Wgrid    ;W grid values


  nx=srf.dim[0]

  n = nx - 1       ;n is the number of the last index of X (in accordance with theory notation)

  ;== KNOTS RENAME==;
  uknots = srf.uknots

  ;== CONTROL MATRIX OF BASIS FUNCTION ==;

  ;the propouse is to use matrix calculation for fasting the operations
  ; RU an is a matrix, coming by the thoery /*(ref)*/
  ;given a grid of 3D point we have that:
  ;S(u)=RU##Pi   (1)
  ;where RU is the control basis matrix
  ;and ## is the IDL operator for row-by-column matrix product
  ;the main idea, is that given the parametric point (UU) and the knots (u_knots, && v_knots)
  ;it's easly to generate the matrix of control point.

  p=srf.p

  ;== GENERATION OF THE CONTROL POLYNOM MATRICES ==;

  RU=__nrbctrlpolymat(upar,uknots, p, nx)  ;RU is the basis matrix along the U direction
  
  det=RU##Wgrid

  ;== EVALUATION OF THE SURFACE BY EACH PARAMETRIC POINTS ==;

  Xeval = RU##(Xgrid)
  Yeval = RU##(Ygrid)
  Zeval = RU##(Zgrid)
  
  IF keyword_set(space) EQ 0 THEN BEGIN
    Xeval = Xeval/det
    Yeval = Yeval/det
    Zeval = Zeval/det
  ENDIF

  nup = n_elements(upar)

  surf=transpose([Xeval,Yeval,Zeval,det])
  ;surf=transpose([Xeval,Yeval,Zeval,Wgrid])

  ;== OUTPUT ==;
  eval=surf(*,0:2)
  w=surf(*,3)

  IF keyword_set(draw) THEN isurface,Zeval, Xeval, Yeval

ENDELSE

return, eval

END
