;+
; NAME:
;	nrbConvSurf
;
; PURPOSE:
;    this function returns a grid matrix (nx,4,ny)-dimension matrix
;	with nx(ny): number of points in x(y).
;	from a either a surface in  the standard surface shadow file, or
;	from x,y and z arrays.
;
; CATEGORY:
;	NURBS library
;
; CALLING SEQUENCE:
;	s = nrbConvSurf(file)
;
; INPUTS:
;    file:  name of the file with data (between quotes).
;           Insert all folder path if is not in the current directory.
;
;	    The SHADOW's file is defined in the first row by 2 numbers
;	    nx and ny the 2nd row defines the ny values. From the 3rd
;	    row till the end are defined:
;	    in the first coloumn the nx values and all the other columns
;	    defines the z grid values. z is defined by a grid of nx*ny values.
;
;	    Alternatively, if file is not defined, the z-vales
;           (elevation) of the surface, x and y values
;	    should be entered in the zz, xx and yy keywords.
;
;
; INPUT (file undefined) OR OUTPUT (file defines) KEYWORDS:
;    xx: the x array (1-dim)
;    yy: the y array (1-dim)
;    zz: the z array (2-dim; z(ny,nx) )
;    weights: the weights array (2-dim) weights(ny,nx)
;
; OUTPUTS:
;	A (nx,4,ny) matrix, with x,y,z,weights in the second index.
;
; PROCEDURE:
;	Uses internal procedure __nrbReadShadowSurf
;
; EXAMPLE:
;	;
;	; create a sphere
;	;
;	nx=20
;	ny=40
;	lx=100D0
;	ly=400D0
;	rr = 10d2 ; in cm
;	;
;	xx = MakeArray1(nx,-lx/2,lx/2,/Double)
;	yy = MakeArray1(ny,-ly/2,ly/2,/Double)
;	zz = DblArr(nx,ny)
;	FOR i=0L,nx-1 DO BEGIN
;	  FOR j=0L,ny-1 DO BEGIN
;	    zz[i,j]= -Sqrt(rr^2-xx[i]^2-yy[j]^2)
;	    zz[i,j]= zz[i,j]+rr
;	  ENDFOR
;	ENDFOR
;	matr2presur,zz,xx,yy,File='testsurf.dat'
;	;
;	; read and plot surface
;	;
;	a=nrbconvsurf('testsurf.dat')
;	nrbplot,a
;
;	; alternatively:
;	a1=nrbconvsurf(zz=transpose(zz), xx=xx, yy=yy)
;	nrbplot,a1
;
; MODIFICATION HISTORY:
;	written by Davide Bianchi. 06/16/2006. ESRF. 2006
;
;-


function nrbconvsurf, coefs,weights=weights,xx=xx,yy=yy,zz=zz

catch, error_status
if error_status ne 0 then begin
   message,/info,'error caught: '+!err_string
   if sdep(/w) then itmp = Dialog_Message(/Error,$
	'NURBS: nrbConvSurf: error caught: '+!err_string)
   catch, /cancel
   on_error,2
   RETURN,0
endif



;==coefs default==;
IF keyword_set(coefs) THEN BEGIN
  IF size(coefs,/type) NE 7 THEN BEGIN
    message,'file must be a string'
  ENDIF
  __nrbreadshadowsurf, coefs, xx=xx, yy=yy, zz=zz, nx=nx, ny=ny
ENDIF
;---------------------------------------------------;


nx=n_elements(zz[0,*])
ny=n_elements(zz[*,0])


grid=dblarr(nx,4,ny,/nozero)

xdim = size(xx,/n_dimensions)

ydim = size(yy,/n_dimensions)

zdim = size(zz,/n_dimensions)

CASE xdim OF

    1:	BEGIN
    	IF  zdim EQ 2 THEN FOR ii=0,ny-1 DO grid(*,0,ii) = xx ELSE BEGIN
    	FOR ii=0,ny-1 DO grid(*,0,ii) = xx[ii]
	ENDELSE
	END

    2: FOR ii=0,ny-1 DO grid(*,0,ii) = xx(ii,*)

ENDCASE

CASE ydim OF

    1: BEGIN
       IF  zdim EQ 2 THEN FOR jj=0,ny-1 DO grid(*,1,jj)= yy[jj] ELSE BEGIN
       FOR jj=0,ny-1 DO grid(*,1,jj)= yy[jj]
       ENDELSE
       END

    2: FOR jj=0,ny-1 DO grid(*,1,jj) = yy(jj,*)


ENDCASE

FOR ii=0,ny-1 DO BEGIN
grid(*,2,ii) = zz(ii,*)
ENDFOR

;== read the weights file==;
IF keyword_set(weights) THEN BEGIN
IF size(weights,/type) EQ 7 THEN BEGIN
readgridfile, weights, zz=wgrid
ENDIF ELSE BEGIN
wgrid=weights
ENDELSE
    FOR jj=0,ny-1 DO BEGIN

    grid(*,3,jj) = wgrid(jj,*)

    ENDFOR
ENDIF ELSE BEGIN
grid(*,3,*)=1
ENDELSE

return,grid

END ; nrbconvsurf
