FUNCTION vr_2slits,slit1,slit2,dist, $
    file=file,nrays=nrays, iseed=iseed,verbose=verbose, retrace=retrace1, $
    wcols=wcols,wsigma=wsigma

;+
; NAME:
;	VR_2SLITS
; PURPOSE:
;	This function creates a SHADOW source using a variance reduction method:
;	  The rays are forced to pass through two slits.
; CATEGORY:
;	SHADOW's utilities.
; CALLING SEQUENCE:
;	output=vr_2slits(slit1,slit2)
; OUTPUTS:
;	output is a SHADOW-IDL structure containing the source
; INPUTS:
;	slit1: a 2-element array containing the entrance slit width (in X) and 
;		height (in Z):  slit1=[w,h]
;	slit2: a 2-element array containing the exit slit width (in X) and 
;		height (in Z):  slit2=[w,h]
;	dist: the distance from slit1 to slit2
;       
; KEYWORD PARAMETERS:
;	file: set this keyword to a string containing the name of the
;		file for writing the output (Default: no file)
;	nrays: the number of generated rays. Note that this number is
;		not limeted here, but it is limited in the SHADOW kernel
;		at compilation time. 
;	iseed: a seed for the random number generator
;	verbose: Set this keyword to produce a text window with
;		information on the weihghting process (if selected):
;		verbose=1 creates xdisplayfile1 window
;		verbose=2 appends to last window
;	
;
;	retrace: The distance from slit 1 where to trace the rays
;		(usually the source plane). The output and output files
;		refer to this point. Default=0 
;	
;	---  Weighting parameters: 
;		Define these two parameters if you want to give a Gaussian 
;		weight to the rays
;		
;	wcols: array with SHADOW columns where to apply weights. 
;	wsigma: array of the same dimension than wcols containg
;		the sigma of the Gaussians for weighting. 
; PROCEDURE:
;	Creeates a random point inside slit 1 and another one in slit2
;	Link the two points with a straight line. This defines one ray.
;	Repeat the process for NRAYS. Retrace the rays to the point 
;	defined in retrace (if not set, rays are defined at the slit1
;	plane)at the slit1 plane).
;
;	If weights are selected, weight each ray by the product
;	of exp(-r**2/2/sigmar**2) where r is summed ever the selected
;	SHADOW columns. 
;
; MODIFICATION HISTORY:
;	M. Sanchez del Rio. ESRF. Grenoble, May 2006
;-
;

IF N_Elements(verbose) EQ 0 THEN verbose=0
IF N_Elements(nrays) EQ 0 THEN nrays=5000L
x1 =(-slit1[0]/2)+RandomU(iseed,nrays,/Double)*slit1[0]
y1=x1*0
z1 =(-slit1[1]/2)+RandomU(iseed,nrays,/Double)*slit1[1]
x2 =(-slit2[0]/2)+RandomU(iseed,nrays,/Double)*slit2[0]
y2=x2*0
z2 =(-slit2[1]/2)+RandomU(iseed,nrays,/Double)*slit2[1]

vx =  x2-x1
vy =  DIST
vz =  z2-z1
v_mod = sqrt(vx*vx+vy*vy+vz*vz)
vx=vx/v_mod
vy=vy/v_mod
vz=vz/v_mod
intens=x1*0+1

; 
; create shadow structure (str) at the slit1 position
;
a1 = Make_Set(x1,y1,z1,vx,vy,vz,intens)
writesh,'  ',a1,[1,2,3,4,5,6,7],str=src,/NoFile

;
; retrace to source plane
;
IF Keyword_Set(retrace1) THEN  $
  src1=retrace(src,dist=retrace1,/ResetY) ELSE $
  src1=src


txt='-- Source Generation Using Variance-reduction algorithm  --'
txt= [ '',txt,'', $
      'Slit1 aperture [cm] ='+Vect2string(slit1), $
      'Slit2 aperture [cm] ='+Vect2string(slit2), $
      'distance slit1-slit2 [cm]='+String(dist)]
      IF keyword_Set(retrace1) THEN txt=[txt,'retrace to source plane at [cm]='+String(retrace1)]

; 
; apply weights
;
IF N_Elements(wcols) NE 0 THEN BEGIN
   IF N_Elements(wcols) NE N_Elements(wsigma) THEN BEGIN
     Message,/Info,'Weight information (WCols, WSigma) must have the '+$
	'same dimensions. Not weigthed'
   ENDIF
   i1=getshcol(src1,7)

   txt=[txt,'','','------------  Gaussian weight information -----------------']
   txt=[txt,String(['col','total(w)','%','accumulated','%'],$
	Format='(A3,4A12)')]
   FOR i=0L,N_Elements(wcols)-1 DO BEGIN
     icol=wcols[i]
     xx = getshcol(src1,icol)
     ; Note the factor 4 because weight_amplitures = sqrt(weight_intensities)
     ;ii=exp(-xx^2/2/wsigma[i]^2)
     ii=exp(-xx^2/4/wsigma[i]^2)
     i1=i1*ii
     
     txt=[txt, String(icol,total(ii^2),total(ii^2)/nrays*100,$
	                   total(i1^2),total(i1^2)/nrays*100, $
			   Format='(I3,4F12.2)')]
   ENDFOR
   r=src1.ray
   r[6,*]=i1
   src1.ray=r 
ENDIF
 
CASE verbose OF
  0:
  1: XDisplayFile1,Text=txt,Title='vr_2slits info'
  2: XDisplayFile1_append,txt
  else:
ENDCASE

IF Keyword_Set(file) THEN putrays,src1,file
Return,src1
END

