;+
; 
; NAME: 
;	BEAM__DEFINE
;
; PURPOSE:
;	Object class and methods to handle SHADOW beams.
;	Used with ShadowVUI application. 
;
; CATEGORY:
;	Object tools
;
; CALLING SEQUENCE:
;	o1 = Obj_New('beam',input)
;	o1-><method>
;
; F=Function, P=Procedure
;
; METHODS:
;	INIT(F)	called when the object is created with o1=Obj_New('beam',input)
;		INPUTS:
;			input: a SHADOW file name or shadow idl/structure
;		KEYWORDS:
;			keywords are passed to readsh()
;
;	CLEANUP(P)	called when the object is destroyed with Obj_destroy,o1
;
;	INFO(P)	just prints a help
;
;	GETVALUE(F)   used to return arrays and parameters in the form
;                     of array [ncol,npoint]
;
;	GENSOURCE(P)  calls SHADOW and creates the source
;		INPUTS:
;			a "SOURCE" object with calculation parameters
;	TRACEOE(P)  calls SHADOW and creates the source
;		INPUTS:
;			input: a "OE" object with calculation parameters
;                       element: the number of oe (starting from 1)
;		OLDBEAM:
;			OLDbeam=a, returns a named variable with the input 
;                       beam (the output one overwrites the "BEAM" data at 
;                       the input)
;	
;
; EXAMPLE:
;
;	src=obj_new('source','start.00')
;	oe1=obj_new('oe')
;	oe1->load,'start.01'
;	beam1=obj_new('beam')
;	beam1->gensource,src
;	beam1->traceOE,oe1,1L
;	plotxy,beam,1,3
; 
;       obj_destroy,beam1
;       obj_destroy,stc
;       obj_destroy,oe1
;
;
; MODIFICATION HISTORY:
;	Initial version  by M. Sanchez del Rio, September 2011,
;
;-

;
;===============================================================================
;
PRO beam::load,file
Catch, error_status
IF error_status NE 0 THEN BEGIN
   Message,/Info,'error caught: '+!err_string
   IF SDep(/w) THEN itmp = Dialog_Message(/Error,$
	'beam::load: error caught: '+!err_string)
   Catch, /Cancel
   On_Error,2
   RETURN
ENDIF

tmp = readsh(file)

*(self.str) = tmp.ray

END

;
;===============================================================================
;
PRO beam::write,file
Catch, error_status
IF error_status NE 0 THEN BEGIN
   Message,/Info,'error caught: '+!err_string
   IF SDep(/w) THEN itmp = Dialog_Message(/Error,$
	'beam::write: error caught: '+!err_string)
   Catch, /Cancel
   On_Error,2
   RETURN
ENDIF

s = size(*(self.str))
tmp={name:'',ncol:s[1],npoint:s[2],ray:*(self.str) }
putrays,tmp,file

END
;
;===============================================================================
;
FUNCTION beam::getvalue,input,nolost=nolost

Catch, error_status
IF error_status NE 0 THEN BEGIN
   Message,/Info,'error caught: '+!err_string
   IF SDep(/w) THEN itmp = Dialog_Message(/Error,$
	'beam::getvalue: error caught: '+!err_string)
   Catch, /Cancel
   On_Error,2
   RETURN,0
ENDIF

s = size(*(self.str))
tmp={name:'',ncol:s[1],npoint:s[2],ray:*(self.str) }
case type(input) of
 0: return,tmp.ray  ; none
 2: BEGIN ; integer
    return,getshcol(tmp,input,nolost=nolost)
    END
 3: BEGIN ; long
    return,getshcol(tmp,input,nolost=nolost)
    END
endcase
 
END

;
;===============================================================================
;
PRO beam::setvalue,value

Catch, error_status
IF error_status NE 0 THEN BEGIN
   Message,/Info,'error caught: '+!err_string
   IF SDep(/w) THEN itmp = Dialog_Message(/Error,$
	'beam::setvalue: error caught: '+!err_string)
   Catch, /Cancel
   On_Error,2
   RETURN
ENDIF

*(self.str)=value
case type(value) of
 8: *(self.str)=value.ray
 else: *(self.str)=value
endcase
 

END

;
;===============================================================================
;
PRO beam::info

Catch, error_status
IF error_status NE 0 THEN BEGIN
   Message,/Info,'error caught: '+!err_string
   IF SDep(/w) THEN itmp = Dialog_Message(/Error,$
	'beam::info: error caught: '+!err_string)
   Catch, /Cancel
   On_Error,2
   RETURN
ENDIF
help,/obj,self
help,/str,*self.str

END


;
;===============================================================================
;
PRO beam::gensource,src

forward_function gensourcegeom, gensourcesync
Catch, error_status
IF error_status NE 0 THEN BEGIN
   Message,/Info,'error caught: '+!err_string
   IF SDep(/w) THEN itmp = Dialog_Message(/Error,$
	'beam::gensource: error caught: '+!err_string)
   Catch, /Cancel
   On_Error,2
   RETURN
ENDIF

if n_elements(src) eq 0 then src=obj_new('source')
str=src->getvalue()
IF ((str.FDISTR EQ 4) OR (str.FSOURCE_DEPTH EQ 4) OR (str.F_WIGGLER GT 0)) THEN BEGIN
  ray = gensourcesync(str)
ENDIF ELSE BEGIN
  ray = gensourcegeom(str)
ENDELSE
*(self.str)=ray

END

;
;===============================================================================
;
PRO beam::traceoe,oe1,element,oldbeam=ray

forward_function traceoe

Catch, error_status
IF error_status NE 0 THEN BEGIN
   Message,/Info,'error caught: '+!err_string
   IF SDep(/w) THEN itmp = Dialog_Message(/Error,$
	'beam::traceoe: error caught: '+!err_string)
   Catch, /Cancel
   On_Error,2
   RETURN
ENDIF

if n_elements(element) eq 0 then element=1L
if n_elements(oe1) eq 0 then oe1=obj_new('oe')
str=oe1->getvalue()
ray=*(self.str)
ray2 = traceoe(str, ray,  element)
*(self.str)=ray2

END


;
;===============================================================================
;
PRO beam::cleanup

Catch, error_status
IF error_status NE 0 THEN BEGIN
   Message,/Info,'error caught: '+!err_string
   IF SDep(/w) THEN itmp = Dialog_Message(/Error,$
	'beam::cleanup: error caught: '+!err_string)
   Catch, /Cancel
   On_Error,2
   RETURN
ENDIF

IF Ptr_Valid(self.str) THEN Ptr_Free,self.str
END

;
;===============================================================================
;
FUNCTION beam::init, input , _extra=extra



Catch, error_status
IF error_status NE 0 THEN BEGIN
  Message,/Info,'error caught: '+!err_string
  IF SDep(/w) THEN itmp = Dialog_Message(/Error,$
    'beam::init: error caught: '+!err_string)
  Catch, /Cancel
  On_Error,2
  RETURN,0
ENDIF


CASE Type(input) OF 
 0: BEGIN       ; no input
    self.str=Ptr_New(0.0D0)
    END
 7: BEGIN       ; string
  tmp=readsh(input, _extra=extra)
  self.str=Ptr_new(tmp.ray)
  END
 8: BEGIN       ; structure
  self.str=Ptr_new(input.ray)
  END
else: 
ENDCASE

RETURN,1
END

;
;===============================================================================
;
PRO beam__define
  tmp = {beam, str:Ptr_New() }
END

