FUNCTION cartesian_oval,y,z,p=p,q=q,n1=n1,n2=n2,approximation=approximation,coeffs=coeffs

;+
;
; NAME:
;	CARTESIAN_OVAL
; PURPOSE:
;	Calculate the Cartesian oval and its approximations
; CATEGORY:
;       SHADOW's utilities
; CALLING SEQUENCE:
;	result = cartesian_oval(y [,z], n1=n1, n2=n2, p=p, q=q, approximation = 0)
; INPUTS:
;	y: the coordinate along the lens (it may be an array).
; OPTIONAL INPUT PARAMETERS:
;       z: set this to the z (elevation) of the lens. In this case
;                      it returns the Optical Path Difference
;                      if undefined (default) it returns the lens elevation z
; KEYWORD PARAMETERS:
;        approximation =        0 (default): return the Cartesian oval (no approximations)
;		                1 returns the surface obtained from a first order series
;                                 expansion of the OPD
;		                2 returns the surface obtained from a second order series
;                                 expansion of the OPD
;        coeffs =  set this value to a named variable to return the 5x5 matrix with 
;                  the coefficients of the Cartesian oval (quadric equation).
; OUTPUTS:
;	the z values of the lens profile z(y)
; PROCEDURE:
;	See M. Sanchez del Rio and L. Alianelli paper in J. Synchrotron Rad.
;       (Submitted, 2012)
; MODIFICATION HISTORY:
;	by M. Sanchez del Rio. ESRF. Grenoble, January 20, 2012
;	20120228 srio@esrf.eu changed formula for quadratic term (easier)
;	
;-

on_error,2
;
; see RevisionPaperCartesianOval3d.nb for formulas
;

if n_elements(p) EQ 0 then p=4700d0 else p=double(p)
if n_elements(q) EQ 0 then q=10d0 else q=double(q)
if n_elements(n1) EQ 0 then n1=0.99999231d0 else n1=double(n1) ; Si at 10 keV
if n_elements(n2) EQ 0 then n2=1d0 else n2=double(n2)          ; Vacuum
if n_elements(y) EQ 0 then y=0d0 else y=double(y)
if n_elements(approximation) EQ 0 then approximation=0 ; no approx by default

;
; these are the coefficients of the Cartesian oval 
;
a=dblarr(6,6) ; zero indices not used, thus indices starting at 1, like in Mathematica
a[1,2] =  (8*n1*n2*p*q*(n1*p + n2*q))/((n1 - n2)*(n1 + n2)^2)
a[1,3] =  (4*n1^2*(n1 + n2)*p^2 + 4*n1*n2*(-n1 + n2)*p*q - 4*n2^2*(n1 + n2)*q^2)/((n1 - n2)*(n1 + n2)^2)
a[1,4] = (-4*(n1^2*p + n2^2*q))/(n1^2 - n2^2)
a[1,5] = 1d0
a[3,1] =  (-4*n1*n2*(n2*p + n1*q)*(n1*p + n2*q))/(n1^2 - n2^2)^2
a[3,2] =  (-4*(n1^2*p + n2^2*q))/(n1^2 - n2^2)
a[3,3] =  2d0
a[5,1] = 1d0 
factor=(n1^4 - 2*n1^2*n2^2 + n2^4) ; normalization factor cl[[1, 5]]]
coeffs = a[1:5,1:5]
;print,'a[1,2] = ', a[1,2] 
;print,'a[1,3] = ', a[1,3] 
;print,'a[1,4] = ', a[1,4] 
;print,'a[1,5] = ', a[1,5] 
;print,'a[3,1] = ', a[3,1] 
;print,'a[3,2] = ', a[3,2] 
;print,'a[3,3] = ', a[3,3] 
;print,'a[5,1] = ', a[5,1] 


;
; if z is not given, returns OPD
;
if n_elements(z) NE 0 then begin ; returns OPD
  ; OPD is just the evaluation of the Cartesian oval coefficients
  ; times the normalization
  z=double(z)
  opd=0d0
  for i=1,5 do begin
    for j=1,5 do begin
      opd = opd + a[i,j]*(y^(double(i)-1d0))*(z^(double(j)-1d0))
    endfor
  endfor
  RETURN,opd*factor
endif 


;
; returns the surface z(y)
;

result = y*0d0
CASE approximation OF
0:begin ; full oval, no approximations
  FOR k=0L,N_Elements(y)-1 DO BEGIN
    yy = y[k]
    c=dblarr(5)
    c[4] = a[1,5]  ; z^4
    c[3] = a[1,4]  ; z^3
    c[2] = a[1,3] + yy^2 * a[3,3]  ; z^2
    c[1] = a[1,2] + yy^2 * a[3,2]  ; z^1
    c[0] =          yy^2 * a[3,1] + yy^4 * a[5,1]  ; z^0
    ; solve the fourth degree polynomial equation in z
    zeroes = FZ_ROOTS(c,/Double,eps=1d-10)
    ;print,'>>>>',zeroes
    ; from all the solutions, keep the real ones
    igood = where(imaginary(zeroes) EQ 0)
    IF  igood[0] EQ -1 THEN result[k]=0d0/0d0
    zeroes = zeroes[igood]
    ; cast im->real
    zeroes = real_part(zeroes)
    ; choose the smaller one
    tmp = min(abs(zeroes),itmp)
    result[k] = zeroes[itmp]
  ENDFOR
  end
1: begin ; first order
   ; From series expansion of opt_path (see CartesianOval3.nb)
   o_cte2 = -(n1*p) - n2*q + $
         n1*Sqrt(p^2 + y^2) + $
         n2*Sqrt(q^2 + y^2)
   o_pendent2 = -((n1*p)/Sqrt(p^2 + y^2)) + $
         (n2*q)/Sqrt(q^2 + y^2)
   result = -o_cte2/o_pendent2
  END
2: BEGIN ; second order
   ; From series expansion of opt_path (see CartesianOval3.nb)
   o_cte2 = -(n1*p) - n2*q + $
         n1*Sqrt(p^2 + y^2) + $
         n2*Sqrt(q^2 + y^2)
   o_linear2 = -((n1*p)/Sqrt(p^2 + y^2)) + $
         (n2*q)/Sqrt(q^2 +  y^2)
   ;o_quadratic2 = -(n1*p^2)/(2.*(p^2 + y^2)^1.5) +  $
   ;      n1/(2.*Sqrt(p^2 + y^2)) - $
   ;      (n2*q^2)/(2.*(q^2 + y^2)^1.5) + $
   ;      n2/(2.*Sqrt(q^2 + y^2))
   ; easier expression
   o_quadratic2 = (n1*y^2)/(2.*(p^2 + y^2)^1.5) + (n2*y^2)/(2.*(q^2 + y^2)^1.5)
   result = ( -o_linear2 + Sqrt(o_linear2^2-4*o_quadratic2*o_cte2) )/2/o_quadratic2
  end
2: begin
  end
else: 
ENDCASE

RETURN,result
END

