PRO xsh_spectrum,parent=iparent,sourceType=sourceType

;+
; NAME:
;	XSH_SPECTRUM
;
; PURPOSE:
;	Computes and visualizes the source spectrum for the current 
;	selected source in the ShadowVui interface
;
; CATEGORY:
;	ShadowVUI  application.
;
; CALLING SEQUENCE:
;
;	XSH_SPECTRUM 
;
; INPUTS:
;	
;
; KEYWORD PARAMETERS:
;	PARENT = the widget if of the parent ShadowVUI window
;		(default: it uses the id of the last opened ShadowVUI window)
;	SOURCE_TYPE = type of source (0=geometrical, 1=bm, 2=wiggler,
;		3=undulator)
;		(default: it uses the selected source in ShadowVui)
;
; OUTPUTS:
;	An xplot window with the plot
;
; COMMON BLOCKS:
;       COMMON shadow_ifc,parent1
;
; SIDE EFFECTS:
;	creates temporary files. 
;	These files could interfere with the source utilities
;
; RESTRICTIONS: 
;
; PROCEDURE:
;	runs srcomp, wiggler_spectrum and urgent, depending on the 
;	synchrotron source selected.
;
; TODO: 
;       Unify the file names.
;	Introduce the electron current in ShadowVUI and retrieve it here.
;	
; MODIFICATION HISTORY:
;	Idea: Bernd Meyer (LNLS) bmeyer@lnls.br
; 	Written by: M. Sanchez del Rio using some code from B.Meyer
;	2009/03/06 srio@esrf.eu first test version
;
;-
COMMON shadow_ifc,parent1
common shadow3,shadow3_mode

catch, error_status
 if error_status ne 0 then begin
   catch, /cancel
   message,/info,'Error caught: '+!err_string
   if sdep(/w) then itmp = dialog_message('XSH_SPECTRUM: Error caught: '+$
        !err_string,/error)
   return
endif

IF shadow3_mode NE 0 THEN BEGIN
   if sdep(/w) then itmp = dialog_message('XSH_SPECTRUM: Not yet implemented for Shadow3',/error)
   return
ENDIF


;
; inputs: define defaults
;
IF N_Elements(iparent) EQ 0 THEN iparent=parent1

IF N_Elements(sourceType) EQ 0 THEN BEGIN
  widget_control,iparent,get_uvalue=state
  widget_control,state.wids.src,get_val=val
  sourceType=val
  print,'XSH_SPECTRUM: Calculating source index: ',sourceType
ENDIF


CASE sourceType OF
  0: BEGIN  ; geometrical
     rcurrent=100.0 ; use by now 100mA
     ; 
     ; delete old spectrum file
     ;
     delete_files,'srcgeo.out'
     ;
     shadow_SetIfcPar,Parent=iParent,'F_COLOR',f_color,source=sourceType,/Get

     ; 
     ; calculate flux for the different geometric sources
     ;
     case F_COLOR of
        1: begin ; single energy 
                shadow_SetIfcPar,Parent=iParent,'PH1',ph1,source=sourceType,/Get
                ;spec=dblarr(2,1)
                ; now use two points to avoid plot crashes
                spec=dblarr(2,2)
                spec[0,0]=PH1
                spec[0,1]=PH1+1
                ;spec(1,0)=str.PH1/1.602189E-10*tmp1.rcurrent
                spec[1,0]=PH1/physical_constants('ec')*1d9*rcurrent
                spec[1,1]=(PH1+1)/physical_constants('ec')*1d9*rcurrent
             end
        2: begin ; mutliple discrete energies, up to 10 energies
                shadow_SetIfcPar,Parent=iParent,'N_COLOR',n_color,source=sourceType,/Get
                spec=dblarr(2,n_color)
                for ii=0,n_color-1 do begin
                   shadow_SetIfcPar,Parent=iParent, $
                    'PH'+StrCompress(ii+1,/Remove_All),ph,source=sourceType,/Get
                   spec[0,ii]=ph
                   spec[1,ii]=ph/physical_constants('ec')*1d9*rcurrent
                endfor
             end
        3: begin ; uniform energy distribution
                spec=dblarr(2,2)
                for ii=0,1 do begin
                   shadow_SetIfcPar,Parent=iParent, $
                    'PH'+StrCompress(ii+1,/Remove_All),ph,source=sourceType,/Get
                   spec[0,ii]=ph
                   spec[1,ii]=ph/physical_constants('ec')*1d9*rcurrent
                   ;spec(0,ii)=str.(nn+ii)
                   ;spec(1,ii)=str.(nn+ii)/1.602189E-10*tmp1.rcurrent
                endfor
             end
        4: begin ; relative intensities
                shadow_SetIfcPar,Parent=iParent,'N_COLOR',n_color,source=sourceType,/Get
                ;nn=where(tag_names(str) eq 'PH1')
                ;nl=where(tag_names(str) eq 'RL1')
                spec=dblarr(2,n_color)
                for ii=0,n_color-1 do begin
                   shadow_SetIfcPar,Parent=iParent, $
                    'PH'+StrCompress(ii+1,/Remove_All),ph,source=sourceType,/Get
                   shadow_SetIfcPar,Parent=iParent, $
                    'RL'+StrCompress(ii+1,/Remove_All),rl,source=sourceType,/Get
                   spec[0,ii]=ph
                   ;spec[1,ii]=str.(nn+ii)/1.602189E-10*tmp1.rcurrent*str.(nl+ii)
                   spec[1,ii]=ph/physical_constants('ec')*1d9*rcurrent*rl
                endfor
             end
        else:
     endcase  
     ; 
     ; write output file
     ;
     openw,lunf,'srcgeo.out',/get_lun
     printf,lunf,spec
     free_lun,lunf
     END
  1: BEGIN  ; bending magnet
     ; 
     ; delete old spectrum file
     ;
     delete_files,'SRCOMP'
     ; 
     ; get energy limits
     ;
     npoints = 200 ; number of spectral points
     shadow_SetIfcPar,Parent=iParent,'PH1',ph1,source=sourceType,/Get
     shadow_SetIfcPar,Parent=iParent,'PH2',ph2,source=sourceType,/Get

     ; 
     ; if source is monochromatic srcomp fails in the calculation
     ; To avoid that, include two points, separated 1eV
     ;
     IF abs(ph1-ph2) LE 1d-6 THEN BEGIN
       ph2=ph1+1.0
       npoints = 2
     ENDIF
     ;
     ; write input file for srcomp
     ;
     OPENW,unit,'xsh_spectrum_tmp.inp',/GET_LUN
     PRINTF,unit,'Bending Magnet'
     PRINTF,unit,'Bending Magnet'

     shadow_SetIfcPar,Parent=iParent,'R_MAGNET',tmp,source=sourceType,/Get
     PRINTF,unit,STRCOMPRESS(tmp,/REMOVE_ALL)

     shadow_SetIfcPar,Parent=iParent,'BENER',tmp,source=sourceType,/Get
     PRINTF,unit,STRCOMPRESS(tmp,/REMOVE_ALL)

     ;retrieve the electron current. Not yet available: use 0.1A
     ;shadow_SetIfcPar,Parent=iParent,'RCURRENT',tmp,source=sourceType,/Get
     ;tmp=tmp*1e-3
     ;PRINTF,unit,STRCOMPRESS(tmp,/REMOVE_ALL)
     PRINTF,unit,STRCOMPRESS('0.1',/REMOVE_ALL)

     shadow_SetIfcPar,Parent=iParent,'HDIV1',tmp1,source=sourceType,/Get
     shadow_SetIfcPar,Parent=iParent,'HDIV2',tmp2,source=sourceType,/Get
     PRINTF,unit,STRCOMPRESS((tmp1+tmp2)*1000.0,/REMOVE_ALL)

     PRINTF,unit,STRCOMPRESS(ph1,/REMOVE_ALL)
     PRINTF,unit,STRCOMPRESS(ph2,/REMOVE_ALL)
     PRINTF,unit,STRCOMPRESS((ph2-ph1)/(npoints-1),/REMOVE_ALL) ; E step
     PRINTF,unit,'3'  ; [ Photons/sec/%bandwidth ]
     PRINTF,unit,'0.1'; 0.1%bw
     PRINTF,unit,'1'  ; use vertical acceptance
     shadow_SetIfcPar,Parent=iParent,'VDIV1',tmp1,source=sourceType,/Get
     shadow_SetIfcPar,Parent=iParent,'VDIV2',tmp2,source=sourceType,/Get
     PRINTF,unit,STRCOMPRESS(tmp1+tmp2,/REMOVE_ALL)
     FREE_LUN,unit
     ;
     ; run srcomp
     ;
     xsh_run,'srcomp < xsh_spectrum_tmp.inp'
     IF checkFile('SRCOMP') NE 1 THEN BEGIN
        itmp = Dialog_Message('Error calculating source spectrum with srcomp')
        RETURN
     ENDIF 

     END
  2: BEGIN  ; wiggler

     ; 
     ; delete old spectrum file
     ;
     delete_files,[$
        'xsh_wiggler_spectrum_tmp.inp',$
        'xsh_wiggler_spectrum_tmp.out',$
        'xsh_ws_epath_tmp.inp',$
        'xsh_ws_epath_tmp.par',$
        'xsh_ws_epath_tmp.traj']

     ; 
     ; write input files for epath and wiggler_spectrum
     ;

     OPENW,unit,'xsh_ws_epath_tmp.inp',/GET_LUN
     ;
     ; write xsh_ws_epath_tmp.inp in the case of 
     ; standard wiggler (sinusoidal)
     ; Not yet implemented: 
     ;    - Magnetic field in external file
     ;    - Magnetic field from file with harmonics
     ;
     PRINTF,Unit,'1'
     shadow_SetIfcPar,Parent=iParent,'PERIODS',tmp,source=sourceType,/Get
     PRINTF,unit,tmp
     PRINTF,Unit,'1'
     shadow_SetIfcPar,Parent=iParent,'WAVLEN',tmp,source=sourceType,/Get
     PRINTF,unit,tmp
     shadow_SetIfcPar,Parent=iParent,'K',tmp,source=sourceType,/Get
     PRINTF,unit,tmp
     shadow_SetIfcPar,Parent=iParent,'BENER',tmp,source=sourceType,/Get
     PRINTF,unit,STRCOMPRESS(tmp,/REMOVE_ALL)
     PRINTF,unit,'101'
     PRINTF,unit,'1'
     PRINTF,unit,'xsh_ws_epath_tmp.par'
     PRINTF,unit,'xsh_ws_epath_tmp.traj'
     FREE_LUN,unit
     ;
     OPENW,unit,'xsh_wiggler_spectrum_tmp.inp',/GET_LUN
     PRINTF,unit,'xsh_ws_epath_tmp.traj'
     PRINTF,unit,'1'
     shadow_SetIfcPar,Parent=iParent,'PH1',ph1,source=sourceType,/Get
     PRINTF,unit,STRCOMPRESS(ph1,/REMOVE_ALL)
     shadow_SetIfcPar,Parent=iParent,'PH2',ph2,source=sourceType,/Get
     PRINTF,unit,STRCOMPRESS(ph2,/REMOVE_ALL)

     PRINTF,unit,STRCOMPRESS((ph2-ph1)/200,/REMOVE_ALL)
     PRINTF,unit,'xsh_wiggler_spectrum_tmp.out'
     PRINTF,unit,'2'
     ; By now, fixed current to 100mA
     ; PRINTF,unit,STRCOMPRESS(tmp1.rcurrent,/REMOVE_ALL)
     PRINTF,unit,'100.0'
     FREE_LUN,unit
     ; 
     ; run epath and wiggler_spectrum
     ;
     widget_control,/hourglass
     xsh_run,'epath < xsh_ws_epath_tmp.inp'
     ;xsh_run,'btraj < xsh_ws_epath_tmp.inp'
     widget_control,/hourglass
     xsh_run,'wiggler_spectrum < xsh_wiggler_spectrum_tmp.inp'
     END
  3: BEGIN  ; undulator
     ;
     ; write the contents of the structure with parameters for running URGENT
     ;
     name='urgent.inp'
     
     OPENW,unit,name,/GET_LUN

     PrintF,unit,STRCOMPRESS(1,/REMOVE_ALL)
     shadow_SetIfcPar,Parent=iParent,'LAMBDAU',tmp,source=sourceType,/Get
     PrintF,unit,STRCOMPRESS(tmp,/REMOVE_ALL)
     PrintF,unit,STRCOMPRESS(0.0,/REMOVE_ALL)    ;kx !!!!!!!!!!

     shadow_SetIfcPar,Parent=iParent,'K',tmp,source=sourceType,/Get
     PrintF,unit,STRCOMPRESS(tmp,/REMOVE_ALL)
     PrintF,unit,STRCOMPRESS(0.0,/REMOVE_ALL)

     shadow_SetIfcPar,Parent=iParent,'NPERIODS',tmp,source=sourceType,/Get
     PrintF,unit,STRCOMPRESS(tmp,/REMOVE_ALL)
   
     shadow_SetIfcPar,Parent=iParent,'EMIN',tmp,source=sourceType,/Get
     PrintF,unit,STRCOMPRESS(tmp,/REMOVE_ALL)
     shadow_SetIfcPar,Parent=iParent,'EMAX',tmp,source=sourceType,/Get
     PrintF,unit,STRCOMPRESS(tmp,/REMOVE_ALL)
     PrintF,unit,STRCOMPRESS(1000,/REMOVE_ALL)
   
     shadow_SetIfcPar,Parent=iParent,'E_ENERGY',tmp,source=sourceType,/Get
     PrintF,unit,STRCOMPRESS(tmp,/REMOVE_ALL)

     ; use by now 0.1A
     ; PrintF,unit,STRCOMPRESS(tmp1.rcurrent/1000.0,/REMOVE_ALL)
     PrintF,unit,'0.1'
     shadow_SetIfcPar,Parent=iParent,'SX',sigmax,source=sourceType,/Get
     PrintF,unit,STRCOMPRESS(sigmax*10.0,/REMOVE_ALL)
     shadow_SetIfcPar,Parent=iParent,'SZ',sigmaz,source=sourceType,/Get
     PrintF,unit,STRCOMPRESS(sigmaz*10.0,/REMOVE_ALL)
     shadow_SetIfcPar,Parent=iParent,'EX',epsi_x,source=sourceType,/Get
     shadow_SetIfcPar,Parent=iParent,'EZ',epsi_z,source=sourceType,/Get
     PrintF,unit,STRCOMPRESS(epsi_x/sigmax*1000.0,/REMOVE_ALL)
     PrintF,unit,STRCOMPRESS(epsi_z/sigmaz*1000.0,/REMOVE_ALL)
   
     PrintF,unit,STRCOMPRESS(10.0,/REMOVE_ALL)
     PrintF,unit,STRCOMPRESS(0.0,/REMOVE_ALL)
     PrintF,unit,STRCOMPRESS(0.0,/REMOVE_ALL)
     PrintF,unit,STRCOMPRESS(1.0,/REMOVE_ALL)
     PrintF,unit,STRCOMPRESS(1.0,/REMOVE_ALL)
     PrintF,unit,STRCOMPRESS(25,/REMOVE_ALL)
     PrintF,unit,STRCOMPRESS(25,/REMOVE_ALL)
   
     PrintF,unit,STRCOMPRESS(5,/REMOVE_ALL)
     PrintF,unit,STRCOMPRESS(2,/REMOVE_ALL)
     PrintF,unit,STRCOMPRESS(-1,/REMOVE_ALL)
   
     PrintF,unit,STRCOMPRESS(0,/REMOVE_ALL)
     PrintF,unit,STRCOMPRESS(0,/REMOVE_ALL)
     PrintF,unit,STRCOMPRESS(0,/REMOVE_ALL)
     PrintF,unit,STRCOMPRESS(0.0,/REMOVE_ALL)
     PrintF,unit,STRCOMPRESS(0,/REMOVE_ALL)
     PrintF,unit,STRCOMPRESS(0.0,/REMOVE_ALL)
     FREE_LUN,unit

     ; 
     ; run urgent
     ; 
     xop_spawn,'urgent < urgent.inp'

     END
  ELSE: 
ENDCASE

; 
; reads file for visualization
;
CASE sourceType OF
  0: spectrum = rascii('srcgeo.out')
  1: spectrum = rascii('SRCOMP')
  2: spectrum=rascii('xsh_wiggler_spectrum_tmp.out')
  3: BEGIN  ; undulator
     a = rascii('urgent.out')
     spectrum=a[[0,2],*]
     END
  ELSE:
ENDCASE
;
; visualization
;
spec2=dblarr(5,n_elements(spectrum[0,*]))
spec2[0,*]=spectrum[0,*]
spec2[1,*]=spectrum[1,*]
spec2[2,*]=spectrum[1,*]/spectrum[0,*]*1000.0
spec2[3,*]=spectrum[1,*]*physical_constants('ec')/(0.1D-2)
spec2[4,*]=spectrum[1,*]*spectrum[0,*]*physical_constants('ec')
titles=['Photon Energy [eV]', $
        'Flux[Phot/sec/0.1%bw]', $
        'Flux[Phot/sec/eV]',$
        'Power[Watts/0.1%bw]', $
        'Power[Watts/eV]']
Xplot,spec2,coltitles=titles,xtitle='-1',ytitle='-1',  $
        wtitle='Spectrum for current SHADOW source',$
        xcol=1,ycol=2,/XLOG,/YLOG

END

