;+
; =================================  xaid_calib ===============================
;
;   The documentation for XAID (the XOP's XAFS toolbox) can be found at:
;   http://www.esrf.fr/computing/scientific/xop/xaid/doc/
;
;		
;   xaid_calib is a widget based graphical interface to perform the
;   XAFS calibration. It allows to load two spectra (reference spectrum
;   and spectrum to calibrate) and allows to interactively shift the
;   one to be calibrated over the reference one. The calibration is
;   done with the formula:
;   t[i] = Xuncalibrated[i]-Xuncalibrated[0]
;   Xcalibrated[i] = a + b*( t[i] ) + c*(t[i])^2
;	
;   (The substraction of Xuncalibrated[0] is essential for TurboXAS).
;		
;
; ----------------------------------------------------------------------
;
;	NAME:
;		XAID_CALIB
;	PURPOSE:
;		 Calibrate Dispersive EXAFS data.
; 	CATEGORY:
;		Widgets. XAFS data analysis.
;	CALLING SEQUENCE:
;		XAID_CALIB 
;	KEYWORD PARAMETERS:
;		GROUP = The widget ID of the widget that calls Xplot.
;		When this ID is specified, a death of the caller results 
;		in a death of XAID_Mu2Chi.
;		DELIAID = a structure with the widget id's from DELIA
;		  application. To be used when XAID_CALIB is called from DELIA.
;		XPLOTID = the id of an existing xplot window where XAID_CALIB
;		  will send the plots. If undefined, XAID_CALIB creates a new
;		  xplot window.
;	OUTPUTS:
;		Opens XPLOT windows and allows changing interactively the 
;		calibration coefficients a,b and c. The results are 
;		automatically copied to the DELIA application when DeliaId 
;		keyword is used.
;	SIDE EFFECTS:
;		If not active, starts Xmanager
;	RESTRICTIONS:
;		Unknown.
;	PROCEDURE:
;		Uses Xplot
;	MODIFICATION HISTORY:
;		Written by  Manuel Sanchez del Rio. ESRF. 2 October 1998
;		2006-03-01 srio@esrf.fr Use Max(Deriv) to set the x0
;			(following the ideas of Moniek Tromp). 
;			Some other small improvements.
;		2006-07-19 srio@esrf.fr Some changes to automatic adjust
;			Y clipping, etc. (after Gemma Guilera)
;
;-

PRO xaid_calib_event,event

catch, error_status
if error_status ne 0 then begin
   message,/info,'error caught: '+!err_string
   if sdep(/w) then itmp = Dialog_Message(/Error,$
	'XAID_CALIB_EVENT: error caught: '+!err_string)
   catch, /cancel
   on_error,2
   goto,out
endif

;
Widget_Control, event.id, get_UValue=eventUValue
Widget_Control, event.id, get_Value=Value
 
if n_elements(eventuvalue) EQ 0 then eventuvalue = ''
if not(keyword_set(eventuvalue)) then eventuvalue = ''
 
stateid = Widget_Info(event.handler,/Child)
Widget_Control, stateid, get_UValue=state, /No_Copy
 

iGuessClip = 0
iapply = 0
isavefile = 0
Case eventUValue OF
    'DISMISS': BEGIN
	Widget_Control,event.handler,/destroy
	END
    'RESET': BEGIN
	widget_control,state.wids.cal_a,set_value=0.0
	widget_control,state.wids.cal_b,set_value=1.0
	widget_control,state.wids.cal_c,set_value=0.0
	widget_control,state.wids.cal_x0,set_value=0.0
	if type(state.wids.delia) EQ 8 then begin
	  widget_control,state.wids.delia.mu.cal_a,set_value=0.0
	  widget_control,state.wids.delia.mu.cal_b,set_value=1.0
	  widget_control,state.wids.delia.mu.cal_c,set_value=0.0
	  widget_control,state.wids.delia.mu.cal_x0,set_value=0.0
	endif
	iapply = 1
	END

     'HELP': BEGIN
        xhelp,'xaid_calib',GROUP=event.top
       END
    'FILE': BEGIN
	fileref=''
	file=''
	widget_control,state.wids.fileref,get_value=fileref
	widget_control,state.wids.file,get_value=file
	if ( (checkfile(fileref) EQ 1) and (checkfile(file) EQ 1) ) then begin
	  iapply=1
	endif else begin
	  itmp = Dialog_Message(/Info,Dialog_Parent=event.top, $
	    'Please load reference and/or data file(s)')
	  goto,out
	endelse
	END
    'EDITCOEFF': BEGIN
	iapply=1
	if type(state.wids.delia) EQ 8 then begin
	  a=0 & b=0 & c=0 & x0=0
	  widget_control,state.wids.cal_a,get_value=a
	  widget_control,state.wids.cal_b,get_value=b
	  widget_control,state.wids.cal_c,get_value=c
	  widget_control,state.wids.cal_x0,get_value=x0
	  widget_control,state.wids.delia.mu.cal_a,set_value=a
	  widget_control,state.wids.delia.mu.cal_b,set_value=b
	  widget_control,state.wids.delia.mu.cal_c,set_value=c
	  widget_control,state.wids.delia.mu.cal_x0,set_value=x0
	endif
	END
    'SWITCH': BEGIN
	  itmp = 0
	  widget_control,state.wids.switch1,get_value=itmp
          widget_control,state.wids.step,set_value=state.move_steps[itmp]
	END
    'MOVE': BEGIN
	iapply = 1
	step=0
	tmp=0.0
	itmp=0
	widget_control,state.wids.step,get_value=step
	widget_control,state.wids.switch1,get_value=itmp
	widget_control,event.id,get_value=tmp
	if  tmp EQ ' << ' then isign=-1.0 else isign=1.0
	case itmp of
	  0: wid = state.wids.cal_a
	  1: wid = state.wids.cal_b
	  2: wid = state.wids.cal_c
	  else:
	endcase
	widget_control,wid,get_value=tmp
	widget_control,wid,set_value=( tmp + isign*step )
	if type(state.wids.delia) EQ 8 then begin
	  case itmp of
	    0: wid = state.wids.delia.mu.cal_a
	    1: wid = state.wids.delia.mu.cal_b
	    2: wid = state.wids.delia.mu.cal_c
	    else:
	  endcase
	  widget_control,wid,get_value=tmp
	  widget_control,wid,set_value=( tmp + isign*step )
	endif
	
	END
	'REFRESH': iapply = 1
	'PLOTREF': BEGIN
	  iapply = 0
	  widget_control,state.wids.fileref,get_value=fileref
	  if (checkfile(fileref) EQ 0)  then begin
	    itmp = Dialog_Message(/Info,Dialog_Parent=event.top, $
	    'Please load reference file')
	    goto,out
	  endif
	  dataref=rascii(fileref)
	  xplot,dataref,/No,Wtitle=fileref
	  END
	'PLOTDATA': BEGIN
	  iapply = 0
	  widget_control,state.wids.file,get_value=file
	  if (checkfile(file) EQ 0)  then begin
	    itmp = Dialog_Message(/Info,Dialog_Parent=event.top, $
	    'Please load data file')
	    goto,out
	  endif
	  dataref=rascii(file)
	  xplot,dataref,/No,Wtitle=file
	  END
	'DERIV': BEGIN
	  iapply = 1
	  widget_control,state.wids.fileref,get_value=fileref
	  widget_control,state.wids.file,get_value=file
	  if ( (checkfile(fileref) EQ 0) or (checkfile(file) EQ 0) ) then begin
	    itmp = Dialog_Message(/Info,Dialog_Parent=event.top, $
	    'Please load reference and/or data file(s)')
	    goto,out
	  endif
	  dataref=rascii(fileref)
	  data   =rascii(file)
	  derivset,dataref,datarefprime
	  derivset,data,dataprime
	  tmp = Max(datarefprime[1,*],idataref)
	  tmp = Max(dataprime[1,*],idata)
	  vref=dataref[0,idataref]
	  Widget_Control,state.wids.cal_x0,Set_Value=idata
	  Widget_Control,state.wids.cal_a,Set_Value=vRef
	  if type(state.wids.delia) EQ 8 then begin
	    widget_control,state.wids.delia.mu.cal_a,set_value=vRef
	    ;widget_control,state.wids.delia.mu.cal_b,set_value=1.0
	    ;widget_control,state.wids.delia.mu.cal_c,set_value=0.0
	    widget_control,state.wids.delia.mu.cal_x0,set_value=idata
	  endif
	  END
    'SAVEFILE': BEGIN
	iapply=1
	isavefile=1
	END
    'GUESSCLIP': BEGIN
	iapply=1
	iGuessClip=1
	END
    'RESETCLIP': BEGIN
        widget_control,state.wids.expandY,Set_value=1.0
        widget_control,state.wids.shiftY,Set_value=0.0
	iapply=1
	END
    else:
endcase

if iapply then begin
  a=0 & b=0 & c=0
  widget_control,state.wids.cal_a,get_value=a
  widget_control,state.wids.cal_b,get_value=b
  widget_control,state.wids.cal_c,get_value=c
  widget_control,state.wids.cal_x0,get_value=x0
  fileref=''
  file=''
  widget_control,state.wids.fileref,get_value=fileref
  widget_control,state.wids.file,get_value=file
  if ( (checkfile(fileref) EQ 0) or (checkfile(file) EQ 0) ) then begin
    itmp = Dialog_Message(/Info,Dialog_Parent=event.top, $
    'Please load reference and/or data file(s)')
    goto,out
  endif
  ; 
  ; adjust Y expansion and shift coeffs if requires. 
  ;
  IF iGuessClip EQ 1 THEN BEGIN
    dataRef = rascii(fileref)
    minYR=min(dataRef[1,*],Max=maxYR)
    data = rascii(file)
    minY=min(data[1,*],Max=maxY)
    expandY=(maxYR-minYR)/(maxY-minY)
    shiftY=minYR-minY
    widget_control,state.wids.expandY,Set_value=expandY
    widget_control,state.wids.shiftY,Set_value=shiftY
  ENDIF ELSE BEGIN
    widget_control,state.wids.expandY,get_value=expandY
    widget_control,state.wids.shiftY,get_value=shiftY
  ENDELSE
  ; 
  ; load reference data and open xplot window if not done
  ;
  if Widget_Info(state.wids.xplot,/Valid_Id) NE 1 then begin
    xplot,p=p,No_Block=state.no_block,group=event.top
    state.wids.xplot=p
    ;	    xplot_plotmgr,state.wids.xplot,/Delete_all
    dataRef = rascii(fileref)
    if a EQ 0 then a = dataRef[0,0]
    widget_control,state.wids.cal_a,set_value=a
    if type(state.wids.delia) EQ 8 then $
      widget_control,state.wids.delia.mu.cal_a,set_value=a

    xplot_loadfile,state.wids.xplot,dataRef,/norefresh
    xplot_controls_action,state.wids.xplot,linestyle='3' ; , $
;	clr_lines=4,clr_axis=3
    xplot_savecurrent,state.wids.xplot
  endif
  ;
  ; now file to calibrate
  ;
  data=rascii(file)
  xx = reform(data[0,*]-x0)
  data[0,*] = xx*xx*c+xx*b+a
  data[1,*]= (data[1,*]+shiftY)*expandY

  xplot_loadfile,state.wids.xplot,data,/NoRefresh
  xplot_controls_action,state.wids.xplot ; ,clr_axis=3
  ;buffer = ["legend,['Reference','Calibrated'],linestyle=[0,2],Color=[4,5]",$
  buffer = ["legend,['Reference','Calibrated'],linestyle=[3,0]",$
	"xyouts,/norm,0.5,0.40,'Xnew[i]=a+b*(X[i]-X[0])+c*(X[i]-X[0])^2'",$
	"xyouts,/norm,0.5,0.35,'a = "+strcompress(a)+"'",$
	"xyouts,/norm,0.5,0.30,'b = "+strcompress(b)+"'",$
	"xyouts,/norm,0.5,0.25,'c = "+strcompress(c)+"'",$
	"xyouts,/norm,0.4,0.20,'reference file: "+fileref+"'" ]
  xplot_exebuffer,state.wids.xplot,setbuffer=buffer
  xplot_controls_action,state.wids.xplot,linestyle='0' ; $
;	,clr_lines=5,clr_axis=3
  ; 
  ; update nove_steps
  ;
  step=0 & itmp=0
  widget_control,state.wids.switch1,get_value=itmp
  widget_control,state.wids.step,get_value=step
  state.move_steps[itmp] = step
  IF ISAVEFILE EQ 1 THEN BEGIN
	back:
	fileout=Dialog_Pickfile(/Write,Dialog_Parent=event.top,$
		File='xsh_calib.dat')
	IF fileout EQ '' THEN GoTo,out
	IF CheckFile(fileout) THEN BEGIN
	  itmp = Dialog_Message(/Question,Dialog_Parent=event.top, $
	  ['File exists: '+fileout,$
	    ' ','Overwrite it?'])
	IF itmp EQ 'No' THEN GoTo,back
	ENDIF
	Openw,unit,fileout,/Get_Lun
	  PrintF,unit,'#D '+Systime()
	  PrintF,unit,'#F '+fileout
	  PrintF,unit,'#C File written by XAID_CALIB'
	  PrintF,unit,'#C It contains the data calibration using the files:'
	  PrintF,unit,'#C '+fileref+' (reference)'
	  PrintF,unit,'#C '+file+' (data)'
	  PrintF,unit,'#C Xnew[i] = a + b*[X[i]-X[0])+ c*(X[i]-X[0])^2'
	  PrintF,unit,'#C x[0] = '+StrCompress(x0,/Rem)
	  PrintF,unit,'#C a = '+StrCompress(a,/Rem)
	  PrintF,unit,'#C b = '+StrCompress(b,/Rem)
	  PrintF,unit,'#C c = '+StrCompress(c,/Rem)
	  PrintF,unit,'#UX0 '+StrCompress(x0,/Rem)
	  PrintF,unit,'#UA '+StrCompress(a,/Rem)
	  PrintF,unit,'#UB '+StrCompress(b,/Rem)
	  PrintF,unit,'#UC '+StrCompress(c,/Rem)
	  PrintF,unit,'#S 1 Calibrated data'
	  PrintF,unit,'#N 3'
	  PrintF,unit,'#L pixel_index  Energy  Signal'
	  FOR i=0,N_Elements(data[0,*])-1 DO Printf,unit, $
		i,data[0,i],data[1,i], Format='(I5,2F19.9)'

	Free_Lun,unit
	itmp = Dialog_Message(/Question,Dialog_Parent=event.top, $
	  ['File in SPEC format with calibrated data written to disk: ',$
	    fileout,' ','Open it (for checking)?'])
	IF itmp EQ 'Yes' THEN XDisplayFile1,fileout
  ENDIF
endif



out:
if Widget_Info(stateid,/Valid_ID) then $
  Widget_Control, stateid, set_UValue=state, /No_Copy
end ; xaid_calib_event
;
;====================================================================
;

PRO xaid_calib, XplotId=xplotId, DeliaId=deliaId, group=group, $
  No_Block=no_block

catch, error_status
if error_status ne 0 then begin
   message,/info,'error caught: '+!err_string
   if sdep(/w) then itmp = Dialog_Message(/Error,$
	'XAID_CALIB: error caught: '+!err_string)
   if n_elements(xplotstate) NE 0 then $
	widget_control, id , SET_UVALUE = xplotstate,/no_copy
   catch, /cancel
   on_error,2
   return
endif

IF N_Elements(no_block) EQ 0 THEN no_block=1


;
; color setting
;
Device,Get_Decomposed=usingDecomposed
IF usingDecomposed EQ 1 THEN BEGIN
  itmp = Dialog_Message(/Question, $
     ['XAID_CALIB does not work properly with decomposed colors',$
     'Turn decomposed colors off?'])
  IF itmp EQ 'Yes' THEN Device,Decomposed=0  
ENDIF
Tek_Color

if xregistered('xaid_calib') then begin
  itmp = Dialog_Message(Dialog_Parent=id, 'A "shift plot" window '+$
    'is already open. Open a new one?',/Question,/Default_No)
  if itmp EQ 'No' then return
endif

;
; define widgets
;
wbase=widget_base(/col,title='XAID_CALIB 1.1 (spectrum calibration)')

wtmp = widget_base(wbase) ; to store state

;
; main buttons
;
wtmp0 = widget_base(wbase,/Row,/Frame)
  wtmp = widget_button(wtmp0,Value='Dismiss',UValue='DISMISS')
;  wtmp = widget_button(wtmp0,Value='Reset',UValue='RESET')
  wtmp = widget_button(wtmp0,Value='Help',UValue='HELP')

wtmp0 = widget_base(wbase,/Column)
  wtmp1 = widget_base(wtmp0,/Row)
    wfileref=cw_pickfile(wtmp1,Title='Reference file:   ',Value='<none>',$
        UValue='FILE',/Frame,XSize=40)
    wtmp = Widget_Button(wtmp1,Value='Plot',UValue='PLOTREF')
  wtmp1 = widget_base(wtmp0,/Row)
    wfile=cw_pickfile(wtmp1,   Title='File to calibrate:',Value='<none>',$
        UValue='FILE',/Frame,XSize=40)
    wtmp = Widget_Button(wtmp1,Value='Plot',UValue='PLOTDATA')



a=0.0 & b=1.0 & c=0.0 & x0=0.0
if keyword_set(deliaid) then begin
  widget_control,deliaid.mu.cal_a,get_value=a
  widget_control,deliaid.mu.cal_b,get_value=b
  widget_control,deliaid.mu.cal_c,get_value=c
  widget_control,deliaid.mu.cal_x0,get_value=x0
endif 
  wtmp1 = widget_label(wbase,/Align_Left,Value=$
     'Calibration:  Xnew[i] = a + b*(X[i]-X[0]) + c*(X[i]-X[0])^2 ')
  wtmp0 = widget_base(wbase,/Row)
    wcal_x0 = cw_field(wtmp0,Title='Index for x[0] pixel: ',Value=x0,/Float,$
	XSize=12,/return_events,UValue='EDITCOEFF')
    wtmp = Widget_Button(wtmp0,Value='Use Max(Deriv)',UValue='DERIV')
    wtmp = widget_button(wtmp0,Value='Reset',UValue='RESET')
    wtmp = Widget_Button(wtmp0,Value='Refresh',UValue='REFRESH')
    wtmp = Widget_Button(wtmp0,Value='Save File...',UValue='SAVEFILE')
  wtmp0 = widget_base(wbase,/Row)
    wcal_a = cw_field(wtmp0,Title='a: ',Value=a,/Float,XSize=12,$
	/return_events,UValue='EDITCOEFF')
    wcal_b = cw_field(wtmp0,Title='b: ',Value=b,/Float,XSize=12,$
	/return_events,UValue='EDITCOEFF')
    wcal_c = cw_field(wtmp0,Title='c: ',Value=c,/Float,XSize=12,$
	/return_events,UValue='EDITCOEFF')
 
move_steps=[10,0.001,1e-7]
wtmp0 = widget_base(wbase,/Column,/Frame)

  wtmp = widget_label(wtmp0, /Align_Left, Value= $
    'Move in Y the calibrated spectrum for better overplot: ')
  wtmp1 = Widget_Base(wtmp0,/Row)
    wshiftY = cw_field(wtmp1,Title='Y shift: ',$
	Value='0.0',UVALUE='REFRESH',/Float,XSize=12,/Return_Event)
    wexpandY = cw_field(wtmp1,Title='Y expansion: ',$
	Value='1.0',UVALUE='REFRESH',/Float,XSize=12,/Return_Event)
    wtmp = Widget_button(wtmp1,Value='Guess',UValue='GUESSCLIP')
    wtmp = Widget_button(wtmp1,Value='Reset',UValue='RESETCLIP')

wtmp0 = widget_base(wbase,/Column,/Frame)
  wtmp = widget_label(wtmp0, /Align_Left, Value='Change shift values: ')
  wswitch = cw_bgroup(wtmp0,/Exclusive,['a','b','c'],$
	UVALUE='SWITCH',Col=3,Set_Value=1)

  wtmp1 = widget_base(wtmp0,/Row)
  wstep = cw_field(wtmp1,Title='Moving step: ',Value=move_steps[1],$
	UValue='STEP',/Float,XSize=12,/Return_Event)
  wtmp = widget_button(wtmp1,Value=' << ',UValue='MOVE',XSize=60)
  wtmp = widget_button(wtmp1,Value=' >> ',UValue='MOVE',XSize=60)

if n_elements(deliaid) EQ 0 then deliaid=0L
if n_elements(xplotid) EQ 0 then xplotid=0L
wids = { switch1:wswitch, step:wstep, delia:deliaid, xplot:xplotid, $
  cal_a:wcal_a, cal_b:wcal_b, cal_c:wcal_c, cal_x0:wcal_x0, $
  fileref:wfileref, file:wfile, expandY:wexpandY, shiftY:wshiftY}
state = {wids:wids, move_steps:move_steps, no_block:no_block}

;
;
widget_control,Widget_Info(wbase,/Child),set_uvalue=state,/no_copy

widget_control,wbase,/realize
xmanager,'xaid_calib',wbase,GROUP=group,No_Block=no_block

end ; xaid_calib
