pro INTENS,shadow_in,intens_out,NFILES=nfiles,POLARIZATION=pol,   $
NOLOST=nolost, block=block, verbose=verbose,  $
nSplit=nSplit, splitRes=splitRes,splitMean=splitMean,splitError=splitError
;+
; NAME:
;	INTENS
; PURPOSE:
;	calculates the intensity from a file or set of files
; CATEGORY:
;	SHADOW's utilities.
; CALLING SEQUENCE:
;	intens,shadow_in,intens_out,Keyword_parameters
; INPUTS:
;	shadow_in  may be:
;          a SHADOW file (between quotes): eg. 'begin.dat' or
;          an IDL-SHADOW structure (without quotes)
; OPTIONAL INPUT PARAMETERS:
;	intens_out the name of an IDL variable with the intensity values
; KEYWORD PARAMETERS:
;		NOLOST = consider losses
;			0 all rays (default option)
;			1 exclude losses;
;               POLARIZATION = when set to 1 calculates the polarization 
;                              parameters:
;                              Is (total intensity for s-polarization)
;                              Ip (total intensity for p-polarization)
;                              S0-stokes = |Es|^2 + |Ep|^2
;                              S1-stokes = |Es|^2 - |Ep|^2
;                              S2-stokes = 2 |Es| |Ep| cos(phase_s-phase_p)
;                              S3-stokes = 2 |Es| |Ep| sin(phase_s-phase_p)
;               NSPLIT = set this keyword to an integer to make the
;                        calculations in NSPLIT blocks to increase statistics. 
;                        For example, using 25000 rays and NSPLIT=5, 
;                        the intensity will be calculated for 5 blocks
;                        of 5000 rays each.
;               BLOCK =This keywors is passed to GetShCol (do not use it
;                      together with NSPLIT)
;               VERBOSE = Set this keyword to 1 to print some results
; OUTPUTS (When NSPLIT is set):
;               SPLITRES = an array of NSPLIT dimension with the 
;                        intensity of each group. Only works if NSPLIT
;			 is defined. 
;               SPLITMEAN = The mean intensity times NSPLIT
;               SPLITERROR = The standard deviation of the  intensity 
;			times NSPLIT
; COMMON BLOCKS:
;	None.
; SIDE EFFECTS:
;	None.
; RESTRICTIONS:
;	None.
; PROCEDURE;
;	Similar to SHADOW's original.
; MODIFICATION HISTORY:
;	M. Sanchez del Rio. ESRF. Grenoble, Oct 1992
;	96/04/16 MSR change fix() by long() in line 16 (to work for
;		large number of rays)
;       2006/12/08 srio@esrf.fr fixes a bug when nolost=1
;       2009/01/12 srio@esrf.eu rewrites it using getShCol. Added NSPLIT
;-

on_error,2


;
; autocall intens in the casethat NSPLIT is set
;
IF Keyword_Set(nSplit) THEN BEGIN
  IF nSplit EQ 1 THEN message,'NSPLIT must be GT 1'
  splitRes = DblArr(nSplit)
  FOR j=0,nSplit-1 DO BEGIN
     block=[j,nSplit]
     intens, shadow_in,intens_out,NFILES=nfiles,POLARIZATION=pol,   $
        NOLOST=nolost,block=block,VERBOSE=verbose
     splitRes[j] = intens_out
  ENDFOR
  ; full intensity
  intens, shadow_in,intens_out,NFILES=nfiles,POLARIZATION=pol,   $
        NOLOST=nolost,VERBOSE=verbose
  mm = moment(splitRes)
  splitMean = mm[0]*nSplit
  splitError = sqrt(mm[1])*nSplit
  IF Verbose THEN BEGIN
    print,' '
    print,'    ----  splitted results ---- '
    message,/info,'Total intensity: '+StrCompress(intens_out)
    message,/info,'nBlocks: '+StrCompress(nSplit)
    message,/info,'Intensity per  block: '+vect2string(splitRes)
    message,/info,'Mean * nBlocks: '+StrCompress(splitMean)
    message,/info,'stDev * nBlocks: '+StrCompress(splitError)
    print,'    --------------------------- '
  ENDIF
  RETURN
ENDIF 

;
; main calculation
;

IF N_Elements(verbose) EQ 0 THEN verbose=1 

;
; start loop through the number of files
;
if not(keyword_set(nfiles)) then nfiles = 1
intens_out=fltarr(nfiles)
for i=1,nfiles do begin
istr = strtrim(i,2)

;
; load shadow-idl structure and define arrays and constants
;
if (nfiles eq 1) then shadow=shadow_in else $
shadow = shadow_in+'-'+istr
shadow_out=readsh(shadow,nolost=nolost,verbose=verbose)

int = getShCol(shadow_out,23,block=block)
;
; calculate the polarization values
;
if (keyword_set(pol) and shadow_out.ncol eq 12) then begin
 message,/info,'Polarization calculations are not posible using only 12 columns'
 exit
endif
if keyword_set(pol) then begin
  as2_pol = getShCol(shadow_out,24,block=block)
  ap2_pol = getShCol(shadow_out,25,block=block)
  s0 = getShCol(shadow_out,30,block=block)
  s1 = getShCol(shadow_out,31,block=block)
  s2 = getShCol(shadow_out,32,block=block)
  s3 = getShCol(shadow_out,33,block=block)
endif
;
;
;
intens_out(i-1)=total(int)

IF (verbose EQ 1 ) THEN BEGIN

if (nfiles gt 1) then message,/info,' ------------------File number '+$
  strcompress(i,/Rem)

message,/info,'Total intensity is              = '+strcompress(total(int))

;
if keyword_set(pol) then begin
 print,'INTENS: ---------------------------------------------------------'
 print,'INTENS: Polarization parameters:    '
 print,'INTENS: |Es|^2 (total intensity for s-polarization) = ',total(as2_pol)
 print,'INTENS: |Ep|^2 (total intensity for p-polarization) = ',total(ap2_pol)
 print,'INTENS: S0-stokes = |Es|^2 + |Ep|^2 = ',total(s0)
 print,'INTENS: S1-stokes = |Es|^2 - |Ep|^2 = ',total(s1)
 print,'INTENS: S2-stokes = 2 |Es| |Ep| cos(phase_s-phase_p) = ',total(s2)
 print,'INTENS: S3-stokes = 2 |Es| |Ep| sin(phase_s-phase_p) = ',total(s3)
 print,'INTENS: ---------------------------------------------------------'
endif

ENDIF ; verbose
;
endfor
return
end
