;+
; NAME:
;       OPTICAL_MODULE__DEFINE.
;
; PURPOSE:
;       This procedure defines a class. The objects of this class made 
;		possible to keep data about an optic element created by XOP, 
;		and to build a graphic model to represent this data.
;
; FIELD:
;		dimension_scale:the scale of the module (all its conponents)
;		dimensionElement_scale: the scale of the optical element
;		distance_scale: the scale of the distance in the module
;		log: the type of scale for the distance in the module 
;		0 ->linear, 1->log
;		show_X: does X must be draw: 0->yes, 1->no
;			show_axis
;			show_plane_added
;			show_symbol
;			show_normal
;			show_axes_element
;		trans: does the optical element must be transparent:
;			0->yes, 1->no
;		rays_style: how does the rays must be draw:
;			0->points, 1->lines, 2->vectors
;		rays_type_toDraw: whitch ray must be draw: 
;			0->all, 1->good only, 2->lost only, 3->none
;		vector_sensitive_source:scall to apply on the vector 
;			arrow of the source plane
;		vector_sensitive_element: the optical element
;		vector_sensitive_image: the image plane
;		normal_scale:the scale of the normal
;		axes_element_scale: the scale of the element's axes
;		X_color: the RVB color of X
;			optical_element_color
;			optical_axes_color
;			optical_plane_color
;			normal_color
;			axes_element_color
;			rays_good_color
;			rays_lost_color
;		parent:ref on a system in witch the module is inclued
;		this Field  respect the XOP definition
;			t_source, t_image, t_incidence,	t_reflection, alpha
;			rwidx1,	rwidx2, rlen1, rlen2
;		prof: the depth of the optical element.
;		type_name:a string to define the type name of the object: 
;			'Rectangle',
;			'Mirror Ellip',
;			'Mirror Ellip Hole'
;		file_name:the file associed whith the object
;		out_transform:a geometric matrix transformation to apply 
;			to the next optical_module 
;		sModel: a 'idlGrModel' to draw the 3D representation.
;		selected: Does the element must be draw as a hightlighted 
;			module:0->no, 1->yes
;		rays_element:a 'beam' object. The beam on the optical Element.
;		planesSource:a 'plane' object. Source's plane.
;		planesImage:a 'plane' object. Image's plane.
;		pRays_list: list rays'position to draw.
;
;
; CATEGORY:
;       BLViewer
;
; CALLING SEQUENCE:
;       Optic=obj_new('OPTICAL_MODULE')
;
; INPUTS:
;		None
;
; KEYWORD PARAMETERS:
;		None.
;
; EXAMPLE:;
;		Optic=obj_new('OPTICAL_MODULE')
;
; PROCEDURE:
;		All the field of an OPTIC_SYSTEM object can be GET/SET hanks to
;		the setPropery procedure and getProperty procedure
;
; REFERENCE:
;
; MODIFICATION HISTORY:
;       Written by CP:  ESRF, EXP, May 2001
;-

;*******************Optical system Definition***********************
;;PURPOSE: init the optical system with default value
function optical_module::Init



self.parent=obj_new()
self.planesSource=obj_new('idl_container')
self.planesImage=obj_new('idl_container')
self.planeElement=obj_new('optical_plane')

self.smodel.element=obj_new('idlgrmodel')
self.smodel.source=obj_new('idlgrmodel')
self.smodel.image=obj_new('idlgrmodel')
self.smodel.rays=obj_new('idlgrmodel')
self.smodel.graphic=obj_new('idlgrmodel')

;*name and file
self.type_name='unknow'
self.file_name=string('')
self.mesh_file='undefined'
;*draw Property

self.sPrivateParameter.dimension_scale=1.
self.sPrivateParameter.dimensionElement_scale=1.
self.sPrivateParameter.distance_scale=1
self.sprivateParameter.log=0
self.sprivateParameter.trans=0
self.sprivateParameter.show_axis=0
self.sprivateParameter.show_plane_added=0
self.sprivateParameter.show_symbol=0
self.sprivateParameter.show_normal=0
self.sprivateParameter.show_axes_element=0
self.sprivateParameter.rays_style=0
self.sprivateParameter.rays_type_toDraw=0
self.sprivateParameter.vector_sensitive_source=1.
self.sprivateParameter.vector_sensitive_element=1.
self.sprivateParameter.vector_sensitive_image=1.
self.sprivateParameter.optical_element_color=[110,110,255]
self.sprivateParameter.optical_axes_color=[255,110,110]
self.sprivateParameter.optical_plane_color=[110,255,110]
self.sprivateParameter.normal_color=[255,255,0]
self.sprivateParameter.normal_scale=1.0
self.sprivateParameter.axes_element_scale=1.0
self.sprivateParameter.axes_element_color=[255, 255, 255]
self.sprivateParameter.rays_good_color=[255, 100, 100]
self.sprivateParameter.rays_lost_color=[255, 255, 50]
self.selected=0
self.rwidx1=0.0
self.rwidx2=0.0
self.rlen1=0.0
self.rlen2=0.0
self.prof=0.0
self.color=[100,100,250]
;*source
optical_axis=obj_new('idlgrpolyline',[0,0],[0,0],[0,0],name='optical axis', $
				color=self.sprivateParameter.optical_axes_color,LINESTYLE=4)
(self.smodel.source)->add, optical_axis

;*element
self.planeElement->getProperty, graphic=model
self.sModel.element->add, model
self.planeElement->setProperty, parent=self, t_location=0, default_lineStyle=6, $
								orientation=[0,2,1], name='element'


;la normal
l=0.07
h=0.15
nb_cotte=9
s=[0,0,0.2+h]
da=(2*!dpi/nb_cotte)
color=self.sprivateParameter.normal_color
point=fltarr(nb_cotte+1,3)
a=0
for p=0, nb_cotte-1 do begin
	point[p,*]=[l*cos(a),l*sin(a),0.2]
	a=a+da
endfor
point[nb_cotte,*]=s
connex=intarr(4,nb_cotte)
for p=0, nb_cotte-2 do begin
	connex[*,p]=[ 3, p, p+1, nb_cotte]
end
connex[*,nb_cotte-1]=[3,nb_cotte-1,0,nb_cotte]

normal=obj_new('idlgrmodel',name='normal')
normal->add, obj_new('idlgrpolyline', [0,s[0]],[0,s[1]],[0,s[2]/10*8], $
						 Color=color)
normal->add, obj_new('idlgrpolygon', Point[*,0],Point[*,1],Point[*,2], $
						POLYGONS =connex, Color=color, /SHADING)
self.smodel.element->add, normal
;axes X-Y-Z
ox_title=obj_new('idlgrtext','X',CHAR_DIMENSIONS=[0.3,0.3])
oy_title=obj_new('idlgrtext','Y',CHAR_DIMENSIONS=[0.3,0.3])
oz_title=obj_new('idlgrtext','Z',CHAR_DIMENSIONS=[0.3,0.3])
oaxisX=obj_new('idlgraxis',direction=0,range=[0,0.4],color=self.sprivateParameter.axes_element_color,major=0,minor=0,title=ox_title)
oaxisY=obj_new('idlgraxis',direction=1,range=[0,0.4],color=self.sprivateParameter.axes_element_color,major=0,minor=0,title=oy_title)
oaxisZ=obj_new('idlgraxis',direction=2,range=[0,0.4],color=self.sprivateParameter.axes_element_color,major=0,minor=0,title=oz_title)
oAxes=obj_new('idlgrmodel', name='axesX-Y-Z')
oAxes->add, oaxisX
oAxes->add, oaxisy
oAxes->add, oaxisZ
self.smodel.element->add, oAxes

;symbol
symbol=obj_new('idlgrmodel', name='symbol')
self.smodel.element->add, symbol

;*image
optical_axis=obj_new('idlgrpolyline',[0,0],[0,0],[0,0], name='optical axis', $
					color=self.sprivateParameter.optical_axes_color,LINESTYLE=4)


self.smodel.image->add, optical_axis

;********bord
larg=0.5
bord=blv_cube(larg,larg,larg)
bord->setProperty, name='bord', hide=1
self.smodel.graphic->add, bord
;*graphic transformation out
self.out_transform=identity(4)
;*************
self.smodel.graphic->add, self.smodel.rays
self.smodel.graphic->add, self.smodel.element
self.smodel.graphic->add, self.smodel.source
self.smodel.graphic->add, self.smodel.image

;*rays
self.pRays_list=ptr_new(intArr(1))
self->add_plane, -1 ;plan source
self->add_plane, 1 ;plan image
return, 1
end

;****************cleanUP*****************
;
;	PURPOSE: use to clean memory when object is destroy
;
pro optical_module::cleanUp

if obj_isa(self.parent,'blv_optical_module') then $
		self.parent->remove, self

m=self.sModel.element->getbyname('axesX-Y-Z')
n=m->count()
for r=0, n-1 do begin
	o=m->get(pos=r)
	o->getProperty, title=t
	obj_destroy, t
endfor

obj_destroy, self.sModel.element
obj_destroy, self.sModel.image
obj_destroy, self.sModel.source
obj_destroy, self.sModel.graphic

obj_destroy, self.planesSource
obj_destroy, self.planesImage
obj_destroy, self.planeElement

ptr_free, self.pRays_list

end
;*********************************************
pro optical_module::add_ray_toDraw, r
	rays_list=[*(self.pRays_list),r]
	ptr_free, self.pRays_list
	self.pRays_list=ptr_new(rays_list,/no_copy)
end
;**********************************************
pro optical_module::clear_rays_toDraw
	ptr_free,self.pRays_list
	self.pRays_list=ptr_new(intarr(1))
end
;*******************************************
function optical_module::count_ray_toDraw
	n=size(*self.pRays_list)
	return, n[1]-1
end
;***********setProperty****************
;
;	PURPOSE: modify the field of the object when the keyWord of the field is used
;
pro optical_module::setProperty,Type_name=type_name, $
								File_name=file_name, $
								T_source=t_source, $
								T_image=t_image, $
								T_incidence=t_incidence, $
								T_reflection=t_reflection, $
								Alpha=alpha, $
								Symbol=Symbol, $
								Graphic=graphic, $
								Transform=transform, $
								Out_transform=out_transform, $
								dimensionElement_scale=dimensionElement_scale, $
								dimension_scale=dimension_scale, $
								distance_scale=distance_scale, $
								Selected=selected, $
								show_axis=show_axis, $
								show_plane_added=show_plane_added, $
								show_symbol=show_symbol, $
								show_normal=show_normal, $
								show_axes_element=show_axes_element, $
								Log=log, $
								transparen=transparen, $
								Rwidx1=rwidx1, $
								Rwidx2=rwidx2, $
								Rlen1=rlen1, $
								Rlen2=rlen2, $
								Prof=prof, $
								Color=color, $
								rays_source=rays_source, $
								rays_element=rays_element, $
								rays_image=rays_image, $
								parent=parent, $
								rays_style=rays_style, $
								rays_type_toDraw=rays_type_toDraw, $
								vector_sensitive_source=vector_sensitive_source, $
								vector_sensitive_element=vector_sensitive_element, $
								vector_sensitive_image=vector_sensitive_image, $
								optical_element_color=optical_element_color, $
								optical_axes_color=optical_axes_color, $
								optical_plane_color=optical_plane_color, $
								normal_color=normal_color, $
								normal_scale=normal_scale, $
								axes_element_scale=axes_element_scale, $
								axes_element_color=axes_element_color, $
								rays_good_color=rays_good_color, $
								rays_lost_color=rays_lost_color, $
								mesh_file=mesh_file


if n_elements(type_name) NE 0 then self.type_name=type_name
if n_elements(file_name) NE 0 then self.file_name=file_name
if n_elements(t_source) NE 0 then begin
	self.t_source=t_source
	plane=self.planesSource->get(pos=0)
	plane->setProperty, t_location=-t_source
endif
if n_elements(t_image) NE 0 then begin
		self.t_image=t_image
		plane=self.planesImage->get(pos=0)
		plane->setProperty, t_location=t_image
endif
if n_elements(t_incidence) NE 0 then self.t_incidence=t_incidence
if n_elements(t_reflection) NE 0 then self.t_reflection=t_reflection
if n_elements(alpha) NE 0 then self.alpha=alpha
if n_elements(graphics) NE 0 then self.smodel.graphic=graphic
if n_elements(transform) NE 0 then self.smodel.graphic->setProperty, transform=transform
if n_elements(out_transform) NE 0 then self.out_transform=out_transform
if n_elements(dimension_scale) NE 0 then self.sPrivateParameter.dimension_scale=dimension_scale
if n_elements(dimensionElement_scale) NE 0 then self.sPrivateParameter.dimensionElement_scale=dimensionElement_scale
if n_elements(distance_scale) NE 0 then self.sPrivateParameter.distance_scale=distance_scale
if n_elements(show_axis) NE 0 then self.sPrivateParameter.show_axis=show_axis
	if n_elements(show_plane_added) NE 0 then begin
	self.sPrivateParameter.show_plane_added=show_plane_added
	nb_planes=self.planesSource->count()
	for p=0, nb_planes-1 do begin
		plane=self.planesSource->get(pos=p)
		plane->setProperty, show=self.sPrivateParameter.show_plane_added
	endfor
	nb_planes=self.planesImage->count()
	for p=0, nb_planes-1 do begin
		plane=self.planesImage->get(pos=p)
		plane->setProperty, show=self.sPrivateParameter.show_plane_added
	endfor
endif
if n_elements(show_normal) NE 0 then self.sPrivateParameter.show_normal=show_normal
if n_elements(show_symbol) NE 0 then self.sPrivateParameter.show_symbol=show_symbol
if n_elements(show_axes_element) NE 0 then self.sPrivateParameter.show_axes_element=show_axes_element
if n_elements(log) NE 0 then self.sPrivateParameter.log=log
if n_elements(transparen) NE 0 then self.sPrivateParameter.trans=transparen
if n_elements(rays_style) NE 0 then self.sPrivateParameter.rays_style=rays_style
if n_elements(rays_type_toDraw) NE 0 then self.sPrivateParameter.rays_type_toDraw=rays_type_toDraw
if n_elements(vector_sensitive_source) NE 0 then self.sPrivateParameter.vector_sensitive_source=vector_sensitive_source
if n_elements(vector_sensitive_element) NE 0 then self.sPrivateParameter.vector_sensitive_element=vector_sensitive_element
if n_elements(vector_sensitive_image) NE 0 then self.sPrivateParameter.vector_sensitive_image=vector_sensitive_image

if n_elements(optical_element_color) NE 0 then begin
	if self.type_name ne 'External (VRML)' then begin
		self.sPrivateParameter.optical_element_color=optical_element_color
		obj=self.smodel.element->getbyname('symbol')
		obj->setProperty, hide=self.sPrivateParameter.show_symbol
		color_model, obj, self.sPrivateParameter.optical_element_color
	end
endif
if n_elements(optical_axes_color) NE 0 then self.sPrivateParameter.optical_axes_color=optical_axes_color
if n_elements(optical_plane_color) NE 0 then begin
self.sPrivateParameter.optical_plane_color=optical_plane_color
	nb_planes=self.planesSource->count()
	for p=0, nb_planes-1 do begin
		plane=self.planesSource->get(pos=p)
		plane->setProperty, color=self.sPrivateParameter.optical_plane_color
	endfor
	nb_planes=self.planesImage->count()
	for p=0, nb_planes-1 do begin
		plane=self.planesImage->get(pos=p)
		plane->setProperty, color=self.sPrivateParameter.optical_plane_color
	endfor
endif
if n_elements(normal_color) NE 0 then self.sPrivateParameter.normal_color=normal_color
if n_elements(normal_scale) NE 0 then self.sPrivateParameter.normal_scale=normal_scale
if n_elements(axes_element_scale) NE 0 then self.sPrivateParameter.axes_element_scale=axes_element_scale
if n_elements(axes_element_color) NE 0 then self.sPrivateParameter.axes_element_color=axes_element_color
if n_elements(rays_good_color) NE 0 then self.sPrivateParameter.rays_good_color=rays_good_color
if n_elements(rays_lost_color) NE 0 then self.sPrivateParameter.rays_lost_color=rays_lost_color

if n_elements(rwidx1) ne 0 then self.rwidx1=rwidx1
if n_elements(rwidx2) ne 0 then self.rwidx2=rwidx2
if n_elements(rlen1) ne 0 then self.rlen1=rlen1
if n_elements(rlen2) ne 0 then self.rlen2=rlen2
if n_elements(prof) ne 0 then self.prof=prof
if n_elements(selected) ne 0 then begin
	self.selected=selected
	bord=self.smodel.graphic->getbyname('bord')
	bord->setProperty, hide=abs(self.selected-1)
endif
if n_elements(color) ne 0 then begin
	self.color=color
	obj=self.smodel.element->getbyname('symbol')
	color_model, obj, color
endif
if n_elements(rays_source) ne 0 then begin
	plane=self.planesSource->get(pos=0)
	plane->setProperty, rays=rays_source
endif
if n_elements(rays_element) ne 0 then begin
	self.planeElement->setProperty, rays=rays_element
endif
if n_elements(rays_image) ne 0 then begin
	plane=self.planesImage->get(pos=0)
	plane->setProperty, rays=rays_image
endif
if n_elements(parent) ne 0 then self.parent=parent
if n_elements(mesh_file) ne 0 then self.mesh_file=mesh_file
if n_elements(Symbol) NE 0 then begin
			Oldsymbol=self.smodel.element->getbyname('symbol')
			self.smodel.element->remove, Oldsymbol
			obj_destroy, Oldsymbol
			symbol->setProperty, name='symbol'
			self.smodel.element->add, symbol
			;normal=obj_new('idlgrpolyline',[0,0],[0,0.3],[0,0],color=[255,255,0],name='normal')
			;self.element->add, normal

	endif

self->upDate_parameter
end
;*********************************************
pro optical_module::flip, show_axis=sa, $
			show_plane_added=spa, $
			show_symbol=ss, $
			show_normal=sn, $
			transparen=transparen, $
			show_axes_element=sae, $
			log=log

if n_elements(sa) ne 0 then self.sPrivateParameter.show_axis=abs(self.sPrivateParameter.show_axis-1)
if n_elements(spa) ne 0 then self.sPrivateParameter.show_plane_added=abs(self.sPrivateParameter.show_plane_added-1)
if n_elements(ss) ne 0 then self.sPrivateParameter.show_symbol=abs(self.sPrivateParameter.show_symbol-1)
if n_elements(sn) ne 0 then self.sPrivateParameter.show_normal=abs(self.sPrivateParameter.show_normal-1)
if n_elements(transparen) ne 0 then self.sPrivateParameter.trans=abs(self.sPrivateParameter.trans-1)
if n_elements(log) ne 0 then self.sPrivateParameter.log=abs(self.sPrivateParameter.log-1)
if n_elements(sae) ne 0 then self.sPrivateParameter.show_axes_element=abs(self.sPrivateParameter.show_axes_element-1)

self->upDate_parameter
end


;***********getProperty****************
;
;	PURPOSE: return the field of the object when the keyWord of the field is used
;



pro optical_module::getProperty, Type_name=type_name, $
								File_name=file_name, $
								T_source=t_source, $
								T_image=t_image, $
								T_incidence=t_incidence, $
								T_reflection=t_reflection, $
								Alpha=alpha, $
								element=element, $
								Source=source, $
								Image=image, $
								rays_model=rays, $
								Graphic=graphic, $
								Out_transform=out_transform, $
								dimension_Scale=dimension_scale, $
								distance_scale=distance_scale, $
								Selected=selected, $
								Log=log, $
								Transparen=trans, $
								Rwidx1=rwidx1, $
								Rwidx2=rwidx2, $
								Rlen1=rlen1, $
								Rlen2=rlen2, $
								Prof=prof, $
								Color=color, $
								rays_source=rays_source, $
								rays_element=rays_element, $
								rays_image=rays_image, $
								parent=parent, $
								show_plane_added=show_plane_added, $
								show_add_plane=show_add_plan, $
								show_axes_element=show_axes_element, $
								rays_style=rays_style, $
								rays_type_toDraw=rays_type_toDraw, $
								vector_sensitive_source=vector_sensitive_source, $
								vector_sensitive_element=vector_sensitive_element, $
								vector_sensitive_image=vector_sensitive_image, $
								optical_element_color=optical_element_color, $
								optical_axes_color=optical_axes_color, $
								optical_plane_color=optical_plane_color, $
								normal_color=normal_color, $
								normal_scale=normal_scale, $
								axes_element_scale=axes_element_scale, $
								axes_element_color=axes_element_color, $
								show_normal=show_normal, $
								show_symbol=show_symbol, $
								show_axis=show_axis, $
								planesSource=planesSource, $
								planesImage=planesImage, $
								planeElement=planeElement, $
								rays_good_color=rays_good_color, $
								rays_lost_color=rays_lost_color, $
								mesh_file=mesh_file

type_name=self.type_name
file_name=self.file_name
t_source=self.t_source
t_image=self.t_image
t_incidence=self.t_incidence
t_reflection=self.t_reflection
alpha=self.alpha
element=self.smodel.element
source=self.smodel.source
image=self.smodel.image
rays=self.smodel.rays
graphic=self.smodel.graphic
out_transform=self.out_transform
dimension_scale=self.sPrivateParameter.dimension_scale
distance_scale=self.sPrivateParameter.distance_scale
selected=self.selected
log=self.sPrivateParameter.log
trans=self.sPrivateParameter.trans
rwidx1=self.rwidx1
rwidx2=self.rwidx2
rlen1=self.rlen1
rlen2=self.rlen2
prof=self.prof
color=self.color

plane=self.planesSource->get(pos=0)
plane->getProperty, rays=rays_source

self.planeElement->getProperty, rays=rays_element

plane=self.planesImage->get(pos=0)
plane->getProperty, rays=rays_Image

parent=self.parent
show_normal=self.sPrivateParameter.show_normal
show_symbol=self.sPrivateParameter.show_symbol
show_axis=self.sPrivateParameter.show_axis
show_plane_added=self.sPrivateParameter.show_plane_added
show_axes_element=self.sPrivateParameter.show_axes_element
rays_style=self.sPrivateParameter.rays_style
rays_type_toDraw=self.sPrivateParameter.rays_type_toDraw
vector_sensitive_source=self.sPrivateParameter.vector_sensitive_source
vector_sensitive_element=self.sPrivateParameter.vector_sensitive_element
vector_sensitive_image=self.sPrivateParameter.vector_sensitive_image
optical_element_color=self.sPrivateParameter.optical_element_color
optical_axes_color=self.sPrivateParameter.optical_axes_color
optical_plane_color=self.sPrivateParameter.optical_plane_color
normal_color=self.sPrivateParameter.normal_color
normal_scale=self.sPrivateParameter.normal_scale
axes_element_scale=self.sPrivateParameter.axes_element_scale
axes_element_color=self.sPrivateParameter.axes_element_color
rays_good_color=self.sPrivateParameter.rays_good_color
rays_lost_color=self.sPrivateParameter.rays_lost_color

planesSource=self.planesSource
planesImage=self.planesImage
planeElement=self.planeElement

mesh_file=self.mesh_file

end

;;***********trace****************
;
;	PURPOSE: build the element necessary to the graphic representation
;	function to the current property
;
pro optical_module::trace, evolution=evolModule
if n_elements(evolModule) eq 0 then evolModule=100.
if evolModule lt 0 then evolModule=0.
if evolModule gt 100 then evolModule=100.
self->update_parameter
;module has a parent?
if obj_isa(self.parent,'optical_system') then begin
	self.parent->getProperty, distance_scale=g_distance_scale, $
							dimension_scale=g_dimension_scale, $
							DistEqual=DistEqual, $
							flagEqual=flagEqual
endif else begin
	g_distance_scale=1.
	g_dimension_scale=1.
	flagEqual=0
endelse

if flagEqual then begin
t_source=DistEqual
t_image=DistEqual
endif else begin
t_source=self.t_source
t_image=self.t_image
endelse

;scale log or linear?
if self.sPrivateParameter.log then begin
	if t_source ne 0 then source=g_distance_scale * self.sPrivateParameter.distance_scale * alog10(t_source) $
			else source=0
	if t_image ne 0 then image=g_distance_scale * self.sPrivateParameter.distance_scale * alog10(t_image) $
			else image=0
endif else begin
	source=g_distance_scale * self.sPrivateParameter.distance_scale * t_source
	image=g_distance_scale * self.sPrivateParameter.distance_scale * t_image
endelse

;*********element*******
self.smodel.element->reset
axis=[1,0,0]
self.smodel.element->rotate, axis, 90-self.t_incidence
axis=[0,1,0]
self.smodel.element->rotate, axis, -self.alpha
self.smodel.element->scale, self.sPrivateParameter.dimension_scale * g_dimension_scale * self.sPrivateParameter.dimensionElement_scale, $
							self.sPrivateParameter.dimension_scale * g_dimension_scale * self.sPrivateParameter.dimensionElement_scale, $
							self.sPrivateParameter.dimension_scale * g_dimension_scale * self.sPrivateParameter.dimensionElement_scale
self.smodel.element->translate, 0, source, 0
;*********source*********
;*optical axis
self.sModel.source->reset

optical_axis=self.smodel.source->getbyname('optical axis')
optical_axis->getProperty, hide=show_axis
self.smodel.source->remove, optical_axis
obj_destroy, optical_axis
optical_axis=obj_new('idlgrpolyline',[0,0],[0,source],[0,0],name='optical axis', $
					color=self.sPrivateParameter.optical_axes_color,LINESTYLE=4, hide=show_axis)
self.smodel.source->add, optical_axis

;*********image********
self.smodel.image->reset
;*optical axis
optical_axis=self.smodel.image->getbyname('optical axis')
self.smodel.image->remove, optical_axis
obj_destroy, optical_axis
optical_axis=obj_new('idlgrpolyline',[0,0],[0,-image],[0,0],name='optical axis', $
				color=self.sPrivateParameter.optical_axes_color,LINESTYLE=4,hide=show_axis)
self.smodel.image->add, optical_axis
;location
axis=[1,0,0]
self.smodel.image->rotate, axis,  180 - (self.t_incidence + self.t_reflection)
axis=[0,1,0]
self.smodel.image->rotate, axis, -self.alpha
deg=180 - (self.t_incidence + self.t_reflection)
x=0.
y=source + image*cos(deg*!pi/180)
z=image*sin(deg*!pi/180)
l=z
x=l*sin(-self.alpha*!pi/180)
z=z*cos(-self.alpha*!pi/180)
self.smodel.image->translate, x,y,z
;*bord*********
bord=self.smodel.graphic->getbyname('bord')
bord->reset
bord->scale, g_dimension_scale * self.sPrivateParameter.dimension_scale * self.sPrivateParameter.dimensionElement_scale, $
			g_dimension_scale * self.sPrivateParameter.dimension_scale * self.sPrivateParameter.dimensionElement_scale, $
			g_dimension_scale * self.sPrivateParameter.dimension_scale * self.sPrivateParameter.dimensionElement_scale
bord->translate, 0,source,0
;***planes
nb_plane=self.planesSource->count()
for p=0, nb_plane-1 do begin
plane=self.planesSource->get(pos=p)
plane->trace, source
endfor
nb_plane=self.planesImage->count()
for p=0, nb_plane-1 do begin
plane=self.planesImage->get(pos=p)
plane->trace, image
endfor
self.planeElement->trace,0
;**save plan out*****
self.smodel.graphic->getProperty, transform=Tm
self.smodel.image->getProperty, transform=Ti
self.out_transform=Ti#Tm
;********************
self->draw_rays, evolution=evolModule
end
;**************************************
;********draw rays********************
pro optical_module::draw_rays, evolution=evolModule
;delete old rays
n=self.sModel.rays->count()
for p=0,n-1 do begin
	r=self.sModel.rays->get(pos=0)
	self.sModel.rays->remove, r
	obj_destroy, r
endfor

if self.sPrivateParameter.rays_type_toDraw eq 3 then return

rays_tab=fltarr(3,3)

plane=self.planesSource->get(pos=0)
plane->getProperty, rays=rays_source

self.planeElement->getProperty, rays=rays_element

plane=self.planesImage->get(pos=0)
plane->getProperty, rays=rays_image


nb_r=rays_source->count()
if nb_r eq 0 then return

nb_r_to_draw=(size(*(self.pRays_list)))[1]-1
if nb_r_to_draw eq 0 then return

rays_source_to_draw=fltarr(4,nb_r_to_draw+1)
rays_element_to_draw=fltarr(4,nb_r_to_draw+1)
rays_image_to_draw=fltarr(4,nb_r_to_draw+1)

if self.sPrivateParameter.rays_style eq 1 then begin
	vect_source_to_draw=fltarr(3,nb_r_to_draw+1)
	vect_element_to_draw=fltarr(3,nb_r_to_draw+1)
	vect_image_to_draw=fltarr(3,nb_r_to_draw+1)
endif

nb_r_to_draw_really=0
for p=1,nb_r_to_draw do begin
	;***select necessary rays
	rayS=rays_source->get((*self.pRays_list)[p])
	rayE=rays_element->get((*self.pRays_list)[p])
	rayI=rays_image->get((*self.pRays_list)[p])

	case self.sPrivateParameter.rays_type_toDraw of
		0:begin
			rays_source_to_draw[0:2,nb_r_to_draw_really]=rayS[0:2]
			rays_element_to_draw[0:2,nb_r_to_draw_really]=rayE[0:2]
			rays_image_to_draw[0:2,nb_r_to_draw_really]=rayI[0:2]
			rays_source_to_draw[3,nb_r_to_draw_really]=rayS[6]
			rays_element_to_draw[3,nb_r_to_draw_really]=rayE[6]
			rays_image_to_draw[3,nb_r_to_draw_really]=rayI[6]
			if self.sPrivateParameter.rays_style eq 1 then begin
				vect_source_to_draw[*,nb_r_to_draw_really]=rayS[3:5]
				vect_element_to_draw[*,nb_r_to_draw_really]=rayE[3:5]
				vect_image_to_draw[*,nb_r_to_draw_really]=rayI[3:5]
			endif
			nb_r_to_draw_really=nb_r_to_draw_really+1
			end
		1:begin
			if rayE[6] eq 1. then begin
			rays_source_to_draw[0:2,nb_r_to_draw_really]=rayS[0:2]
			rays_element_to_draw[0:2,nb_r_to_draw_really]=rayE[0:2]
			rays_image_to_draw[0:2,nb_r_to_draw_really]=rayI[0:2]
			rays_source_to_draw[3,nb_r_to_draw_really]=rayS[6]
			rays_element_to_draw[3,nb_r_to_draw_really]=rayE[6]
			rays_image_to_draw[3,nb_r_to_draw_really]=rayI[6]
			if self.sPrivateParameter.rays_style eq 1 then begin
				vect_source_to_draw[*,nb_r_to_draw_really]=rayS[3:5]
				vect_element_to_draw[*,nb_r_to_draw_really]=rayE[3:5]
				vect_image_to_draw[*,nb_r_to_draw_really]=rayI[3:5]
			endif
			nb_r_to_draw_really=nb_r_to_draw_really+1
			endif
			end
		2:begin
			if rayE[6] ne 1. then begin
			rays_source_to_draw[0:2,nb_r_to_draw_really]=rayS[0:2]
			rays_element_to_draw[0:2,nb_r_to_draw_really]=rayE[0:2]
			rays_image_to_draw[0:2,nb_r_to_draw_really]=rayI[0:2]
			rays_source_to_draw[3,nb_r_to_draw_really]=rayS[6]
			rays_element_to_draw[3,nb_r_to_draw_really]=rayE[6]
			rays_image_to_draw[3,nb_r_to_draw_really]=rayI[6]
			if self.sPrivateParameter.rays_style eq 1 then begin
				vect_source_to_draw[*,nb_r_to_draw_really]=rayS[3:5]
				vect_element_to_draw[*,nb_r_to_draw_really]=rayE[3:5]
				vect_image_to_draw[*,nb_r_to_draw_really]=rayI[3:5]
			endif
			nb_r_to_draw_really=nb_r_to_draw_really+1
			endif
			end
	endcase
endfor
if nb_r_to_draw_really eq 0 then return

if self.sPrivateParameter.rays_style eq 1 then begin
	vect_source_to_draw=vect_source_to_draw[*,0:nb_r_to_draw_really]
	vect_element_to_draw=vect_element_to_draw[*,0:nb_r_to_draw_really]
	vect_image_to_draw=vect_image_to_draw[*,0:nb_r_to_draw_really]

	;sensitive
	vect_source_to_draw[0,*]=vect_source_to_draw[0,*]*self.sPrivateParameter.vector_sensitive_source
	vect_source_to_draw[2,*]=vect_source_to_draw[2,*]*self.sPrivateParameter.vector_sensitive_source
	vect_element_to_draw[0,*]=vect_element_to_draw[0,*]*self.sPrivateParameter.vector_sensitive_element
	vect_element_to_draw[1,*]=vect_element_to_draw[1,*]*self.sPrivateParameter.vector_sensitive_element
	vect_image_to_draw[0,*]=vect_image_to_draw[0,*]*self.sPrivateParameter.vector_sensitive_image
	vect_image_to_draw[2,*]=vect_image_to_draw[2,*]*self.sPrivateParameter.vector_sensitive_image

	vect_source_to_draw=vect_source_to_draw + rays_source_to_draw[0:2,0:nb_r_to_draw_really]
	vect_element_to_draw=vect_element_to_draw + rays_element_to_draw[0:2,0:nb_r_to_draw_really]
	vect_image_to_draw=vect_image_to_draw + rays_image_to_draw[0:2,0:nb_r_to_draw_really]

	plane=self.planesSource->get(pos=0)
	Tp=plane->get_transform()
	vect_source_to_draw=vert_t3d(vect_source_to_draw, matrix=Tp)

	self.smodel.element->getProperty, transform=Te
	vect_element_to_draw=vert_t3d(vect_element_to_draw, matrix=Te)

	plane=self.planesImage->get(pos=0)
	Tp=plane->get_Transform()
	self.smodel.image->getProperty, transform=Ti
	vect_image_to_draw=vert_t3d(vect_image_to_draw, matrix=Tp#Ti)
endif

rays_source_to_draw=rays_source_to_draw[*,0:nb_r_to_draw_really]
rays_element_to_draw=rays_element_to_draw[*,0:nb_r_to_draw_really]
rays_image_to_draw=rays_image_to_draw[*,0:nb_r_to_draw_really]
rays_source_to_draw[0:2,nb_r_to_draw_really]=[1.,1.,1.]
rays_element_to_draw[0:2,nb_r_to_draw_really]=[1.,1.,1.]
rays_image_to_draw[0:2,nb_r_to_draw_really]=[1.,1.,1.]

plane=self.planesSource->get(pos=0)
Tp=plane->get_transform()
rays_source_to_draw[0:2,*]=vert_t3d(rays_source_to_draw[0:2,*], matrix=Tp)

self.smodel.element->getProperty, transform=Te
rays_element_to_draw[0:2,*]=vert_t3d(rays_element_to_draw[0:2,*], matrix=Te)

plane=self.planesImage->get(pos=0)
Tp=plane->get_Transform()
self.smodel.image->getProperty, transform=Ti
rays_image_to_draw[0:2,*]=vert_t3d(rays_image_to_draw[0:2,*], matrix=Tp#Ti)


case self.sPrivateParameter.rays_style of
	0:	for r=0, nb_r_to_draw_really-1 do begin
			flag_out=0
			for t=0, 2 do begin
				rays_tab[t,0]=rays_source_to_draw[t,r]
				rays_tab[t,1]=rays_element_to_draw[t,r]
				rays_tab[t,2]=rays_image_to_draw[t,r]
				endfor
			if rays_element_to_draw[3,r] ne 1. then color=self.sPrivateParameter.rays_lost_color else color=self.sPrivateParameter.rays_good_color
			line= obj_new('idlgrpolygon', $
						rays_tab[0,*], rays_tab[1,*], rays_tab[2,*], $
						color=color,style=0)
			self.smodel.rays->add, line
		endfor

	1:	for r=0, nb_r_to_draw_really-1 do begin
			if rays_element_to_draw[3,r] ne 1. then color=self.sPrivateParameter.rays_lost_color else color=self.sPrivateParameter.rays_good_color

			vectS=obj_new('myidlgrarrowlight')
			vectS->setProperty, [rays_source_to_draw[0,r],vect_source_to_draw[0,r]], $
							[rays_source_to_draw[1,r],vect_source_to_draw[1,r]], $
							[rays_source_to_draw[2,r],vect_source_to_draw[2,r]], $
							color=color
			modelS=vectS->get_model()

			vectE=obj_new('myidlgrarrowlight')
			vectE->setProperty, [rays_element_to_draw[0,r],vect_element_to_draw[0,r]], $
							[rays_element_to_draw[1,r],vect_element_to_draw[1,r]], $
							[rays_element_to_draw[2,r],vect_element_to_draw[2,r]], $
							color=color
			modelE=vectE->get_model()
			vectI=obj_new('myidlgrarrowlight')

			vectI->setProperty, [rays_image_to_draw[0,r],vect_image_to_draw[0,r]], $
							[rays_image_to_draw[1,r],vect_image_to_draw[1,r]], $
							[rays_image_to_draw[2,r],vect_image_to_draw[2,r]], $
							color=color
			modelI=vectI->get_model()

			self.smodel.rays->add, modelS
			self.smodel.rays->add, modelE
			self.smodel.rays->add, modelI
			obj_destroy, vectS
			obj_destroy, vectE
			obj_destroy, vectI
		endfor

	2:	for r=0, nb_r_to_draw_really-1 do begin
			for t=0, 2 do begin
				rays_tab[t,0]=rays_source_to_draw[t,r]
				rays_tab[t,1]=rays_element_to_draw[t,r]
				rays_tab[t,2]=rays_image_to_draw[t,r]
				if evolModule lt 100. then begin
					if evolModule gt 50. then begin
						evolRays=(evolModule-50)*2
						rays_tab[t,2]=(rays_tab[t,2]-rays_tab[t,1])*evolRays/100. + rays_tab[t,1]
					endif else begin
						evolRays=evolModule*2
						rays_tab[t,1]=(rays_tab[t,1]-rays_tab[t,0])*evolRays/100. + rays_tab[t,0]
						rays_tab[t,2]=rays_tab[t,1]
					endelse
				endif
			endfor
			if rays_element_to_draw[3,r] ne 1. then color=self.sPrivateParameter.rays_lost_color else color=self.sPrivateParameter.rays_good_color
			line= obj_new('idlgrpolyline', $
						rays_tab[0,*], rays_tab[1,*], rays_tab[2,*], $
						 color=color)
			self.smodel.rays->add, line
		endfor
	else:;nothing

	endcase
end
;********************************************
;******************************************
pro optical_module::update_parameter

	obj=self.smodel.source->getbyname('optical axis')
	obj->setProperty, hide=self.sPrivateParameter.show_axis

	obj=self.smodel.image->getbyname('optical axis')
	obj->setProperty, hide=self.sPrivateParameter.show_axis


	obj=self.smodel.element->getbyname('normal')
	obj->reset
	obj->setProperty, hide=self.sPrivateParameter.show_normal
	color_model, obj, self.sPrivateParameter.normal_color
	obj->scale,self.sPrivateParameter.normal_scale, $
				self.sPrivateParameter.normal_scale, $
				self.sPrivateParameter.normal_scale

	obj=self.smodel.element->getbyname('axesX-Y-Z')
	obj->reset
	obj->setProperty, hide=self.sPrivateParameter.show_axes_element
	color_model, obj,self.sPrivateParameter.axes_element_color
	obj->scale,self.sPrivateParameter.axes_element_scale, $
				self.sPrivateParameter.axes_element_scale, $
				self.sPrivateParameter.axes_element_scale


	obj=self.smodel.element->getbyname('symbol')
	obj->setProperty, hide=self.sPrivateParameter.show_symbol

	if self.sPrivateParameter.trans then transparency, self.smodel.element else opacity, self.smodel.element


end
;****************************************
pro optical_module::delete_beam
obj_destroy, self.rays_element
self.rays_element=obj_new('beam')
nb_plane=self.planesSource->count()
for p=0, nb_plane-1 do begin
	plane=self.planesSource->get(pos=p)
	plane->delete_beam
endfor
nb_plane=self.planesImage->count()
for p=0, nb_plane-1 do begin
	plane=self.planesImage->get(pos=p)
	plane->delete_beam
endfor

end
;***************************************
pro optical_module::add_plane, location, file=file
plane=obj_new('optical_plane')
if n_elements(file) ne 0 then plane->setProperty, beam_file=file
plane->getProperty, graphic=model
plane->setProperty, parent=self, t_location=location
if location gt 0 then begin
	self.smodel.image->add, model
	self.planesImage->add, plane
endif else begin
	self.smodel.source->add, model
	self.planesSource->add, plane
endelse
end
;****************Main********************
pro optical_module__define

parent=obj_new()
rays_element=obj_new()

sModel={model_module, $
		element:obj_new(), $
		source:obj_new(), $
		image:obj_new(), $
		graphic:obj_new(), $
		rays:obj_new() $
		}

pRays_list=ptr_new(intArr(1)) ; the list to rays to draw
planesSource=obj_new()
planesImage=obj_new()


flt=0.
int=0

sPrivateParameter={private_parameters, $
			dimension_scale:flt, $
			dimensionElement_scale:flt, $
			distance_scale:flt, $

			log:int, $
			show_axis:int, $
			show_plane_added:int, $
			show_symbol:int, $
			show_normal:int, $
			show_axes_element:int, $
			trans:int, $
			rays_style:int, $
			rays_type_toDraw:int, $
			vector_sensitive_source:flt, $
			vector_sensitive_element:flt, $
			vector_sensitive_image:flt, $
			rays_good_color:[int, int, int], $
			rays_lost_color:[int, int, int], $

			optical_element_color:[int,int,int], $
			optical_axes_color:[int,int,int], $
			optical_plane_color:[int,int,int], $
			normal_color:[int,int,int], $
			normal_scale:flt, $
			axes_element_scale:flt, $
			axes_element_color:[int, int, int] $
			}

struct={	optical_module, $ ;name of the structure
		parent:parent ,$ ; ref on a system in witch the module is inclued
		t_source:0.0, $ ;position/location
		t_image:0.0, $
		t_incidence:0.0, $
		t_reflection:0.0, $
		alpha:0.0, $
		rwidx1:0.0, rwidx2:0.0, rlen1:0.0, rlen2:0.0, prof:0.0, $ ;dimension
		type_name:string(''), $ ;the type name of the object
		file_name:string(''), $ ;the file associed whith the object
		mesh_file:string(''), $ ;the wrl file (use only with External (VRML) type)
		out_transform:fltarr(4,4), $ ;a geometric matrix transformation to apply to the
									;next optical_module
		sModel:sModel, $
		sPrivateParameter:sPrivateParameter, $
		selected:1, $
		color:[0,0,0], $
		rays_element:rays_element, $
		pRays_list:pRays_list, $
		planeElement:obj_new(), $
		planesSource:planesSource, $
		planesImage:planesImage $
		}
end
