#! /bin/sh
#
# excon-seed: EXCON wrapper for SHADOW PWR_DENS program.
#
# Author: Mumit Khan <khan@xraylith.wisc.edu>
#
# Source: src/utils/excon/excon-pwr
#
# ----------------------------------------------
#	       SHADOW
#    Center for X-ray Lithography
#  University of Wisconsin-Madison
#  3731 Schneider Dr., Stoughton, WI, 53589
# ------------------------------------------------
#
# Log: excon-pwr
# Revision 1.2  1994/12/14  21:35:59  cwelnak
# excon-pwr_dens to pwr_dens
#
# Revision 1.1  1994/01/08  21:20:45  khan
# Initial revision
#
#
#

# environment variables that this shell script sets/changes:
# export VARIABLES.

# error message function
error () {
    echo "`basename $0`: $@" 1>&2
    exit 1
}

# usage message function
usage () {
    if [ ! -z "$@" ]; then
	echo "`basename $0`: $@" 1>&2
    fi
    echo "Usage: `basename $0` -g G_FILE_NAME [-help] [-debug]" 1>&2
    exit 1
}


# don't bother to continue if SHADOW environmet isn't set up properly.
if [ -z "${SHADOW_ROOT}" ]; then
	error \
"SHADOW environment is not properly setup for this script
to run.  Set the environment variables via the \`\`.shadowrc'' shell script 
provided with SHADOW distribution."
fi

if [ -z "${SHADOW_BIN}" ]; then
    SHADOW_BIN="${SHADOW_ROOT}/bin"
fi

# Now all the programs that we need to run.
#PWR_DENS=${SHADOW_BIN}/excon-pwr_dens
PWR_DENS=${SHADOW_BIN}/pwr_dens

#
# set up the variables, including the ones from EXCON.
#
GFILE=				# name of gfile to read from.
RETVAL=0			# return codes from programs.
NRUNS=				# how many runs of pwr_dens done?
STATEFILE=pwr_dens.state.g	# this is where the pwr_dens is stored 
ITERATION=
DUMP_TO_FILE=0			# dump debugging info to file?
DUMP_FILE_NAME=

# Parse command line args.
while [ $# -gt 0 ]; do
    case "$1" in
	-g)
	    if [ $# -lt 2 ]; then
		usage "$1 option requires a gfile name"
	    fi
	    shift
	    GFILE="$1"
	    ;;
	-debug|-d)
	    set -x
	    ;;
	-dump)
	    if [ $# -lt 2 ]; then
		usage "$1 option requires a debugging dump file name"
	    fi
	    shift
	    DUMP_TO_FILE=1
	    DUMP_FILE_NAME="$1"
	    ;;
	-help|-h)
	    usage ""
	    ;;
	*)
	    usage "Illegal command line option $1"
	    ;;
    esac
    shift
done

#
# check sanity and init state info
#

if [ -z "${GFILE}" ]; then
    usage "No GFILE name supplied."
fi

if [ ! -r "${GFILE}" ]; then
    error "ERROR: GFILE \`\`${GFILE}'' is not readable or does not exist."
fi

# 
# FUN begins.
#

# FUNCTION to get a parameter value from a gfile.
getgparam() {
    echo `cat "$GFILE" | grep -i "$1" | awk -F= '{print $2}'`
}

STATEFILE="`getgparam statefile`"
ITERATION="`getgparam '$ITERATION'`"

if [ ! "$STATEFILE" ]; then		# null string
    error "ERROR: GFILE \`${GFILE}' does not have \`statefile' value."
elif [ ! -r "$STATEFILE" ]; then		# null string
    error "ERROR: Cannot read statefile \`${STATEFILE}'."
fi

if [ ! "$ITERATION" ]; then		# null string
    error "ERROR: GFILE \`${GFILE}' does not have \`\$ITERATION' value."
fi


# 
# get the rest of the parameters from the statefile, not the excon gfile
#

unset getgparam
# FUNCTION to get a parameter value from a gfile.
getgparam() {
    echo `cat "$STATEFILE" | grep -i "$1" | awk -F= '{print $2}'`
}

NRUNS="`getgparam 'iter[^a]'`"		# don't pick up iteration!

#
# the whole thing is ruined if the GFILE supplied is a bad one ...
# so check for all the essential variables.
#

if [ ! "$NRUNS" ]; then			# null string
    error "ERROR: GFILE \`${GFILE}' does not have \`iter' value."
fi

if [ $NRUNS -eq 0 ]; then
    error "ERROR: Must run PWR_DENS once before EXCON to initialize."
fi


#
# do the right thing ... 
#

INFIL1="`getgparam infil1`"
INFIL2="`getgparam infil2`" 
INFIL3="`getgparam infil3`" 

unset getgparam				# not needed any more after this

Io_NEEDED=0

OENUMBER1=0
PREFIX_1=
INFIL2_IS_SCREEN=0

OENUMBER2=0
PREFIX_2=
INFIL3_IS_SCREEN=0

#
# FUNCTION to get a OE number from a file name.
# eg., getoenumber star.02 will return 2
#
getoenumber() {
    echo `echo "$1" | awk -F. '{printf "%d", $2}'`
}
#
# FUNCTION to get a OE+SCREEN number from a file name.
# eg., get_oe_of_scr_number star.0301 will return 03
#
get_oe_of_scr_number() {
    echo `echo "$1" | nawk '{
	oenum=substr($0, 8, 2); printf "%d", oenum
    }'`
}

#
# get_scr_of_scr_number star.0301 will return 01
#
get_scr_of_scr_number() {
    echo `echo "$1" | nawk '{
	scrnum=substr($0, 10, 2); printf "%d", scrnum
    }'`
}

# link the appropriate files

case "$INFIL2" in
    star.*)
	PREFIX_1=star
	;;
    mirr.*)
	PREFIX_1=mirr
	;;
    screen.*)
	PREFIX_1=$INFIL2
	INFIL2_IS_SCREEN=1
	;;
    *)
	error "Illegal file type for INFIL2: $INFIL2"
	;;
esac

if [ $INFIL2_IS_SCREEN -eq 1 ]; then
    OENUMBER1="`get_oe_of_scr_number $INFIL2`"
    SUFFIX_1=screen.trace${OENUMBER1}+${ITERATION}
else
    OENUMBER1="`getoenumber $INFIL2`"
    SUFFIX_1=trace${OENUMBER1}+${ITERATION}
fi

if [ "$INFIL3" ]; then
    Io_NEEDED=1
    case "$INFIL3" in
	star.*)
	    PREFIX_2=star
	    ;;
	mirr.*)
	    PREFIX_2=mirr
	    ;;
	screen.*)
	    PREFIX_2=screen
	    INFIL3_IS_SCREEN=1
	    ;;
	NONE*)
	    Io_NEEDED=0
	    ;;
	*)
	    error "Illegal file type for INFIL3: $INFIL3"
	    ;;
    esac
    if [ $INFIL3_IS_SCREEN -eq 1 ]; then
	OENUMBER2="`get_oe_of_scr_number $INFIL3`"
	SUFFIX_2=screen.trace${OENUMBER2}+${ITERATION}
    else
	OENUMBER2="`getoenumber $INFIL3`"
	SUFFIX_2=trace${OENUMBER2}+${ITERATION}
    fi
fi

TMPINFIL1=begin.source+${ITERATION}
TMPINFIL2=${PREFIX_1}.${SUFFIX_1}

if [ $Io_NEEDED -eq 1 ]; then
    TMPINFIL3=${PREFIX_2}.${SUFFIX_2}
fi

if [ ! -f $TMPINFIL1 ]; then
    error "ERROR: INFIL1 $TMPINFIL1 doesn't exist."
fi

if [ ! -f $TMPINFIL2 ]; then
    error "ERROR: INFIL2 $TMPINFIL2 doesn't exist."
fi

if [ $Io_NEEDED -eq 1 ]; then
    if [ ! -f $TMPINFIL3 ]; then
	error "ERROR: INFIL3 $TMPINFIL3 doesn't exist."
    fi
fi

rm -f $INFIL1 $INFIL2 $INFIL3 >/dev/null 2>&1
ln -s $TMPINFIL1 $INFIL1
ln -s $TMPINFIL2 $INFIL2

if [ $Io_NEEDED -eq 1 ]; then
    ln -s $TMPINFIL3 $INFIL3
fi

#########################################################################
#
# this section is for debugging. To activate this, use -dump dumpfile
# on the command line.
#
if [ $DUMP_TO_FILE -eq 1 ]; then
    if [ $ITERATION -eq 1 ]; then
	rm -f $DUMP_FILE_NAME
	touch $DUMP_FILE_NAME
    fi
    cat << __EOB__ >> $DUMP_FILE_NAME
ITERATION: $ITERATION
INFIL1: $INFIL1 ==> $TMPINFIL1
INFIL2: $INFIL2 ==> $TMPINFIL2
INFIL3: $INFIL3 ==> $TMPINFIL3

==============================

__EOB__

fi
#
# END DEBUG
#
#########################################################################

${PWR_DENS} -g $STATEFILE

rm -f $INFIL1 $INFIL2 $INFIL3 >/dev/null 2>&1

RETVAL=$?
exit ${RETVAL}
