PRO WRITESH, filesha,values,columns,ev=ev,str=sh_str,noFile=noFile
;+
; NAME:
;	WRITESH
; PURPOSE:
;	Template to create a SHADOW  file
; CATEGORY:
;	SHADOW's utilities.
; CALLING SEQUENCE:
;	writesh
;	(or) writesh,'file'
;	(or) writesh,'file',values,columns
;
; OPTIONAL INPUTS:
;	file: File name to be written.
;	values: an array (m,npoint) with the column values to be placed
;		in the file. If a column value is not defined, the routine
;		sets the default value (point source, directed in
;		(0,1,0), with energy of 1000eV.)
;	columns: an integer (or integer array) with the column numbers
;		where to place the "values"
; OPTIONAL OUTPUTS:
;	str: a named variable where to put the SHADOW structure
;
; KEYWORD PARAMETERS:
;	EV: When set to 1, converts input column 11 (wavevector) from eV
;		to cm**-1 (shadow's unit)
;	NOFILE: Set this keyword to prevent writing a file (to be used
;		with the STR keyword)
;
; OUTPUTS:
;	A disk file.
; COMMON BLOCKS:
;	None.
; SIDE EFFECTS:
;	Create a file.
; RESTRICTIONS:
;	None.
; PROCEDURE:
;	Defines an idl/shadow structure with default values, change
;	the values for the requires columns and write the file using
;	PUTRAYS procedure.
; EXAMPLE: 
;	Create a source with 1000 rays and "cubic" size in [0,1]x[0,1]x[0,1]
;	;values = randomu(seed,3,1000) & columns=[1,2,3]
;	writesh,'cubic.dat',randomu(seed,3,1000),[1,2,3]
;
;	Create a source with 2000 rays with stepped energy in [0,1999] eV:
;	writesh,'tmp.00',transpose(findgen(2000)),11,/ev
; MODIFICATION HISTORY:
;	M. Sanchez del Rio. ESRF. Grenoble, 31 May 1996
;	2006/04/24 srio@esrf.fr added NoFile and Str keywords
;-
on_error,2
;
;
; define defaults
;

if n_params() EQ 0 then filesha='tmp.dat'

if n_params() EQ 2 then message,"Usage: writesh,'file',values,columns"

if n_params() GT 2 then begin
  NCOLINPUT =  long(n_elements(values(*,0)))
  NPOINT =  long(n_elements(values(0,*)))
  if n_elements(columns) ne ncolinput then message, $
     'Incompatible dimensions between values and columns '+strcompress(ncolinput,/rem)+$
     '  '+strcompress(n_elements(columns))
endif else begin
  NCOLINPUT =  0
  NPOINT=5000L
endelse
NCOL=18L
RAY = dblarr(NCOL,NPOINT)

; cols 1-3 equal to zero (Point source)

; col 5 equal to 1 (all rays in Y direction)
ray(4,*) = 1.0D0

; col 7 equals one (electric vector sigma)
ray(6,*) = 1.0D0

; col 10 equals 1 (all rays are good)
ray(9,*) = 1.0D0

;col 11 equals 1000eV
; ray wavebvector k = 2 pi/lambda [cm^-1]
ray(10,*) = 1000.0D0*6.283185307179586476925287D0/1.239852D-4

; col 12 (ray index)
ray(11,*) = lindgen(NPOINT)+1

;
;puts the data values 
;
if n_params() GT 2 then begin
  for i=0,ncolinput-1 do begin
    colindex = columns(i)-1
    ray(colindex,*) = values(i,0:npoint-1)
  endfor
endif

if keyword_set(ev) then ray(10,*) = $
	ray(10,*)  *6.283185307179586476925287D0/1.239852D-4

; create the structure
sh_str = {  name:filesha, ncol: ncol, npoint: npoint, ray:ray }  ; 
;help,/str,sh_str

; create the file
IF Not(Keyword_Set(noFile)) THEN putrays,sh_str,filesha

end
