
pro INFOSH , shadow_in, NOLOST=nolost,POLARIZATION=pol, $
  GROUP=group, NOWINDOW = NOWINDOW, ASK=ASK, _EXTRA=extra
;+
; NAME:
;	INFOSH
; PURPOSE:
;	INFO about the selected SHADOW structure in IDL
; CATEGORY:
;	SHADOW's utilities.
; CALLING SEQUENCE:
;	infosh,  shadow_in,keyword_parameters
; INPUTS:
;	shadow_in  may be:
;          a SHADOW file (between quotes): eg. 'begin.dat' or
;          an IDL-SHADOW structure (without quotes)
; KEYWORD PARAMETERS:
;		NOLOST = consider losses
;			0 all rays (default option)
;			1 exclude losses;
;			2 consider only losses;
;               POLARIZATION = when set to 1 calculates the polarization 
;                              parameters:
;                              Is (total intensity for s-polarization)
;                              Ip (total intensity for p-polarization)
;                              S0-stokes = |Es|^2 + |Ep|^2
;                              S1-stokes = |Es|^2 - |Ep|^2
;                              S2-stokes = 2 |Es| |Ep| cos(phase_s-phase_p)
;                              S3-stokes = 2 |Es| |Ep| sin(phase_s-phase_p)
;		GROUP = When using widget output, this is the parent id
;		NOWINDOW = when set to 1, eliminates the widget output
;			which is the default with x windows
;		ASK = when set to 1 presents a window to check the
;		      parameters before starting infosh.
;		_EXTRA = extra: other kwywords to be passed to XScrMenu (if ASK 
;			is set).
; OUTPUTS:
;	interactive info on the screen.
; PROCEDURE;
;	Similar to SHADOW's original.
; MODIFICATION HISTORY:
;	M. Sanchez del Rio. ESRF. Grenoble, May 1991
;	1-Sep-1992 MSR change input file/structures
;	30-Sep-1992 MSR introduces the polarization parameters
;	02-Oct-1992 MSR adds GETPHASE to correct phases for stokes parameters
;	10-Feb-1995 MSR makes a widget to display the results
;	12-Nov-1997 MSR users "cd" instead of spawning "pwd". Sdep() also used.
;	18-Mar-1998 MSR checks for the existence of the file.
;	20-Mar-1998 MSR adds the ASK keyword.
;	30-Jun-2000 MSR adds the _EXTRA keyword.
;	24-Mar-2003 srio@esrf.fr adds the "intensity wiighted" options.
;		Rewritten using readsh, xdisplayfile1, etc. 
;
;-
on_error,2
cd,c=pwd
if keyword_set(ask) then begin
  if n_params() GT 0 then file = shadow_in else file='begin.dat'
  if n_elements(nolost) EQ 0 then nolost=1 else nolost=nolost
  if n_elements(polarization) EQ 0 then polarization=0 else polarization=1
  tmp = {file:file, $
     nolost:[strcompress(nolost,/Rem),'All rays','Good only','Lost Only'], $
     polarization:[strcompress(polarization,/Rem),'No','Yes']}
  action = 0
  xscrmenu,tmp,action=action,/notype,/interp,flags=['1','1','1'],$
	titles=['SHADOW input:','Consider rays','Calculate polarization'],$
	wtitle='InfoSh parameters',_Extra=extra
  if action EQ 'DONT' then return
  shadow_in = tmp.file
  nolost = fix(tmp.nolost(0))
  pol = fix(tmp.polarization(0))
  if checkfile(shadow_in) NE 1 then begin
    itmp = Dialog_Message(/Error,/Question,Dialog_Parent=group,$
	['File not found: '+shadow_in, 'Retry?'],Dialog_Parent=group)
    if itmp EQ 'No' then return
    shadow_in = Dialog_Pickfile(title='SHADOW binary file to read',$
	Group=group)
    if shadow_in EQ '' then return
  endif
endif


;
; ray distribution
;

shstr = ReadSh(shadow_in)
npoint = shstr.npoint
flg = getshcol(shstr,10)

tmp = StrPos(shstr.name,'mirr')
CASE tmp[0] OF 
   -1:  BEGIN ; start.xx or screen.xxyy
      cols0 = [1,3,4,6,2,5,11,23]   ; for extrema
      col_name0 = [ $
            "X [cm]",$
            "Z [cm]",$
            "X' [rad]",$
            "Z' [rad]",$
            "Depth Y [cm]",$
            "Collimation Y'",$
            'E [eV]',$
            'Intensity']
      cols = [1,3,4,6,2,5,11,23]  ; for momenta
      col_name = [ $
            "X ",$
            "Z ",$
            "X'",$
            "Z'",$
            "Y ",$
            "Y'",$
            'E ',$
            'I ']
      END

  else: BEGIN  ; mirr.xx
      cols0 = [1,2,4,5,3,6,11,23]   ; for extrema
      col_name0 = [ $
            "X [cm]",$
            "Y [cm]",$
            "X' [rad]",$
            "Y' [rad]",$
            "Z' [rad]",$
            "Z [cm]",$
            'E [eV]',$
            'Intensity']
      cols = [1,2,4,5,11,23]  ; for momenta
      col_name = [ $
            "X ",$
            "Y ",$
            "X'",$
            "Y'",$
            'E ',$
            'I ']
      END
ENDCASE

ngood = total(flg>0)
help,nolost
IF Keyword_Set(noLost) THEN shstr = ReadSh(shstr,NoLost=nolost) 

IF shstr.npoint EQ 0 THEN BEGIN
  itmp = Dialog_Message(/Error,'INFOSH: no rays to analyse.')
  RETURN 
ENDIF

text = [ $
'          Information about SHADOW file: ',$
'    '+pwd+sdep(/ds)+shstr.name,$
'                                                    ',$
' Number of rays is =    '+String(npoint,Format='(I10)'),$
' Number of good rays =  '+String(ngood,Format='(I10)')+$
      ' ('+String(100.0*Float(ngood)/npoint,Format='(F5.1)')+'%)' ,$
' Number of lost rays =  '+String(npoint-ngood,Format='(I10)')+ $
      ' ('+String(100.0*Float(npoint-ngood)/npoint,Format='(F5.1)')+'%)' ,$
' Number of columns =    '+String(shstr.ncol,Format='(I10)'),$
'                                                    ']


IF N_Elements(nolost) EQ 0 THEN nolost=0
text =   [text,'                 ##################################']
CASE nolost OF
  0:text=[text,'                 #   Using ALL rays (good+lost)   #']
  1:text=[text,'                 #   Using GOOD rays only         #']
  2:text=[text,'                 #   Using LOST rays only         #']
ENDCASE
text =   [text,'                 ##################################']


int = getshcol(shstr,23)
text=[text,'','Total intensity = '+StrCompress(Total(int))]


; 
; extrema
;
text = [text,'','***************  Extrema: ','',$
  String(['Col#','name','minimum','maximum'], Format='(A4,A16,2(A13))'),'' ]
itmp  = 0
FOR i=0,N_Elements(cols0)-1 DO BEGIN
  icol = cols0[i]
  vmax = max(getshcol(shstr,icol),min=vmin)
  text = [text,String(icol,Format='(I4)')+ $
               String(col_name0[i],Format='(A16)')+ $
               String( [vmin,vmax],Format='(2G13.6)')]
  itmp = itmp + 1
  IF itmp EQ 2 THEN BEGIN
     text = [text,'']
     itmp = 0
  ENDIF
ENDFOR

;
; ray distribution
;
text=[text,'','***************  Ray distribution momenta:','']
text = [text,String(  $
  ['Col#','mean','variance','skewness','kurtosis','StDev', $
           'StDev*2.35'], Format='(A6,6(A11))'),'' ]

itmp = 0
FOR i=0,N_Elements(cols)-1 DO BEGIN
  icol = cols[i]
  tmp = getshcol(shstr,icol)
  m = moment(tmp)
   tmp = $
   String(icol,Format='(I3)')+String(col_name[i],Format='(A3)')+$
   String([m,sqrt(m[1]),2.*sqrt(2*alog(2))*sqrt(m[1])],Format='(6G11.3)')
   text = [text,tmp]
  itmp = itmp + 1
  IF itmp EQ 2 THEN BEGIN
     text = [text,'']
     itmp = 0
  ENDIF
ENDFOR



;
; intensity (weighted rays) distribution
;
text=[text,'','***************  Intensity (i.e., weighted rays) distribution momenta:','']


IF Abs(total(int)-N_Elements(int)) LE 1e-10 THEN BEGIN
  text=[text,'', $
  '    The same as ray distribution (all rays have intensity one)']
ENDIF ELSE BEGIN
  text = [text,String(  $
  ['Col#','mean','variance','skewness','kurtosis','StDev', $
           '2.35*StDev'], Format='(A6,6(A11))'),'' ]

  itmp = 0
  FOR i=0,N_Elements(cols)-1 DO BEGIN
    icol = cols[i]
    tmp = getshcol(shstr,icol)

    mmean = Total(tmp*int)/Total(int)
    tmp = (tmp*int)-mmean
    m = moment(tmp)
    m[0] = mmean
     tmp = $
     String(icol,Format='(I3)')+String(col_name[i],Format='(A3)')+$
     String([m,sqrt(m[1]),2.*sqrt(2*alog(2))*sqrt(m[1])] , $
       Format='(6G11.3)')
    text = [text,tmp]
    itmp = itmp + 1
    IF itmp EQ 2 THEN BEGIN
       text = [text,'']
       itmp = 0
    ENDIF
  ENDFOR
ENDELSE

;
; calculate the polarization values
;
if (keyword_set(pol) and shstr.ncol eq 12) then begin
 Print,'InfoSh: Polarization calculations are not posible using only 12 columns'
 text = [text,'Polarization calculations are not posible using only 12 columns']
 GoTo,display
endif

if keyword_set(pol) then begin
  
  as_pol = sqrt(shstr.ray(6,*)^2+   $
                shstr.ray(7,*)^2+   $
                shstr.ray(8,*)^2)
  as2_pol = shstr.ray(6,*)^2+       $
            shstr.ray(7,*)^2+       $
            shstr.ray(8,*)^2
  ap_pol = sqrt(shstr.ray(15,*)^2+  $
                shstr.ray(16,*)^2+  $
                shstr.ray(17,*)^2)
  ap2_pol = shstr.ray(15,*)^2+      $
            shstr.ray(16,*)^2+      $
            shstr.ray(17,*)^2
  phase = getphase(shstr)
  cos_pol = cos(phase(0,*)-phase(1,*))
  sin_pol = sin(phase(0,*)-phase(1,*))
  
  s0=as2_pol+ap2_pol
  s1=as2_pol-ap2_pol
  s2= 2*as_pol*ap_pol*cos_pol
  s3= 2*as_pol*ap_pol*sin_pol
endif

if keyword_set(pol) then begin
 text = [text,'','',$
 '                                                    ',$
 '***************  Polarization parameters:    ',' ',$
 ' |Es|^2 (total intensity for s-polarization) = '+ $
        strcompress(total(as2_pol) ),$
 ' |Ep|^2 (total intensity for p-polarization) = '+ $
        strcompress(total(ap2_pol) ),$
 ' S0-stokes = |Es|^2 + |Ep|^2 = '+ $
        strcompress(total(s0) ),$
 ' S1-stokes = |Es|^2 - |Ep|^2 = '+strcompress(total(s1) ),$
 ' S2-stokes = 2 |Es| |Ep| cos(phase_s-phase_p) = '+ $
        strcompress(total(s2) ),$
 ' S3-stokes = 2 |Es| |Ep| sin(phase_s-phase_p) = '+ $
        strcompress(total(s3) ),$
 '']
endif

display:

XDisplayFile1,Text=text,Height=55,Title= 'SHADOW info for file: '+shstr.name
end

