#
# prefs-unix.tcl: routines for SHADOW GUI personal preferences for unix
#
# ------------------------------------------------
# Mumit Khan <khan@xraylith.wisc.edu>
# Center for X-ray Lithography
# University of Wisconsin-Madison
# 3731 Schneider Dr., Stoughton, WI, 53589
# ------------------------------------------------
#
# Copyright (c) 1994-1997 Mumit Khan
#
#

#
# These are private routines called by prefs: module (prefs.tcl)
#
# prefs:load_unix
# prefs:save_unix
#

#
# prefs:_load_unix_prefs:
#
# Loads the preferences file, which under Unix is a simple text file with
# the following format:
#
#    <Variable>	<Separator> <Value>
#
# Where <Separator> = {"="":"}
#
# Split each line, and then assign a($variable) = $value, where "a" is
# passed from the caller, usually "gprefs"
#
proc prefs:_load_unix_prefs {file a} {
    upvar 1 $a array
    if {[catch {open $file r} fin] != 0} {
	return
    }
    while {[gets $fin line] >= 0} {
	# is it a comment or empty line?
	if {[regexp {^[ \t]*#} $line] == 0 &&
	    [regexp {^[ \t]*$} $line] == 0} {
	    set line [split $line "=:"]
	    if {[llength $line] != 2} {
		close $fin
		error \
		    "Illegal syntax in pref \"$file\". wrong format file?"
	    }
	    # use case-indepent variable names.
	    set lname [string tolower [string trim [lindex $line 0]]]
	    set lvalue [string trim [lindex $line 1]]
	    #
	    # now simply add a variable in gvars array with the same
	    # name as the one in the gfile and the corresponding value.
	    vputs "set array($lname) $lvalue"
	    set array($lname) $lvalue
	}
    }
    return
}

#
# LOAD_PREFS: load user-specific preferences
#
proc prefs:load_unix {} {
    global gvars gprefs
    set pref_file $gvars(pref_file)
    vputs "Loading preferences from $pref_file..."
    if ![file readable $pref_file] {
	dialog .pref {Load preferences} \
	    [build_msg \
	    "Preference file \"$pref_file\" does not exist or not readable"] \
	    {} 0 Dismiss
	return 1
    }
    if {[catch {prefs:_load_unix_prefs [list $pref_file] gprefs} msg]} {
	dialog .pref {Load preferences} \
	    [build_msg \
	    "Error loading preferences from \"$pref_file\".($msg)\n" \
	    "This may be caused by files created by an older version of \n"\
	    "SHADOW. Please save the preference file again from the GUI"] \
	    {} 0 Dismiss
	return 1
    }
    vputs "Loaded preferences from $pref_file."
    return 0
}

#
# SAVE_PREFS: save current preference for unix
#
proc prefs:save_unix {} {
    global gvars gprefs
    set pref_file $gvars(pref_file)
    vputs "Saving preferences to $pref_file..."
    if {[catch {open $pref_file w} fid]} {
	dialog .pref {Load preferences} \
	    "Error opening preferences file \"$pref_file\"." \
	    {} 0 Dismiss
	return 1
    }
    parray_names gprefs $fid
    close $fid
    vputs "Saved preferences to $pref_file."
    return 0
}

