function READ_MAP, filename, par=par, intervals=intervals, $
	mapsize=mapsize, mapstart=mapstart, mapindex=mapindex
;+
; NAME:
;	READ_MAP
;
; PURPOSE:
;	Read CCP4-maps
;
; CATEGORY:
;	Crystallography
;
; CALLING SEQUENCE:
; 	map= READ_MAP(filename [,par=par, intervals=intervals, $
;	mapsize=mapsize, mapstart=mapstart, mapindex=mapindex ] )
;
; INPUTS:
;	FILENAME: Name of the CCP4 map file.
;
; KEYWORDED PARAMETERS:
;	None
;
; OUTPUTS:
;	Returns the map as a FLTARR(columns,rows,sections) where PAR.COL
;	PAR.ROW, PAR.SEC indicate which axes corresponds to columns etc. 
;	Note that x means a-axis, y means b-axis and z means c-axis here.
;	(A bit confusing but that is the description in the CCP4-manual.)
;	Note also that the PDB standard concerning orthogonal
;	coordinates is: x/a, y/cstarxa, z/cstar. 
;	In the format description "orthogonal map frame" is mentioned,
;	but I don't know what it means. Everything here is only
;	according to my humble understanding...
;
;	To go from coordinates of an atom (in the PDB file) to coordinates
;	in the map (see restrictions):
;		m= PDBSCALE(par)
;		fc = m # coord
;		mc= ROUND((fc[INV(mapindex]) * intervals) - mapstart)
;	   where INV(mapindex)=[WHERE(mapindex eq 0), $
;                               WHERE(mapindex eq 1), $
;                               WHERE(mapindex eq 2)]
;	=> MAP(mc(0),mc(1),mc(2)) is the density value in the atom 
;	position (or rather at the closest grid point).
;
;	"coord" should be a vector with the coordinates of the atom, the first
;	coordinate being the x-coordinate etc.
;	mc is then the coordinates in array index of the corresponding atom.
;
;	A point in the map corresponds to the following fractional 
;	coordinates:
;		fc= ((mc+mapstart)/intervals)[mapindex]
;
;	Note that MAPSIZE-1 is the maximum coordinates in the map and
;	that the axes of MAP is not (necessarily) with x as the first index.
;
;	PAR: Contains some parameters read from the map header. 
;		Structure.
;	The following four variables are also tags in PAR:
;	INTERVALS: A vector containing [column_intervals, row_intervals, $
;		section_intervals] where column_intervals is PAR.X_INTERVALS
;		if PAR.COL is 'x' etc.
;	MAPSIZE: A vector containing [PAR.COLUMNS, PAR.ROWS, PAR.SECTIONS].
;	MAPSTART: A vector containing [PAR.COL_START, PAR.ROW_START, $
;				PAR.SEC_START].
;	MAPINDEX: A vector to be used to transform from X,Y,Z coordinates
;		to map index (see above).
;
; COMMON BLOCKS:
;	None.
;
; SIDE EFFECTS:
;	None
;
; RESTRICTIONS:
;	Does not read all information in the header and especially if 
;	PAR.SKEW is set to 1 (and not 0) on output then there is a
;	transformation between standard (PDB) orthogonal coordinates
;	to map orthogonal coordinates which isn't read.
;
; PROCEDURE:
;	Straightforward
;
; MODIFICATION HISTORY:
;	Thomas Ursby, 1996
;	Modified header, TU, April, 2000
;-

par= { columns:0, rows:0, sections:0, $
	col_start:0, row_start:0, sec_start:0, $
	x_intervals:0, y_intervals:0, z_intervals:0, $
	a:0.0, b:0.0, c:0.0, alpha:0.0, beta:0.0, gamma:0.0, $
	col:'x', row:'y', sec:'z', $
	min:0.0, max:0.0, mean:0.0, $
	skew:0, sigma:0.0, $
	intervals:[0,0,0], mapsize:[0,0,0], $
	mapstart:[0,0,0], mapindex:[0,0,0] }

dummy= FINDFILE(filename, count=ct)
IF (ct eq 0) THEN MESSAGE, 'File not found.'
OPENR, file, filename, /GET_LUN

header= BYTARR(256*4)
READU, file, header
;swap the bytes ?
swap = 0
orig_header = header
IF LONG(header, 4*(24-1)) LT 0 THEN BEGIN
 PRINT,'Bytes will be swapped !'
 swap = 1
 BYTEORDER,header,/LSWAP
ENDIF

;CASE !version.os OF
; 'linux' : BEGIN
;           END
;  ELSE : orig_header = header 
;ENDCASE

IF (STRING(BYTE(orig_header,4*(-1+53),4)) ne 'MAP ') THEN $
	MESSAGE, 'File not of CCP4 map format.'
symop= BYTARR(LONG(header, 4*(24-1)))

READU, file, symop
map= FLTARR(LONG(header, 4*(1-1)), LONG(header, 4*(2-1)),  $
	LONG(header, 4*(3-1)))

PRINT,'Wait ....'
READU, file, map
IF swap THEN BYTEORDER,map,/LSWAP
;CASE !version.os OF
; 'linux' : BEGIN
;	IF swap THEN BYTEORDER,map,/LSWAP
;           END
;  ELSE : 
;ENDCASE

FREE_LUN, file

; Assig values to the output parameter structure:
par.columns= 	LONG(header, 4*(1-1))
par.rows= 	LONG(header, 4*(2-1))
par.sections= 	LONG(header, 4*(3-1))
par.col_start= 	LONG(header, 4*(5-1))
par.row_start= 	LONG(header, 4*(6-1))
par.sec_start= 	LONG(header, 4*(7-1))
par.x_intervals=LONG(header, 4*(8-1))
par.y_intervals=LONG(header, 4*(9-1))
par.z_intervals=LONG(header, 4*(10-1))
par.a=		FLOAT(header, 4*(11-1))
par.b=		FLOAT(header, 4*(12-1))
par.c=		FLOAT(header, 4*(13-1))
par.alpha=	FLOAT(header, 4*(14-1))
par.beta=	FLOAT(header, 4*(15-1))
par.gamma=	FLOAT(header, 4*(16-1))
par.col=	STRCOMPRESS(STRING(119b+BYTE(LONG(header, 4*(17-1)))), $
		/REMOVE_ALL)
par.row=	STRCOMPRESS(STRING(119b+BYTE(LONG(header, 4*(18-1)))), $
		/REMOVE_ALL)
par.sec=	STRCOMPRESS(STRING(119b+BYTE(LONG(header, 4*(19-1)))), $
		/REMOVE_ALL)
par.min=	FLOAT(header, 4*(20-1))
par.max=	FLOAT(header, 4*(21-1))
par.mean=	FLOAT(header, 4*(22-1))
par.skew=	LONG(header, 4*(25-1))
par.sigma=	FLOAT(header, 4*(55-1))

; The first element in INTERVALS is the number of column intervals etc:
intervals= 	[LONG(header, 4*(6+BYTE(LONG(header, 4*(17-1))))),$
		LONG(header, 4*(6+BYTE(LONG(header, 4*(18-1))))),$
		LONG(header, 4*(6+BYTE(LONG(header, 4*(19-1)))))]
mapsize=	[PAR.COLUMNS,PAR.ROWS,PAR.SECTIONS]
mapstart=	[PAR.COL_START,PAR.ROW_START,PAR.SEC_START]
mapindex= [	BYTE(LONG(header, 4*(17-1)))-1, $
		BYTE(LONG(header, 4*(18-1)))-1, $
		BYTE(LONG(header, 4*(19-1)))-1]

par.intervals= 	intervals
par.mapsize= 	mapsize
par.mapstart=	mapstart
par.mapindex= 	mapindex

RETURN, map

END
