function READ_GEASC2, filename, par
;+
; NAME:
;	READ_GEASC2
;
; PURPOSE:
;
; 	Read in an (ASCII) ".geasc" from the Laue suite CCP4. Returns a 
;	structure array containing the information for each reflection.
;
; CATEGORY:
;	File managing
;
; CALLING SEQUENCE:
;	Array = READ_GEASC2(filename,par) ;
;
; INPUTS:
;	filename : the path name of the file to process (without extension)
;	: BOTH the .geasc and the .gen files must be present. This is because
;	some parameters not present or possibly wrong in the .geasc are found
;	in the .gen file !
;
; OUTPUTS:
;	Array : the structure array containing the data
;	[par] : optionnal output containing general info. 
;		It is also a structure. [see procedure]
;
; COMMON BLOCKS:
;	None.
;
; SIDE EFFECTS:
;	None.
;
; RESTRICTIONS:
;	None.
;
; PROCEDURE:
;	Par -->
;   CRYSTAL	FLTARR(6) : alpha,beta,gamma,a*,b*,c* in reciprocal space
;   PHIS        FLTARR(3) : the crystal setting angles [degrees]
;   SPINDLE	FLOAT  	  : the spindle value
;   CDD         FLOAT     : crystal to detector distance [mm]
;   RASTER      FLOAT     : pixel size [um]
;   RASTX       INT       : number of pixel in X dimension
;   RASTY       INT       : number of pixel in Y dimension
;   CENX        FLOAT     : X position of beam center [pixels] from lower left of image
;   CENY        FLOAT     : Y position of beam center [pixels] from lower left of image
;   RADIUS      FLOAT     : maximum radius from beam center [mm] used for prdicted spots
;   LMIN        FLOAT     : minimum wavelength [A]
;   LMAX        FLOAT     : maximum wavelength [A]
;   DMIN        FLOAT     : resolution limit [A] used for predicted spots
;   YSCALE      FLOAT     : Y scale for pixel non squareness
;   TWIST       FLOAT     : correction for twist [0.01 degrees]
;   TILT        FLOAT     : correction for tilt [0.01 degrees]
;   BULGE       FLOAT     : correction for bulge [0.01 degrees]
;   R_REJ       FLOAT     : minimum radius from beam center [mm] used for prdicted spots
;   GAIN        FLOAT     : gain of the detector at 1.5418 A
;   BG_GAIN     FLOAT     : gain of the detector at the mean background wavelength.
;   BEAMSIZE    FLOAT     : beamsize [mm]
;   SAMPLE_DEPTH    FLOAT : sample depth [mm]
;   DARK_CURRENT    FLOAT : dark current of the detector [counts]
;   BOXSIZE         STRUCTURE : size of box for data integration [pixel]
;   NOISE_PERCENTAG INT   : initial percentage of the pixels taken into account for 
;   BG_MAX_SLOPE FLOAT    : maximum slope for background calculations
;			    background calculation.
;   SATVALUE	LONG	  : saturation value
;   OVLP_DIST   FLOAT     : distance [mm] between spot centers, below which spots are considered overlapped
;   OVLP_MIN_DIST   FLOAT     : distance [mm] between spot centers, below which spots are considered impossible
;				to deconvolute
;      Remark --> The last 8 parameters are entered later with the progrm WRITEPAR_LAUE.PRO
;		  They are used for IDL data integration. 
;
;      Data -->
;	M       INTARR(3) :The Miller indices h,k,l
;  	X       FLOAT     : X position from beam center [mm]
;   	Y       FLOAT     : Y position from beam center [mm]
;   	L       FLOAT     : Wavelength [A]
;   	MUL     INT       : Multiplicity index
;   	NODAL   INT       : 0 if not nodal ; 1 if yes (ie |h|< 12 && |k|<12 && |l|<12)
;   	OVLP    INT       : 0 if not overlap ; 1 if yes. This field will usually be 
;				overwritten if the routine FIND_OVLP.PRO is used
;   	OVLP2   INT       : 0 if not overlap ; 1 if yes. This field will usually be 
;				overwritten if the routine FIND_QUICKOVLP.PRO is used
;   	ID      INT       : Identification # of spot (used for IDL integration)
;   	CONNECT  INTARR(6) : Contains the identification # of the overlapping spots
;   	CLOSE   INT        : 0 if not close, 1 if yes
;  	M2      INTARR(3)  : Nodal Miller indices for second harmonic (ie: 2* .m)
;   	NIDX    INT        : Nodal spot selection index
;   	L2      FLOAT      : Wavelength for second harmonic (for doubles only, otherwise 0)
;   	IPOINT  INT        : Pointer to .ge2 file for a multiple (>2) 
;   	DMINTH  FLOAT      : Dmin threshold squared for next harmonic
;   	INTB    FLTARR(6)  : Box integrated intensities for 6 films
;  	ISIGB   FLTARR(6)  : Sigmas for box integrated intensities for 6 films
;  	INTP    FLTARR(6)  : Profile fitted integrated intensities for 6 films
;  	ISIGP   FLTARR(6)  : Sigmas for profile fitted integrated intensities for 6 films
;  	TTH     FLOAT      : Two theta angle [degrees]
;   	RES     FLOAT      : Resolution [A] in direct space
;	FLAG    INT	   : Flag
;	GFIT    FLOAT      : Goodness of fit
;
; MODIFICATION HISTORY:
;	D.Bourgeois, June 1995.
;-

;on_error,2                              ;Return to caller if an error occurs

;define one reflexion
one_refl={	m: INTARR(3), $
		x: 0.0, $
		y: 0.0, $
		l: 0.0, $
		mul: 0, $
	        strength: 0.0, $
	        xpix : 0.0, $
       	 	ypix : 0.0, $
        	chisquare : 0.0, $
		cosine: 0.0, $
		nodal: 0, $
		ovlp: 0,$
		ovlp2: 0,$
		sector:0,$
		id:0, $
		connect: INTARR(24), $
		close:0, $
		m2: INTARR(3), $
		nidx: 0, $
		l2: 0.0,$
		ipoint: 0, $
		dminth: 0.0, $
		intb: FLTARR(6), $
		isigb: FLTARR(6), $
		intp: FLTARR(6), $
		isigp: FLTARR(6),$
		tth: 0.0, $
		res: 0.0, $
		bc_coeff: FLTARR(3), $
		flag: 0, $
		gfit: 0.0, $
	        lp: 0.0, $
		phi:0.0, $
        	partiality: 0.0, $
		par1 : 0., $
		par2 : 0. }

par = {denzo_par, $
		crystal : FLTARR(6), $
	  	phis : FLTARR(3), $
		spindle : 0.0, $
		mosa : 0.0, $
		low_res : 0.0, $
		hi_res : 0.0, $
		cdd : 0.0, $
		raster : 0.0, $
		rastx : 0, $
		rasty : 0, $
		cenx : 0.0, $
		ceny : 0.0, $
		radius : 0.0, $
		osc_start: 0.0, $
		osc_end: 0.0, $
		lmin : 0.0, $
		lmax : 0.0, $
		dmin : 0.0, $
		yscale : 0.0, $
		twist : 0.0, $
		tilt : 0.0, $
		bulge : 0.0, $
		r_rej : 0.0, $
		gain : 0.0, $
		wave : 1.0, $
		bg_gain : 0.0, $
		beamsize : 0.0, $
		sample_depth : 0.0, $
		dark_current : 0.0, $
	  	boxsize: {box, x:0,y:0} , $
		satvalue : 0L, $
		bc_extend : 2, $
		region_level : 0.5, $
		ovlp_dist : 0.0, $
		ovlp_min_dist : 0.0, $
		noise_percentage : 30.,$
		bg_max_slope : 100.}


;initialize par
dummy=0.0
axis = FLTARR(3)
angle = FLTARR(3)
phis = FLTARR(3)
lmin=0.0 & lmax=0.0 
xc_offset=0.0 & yc_offset=0.0 & centerx=0.0 & centery=0.0
cdd=0.0 & radius=0.0 & dmin=0.0
twist=0.0 & tilt=0.0 & bulge=0.0
yscale=0.0
r_rej=0.0
rastx=0 & rasty=0
raster=0.0

 ;open the 'gen' file 
 genfile=FINDFILE(filename+'.gen',COUNT=count)
 IF count EQ 1 THEN BEGIN
  OPENR,lun,filename+'.gen',/GET_LUN
  PRINT,'Reading .gen file ...'
  READF,lun,FORMAT='(10(/),F12.5)',r_rej
  READF,lun,FORMAT='(/)'
  READF,lun,FORMAT='(/)'
  READF,lun,rastx,rasty,dummy,dummy,dummy,raster
  FREE_LUN,lun
 ENDIF ELSE BEGIN
 ;try to open the 'ldm' file 
  ldmfile=FINDFILE(filename+'.ldm',COUNT=count)
  IF count EQ 1 THEN BEGIN
   line=''
   OPENR,lun,filename+'.ldm',/GET_LUN
   PRINT,'Reading .ldm file ...'
   WHILE (NOT EOF(lun)) DO BEGIN
    READF, lun, line
    IF (STRMID(line, 0, 6) EQ 'NXRAST') THEN BEGIN
     READS,line,FORMAT='(6X,I5,7X,I5)',rastx,rasty
    ENDIF
    IF (STRMID(line, 0, 11) EQ 'RASTER_SIZE') THEN BEGIN
     READS,line,FORMAT='(11X,F7.3)',raster
    ENDIF
   ENDWHILE
   FREE_LUN,lun
  ENDIF ELSE MESSAGE,'Could not find .gen file neither .ldm file !'
 ENDELSE

OPENR,lun,filename+'.geasc',/GET_LUN
PRINT,'Reading .geasc file ...'

;Read the header :
centerx = FLTARR(6)
centery = FLTARR(6)
omega = FLTARR(6)
cell = FLTARR(6)
phi = FLTARR(3)
l = FLTARR(3)
dmin = 0.0
radius = 0.0
spindle = 0.0
xc_shift = FLTARR(6)
yc_shift = FLTARR(6)
wc_shift = FLTARR(6)
delta = 0.0
twist = FLTARR(6)
tilt = FLTARR(6)
bulge = FLTARR(6)
cdd = 0.0
yscale = FLTARR(6)
refl_number = 0L

READF,lun,FORMAT='(3(/),TR5,6(F12.4))',centerx
READF,lun,FORMAT='(TR5,6(F12.4))',centery
READF,lun,FORMAT='(TR5,6(F12.4))',omega
READF,lun,FORMAT='(TR5,6(F12.6))',cell
READF,lun,FORMAT='(/,TR5,F12.4)',dummy ;this is the raster, but it is wrong !!
READF,lun,FORMAT='(/,TR5,3(F10.3))',phi
READF,lun,FORMAT='(TR5,3(F10.4))',l
READF,lun,FORMAT='(TR5,F10.4)',dmin
READF,lun,FORMAT='(TR5,F10.4)',radius
READF,lun,FORMAT='(TR5,F10.4)',spindle
READF,lun,FORMAT='(TR5,6(F12.5))',xc_shift
READF,lun,FORMAT='(TR5,6(F12.5))',yc_shift
READF,lun,FORMAT='(TR5,6(F12.5))',wc_shift
READF,lun,FORMAT='(TR5,F12.4)',delta
READF,lun,FORMAT='(TR5,6(F12.5))',twist
READF,lun,FORMAT='(TR5,6(F12.5))',tilt
READF,lun,FORMAT='(TR5,6(F12.5))',bulge
READF,lun,FORMAT='(TR5,F12.5)',cdd
READF,lun,FORMAT='(TR5,6(F12.5))',yscale

par.crystal=cell
par.phis=phi
par.spindle=spindle
par.osc_start=spindle
par.osc_end=spindle
par.lmin=l(1)
par.lmax=l(2)
par.wave=0.5*(l(1)+l(2)) ; choose half the range for mean wavelength
par.dmin=dmin
par.cdd=cdd
par.radius=radius
par.cenx = 1000*(centerx(0) + xc_shift(0))/raster
par.ceny = 1000*(centery(0) + yc_shift(0))/raster*yscale(0)
par.twist=twist(0)
par.tilt=tilt(0)
par.bulge=bulge(0)
par.yscale=yscale(0)
par.r_rej=r_rej ;this parameter not present in the geasc file
par.rastx=rastx ;this parameter not present in the geasc file
par.rasty=rasty ;this parameter not present in the geasc file
par.raster=raster ;this parameter wrong in the geasc file


READF,lun,FORMAT='(16(/),TR17,A5)',refl_number

PRINT,FORMAT='("Number of reflections :",I0)',refl_number

;initialize all parameters for one reflection
star='*'
m=INTARR(3)
x=0.0
y=0.0
l=0.0
mul=0
nodal=0
ovlp=0
ovlp2=0
close=0
m2=INTARR(3)
nidx=0
l2=0.0
ipoint=0
dminth=0.0
intb=FLTARR(6)
isigb=FLTARR(6)
intp=FLTARR(6)
isigp=FLTARR(6)
tth=0.0
res=0.0
bc_coeff=FLTARR(3)

;initialize the counter
index=0L

cr = STRING("15b)

;produce the right number of reflections
refl_arr = REPLICATE(one_refl,refl_number)

FOR index=LONG(0),refl_number-1 DO BEGIN
;WHILE(NOT EOF(1)) DO BEGIN

  IF index/100 EQ index/100.0 THEN PRINT,FORMAT="($,I5,' ie : ',F3.0,'%',A)",index,100.*index/FLOAT(refl_number),cr

 READF,lun,$
   m,x,y,l,mul,nodal,ovlp,close,$
   m2,nidx,l2,ipoint,dminth,intb,isigb,intp,isigp,star

 refl_arr(index).m = m
 refl_arr(index).x = x
 refl_arr(index).y = y
 refl_arr(index).l = l
 refl_arr(index).mul = mul
 refl_arr(index).nodal = nodal
 refl_arr(index).ovlp = ovlp
 refl_arr(index).id = index
 refl_arr(index).connect(*) = -1
 refl_arr(index).close = close
 refl_arr(index).m2 = m2
 refl_arr(index).nidx = nidx
 refl_arr(index).l2 = l2
 refl_arr(index).ipoint = ipoint
 refl_arr(index).dminth = dminth
 refl_arr(index).intb(1) = intb(0)
 refl_arr(index).isigb(1) = isigb(0)
 refl_arr(index).intp(5) = intp(0)
 refl_arr(index).isigp(5) = isigp(0)
 tth = ATAN(SQRT(x^2+y^2)/cdd)
 refl_arr(index).tth = tth*180/3.141593
 refl_arr(index).res = 0.5*l/SIN(0.5*tth)
 refl_arr(index).bc_coeff = bc_coeff
 refl_arr(index).phi = par.spindle

ENDFOR
;ENDWHILE

;close the file
FREE_LUN,lun

PRINT,'Correcting for twist, tilt and bulge ...'
TILTLAUE,refl_arr,par

;for compatibility with denzo format. VERY IMPORTANT
refl_arr.x=refl_arr.x + 0.001*par.raster/2.
refl_arr.y=refl_arr.y + 0.001*par.raster/2.

refl_arr.xpix = (1000/FLOAT(par.raster))*refl_arr.x + par.cenx-1
refl_arr.ypix = (1000/FLOAT(par.raster))*refl_arr.y + par.ceny-1

RETURN,refl_arr
END

