; Copyright (c) 1991. Research Systems, Inc. All rights reserved.
;	Unauthorized reproduction prohibited.
;+
; NAME:
;	IQ_READ
;
; PURPOSE:
;	Read ImageQuant ".gel" files. A psuedo TIFF format. 
;
; CATEGORY:
;	Input/output.
;
; CALLING SEQUENCE:
;   Result = IQ_READ(Filename)
;
; INPUTS:
;    Filename:	A string containing the name of file to read.
;		The default extension is ".gel".
;
;	If the filename ends with ".gel.Z" the program assumes
;	the file has been compressed with the unix "compress"
;	command and will uncompress the file before reading.
;
; OUTPUTS:
;	IQ_READ returns a long array containing the image data.  The 
;	dimensions of the result are the same as defined in the TIFF 
;	file: (Columns, Rows). The values are corrected according to
;	the inormation in the file header. (Int = val^2 * CorrFactor)
;
; COMMON BLOCKS:
;	TIFF_COM.  Only for internal use.
;
; SIDE EFFECTS:
;	A file is read.
;
; RESTRICTIONS:
;	Will not handle normal TIFF files correctly. Assumes that the
;	file is written on a SmallEndian CPU. (i.e. Intel)
;
; EXAMPLE:
;	Read the file "pattern.gel" in the current directory into the variable
;	IMAGE, 	entering:
;
;		IMAGE = IQ_READ('pattern.gel')
;
;	To view the image, scale to color range to the device and
;	display the data
;	entering:
;
;		TVSCL, IMAGE 
;
;	The image is usually to large to be displayed on the screen.
;	If the image plate has been scanned in low resolution mode,
;	rebin the image using the following commands;
;
;		IMSIZE = SIZE(IMAGE)
;		IMAGE = REBIN(IMAGE, IMSIZE(1)/2, IMSIZE(2)/2)
;		WINDOW, IMSIZE(1), IMSIZE(2)
;		TVSCL, IMAGE
;
; MODIFICATION HISTORY:
;	DMS, Written for VMS in 1985.
;	DMS, April, 1991.  Rewrote and added class R and P images.
;	DMS, Jan, 1992.  Fixed bug for images without a RowsPerStrip field.
;	ESRF, Feb, 1993. Rewritten for Molecular Dynamics' image files.
;-

function tiff_long,a,i,len=len	;return longword(s) from array a(i)
common mytiff_com, lun, order, ifd, count

on_error,2              ;Return to caller if an error occurs

   if n_elements(len) le 0 then len = 1
   if len gt 1 then result = long(a,i,len) $
   else result = long(a,i)
   if order then byteorder, result, /lswap
   return, result
end


function tiff_rational,a,i, len = len  	; return rational from array a(i)
common mytiff_com, lun, order, ifd, count

on_error,2              ;Return to caller if an error occurs

if n_elements(len) le 0 then len = 1
tmp = tiff_long(a, i, len = 2 * len)	;1st, cvt to longwords
if len gt 1 then begin
	subs = lindgen(len)
	rslt = float(tmp(subs*2)) / tmp(subs*2+1)
endif else rslt = float(tmp(0)) / tmp(1)
return, rslt
end

function tiff_int,a,i, len=len	;return short int (2 bytes) from array a(i)
common mytiff_com, lun, order, ifd, count

on_error,2              ;Return to caller if an error occurs
   if n_elements(len) le 0 then len = 1
   if len gt 1 then result = fix(a,i,len) $
   else result = fix(a,i)
   if order then byteorder, result, /sswap
   return, result
end

function tiff_byte, a,i,len=len	;return bytes from array a(i)
common mytiff_com, lun, order, ifd, count

on_error,2              ;Return to caller if an error occurs

   if n_elements(len) le 0 then len = 1
   if len gt 1 then result = a(i:i+len-1) $
   else result = a(i)
   return, result
end

function tiff_read_field, index, tag	;Return contents of field index
; On output, tag = tiff tag index.
;
common mytiff_com, lun, order, ifd, count


on_error,2                      ;Return to caller if an error occurs
TypeLen = [0, 1, 1, 2, 4, 8] ;lengths of tiff types, 0 is null type for indexin

ent = ifd(index * 12: index * 12 + 11)  ;Extract the ifd
tag = tiff_int(ent, 0)		;Tiff tag index
typ = tiff_int(ent, 2)		;Tiff data type
cnt = tiff_long(ent, 4)		;# of elements
nbytes = cnt * TypeLen(typ)	;Size of tag field
IF (nbytes GT 4) THEN BEGIN 	;value size > 4 bytes ?
        offset = tiff_long(ent, 8)	;field has offset to value location
        Point_Lun, lun, offset
        val = BytArr(nbytes) 	;buffer will hold value(s)
        Readu, lun, val
        CASE typ OF		;Ignore bytes, as there is nothing to do
	   1: i = 0		;Dummy
           2: val = String(val)		;tiff ascii type
           3: val = tiff_int(val,0, len = cnt)
	   4: val = tiff_long(val,0, len = cnt)
           5: val = tiff_rational(val,0, len = cnt)
	ENDCASE
ENDIF ELSE BEGIN			;Scalar...
        CASE typ OF
	   1: val = ent(8)
  	   2: val = string(ent(8:8+cnt-1))
	   3: val = tiff_int(ent,8)
	   4: val = tiff_long(ent,8)
        ENDCASE
     ENDELSE
return, val
end


function iq_read, file
common mytiff_com, lun, order, ifd, count


on_error,2                      ;Return to caller if an error occurs

; Check if the file is compressed and act accordingly
; The temporary file is deleted after it has been read
;
tempfile=0
ext= strmid(file,(strlen(file)-2)(0),2)
if ((ext eq '.Z') or (ext eq '.z')) then begin
	spawn, 'logname', logname
	spawn, 'echo $$', pid
	file2= '/tmp/'+logname(0)+'.'+pid(0)	
	print, 'Compressed File! - Uncompressing...'
	spawn,  'zcat ' + file + ' >! '+file2
	file = file2
	tempfile=1
endif

if n_elements(lun) le 0 then lun = -1	;Clean up
if lun gt 0 then free_lun, lun
openr,lun,file, error = i, /GET_LUN, /BLOCK
if i lt 0 then begin 			; Something wrong, 
	if lun gt 0 then free_lun,lun
	lun = -1
	message, 'Unable to open file: ' + file
endif

;	Insert default values:
compression = 1
bits_sample = 1
ord = 1
samples_pixel = 1L
pc = 1
photo = 1
rows_strip = 'fffffff'xl	;Essentially infinity

hdr = bytarr(8)			;Read the header
readu, lun, hdr

typ = string(hdr(0:1))		;Either MM or II
if (typ ne 'MM') and (typ ne 'II') then begin ; Fucked up
	message,'TIFF_READ: File is not a Tiff file: ' + string(file)
	return,0
	endif
order = typ eq 'MM'  		;1 if Motorola 0 if Intel (LSB first or vax)
endian = byte(1,0,2)		;What endian is this?
endian = endian(0) eq 0		;1 for big endian, 0 for little
order = order xor endian	;1 to swap...

;print,'Tiff File: byte order=',typ, ',  Version = ', tiff_int(hdr,2)

offs = tiff_long(hdr, 4)	;Offset to IFD


while (offs ne 0) do begin

point_lun, lun, offs		;Read it

a = bytarr(2)			;Entry count array
readu, lun, a
count = tiff_int(a,0)		;count of entries
;print,count, ' directory entries'
ifd = bytarr(count * 12 + 4)	;Array for IFD's
readu, lun, ifd			;read it

offs = tiff_long(ifd, count * 12) ; Offset to next IFD

for i=0,count-1 do begin	;Print each directory entry
	value = tiff_read_field(i, tag)  ;Get each parameter
	;print , tag
	case tag of		;Decode the tag fields, other tags could be added
256:	Begin
		width = value
		print, 'Width:  ',width
	End
257:	Begin
		length = value
		print, 'Length: ',length
	End
258:	Begin
		bits_sample = value
		print, 'BitsPerSample: ',bits_sample
	End
259:	compression = value
262:	Photo = value
273:	Begin
		StripOff = value
		print, 'StripOffsets: ',StripOff
	End
274:	Ord = value
277:	Begin
		samples_pixel = long(value)
		print, 'SamplesPerPixel: ',samples_pixel
	End
278:	Begin
		Rows_strip = value
		print, 'RowsPerStrip: ',Rows_strip
	End
279:	Begin
		Strip_bytes = value
		print, 'StripByteCounts: ',Strip_bytes
	End
284:	PC = value
320:	ColorMap = value
; Here comes som ImageQuant tags.
33446:	Begin
		CorrFactor = value
		print,'CorrFactor:  ' ,CorrFactor
	End
else:   value = 0		;Throw it away
	endcase
endfor	
endwhile


nbytes = width * long(length)
strips_image = (length + rows_strip -1) / rows_strip
print, 'StripsPerImage: ',strips_image

free_lun, lun

print, 'Byteswapping and Converting to Long Integer.'

image_size = long(width)*long(length)

image = lonarr(width, length, /nozero)

dummy = CALL_EXTERNAL('/users/a/ursby/idl/lib/iq_lib.so', 'iq_swap', image, image_size, file, stripoff(0), CorrFactor)

IF (tempfile) THEN SPAWN, 'rm '+file

print, 'Done'

return, image

end


function iq_pick

on_error, 2

image = iq_read(pickfile(/READ, $
			PATH = '/users/i/opid09/data', $
			FILTER = ['*.gel*']))
return, image
end

