pro MRQMIN, x, y, sig, a, lista, covar, alpha, mbeta, chisq, funcs, alambda, $
	dampfactor, stepfactor=stepfactor
;+
; NAME:
;	MRQMIN
;
; PURPOSE:
;	Perform non-linear least-square refinement following the 
;	Levenberg-Marquardt method as described in Numerical Recipes.
;	This routine is called iteratively until convergence is reached.
;	First call MRQMIN with ALAMBDA negative to initialize and then
;	iterate until convergence. Then call MRQMIN one final time with
;	ALABMDA set to 0.
;
; CATEGORY:
;	Numerical model refinement.
;
; CALLING SEQUENCE:
; 	MRQMIN, x, y, sig, a, lista, covar, alpha, chisq, funcs, alambda
;
; INPUTS:
;	X: Input data, independent variable. Only passed to the user supplied
;		function FUNCS.
;	Y: Input data, dependent variable. Only passed to the user supplied
;		function FUNCS.
;	SIG: The standard deviation of the measurements Y. If not known set
;		to 1.0. (Vector of same size as X or scalar.)
;	A: A vector of the parameters to the function FUNCS (and some of these
;		are refined).
;	LISTA: A vector containing the subscript in A of the parameters to
;		refine (and thus of the same length as the number of parameters
;		to refine).
;	ALPHA: The curvature matrix. Do not modify between call of MRQMIN.
;	MBETA: Vector containing -0.5 times the gradient of ChiSq in respect
;		to the elements in LISTA.
;		Do not modify between call of MRQMIN.
;	CHISQ: The value of the minimisation function. Do not modify between 
;		call of MRQMIN.
;	FUNCS: String containing the name of the function to call to determine
;		the agreement between Y and model Y and to calculate the
;		partial derivatives.
;		The function is called with "d= FUNCS(x,y,a,dyda)" where
;		X,Y and A is as input to MRQMIN (with the values of A possibly
;		modified) and DYDA should be an array of #elementsX x 
;		#elementsA containing the partial derivatives on output. 
;		The function should return the difference between model and 
;		measurements as a vector of same size as Y. When FUNCS is 
;		called DYDA is set to be a vector of same size as A containing
;		zeros at the positions of the parameters not being refined. 
;		This can be used in FUNCS to save time (only calculate the
;		partial derivatives for the parameters who has a non-zero
;		value in the given dyda).
;	ALAMBDA: A parameter in the refinement routine. Should not be modified
;		between calls of MRQMIN except that it should be set to a 
;		negative number the first time the routine is called (the
;		first time with a new set of refinement parameters) and
;		when the refinement is considered converged ALAMBDA should be
;		set to zero and MRQMIN called once more.
;	DAMPFACTOR: The calculated shift is multiplied with this before
;		the shift is applied to A.
;
; KEYWORDED PARAMETERS:
;	STEPFACTOR: The default steps used to calculate the derivatives
;		are multiplied by this factor. (Only passed on to FUNCS.)
;
;
; OUTPUTS:
;	A: See above. Contains the new values. If CHISQ doesn't decrease 
;		then the old values of A and CHISQ are kept.
;	COVAR: The covariance matrix. (Only correct when called with ALAMBDA)
;	ALPHA: The curvature matrix.  (set to zero.)
;	CHISQ: See above. If CHISQ doesn't decrease then the old values 
;		of A and CHISQ are kept. 
;		
; COMMON BLOCKS:
;	None.
;
; SIDE EFFECTS:
;	A, MBETA, CHISQ, ALAMBDA are modified. (X, Y might also be modified
;		if the user supplied function FUNCS does so.)
;
; RESTRICTIONS:
;	None
;
; PROCEDURE:
;	MRQMIN is based on the routine mrqmin described in section 14.4 
;	of Numerical Recipes in C: The Art of Scientific Computing, 
;	W.H. Press, B.P. Flanery, S.A. Teukolsky, W.T. Vetterling,
;	published by Cambridge University Press.
;
; MODIFICATION HISTORY:
;	Adopted by Thomas Ursby, June 1995	
;
;-

ndata= (size(x))(1)
ma= (size(a))(1)
mfit= (size(lista))(1)

; Initialize
IF (alambda lt 0.0) THEN BEGIN 
  kk= mfit
  FOR j=0,ma-1 DO BEGIN ; Check lista
    ihit= 0
    FOR k=0,mfit-1 DO IF (lista(k) eq j) THEN ihit=ihit+1
    IF (ihit eq 0) THEN kk= kk+1 $
    ELSE IF (ihit gt 1) THEN MESSAGE,'Bad LISTA permutation in MRQMIN-1' 
  ENDFOR 
  IF (kk ne ma) THEN MESSAGE, 'Bad LISTA permutation in MRQMIN-2'
  alambda=0.001
  MRQCOF, x, y, sig, a, lista, alpha, mbeta, chisq, funcs, stepfactor=stepfactor
  ; First time we return without any refinement.
ENDIF ELSE BEGIN	
  ; Alter linearized fitting matrix by augmenting diagonal elements.
  covar= alpha
  FOR j=0,mfit-1 DO covar(j,j)=alpha(j,j)*(1.0+alambda)
  
  ; Matrix solution.
  covar= INVERT(covar)
  ; Use SVD instead? (not yet tested)
; What THRESH should we use?
; What useful information can we give the caller?
;  NR_SVD,covar,w,u,v			;Do the svd
;  good = WHERE(w gt (max(w) * thresh), ng) ;Cutoff for sing values
;  sing = m - ng		;# of singular values
;  IF (sing ne 0) THEN BEGIN
;    PRINT, 'Warning: ' + strcompress(sing, /REMOVE) + $
;			' singular (or near singular) values found.'
;    PRINT, 'In: MRQMIN'
;    small=WHERE(w le max(w)*thresh)
;    w(small)=0
;    IF (ng eq 0) THEN MESSAGE, 'Matrix has null rank.'
;  ENDIF
;  wu = DBLARR(mfit,mfit)
;  FOR j=0,mfit-1 DO wu(j,*)= u(*,j)/w(j)
;  covar= v # wu

  da= dampfactor * (covar # mbeta)
  ochisq= chisq
  IF (alambda ne 0) THEN BEGIN 
    ; Did the trial succed?
    atry=a
    atry(lista)= a(lista)+da
    MRQCOF, x, y, sig, atry, lista, covar, da, chisq, funcs, $
  	stepfactor=stepfactor
    IF (chisq lt ochisq) THEN BEGIN
      ; Accept solution
      alambda= 0.1*alambda
      ochisq= chisq
      alpha= covar
      mbeta=da
      a(lista)=atry(lista)
    ENDIF ELSE BEGIN
      ; Failure
      alambda= 10.0*alambda
      chisq= ochisq
    ENDELSE
  ENDIF
ENDELSE
 
END
