pro DRAW_INSPECT_PRED,state
;+
; NAME:
;	DRAW_INSPECT_PRED
;
; PURPOSE:
;	Inspection of integrated spots
;
; CATEGORY:
;	Data processing
;
; CALLING SEQUENCE:
;	DRAW_INSPECT_PRED,state
;
; INPUTS:
;	state : the state (a structure) containing the state of the 
;		widget application.
;
; OUTPUTS:
;	none
;
; COMMON BLOCKS:
;	None.
;
; SIDE EFFECTS:
;	None.
;
; RESTRICTIONS:
;
; PROCEDURE:
;	The user clicks with left mouse on one spot from real data (usually
;	a small portion of the whole array). If there is a simulated spot
;	whose location is near enough the clicked position, the information
;	concerning this spot will be print out from the "preds" parameter.
;
; MODIFICATION HISTORY:
;	D.Bourgeois, May 1997.
;-

COMMON SHARE_PROW,pred,sel_pred,par

;new line character (to display infos)
nl=STRING("12b)

;get the windows
main_win=state.draw_win
zoom_win=state.zoom_win
show_win=state.study_win

;get image size
imageX = state.x_im_sz
imageY = state.y_im_sz

; get beam center
centerX = par.cenx - 1
centerY = par.ceny - 1

;get yscale
scaleY = par.yscale

;original pixel size [micrometers]
pix_size = par.raster
pix_sizeX = 0.001*pix_size
pix_sizeY = 0.001*pix_size/scaleY

;get the boxsize
boxsize=state.boxinteg.boxsize

;create a prediction image
pred_image = BYTARR(imageX,imageY)

;get the binning factor
bin_fac=state.bin_fac

;get the origin of zoom image
relX=state.x0
relY=state.y0

;position of the main image area relative to the center of the beam [mm]
minX = -1*pix_sizeX*centerX
minY = -1*pix_sizeY*centerY
maxX = pix_sizeX*(imageX-1-centerX)
maxY = pix_sizeY*(imageY-1-centerY)

;position of the zoom image area relative to the center of the beam [mm]
minXz = pix_sizeX*(relX-centerX)
minYz = pix_sizeY*(relY-centerY)
maxXz = pix_sizeX*(relX-centerX+imageX-1)
maxYz = pix_sizeY*(relY-centerY+imageY-1)
;PRINT,'Position of lower left corner of image relative to center of beam (X,Y) [mm] :',minX,minY




WHILE 1 DO BEGIN                ;Internal event loop   
 ;wait for an event from the main window
 ev = WIDGET_EVENT([state.draw,state.zoom])
 win_id=ev.id ; from which window
 button = ev.press  ;New button state   
 IF button EQ 4 THEN GOTO,cut_loop_inspect ;leave the routine
 IF ev.press THEN BEGIN
  IF win_id EQ state.draw THEN BEGIN ;in main drawing window
   x = state.bin_fac*ev.x    ;Pixel coordinates
   y = state.bin_fac*ev.y
  ENDIF ELSE BEGIN ; in zoom window
   x = state.x0+ROUND(ev.x/state.scale)    ;Pixel coordinates
   y = state.y0+ROUND(ev.y/state.scale)
  ENDELSE
  box = (*state.orig_image)(MAX([0,x-boxsize.x/2]):MIN([imageX-1,x+boxsize.x/2]),$
	     MAX([0,y-boxsize.y/2]):MIN([imageY-1,y+boxsize.y/2]))

  ;find proper center of the box. Just consider the center of gravity of the
  ;central part
  sbox = (*state.orig_image)(MAX([0,x-boxsize.x/4]):MIN([imageX-1,x+boxsize.x/4]),MAX([0,y-boxsize.y/4]):MIN([imageY-1,y+boxsize.y/4]))

  loc=WHERE(sbox LT MAX(sbox)/1.2,count)
  IF count NE 0 THEN sbox(loc) = 0
  ;get the integrated row and column
  row=TOTAL(sbox,2)>0.1
  column=TOTAL(sbox,1)>0.1
  ;compute center of gravity
  indx=INDGEN((SIZE(sbox))(1))
  indy=INDGEN((SIZE(sbox))(2))
  cenX=-1*boxsize.x/4+TOTAL(indx*row)/TOTAL(row)
  cenY=-1*boxsize.y/4+TOTAL(indy*column)/TOTAL(column)

;  print,'Center of gravity : [pixels in box]',cenX,cenY

  peak_minX = ROUND(MAX([0,cenX+x-boxsize.x/2]))
  peak_maxX = ROUND(MIN([imageX-1,cenX+x+boxsize.y/2]))
  peak_minY = ROUND(MAX([0,cenY+y-boxsize.x/2]))
  peak_maxY = ROUND(MIN([imageY-1,cenY+y+boxsize.y/2]))

  box = (*state.orig_image)(MAX([0,peak_minX]):MIN([imageX-1,peak_maxX]),MAX([0,peak_minY]):MIN([imageY-1,peak_maxY]))

  ;center of gravity of the final box
  fcenX=cenX+x-peak_minX
  fcenY=cenY+y-peak_minY

  ;we want to compare the pixel position inside that box with candidate 
  ;positions for predicted spots
  ;the positions of the corners of the selected box relative to beam center [mm]
  minboxX = minX+pix_sizeX*(MAX([peak_minX,0]))
  minboxY = minY+pix_sizeY*(MAX([peak_minY,0]))
  maxboxX = minX+pix_sizeX*(MIN([peak_maxX,imageX-1]))
  maxboxY = minY+pix_sizeY*(MIN([peak_maxY,imageY-1]))
;  PRINT,'Position of lower left corner of box relative to center of beam (X,Y) [mm] :',minboxX,minboxY

  ;select the possible preds
  poss_index = WHERE((sel_pred(*).x GE minboxX) AND $
		     (sel_pred(*).x LE maxboxX) AND $
		     (sel_pred(*).y GE minboxY) AND $
		     (sel_pred(*).y LE maxboxY),n_poss_index)

  IF n_poss_index EQ 0 THEN BEGIN
   INFO='No reflexion found ...'
   WSET,show_win
   ERASE
  ENDIF
  IF n_poss_index GT 0 THEN BEGIN
   disp_pred=sel_pred(poss_index)
;   PRINT,'Total number of sel_pred selected : ',n_poss_index


   ;display these disp_pred : for each of them, compute the position in 
   ;pred_image in pixels
   FOR i=0L,n_poss_index-1 DO BEGIN
    xpos = ROUND(disp_pred(i).x/pix_sizeX+centerX)
    ypos = ROUND(disp_pred(i).y/pix_sizeY+centerY)
    pred_image(xpos,ypos) = 100
   ENDFOR
      
   pred_box = pred_image(MAX([0,peak_minX]):MIN([imageX-1,peak_maxX]),MAX([0,peak_minY]):MIN([imageY-1,peak_maxY]))

  WSET,show_win
  SURFACE,box,CHARSIZE=1.5,XTITLE='X Dim [Pixels]',YTITLE='Y Dim [Pixels]',ZTITLE='Intensity [Counts]',/SAVE
  CONTOUR,pred_box,CHARSIZE=1.5,ZVALUE=1,/T3D,/NOERASE,LEVELS=[(1+indgen(2))/2.*(MAX(pred_box)>1)]

  INFO=     '*******************************************'+nl
  INFO=INFO+'Number of sel_pred selected : '+STRING(n_poss_index)+nl
  INFO=INFO+'*******************************************'+nl
  FOR index=0,n_poss_index-1 DO BEGIN
	INFO=INFO+''+nl
	INFO=INFO+'@@@@@@'+nl
	INFO=INFO+'Spot # :'+STRING(index+1)+nl
	INFO=INFO+'@@@@@@'+nl
	INFO=INFO+'Miller indices : '+STRING(disp_pred(index).m(0))+STRING(disp_pred(index).m(1))+STRING(disp_pred(index).m(2))+nl
	INFO=INFO+'X and Y position (predicted) [mm]: '+STRING(disp_pred(index).x)+' '+STRING(disp_pred(index).y)+nl
	xpos = disp_pred(index).x/FLOAT(pix_sizeX) + centerX
	ypos = disp_pred(index).y/FLOAT(pix_sizeY) + centerY
	INFO=INFO+'X and Y position (predicted) [pixels in box]: '+STRING(xpos - peak_minX)+' '+STRING(ypos - peak_minY)+nl
	INFO=INFO+'X and Y position (central observed peak) [pixels in box]: '+STRING(fcenX)+' '+STRING(fcenY)+nl
	INFO=INFO+'Wavelength [A] ; Energy [keV] : '+STRING(disp_pred(index).l)+' '+STRING(12.4/disp_pred(index).l)+nl
	INFO=INFO+'Resolution : '+STRING(disp_pred(index).res)+nl
	INFO=INFO+'Profile fitted integrated intensity : '+STRING(disp_pred(index).intp(0))+nl
	IF state.pred.type EQ 2 THEN BEGIN
	 INFO=INFO+'Box integrated intensity : '+STRING(disp_pred(index).intp(2))+nl
	 INFO=INFO+'Combined integrated intensity : '+STRING((1-disp_pred(index).gfit)*disp_pred(index).intp(2)+disp_pred(index).gfit*disp_pred(index).intp(0))+nl
	 INFO=INFO+'Denzo profile fitted intensity : '+STRING(disp_pred(index).intp(5))+nl
        ENDIF
	INFO=INFO+'Standard deviation : '+STRING(disp_pred(index).isigp(0))+nl
	INFO=INFO+'I/Sigma(I) : '+STRING(disp_pred(index).intp(0)/(disp_pred(index).isigp(0)>1))+nl
	;INFO=INFO+'Multiplicity : '+STRING(disp_pred(index).mul)+nl
  ENDFOR

;  PRINT,''
;  PRINT,'*******************************************'
;  PRINT,'Number of sel_pred selected : ',n_poss_index
;  PRINT,'*******************************************'

;  FOR index=0,n_poss_index-1 DO BEGIN
;	PRINT,''
;	PRINT,'@@@@@@'
;	PRINT,'Spot # :',index+1
;	PRINT,'@@@@@@'
;	PRINT,'Miller indices : ', disp_pred(index).m
;	PRINT,'X and Y position (predicted) [mm]: ', disp_pred(index).x,disp_pred(index).y
;	xpos = disp_pred(index).x/FLOAT(pix_sizeX) + centerX
;	ypos = disp_pred(index).y/FLOAT(pix_sizeY) + centerY
;	PRINT,'X and Y position (predicted) [pixels in box]: ', xpos - peak_minX,ypos - peak_minY
;	PRINT,'Wavelength [A] ; Energy [keV] : ', disp_pred(index).l,12.4/disp_pred(index).l
;	PRINT,'Resolution : ', disp_pred(index).res
;	PRINT,'Multiplicity : ', disp_pred(index).mul
;	PRINT,'Box integrated intensity : ', disp_pred(index).intb(0)
;        PRINT,'Profile fitted integrated intensity : ', disp_pred(index).intp(0)
;	PRINT,'Standard deviation : ', disp_pred(index).isigp(0)
;	PRINT,'I/Sigma(I) : ', disp_pred(index).intp(0)/disp_pred(index).isigp(0)
;	PRINT,'Multiplicity : ', disp_pred(index).mul
;  ENDFOR
 
  ENDIF

  pop_info = WIDGET_INFO(state.pop_info, /CHILD)
  WIDGET_CONTROL,pop_info,SET_VALUE=info

  WIDGET_CONTROL,state.info, SET_VALUE='Info : '

 ENDIF
ENDWHILE 
cut_loop_inspect : 
END
