pro RFACTOR_L, arr1, arr2, lambda, w=w, type=type, lbins=lbins
;+
; NAME:
;	RFACTOR_L
;
; PURPOSE:
;	Calculate R-factors in Lambda-bins. Normally called from
;	RFACTOR. 
;
; CATEGORY:
;	Crystallography, data analysis.
;
; CALLING SEQUENCE:
; 	RFACTOR_L, arr1, arr2, lambda [, w=w, type=type, lbins=lbins] 
;
; INPUTS:
;	ARR1,ARR2: The arrays containing the data.
;	LAMBDA: Array containing the lambda values for the reflections.
;
; KEYWORDED PARAMETERS:
;	W: Weighting. Array of the same dimension as ARR1 and ARR2.
;		The weighting is used for the
;		R-factor calculations if weighting is requested.
;	TYPE: Indicates what type of R-factor (default 1):
;		1: R= Sum(abs(arr1-arr2))/Sum(arr1)
;		2: R= Sum(abs(arr1-arr2))/Sum((arr1+arr2)/2)
;		3: R= Sum(w*abs(arr1-arr2))/ Sum(w*arr1)
;		4: R= Sum(w*abs(arr1-arr2))/ Sum(w*(arr1+arr2)/2)
;		5: R= Sum(w*(arr1-arr2)^2) / Sum(w*arr1^2)
;		6: R= Sum(w*(arr1-arr2)^2) / Sum(w*(arr1+arr2)^2/4)
;	LBINS: The R-factor is calculated in LAMBDA bins.
;		LBINS indicates how many bins. The range is lambda=
;		[min(lambda),max(lambda)]. Default 10.
;
; OUTPUTS:
;	Prints a table of R-factors in lambda-bins.
;
; COMMON BLOCKS:
;	None.
;
; SIDE EFFECTS:
;	None.
;
; RESTRICTIONS:
;	None.
;
; PROCEDURE:
;	Straightforward
;
; MODIFICATION HISTORY:
;	Thomas Ursby, 1995
;-

ON_ERROR, 2

IF NOT KEYWORD_SET(w) THEN w=1.0
IF NOT KEYWORD_SET(type) THEN type=1
IF NOT KEYWORD_SET(lbins) THEN lbins=10
r= dblarr(lbins)
racc=dblarr(lbins)
no = intarr(lbins)
noacc = intarr(lbins)

  ; Calculate R-factors in lambda-bins:
  lmin= FIX(10.0*min(lambda, max=lmax))/10.0 
  lmax= FIX(10.0*lmax)/10.0 + 0.1
  lbinsize= (lmax-lmin)/lbins
  FOR i=0,lbins-1 DO BEGIN
    ind= WHERE((lambda gt (lmin+i*lbinsize)) and $
		(lambda le (lmin+(i+1)*lbinsize)), count)
    IF (count ne 0) THEN BEGIN
      no(i) =(size(ind))(1)
      CASE (type) OF
        1: r(i) = TOTAL(abs(arr1(ind)-arr2(ind))) / TOTAL(arr1(ind))
        2: r(i) = TOTAL(abs(arr1(ind)-arr2(ind))) / $
		(TOTAL(arr1(ind)+arr2(ind))/2)
        3: r(i) = TOTAL(w*abs(arr1(ind)-arr2(ind))) / TOTAL(w*arr1(ind))
        4: r(i) = TOTAL(w*abs(arr1(ind)-arr2(ind))) / $
		(TOTAL(w*(arr1(ind)+arr2(ind)))/2)
        5: r(i) = TOTAL(w*(arr1(ind)-arr2(ind))^2) / TOTAL(w*arr1(ind)^2)
        6: r(i) = TOTAL(w*(arr1(ind)-arr2(ind))^2) / $
		(TOTAL(w*(arr1(ind)+arr2(ind))^2)/4)
        ELSE: MESSAGE,'Illegal type'
      ENDCASE
    ENDIF $
    ELSE BEGIN
      r(i)= 0.0
      no(i)= 0 
    ENDELSE
    indacc= WHERE(lambda le (lmin+(i+1)*lbinsize), count)
    IF (count ne 0) THEN BEGIN
      noacc(i) =(size(indacc))(1)
      CASE (type) OF
        1: racc(i) = TOTAL(abs(arr1(indacc)-arr2(indacc))) / $
			TOTAL(arr1(indacc))
        2: racc(i) = TOTAL(abs(arr1(indacc)-arr2(indacc))) / $
		(TOTAL(arr1(indacc)+arr2(indacc))/2)
        3: racc(i) = TOTAL(w*abs(arr1(indacc)-arr2(indacc))) / $
			TOTAL(w*arr1(indacc))
        4: racc(i) = TOTAL(w*abs(arr1(indacc)-arr2(indacc))) / $
			(TOTAL(w*(arr1(indacc)+arr2(indacc)))/2)
        5: racc(i) = TOTAL(w*(arr1(indacc)-arr2(indacc))^2) / $
			TOTAL(w*arr1(indacc)^2)
        6: racc(i) = TOTAL(w*(arr1(indacc)-arr2(indacc))^2) / $
			(TOTAL(w*(arr1(indacc)+arr2(indacc))^2)/4)
        ELSE: MESSAGE,'Illegal type'
      ENDCASE
    ENDIF $
    ELSE BEGIN
      racc(i)= racc((i-1)>0) 
      noacc(i)= noacc((i-1)>0)
    ENDELSE
  ENDFOR

  ; Print R-factors
  PRINT, '       Lmin           Lmax             R          No         Racc       No,acc'
  FOR i=0,lbins-1 DO PRINT,lmin+i*lbinsize, ' ',$
	lmin+(i+1)*lbinsize,' :', r(i), no(i), racc(i), noacc(i)

END
