function READ_SCALELOG, filename, par, logfilm 
;+
; NAME:
;	READ_SCALELOG
;
; PURPOSE:
;
; 	Read in an (ASCII) logfile from SCALEPACK output.
;	Prints out global statistical factors (Rsym, completeness, redundancy ...)
;
; CATEGORY:
;	File managing
;
; CALLING SEQUENCE:
;	logres = READ_SCALELOG(filename,[par],[logfilm]) ;
;
; INPUTS:
;	filename : the path name of the file to process (with extension)
;
; OUTPUTS:
;	logres : the structure array containing the informations in resolution ranges
;	[logfilm] : [optional] the structure array containing the informations in images
;	[par] : optionnal output overall info. 
;
; COMMON BLOCKS:
;	None.
;
; SIDE EFFECTS:
;	None.
;
; RESTRICTIONS:
;	None.
;
; PROCEDURE:
;	Par -->
;		npacks 	: The number of packs or images
;		isigi 	: The mean I/Sigma(I) over the data set
;		red	: The mean redundancy over the data set
;		r	: The global Rmerge over the data set
;		rsquare	: The global square Rmerge over the data set
;		comp	: The global completeness over the data set
;		nmeas	: The total number of MEASURED intensities
;		nref	: The total number of USED reflections
;		nfull	: The total number of "reconstituted" full observations
;		imos	: The initial mosaicity
;		fmos	: The final mosaicity
;		icell	: The initial cell parameters
;		fcell 	: The final celle parameters
;		meani 	: The mean integrated intensity
;		errfull	: The full error estimation
;		errstat	: The statistical error estimation
;
;	Logres -->
;		lres	: The lower resolution
;		hres	: The higher resolution
;		chis	: The Chi square value in the resolution range
;		r	: The Rmerge in the resolution range
;		rsquare	: The square Rmerge in the resolution range
;		comp	: The completeness in the resolution range
;		nmeas	: The number of mesurements in the resolution range
;		nref	: The number of reflections measured in the resolution range
;		red	: The redundancy in the resolution range
;		
;	Logfilm -->
;		film	: The image number
;		scale	: The isotropic scale factor for the image
;		bfactor	: The isotropic B factor for the image
;		partial	: The number of partial reflections in the image
;		full	: The number of full reflections in the image
;		nrej	: Measurements rejected for reasons other than statistical
;		chis	: The Chi square value in the image
;		r	: The Rmerge in the image
;
; MODIFICATION HISTORY:
;	D.Bourgeois, August 1996.
;-

;on_error,2                              ;Return to caller if an error occurs


;open the SCALEPACK  logfile 
dum = FINDFILE(filename,count=count)
IF count EQ 1 THEN PRINT,'Reading ',filename ELSE BEGIN 
PRINT,'File not found : ',filename
GOTO,escape
ENDELSE

;Define the structures :
maxrange = 100
maxfilm = 1000

par = { $
		npacks 	: 0  ,$
		isigi 	: 0. ,$
		red	: 0. ,$
		r	: 0. ,$ 
		rsquare	: 0. ,$ 
		comp	: 0. ,$ 
		nref	: 0L ,$ 
		nmeas	: 0L ,$ 
		nfull	: 0L ,$ 
		imos	: 0. ,$ 
		fmos	: 0. ,$ 
		meani	: 0. ,$ 
		errfull	: 0. ,$ 
		errstat	: 0. ,$ 
		icell	: {a:0.,b:0.,c:0.,alpha:0.,beta:0.,gamma:0.} ,$ 
		fcell 	: {a:0.,b:0.,c:0.,alpha:0.,beta:0.,gamma:0.} $ 
	} 


onerange =	{ $
		lres	: 0., $
		hres	: 0., $
		chis	: 0., $
		r	: 0., $
		rsquare	: 0., $
		comp	: 0., $
		nmeas	: 0L, $
		nref	: 0L, $
		red	: 0. $
		}

logres = REPLICATE(onerange,maxrange) 

onefilm = 	{ $
		film	: 0 , $
		scale 	: 0., $
		bfactor : 0., $
		partial : 0L, $
		full	: 0L, $
		chis	: 0., $
		nrej	: 0., $
		r	: 0.  $
		}

logfilm = REPLICATE(onefilm,maxfilm) 



line = STRING(90)
cycle = 0
filmindex = 0
resindex = 0

OPENR,unit,filename,/GET_LUN

;find the parameters :

WHILE (NOT EOF(unit)) DO BEGIN
  READF, unit, line

;get the number of film packs
IF (STRMID(line, 1, 15) EQ '# of film packs') THEN BEGIN
READS,line,FORMAT='(16X,I5)',npacks
par.npacks = npacks
ENDIF

;get the number of reflection read in :
IF (STRMID(line, 0, 33) EQ ' total number of reflections used') THEN BEGIN
READS,line,FORMAT='(34X,I8)',nmeas
par.nmeas = nmeas
ENDIF

;get cell parameters and mosaicity
IF (STRMID(line, 1, 6) EQ 'Film #') THEN BEGIN
READF,unit,line
READS,line,FORMAT='(7X,6F8.3,27X,F5.3)',a,b,c,alpha,beta,gamma,m
IF cycle EQ 0 THEN BEGIN
par.icell.a = a & par.icell.b = b & par.icell.c = c
par.icell.alpha = alpha & par.icell.beta = beta & par.icell.gamma = gamma
par.imos = m
ENDIF
cycle = cycle + 1
ENDIF

IF (STRMID(line, 0, 43) EQ ' 1 - count of observations deleted manually') THEN BEGIN
par.fcell.a = a & par.fcell.b = b & par.fcell.c = c
par.fcell.alpha = alpha & par.fcell.beta = beta & par.fcell.gamma = gamma
par.fmos = m
ENDIF

;get number of partial and full reflections
IF (STRMID(line, 0, 16) EQ ' IP fitted, no o') THEN BEGIN
READS,line,FORMAT='(18X,I4,F8.4,F8.2,8I5)',film,scale,bfactor,r1,r2,r3,r4,r5,r6,partial,full
logfilm(filmindex).film = film
logfilm(filmindex).scale = scale
logfilm(filmindex).bfactor = bfactor
logfilm(filmindex).nrej = r1+r2+r3+r4+r5+r6
logfilm(filmindex).partial = partial
logfilm(filmindex).full = full
filmindex = filmindex + 1
ENDIF

;get observation redundancies by shells
IF (STRMID(line, 22, 46) EQ 'Summary of observation redundancies by shells:') THEN BEGIN
resindex = 0
READF,unit,line
READF,unit,line

WHILE(STRMID(line,0,4) NE ' All') DO BEGIN
READF,unit,line
IF (STRMID(line,0,4) NE ' All') THEN BEGIN
 READS,line,FORMAT='(1X,2F6.2,10I6,I7)',lres,hres,n0,n1,n2,n3,n4,n5,n7,n9,n13,n20,nref
logres(resindex).lres = lres
logres(resindex).hres = hres
logres(resindex).nref = nref
logres(resindex).nmeas = n1+2*n2+3*n3+4*n4+5.5*n5+7.5*n7+10.5*n9+16*n13+20*n20
IF nref GT 0 THEN logres(resindex).red = (n1+2*n2+3*n3+4*n4+5.5*n5+7.5*n7+10.5*n9+16*n13+20*n20)/nref ELSE logres(resindex).red = 0
resindex = resindex + 1
ENDIF
ENDWHILE
ENDIF


;get R-factors by film
IF (STRMID(line, 46, 26) EQ 'R-factors by "film" number') THEN BEGIN
filmindex = 0
READF,unit,line
READF,unit,line

WHILE(STRMID(line,0,4) NE ' All') DO BEGIN
READF,unit,line
IF (STRMID(line,0,4) NE ' All') THEN BEGIN
 READS,line,FORMAT='(2X,I6,2(4X,I6),5X,F7.1,4X,F6.3,3X,F6.3)',film,n0,n1,isigi,chis,r
logfilm(filmindex).r = 100*r
logfilm(filmindex).chis = chis
filmindex = filmindex + 1
ENDIF
 READS,line,FORMAT='(10X,I8,4X,I6,5X,F7.1,4X,F6.3,3X,F6.3)',nfull,n1,isigi,chi,r
par.nfull = nfull
par.isigi = isigi
ENDWHILE
ENDIF

;get completeness by shell
IF (STRMID(line, 23, 36) EQ 'Summary of observation redundancies:') THEN BEGIN
resindex = 0
READF,unit,line
READF,unit,line

WHILE(STRMID(line,0,4) NE ' All') DO BEGIN
READF,unit,line
IF (STRMID(line,0,4) NE ' All') THEN BEGIN
 READS,line,FORMAT='(1X,2F6.2,10F6.1,F7.1)',lres,hres,p0,p1,p2,p3,p4,p5,p7,p9,p13,p20,comp
logres(resindex).comp = comp
resindex = resindex + 1
ENDIF
 READS,line,FORMAT='(13X,10F6.1,F7.1)',p0,p1,p2,p3,p4,p5,p7,p9,p13,p20,comp
par.comp = comp
ENDWHILE
ENDIF

;get final R-factor by shell
IF (STRMID(line, 15, 58) EQ 'Summary of reflections intensities and R-factors by shells') THEN BEGIN
READF,unit,line
IF (STRMID(line, 5, 8) EQ 'R linear') THEN BEGIN
resindex = 0
READF,unit,line
READF,unit,line
READF,unit,line
READF,unit,line
READF,unit,line
READF,unit,line
WHILE(STRMID(line,1,4) NE ' All') DO BEGIN
IF EOF(unit) THEN MESSAGE,'PROBLEM WITH SCALEPACK ...'
READF,unit,line
IF (STRMID(line,1,4) NE ' All') THEN BEGIN
 READS,line,FORMAT='(4X,2F7.2,3F8.1,3F7.3)',lres,hres,meani,errfull,errstat,chis,r,rsquare
 logres(resindex).chis = chis
 logres(resindex).r = 100*r
 logres(resindex).rsquare = 100*rsquare
 resindex = resindex + 1
ENDIF
 READS,line,FORMAT='(21X,3F8.1,3F7.3)',meani,errfull,errstat,chi,r,rsquare
par.r = 100*r
par.rsquare = 100*rsquare
par.meani = meani
par.errfull = errfull
par.errstat = errstat
ENDWHILE
ENDIF
ENDIF

ENDWHILE

FREE_LUN, unit

logfilm = logfilm(0:npacks-1)
logres = logres(0:resindex-1)
par.nref =  LONG(TOTAL(logres.nref))
par.red = TOTAL(logres.nmeas)/TOTAL(logres.nref)

PRINT,''
PRINT,'&&&&&&& CELL PARAMETERS AND MOSAICITY &&&&&&&'
PRINT,''
PRINT,'Initial cell parameters : ',par.icell.a,par.icell.b,par.icell.c,par.icell.alpha,par.icell.beta,par.icell.gamma
PRINT,'Initial mosaicity : ',par.imos
PRINT,'Final cell parameters : ',par.fcell.a,par.fcell.b,par.fcell.c,par.fcell.alpha,par.fcell.beta,par.fcell.gamma
PRINT,'Final mosaicity : ',par.fmos
PRINT,'Number of refinement cycles : ',cycle 

PRINT,''
PRINT,'&&&&&&& NUMBER OF REFLECTIONS &&&&&&&'
PRINT,''
PRINT,'Number of packs : ',par.npacks
PRINT,'Number of measurements read in : ',par.nmeas
PRINT,'Total number of measurements rejected for reasons other than statistical : ', LONG(TOTAL(logfilm.nrej))
PRINT,'Total number of partials : ', LONG(TOTAL(logfilm.partial))
PRINT,'Total number of fulls : ', LONG(TOTAL(logfilm.full))
PRINT,'Number of reconstituted "full" measurements : ',par.nfull
PRINT,'Total number of reflections used in scaling : ', par.nref
PRINT,'To get the number of reflections rejected for statistical reasons, subtract '
PRINT,'this number : ',par.nref,' to the number of reflections in the .sca file'

PRINT,''
PRINT,'&&&&&&& FINAL PARAMETERS &&&&&&&'
PRINT,''
PRINT,'Average redundancy : ', TOTAL(logres.nmeas)/TOTAL(logres.nref)
PRINT,'Mean <I/Sigma> : ',par.isigi
PRINT,'Mean <I> : ',par.meani
PRINT,'Overall completeness [%] : ',par.comp
PRINT,'Merging Square R-Factor [SUM ( (I - <I>) ** 2) / SUM (I ** 2)] [%] : ',par.rsquare
PRINT,'Merging R-Factor [SUM ( ABS(I - <I>)) / SUM (I)] [%] : ',par.r

RETURN,logres

escape :
END







