function PDBSCALE, par
;+
; NAME:
;	PDBSCALE
;
; PURPOSE:
;	Calculate the SCALE matrix in a PDB file. This matrix multiplied
;	by the orthogonal coordinates of an atom gives the atoms fractional
;	coordinates.
;
; CATEGORY:
;	Crystallography
;
; CALLING SEQUENCE:
; 	m= PDBSCALE(par)
;
; INPUTS:
;	PAR: Structure with the cell parameters (tags A,B,C,ALPHA,BETA,GAMMA).
;		The cell edges should be in Angstroms and the angles in 
;		degrees.
;
; KEYWORDED PARAMETERS:
;	None
;
; OUTPUTS:
;	Returns the matrix as SCALE in a PDB file. 
;		fc= m # oc
;	where fc is the fractional coordinates and oc are the orthogonal
;	coordinates.
;
; COMMON BLOCKS:
;	None.
;
; SIDE EFFECTS:
;	None
;
; RESTRICTIONS:
;	None
;
; PROCEDURE:
;	See appendix A of the PDB entry format description (Feb 1992).
;
; MODIFICATION HISTORY:
;	Thomas Ursby, 1996
;-

a= par.a & b=par.b & c=par.c
alpha= !dtor*par.alpha & beta=!dtor*par.beta & gamma=!dtor*par.gamma

v= a*b*c*sqrt(1-cos(alpha)^2-cos(beta)^2-cos(gamma)^2+ $
	2*cos(alpha)*cos(beta)*cos(gamma))
m= INVERT([[a,0,0], $
	[b*cos(gamma),b*sin(gamma),0], $
	[c*cos(beta), c*(cos(alpha)-cos(beta)*cos(gamma))/sin(gamma), $
		v/(a*b*sin(gamma))]])

RETURN, m

END