function MAP_INTEG, map, coordinates, radius, par, $
		center=center, average=average, index=index, rms=rms
;+
; NAME:
;	MAP_INTEG
;
; PURPOSE:
;	Do box (or rather sphere) integration around a point in a density
;	map. 
;
; CATEGORY:
;	Crystallography (and...)
;
; CALLING SEQUENCE:
; 	integrated_density= MAP_INTEG(map,coordinates,radius,par [, $
;		/center, average=averag, index=index, rms=rms)
;
; INPUTS:
;	MAP: The map as read by READ_MAP (FLTARR with three dimensions).
;	COORDINATES: Orthogonal coordinates of the point around which to
;		integrate (as for the atoms in the PDB file and assuming
;		that SKEW=0 for the map file).
;	RADIUS: Radius around the point in which to integrate. Grid points
;		which fall inside this radius are summed.
;	PAR: The parameters returned by READ_MAP (in the keyword PAR).
;
; KEYWORDED PARAMETERS:
;	CENTER: If set then the integration is done around the grid point
;		with the largest value within the distance RADIUS around
;		the original given coordinates,
;		NOTE that the routine might not handle this correctly
;		(see varible ss below, used to select a small part 
;		of the original map)
;
; OUTPUTS:
;	Returns the integrated density.
;	AVERAGE: Contains the averaged density within the sphere.
;	INDEX: Indices of the map (one-dimensional) of the points in
;		the maps that were included in the integration (i.e. the
;		points in the map within the specified distance from
;		COORDINATES).
;	RMS: The root means square deviation of the intensity where 
;		integration takes place.
;
; COMMON BLOCKS:
;	None.
;
; SIDE EFFECTS:
;	None
;
; RESTRICTIONS:
;	MAP_INTEG can't handle  par.skew not equal 0 (see READ_MAP
;	    and the definition of the CCP4 map format), i.e.
;	    MAP_INTEG is relying on that the axis are as the PDB-standard
;	    (It is using PDBSCALE to get the transformation matrix
;	     between orthogonal and fractional coordinates. If these
;	     fractional coordinates doesn't correspond to the axis
;	     of the maps then everyhing is wrong.)
;	If the map is not containing the full sphere around the integration
;	     point, then it will integrate only those points available
;	     but not give any error message. (I think...)
;
; PROCEDURE:
;	Straightforward
;
; MODIFICATION HISTORY:
;	Thomas Ursby, 1996
;	Major changes, TU, Apr, 2000 (Selecting part of map to
;		speed up calculations).
;-

ON_ERROR, 2

; Inital checks:
IF (par.skew ne 0) THEN $
  MESSAGE, 'Skew not equal zero. MAP_INTEG cant handle this!'

; Transformation from orthogonal coordinates to fractional: 
m0= PDBSCALE(par)
; Transformation from fractional coordinates to orthogonal: 
m= INVERT(m0)

; Some parameters from par that will be used:
;  (The axes can be shifted so we must use mapindex)
;  (After transformation, [0] will correspond to a-axis etc)
inverted_mapindex= [WHERE(par.mapindex eq 0), $
		    WHERE(par.mapindex eq 1), $
		    WHERE(par.mapindex eq 2)]
mapstart=  par.mapstart[inverted_mapindex]
mapsize=   par.mapsize[inverted_mapindex]
intervals= par.intervals[inverted_mapindex]

; Reduce the size of the map to the around the centre coordinate.
; This can increase the speed a lot!
; Calculate the fractional coordinates of all eight corners of a box
; with side 2*radius around "coordinates" to find minimum and maximum
; of the fractional coordinates => Part of map to use.
; Note ss should be bigger than radius if the keyword CENTER is used!
ss= radius ; Note that one extra step is added in all directions below.
minfc= FLOAT(mapstart+mapsize)/intervals
maxfc=  FLOAT(mapstart)/intervals 
FOR i=-1,1,2 DO BEGIN
  FOR j=-1,1,2 DO BEGIN
    FOR k=-1,1,2 DO BEGIN
      fc= m0 # (coordinates + [i*ss, j*ss, k*ss])
      minfc= [fc[0]<minfc[0], fc[1]<minfc[1], fc[2]<minfc[2]]
      maxfc= [fc[0]>maxfc[0], fc[1]>maxfc[1], fc[2]>maxfc[2]]
    ENDFOR
  ENDFOR
ENDFOR
; Get the array limits of the small selected part of the map:
i1= FLOOR(minfc*intervals) - [1,1,1] - mapstart
i2=  CEIL(maxfc*intervals) + [1,1,1] - mapstart
; Make sure the selected limits are inside the map:
i1= i1> [0,0,0]
i2= i2< (mapsize - [1,1,1])
; Note that the small selected part of the map, smap, keeps
; the column,row,section ordering of map (i.e. we use mapindex
; to transform back):
smap= map[i1[par.mapindex[0]]:i2[par.mapindex[0]], $
	  i1[par.mapindex[1]]:i2[par.mapindex[1]], $
	  i1[par.mapindex[2]]:i2[par.mapindex[2]]]
; We also now work with the following variables expressed in
; coloumns,row,sections ordering:
smapstart= (i1+mapstart)[par.mapindex]
smapsize=  (i2-i1 + [1,1,1])[par.mapindex]
sintervals= intervals[par.mapindex]

nc= smapsize(0) & nr=smapsize(1) & ns=smapsize(2)
ic= FLOAT(sintervals(0)) & ir= FLOAT(sintervals(1)) & is= FLOAT(sintervals(2))
; Array of the map size 
smappos= LINDGEN(long(nc)*nr*ns)
; xx[0,*] will step in 1/ic steps from mapstart(0)/ic to 
;        (mapstart(0)+mapsize(0))/ic and then start from 0,
; xx[1,*] will step in 1/ir steps from mapstart(1)/ir to
;        (mapstart(1)+mapsize(1))/ir and then start from 0
;        but each number is repeated nc times,
; xx[2,*] will step in 1/is steps from mapstart(2) to
;	(mapstart(2)+mapsize(2))/is but each number is 
;        repeated nc*nr times (so it will arrive at the "mapend" at the
;       last position and therefore not start again from 0)
xx=TRANSPOSE( $
	[[(smapstart(0)+((smappos mod (nc*nr)) mod nc))/ic],$
	[(smapstart(1)+FIX((smappos mod (nc*nr))/nc))/ir], $
	[(smapstart(2)+FIX(smappos/(nc*nr)))/is]])
; The axes can be shifted so we must use mapindex:
xx= xx(inverted_mapindex,*)
; Transform to orthogonal coordinates:
xx= m # TEMPORARY(xx)
; Now set x to the distance to the required position 
FOR i=0,2 DO xx(i,*)= coordinates(i)-xx(i,*)
x= sqrt(TOTAL(xx^2,1))

index= WHERE(x lt radius, ct)
IF (ct eq 0) THEN MESSAGE, $
	'No grid points of the map within specified distance.'
IF (N_ELEMENTS(center) gt 0) THEN BEGIN
  maxdensity= MAX(smap(index),center)
  PRINT, 'Center moved to :', 2*coordinates-xx(*,index(center))
  FOR i=0,2 DO xx(i,*)= coordinates(i) - xx(i,*) + $
		coordinates(i)-xx(i,index(center))
  x= sqrt(TOTAL(xx^2,1))
  index= WHERE(x lt radius, ct)
ENDIF

integ= TOTAL(smap(index))
;print,smap(index)

average= integ/ct
rms = STDEV(smap(index))

RETURN, integ

END

