function GET_BASELINE,x, y, m, XRANGE=xrange, YRANGE=yrange, $
                 BASELINE_RANGE = baseline_range,$
                 CHI_SQUARE=chi_square, WEIGHTS = weights, $
                 COEFF=coeff, $
                 NOPRINT=noprint, $
                 GET_PLOT=get_plot, PLOT_RESIDUAL=plot_residual, $
                 USE_SVD=use_svd, USE_SPLINE=use_spline, SIGMA_SPLINE=sigma_spline
;+           
; NAME:
;	GET_BASELINE
;
; PURPOSE:
;
;	Returns the baseline of an experimental data-set
;
; CATEGORY:
;	Data processing
;
; CALLING SEQUENCE:
;	baseline = GET_BASELINE(x, y, m, XRANGE=xrange, YRANGE=yrange,$
;                BASELINE_RANGE = baseline_range, YFIT=yfit, $
;                 CHI_SQUARE=chi_square, WEIGHTS = weights, $
;                 GET_PLOT=get_plot, PLOT_RESIDUAL=plot_residual,
;                 USE_SVD=use_svd,USE_SPLINE=use_spline, SIGMA_SPLINE=sigma_spline)
;
; INPUTS:
;	x : the x-values of the experimental data set 
;	y : the y-values of the experimental data set 
;       m : the order of the polynom to fit
;       XRANGE : the range of x-values over which the fit is performed
;       YRANGE : the range of y-values for plotting
;       BASELINE_RANGE : the window of baseline data (A structure array)
;                        of type REPLICATE({FLTARR(2)},N_windows) 
;       CHI_SQUARE : A chi-square quality factor for the fit.
;       COEFF : the coefficient of fit polynoms if spline fitting not used
;       NOPRINT: set this keyword to avoid printing anything
;       GET_PLOT : plot the experimental data together with the fitting
;       data
;       PLOT_RESIDUAL: overplots the residual.
;       USE_SVD: use this keyword to use SVD deconvolution instead of
;       polynom fit
;       USE_SPLINE: use this keyword to use CUBIC SPLINE deconvolution
;       instead of
;       SIGMA_SPLINE: tension applied to spline fitting (0: pure
;       spline, 10: more like polynom) 
;       polynom fit
;
; OUTPUTS:
;	coeff : the coefficient of the fitting function.
;
; COMMON BLOCKS:
;	None.
;
; SIDE EFFECTS:
;	None.
;
; RESTRICTIONS:
;	None
;
; PROCEDURE:
;	Uses the function SVDFIT, POLY_FIT, SPLINE from the IDL library.
;
; MODIFICATION HISTORY:
;	D.Bourgeois, February 2006.
;-

IF KEYWORD_SET(use_spline) OR KEYWORD_SET(use_svd) THEN use_poly_fit = 0 ELSE use_poly_fit = 1

w1 = MIN(WHERE(x GE xrange(0)))
w2 = MAX(WHERE(x LE xrange(1)))
x2=x(w1:w2)
y2=y(w1:w2)
y_baseline=FLTARR(N_ELEMENTS(y))

IF KEYWORD_SET(BASELINE_RANGE) THEN BEGIN
 n_windows = (SIZE(baseline_range))(1)
 n_indices = FLTARR(n_windows,2)
 n_points = (SIZE(x))(1)
 x_byte = BYTARR(n_points)
 FOR i=0,n_windows-1 DO BEGIN
  n_indices(i,0) = MIN(WHERE(x GE baseline_range(i).r(0)))
  n_indices(i,1) = MAX(WHERE(x LE baseline_range(i).r(1)))
  x_byte(n_indices(i,0):n_indices(i,1))=1
ENDFOR

  w_baseline_fit = WHERE(x_byte EQ 1)
  x_baseline_fit = x(w_baseline_fit)
  y_baseline_fit = y(w_baseline_fit)

ENDIF ELSE BEGIN 
 x_baseline_fit = x2
 y_baseline_fit = y2
ENDELSE

;w=1.0/y			;Weights
w=replicate(1.0, N_ELEMENTS(y_baseline_fit))

IF KEYWORD_SET(use_svd) THEN BEGIN
 IF NOT KEYWORD_SET(noprint) THEN PRINT,'Using SVD fit ...'
 coeff=SVDFIT(x_baseline_fit,y_baseline_fit,m+1,WEIGHTS=w,CHISQ=chisq,YFIT=yfit) 
 IF NOT KEYWORD_SET(noprint) THEN PRINT,'Chi square value for SVD deconvolution: ', chisq
 y_baseline2 = POLY(x2,coeff)
 y_baseline = POLY(x,coeff)
ENDIF ;use_svd

IF KEYWORD_SET(use_spline) THEN BEGIN
 IF NOT KEYWORD_SET(sigma_spline) THEN sigma_spline = 1 ; default
 IF NOT KEYWORD_SET(noprint) THEN PRINT,'Using SPLINE fit ...'
 y_baseline2=SPLINE(x_baseline_fit,y_baseline_fit,x2,sigma_spline)
 y_baseline(w1:w2)=y_baseline2
 coeff=0
ENDIF ; use_spline

IF KEYWORD_SET(use_poly_fit) THEN BEGIN
 IF NOT KEYWORD_SET(noprint) THEN PRINT,'Using ploynomial fit ...'
 coeff=POLY_FIT(x_baseline_fit,y_baseline_fit,m,YFIT=yfit)
 y_baseline2 = POLY(x2,coeff)
 y_baseline = POLY(x,coeff)
ENDIF ; use_poly_fit


 !P.MULTI = [0,1,2,0,0]

IF KEYWORD_SET(get_plot) THEN BEGIN
 PLOT,x2,y2, YRANGE=yrange, XRANGE=xrange, CHARSIZE=1.5, THICK=2.0, XTITLE = 'Wavelength [nm]', YTITLE='Raw data [AU]'
 OPLOT,x_baseline_fit,y_baseline_fit, PSYM=1
 OPLOT,x2,y_baseline2, THICK=1.5
 IF KEYWORD_SET(plot_residual) THEN BEGIN
  PLOT, x2, y2-y_baseline2, LINESTYLE=0, XRANGE=xrange, CHARSIZE=1.5, THICK=2.0, XTITLE = 'Wavelength [nm]', YTITLE='Baseline Corrected Data [AU]'
 ENDIF
ENDIF; get_plot

 !P.MULTI = [0,1,1,0,0]


RETURN, y_baseline

END
