PRO EQUIV_HKL, PTGP, HKLARR
;+
; NAME:
;	EQUIV_HKL
;
; PURPOSE:
;   Calculate equivalent, unique reflection indices for 
;   the crystallographic point groups.
;
; CATEGORY:
;	Crystallography.
;
; CALLING SEQUENCE:
;   EQUIV_HKL, PTGP, HKLARR
;
; INPUTS:
;	TO FIGURE OUT WHICH NUMBER TO ENTER :
;	  --> What we need is the LAUE symmetry (symmetry in the diffraction
;		pattern), NOT the POINT GROUP symmetry.
;	  --> To figure out the LAUE symmetry, use CAD. Exemple :
;		1/ On NICE in  /disk32/bourgeoi/ccp4 run :
;			A/ f2mtz_create.com
;			B/ cad_create.com > ! cad.out
;			C/ Look at cad.out.
;	  --> For exemple for P6222 (sg=180) Laue symmetry = 6/mmm
;
;    PTGP: Point group number.
;
;     (SEE INTERNATIONAL TABLES FOR CRYSTALLOGRAPHY (1983).  VOLUME A,
;     PP. 750-770.)
;
;     THERE ARE TEN CASES OF ALTERNATIVE AXES FOR SEVEN OF THE 32
;     CRYSTALLOGRAPHIC POINT GROUPS, THUS A TOTAL OF 42 CASES IS
;     TABULATED.
;
;     TRICL. MONOCL. ORTHO.  TETRAG.    TRIG.     RHOMB.    CUB.
;
;     (1) 1  (3) 2   (6) 222  (9) 4     (17)  3   (33) R3   (38) 23
;     (2) -1 (4) M   (7) MM2 (10) -4    (18)  -3  (34) R-3  (39) M3
;            (5) 2/M (8) MMM (11) 4/M
;                                       (19) 312  (35) R32  (40) 432
;                            (12) 422   (20) 321  (36) R3M  (41) -43M
;                            (13) 4MM   (21) 31M  (37) R-3M (42) M3M
;                            (14) -42M  (22) 3M1
;                            (15) -4M2  (23) -31M
;                            (16) 4/MMM (24) -3M1
;
;                                       HEXAG.
;
;                                       (25)  6
;                                       (26)  -6
;                                       (27)  6/M
;
;                                       (28)  622
;                                       (29)  6MM
;                                       (30) -6M2
;                                       (31) -62M
;                                       (32)  6/MMM
;
;   HKLARR: Array of one or several reflection indices, see example.
;
;
; OUTPUTS:
;   The hkl-indices in HKLARR are replaced with their equivalent ones.
;
;
; SIDE EFFECTS:
;	None.
;
;
; EXAMPLE 1:
;	
;   HKLARR = [-1,-1,-1]
;   EQUIV_HKL, 2, HKLARR
;   PRINT, HKLARR
;     1      1      1
;
;
;
; EXAMPLE 2:
;	
;   HKLARR = [[-1,-1,-1],[1,2,1],[-1,2,-3]
;   EQUIV_HKL, 2, HKLARR
;   PRINT, HKLARR
;     1      1      1
;     1      2      1
;     1     -2      3
;
; MODIFICATION HISTORY:
;   Originally written by:    ROBERT H. BLESSING
;                             MEDICAL FOUNDATION OF BUFFALO
;                             73 HIGH STREET
;                             BUFFALO, NEW YORK 14203, USA
;                             TELEPHONE:  (716) 856-9600
;                             ELECTRONIC MAIL:  Blessing@MFB.Buffalo.Edu
;
;                             JULY 1993
;
;  Translated from FORTRAN to IDL by Olof Svensson October 1995.
;
;-
;on_error,2 ;Return to caller if an error occurs

IF (N_ELEMENTS(PTGP) EQ 0) THEN BEGIN
  PRINT, "No point group number passed to HKL_EQUIV, aborting."
  STOP
ENDIF ELSE IF ((PTGP LE 0) OR (PTGP GT 42)) THEN BEGIN
  PRINT, "Illegal point group number, aborting"
  STOP
ENDIF

IF (N_ELEMENTS(HKLARR) EQ 0) THEN BEGIN
  PRINT, "No hkl-array given to HKL_EQUIV, aborting"
  STOP
ENDIF

size_hklarr = SIZE(HKLARR)
IF ( (size_hklarr(0) EQ 1) AND (size_hklarr(1) EQ 3) ) THEN BEGIN
  arr_type = 1
  size_hkl_arr = 1
ENDIF ELSE IF ( (size_hklarr(0) EQ 2) AND $
                (size_hklarr(1) EQ 3) ) THEN BEGIN
  arr_type = 2
  size_hkl_arr = size_hklarr(2)
ENDIF ELSE BEGIN
  PRINT, "Wrong dimensions of HKLARR, aborting."
  STOP
ENDELSE

FOR index=0L,size_hkl_arr-1 DO BEGIN

IF arr_type EQ 1 THEN BEGIN
  H = HKLARR(0)
  K = HKLARR(1)
  L = HKLARR(2)
ENDIF ELSE BEGIN
  H = HKLARR(0,index)
  K = HKLARR(1,index)
  L = HKLARR(2,index)
ENDELSE

CASE PTGP OF
;
;     TRICLINIC
;
;     1
;
  1: GOTO, jumpendcase
;
;     -1
;
  2: BEGIN
      IF (L GT 0) THEN GOTO, jumpendcase
      H=-H
      K=-K
      L=-L
      IF (L GT 0) THEN GOTO, jumpendcase
      IF (K GT 0) THEN GOTO, jumpendcase
      H=-H
      K=-K
      IF (K GT 0) THEN GOTO, jumpendcase
      IF (H GE 0) THEN GOTO, jumpendcase
      H=-H
      GOTO, jumpendcase
     END
;
;     MONOCLINIC (B-AXIS UNIQUE)
;
;     2
;
  3: BEGIN
jump3:
      IF (L GT 0) THEN GOTO, jumpendcase
      H=-H
      L=-L
      IF (L GT 0) THEN GOTO, jumpendcase
      IF (H GE 0) THEN GOTO, jumpendcase
      H=-H
      GOTO, jumpendcase
     END
;
;     M
;
  4: BEGIN
      K=ABS(K)
      GOTO, jumpendcase
     END
;
;     2/M
;
  5: BEGIN
      K=ABS(K)
      GOTO, jump3
     END
;
;     ORTHORHOMBIC
;
;     222
;
  6: BEGIN
      IF (H EQ 0) OR (K EQ 0) OR (L EQ 0) THEN BEGIN
        H=ABS(H)
        K=ABS(K)
        L=ABS(L)
      ENDIF
      IF (H GE 0) AND (K GE 0) THEN GOTO, jumpendcase
      X=H
      Y=K
      Z=L
      H=-X
      K=-Y
      L=Z
      IF (H GE 0) AND (K GE 0) THEN GOTO, jumpendcase
      H=-X
      K=Y
      L=-Z
      IF (H GE 0) AND (K GE 0) THEN GOTO, jumpendcase
      H=X
      K=-Y
      L=-Z
      GOTO, jumpendcase
     END
;
;     MM2
;
  7: BEGIN
jump7:
      H=ABS(H)
      K=ABS(K)
      GOTO, jumpendcase
     END
;
;     MMM
;
  8: BEGIN
      L=ABS(L)
      GOTO, jump7
     END
;
;     TETRAGONAL
;
;     4
;
  9: BEGIN
jump9:
      IF (H EQ 0) AND (K EQ 0) THEN GOTO, jumpendcase
      IF (H GT 0) AND (K GE 0) THEN GOTO, jumpendcase
      X=H
      Y=K
      H=-Y
      K=X
      IF (H GT 0) AND (K GE 0) THEN GOTO, jumpendcase
      H=-X
      K=-Y
      IF (H GT 0) AND (K GE 0) THEN GOTO, jumpendcase
      H=Y
      K=-X
      GOTO, jumpendcase
     END
;
;     -4
;
 10: BEGIN
      IF (H EQ 0) AND (K EQ 0) THEN L=ABS(L)
      IF (H EQ 0) AND (K EQ 0) THEN GOTO, jumpendcase
      IF (H GT 0) AND (K GE 0) THEN GOTO, jumpendcase
      X=H
      Y=K
      H=-X
      K=-Y
      IF (H GT 0) AND (K GE 0) THEN GOTO, jumpendcase
      H=Y
      K=-X
      L=-L
      IF (H GT 0) AND (K GE 0) THEN GOTO, jumpendcase
      H=-Y
      K=X
      GOTO, jumpendcase
     END
;
;     4/M
;
 11: BEGIN
      L=ABS(L)
      GOTO, jump9
     END
;
;     422
;
 12: BEGIN
jump12:
      IF (H EQ 0) OR (K EQ 0) OR (ABS(H) EQ ABS(K)) THEN L=ABS(L)
      IF (H GE K) AND (K GE 0) THEN GOTO, jumpendcase
      X=H
      Y=K
      H=-X
      K=-Y
      IF (H GE K) AND (K GE 0) THEN GOTO, jumpendcase
      H=-Y
      K=X
      IF (H GE K) AND (K GE 0) THEN GOTO, jumpendcase
      H=Y
      K=-X
      IF (H GE K) AND (K GE 0) THEN GOTO, jumpendcase
      H=-X
      K=Y
      L=-L
      GOTO, jump12
     END
;
;     4MM
;
 13: BEGIN
jump13:
      H=ABS(H)
      K=ABS(K)
      IF (H GE K) THEN GOTO, jumpendcase
      X=H
      H=K
      K=X
      GOTO, jumpendcase
     END
;
;     -42M
;
 14: BEGIN
jump14:
      IF (H EQ 0) OR (K EQ 0) THEN L=ABS(L)
      IF (H GE K) AND (K GE 0) THEN GOTO, jumpendcase
      X=H
      Y=K
      H=-X
      K=-Y
      IF (H GE K) AND (K GE 0) THEN GOTO, jumpendcase
      H=Y
      K=-X
      L=-L
      IF (H GE K) AND (K GE 0) THEN GOTO, jumpendcase
      H=-Y
      K=X
      IF (H GE K) AND (K GE 0) THEN GOTO, jumpendcase
      H=-X
      K=Y
      GOTO, jump14
     END
;
;     -4M2
;
 15: BEGIN
      H=ABS(H)
      K=ABS(K)
      IF (H EQ K) THEN L=ABS(L)
      IF (H GE K) THEN GOTO, jumpendcase
      X=H
      H=K
      K=X
      L=-L
      GOTO, jumpendcase
     END
;
;     4/MMM
;
 16: BEGIN
      L=ABS(L)
      GOTO, jump13
     END
;
;     TRIGONAL - HEXAGONAL AXES
;
;     3
;
 17: BEGIN
jump17:
      IF (H EQ 0) AND (K EQ 0) THEN GOTO, jumpendcase
      IF (H LT 0) AND (K GE 0) THEN GOTO, jumpendcase
      X=H
      Y=K
      I=-H-K
      H=I
      K=X
      IF (H LT 0) AND (K GE 0) THEN GOTO, jumpendcase
      H=Y
      K=I
      GOTO, jumpendcase
     END
;
;     -3
;
 18: BEGIN
      IF (H EQ 0) AND (K EQ 0) THEN L=ABS(L)
      IF (H EQ 0) AND (K EQ 0) THEN GOTO, jumpendcase
 jump518: 
      IF (H LT 0) AND (K GE 0) AND ((L GT 0) OR ((L EQ 0) AND (H LT -K))) THEN GOTO, jumpendcase
      X=H
      Y=K
      I=-H-K
      H=I
      K=X
      IF (H LT 0) AND (K GE 0) AND ((L GT 0) OR ((L EQ 0) AND (H LT -K))) THEN GOTO, jumpendcase
      H=Y
      K=I
      IF (H LT 0) AND (K GE 0) AND ((L GT 0) OR ((L EQ 0) AND (H LT -K))) THEN GOTO, jumpendcase
      H=-X
      K=-Y
      L=-L
      GOTO, jump518
     END
;
;     312
;
  19: BEGIN
jump19:
      IF (H EQ 0) OR (K EQ 0) OR (-H EQ K) THEN L=ABS(L)
      IF (H GE 0) AND (K GE 0) THEN GOTO, jumpendcase
      X=H
      Y=K
      I=-H-K
      H=I
      K=X
      IF (H GE 0) AND (K GE 0) THEN GOTO, jumpendcase
      H=Y
      K=I
      IF (H GE 0) AND (K GE 0) THEN GOTO, jumpendcase
      H=-Y
      K=-X
      L=-L
      GOTO, jump19
     END
;
;     321
;
 20: BEGIN
jump20:
      IF (H EQ K) OR (-2*H EQ K) OR (-H EQ 2*K) THEN L=ABS(L)
      IF (H GE 0) AND (-2*K LE H) AND (H LE K) THEN GOTO, jumpendcase
      X=H
      Y=K
      I=-H-K
      H=I
      K=X
      IF (H GE 0) AND (-2*K LE H) AND (H LE K) THEN GOTO, jumpendcase
      H=Y
      K=I
      IF (H GE 0) AND (-2*K LE H) AND (H LE K) THEN GOTO, jumpendcase
      H=Y
      K=X
      L=-L
      GOTO, jump20
     END
;
;     31M
;
 21: BEGIN
jump21:
      IF (H GE 0) AND (-2*K LE H) AND (H LE K) THEN GOTO, jumpendcase
      X=H
      Y=K
      I=-H-K
      H=I
      K=X
      IF (H GE 0) AND (-2*K LE H) AND (H LE K) THEN GOTO, jumpendcase
      H=Y
      K=I
      IF (H GE 0) AND (-2*K LE H) AND (H LE K) THEN GOTO, jumpendcase
      H=Y
      K=X
      GOTO, jump21
     END
;
;     3M1
;
 22: BEGIN
jump22:
      IF (H GE 0) AND (K GE 0) THEN GOTO, jumpendcase
      X=H
      Y=K
      I=-H-K
      H=I
      K=X
      IF (H GE 0) AND (K GE 0) THEN GOTO, jumpendcase
      H=Y
      K=I
      IF (H GE 0) AND (K GE 0) THEN GOTO, jumpendcase
      H=-Y
      K=-X
      GOTO, jump22
     END
;
;     -31M
;
 23: BEGIN
jump23:
      IF (H EQ 0) OR (K EQ 0) OR (-H EQ K) THEN L=ABS(L)
      IF (H GE 0) AND (-2*K LT H) AND (H LE K) AND ((L GT 0) OR ((L EQ 0) AND (K GE 0))) THEN GOTO, jumpendcase
      X=H
      Y=K
      I=-H-K
      H=I
      K=X
      IF (H GE 0) AND (-2*K LT H) AND (H LE K) AND ((L GT 0) OR ((L EQ 0) AND (K GE 0))) THEN GOTO, jumpendcase
      H=Y
      K=I
      IF (H GE 0) AND (-2*K LT H) AND (H LE K) AND ((L GT 0) OR ((L EQ 0) AND (K GE 0))) THEN GOTO, jumpendcase
      H=Y
      K=X
      IF (H GE 0) AND (-2*K LT H) AND (H LE K) AND ((L GT 0) OR ((L EQ 0) AND (K GE 0))) THEN GOTO, jumpendcase
      H=X
      K=I
      IF (H GE 0) AND (-2*K LT H) AND (H LE K) AND ((L GT 0) OR ((L EQ 0) AND (K GE 0))) THEN GOTO, jumpendcase
      H=I
      K=Y
      IF (H GE 0) AND (-2*K LT H) AND (H LE K) AND ((L GT 0) OR ((L EQ 0) AND (K GE 0))) THEN GOTO, jumpendcase
      H=-X
      K=-Y
      L=-L
      GOTO, jump23
     END
;
;     -3M1
;
 24: BEGIN
jump24:
      IF (H EQ K) OR (-2*H EQ K) OR (-H EQ 2*K) THEN L=ABS(L)
      IF (H GE 0) AND (K GE 0) AND ((L GT 0) OR ((L EQ 0) AND (H GE K))) THEN GOTO, jumpendcase
      X=H
      Y=K
      I=-H-K
      H=I
      K=X
      IF (H GE 0) AND (K GE 0) AND ((L GT 0) OR ((L EQ 0) AND (H GE K))) THEN GOTO, jumpendcase
      H=Y
      K=I
      IF (H GE 0) AND (K GE 0) AND ((L GT 0) OR ((L EQ 0) AND (H GE K))) THEN GOTO, jumpendcase
      H=Y
      K=X
      L=-L
      IF (H GE 0) AND (K GE 0) AND ((L GT 0) OR ((L EQ 0) AND (H GE K))) THEN GOTO, jumpendcase
      H=X
      K=I
      IF (H GE 0) AND (K GE 0) AND ((L GT 0) OR ((L EQ 0) AND (H GE K))) THEN GOTO, jumpendcase
      H=I
      K=Y
      IF (H GE 0) AND (K GE 0) AND ((L GT 0) OR ((L EQ 0) AND (H GE K))) THEN GOTO, jumpendcase
      H=-X
      K=-Y
      GOTO, jump24
     END
;
;     HEXAGONAL
;
;     6
;
 25: BEGIN
jump25:
      IF (H EQ 0) AND (K EQ 0) THEN GOTO, jumpendcase
      IF (H GT 0) AND (K GE 0) THEN GOTO, jumpendcase
      X=H
      Y=K
      I=-H-K
      H=I
      K=X
      IF (H GT 0) AND (K GE 0) THEN GOTO, jumpendcase
      H=Y
      K=I
      IF (H GT 0) AND (K GE 0) THEN GOTO, jumpendcase
      H=-X
      K=-Y
      IF (H GT 0) AND (K GE 0) THEN GOTO, jumpendcase
      H=-I
      K=-X
      IF (H GT 0) AND (K GE 0) THEN GOTO, jumpendcase
      H=-Y
      K=-I
      GOTO, jumpendcase
     END
;
;     -6
;
 26: BEGIN
      L=ABS(L)
      GOTO, jump17
     END
;
;     6/M
;
 27: BEGIN
      L=ABS(L)
      GOTO, jump25
     END
;
;     622
;
 28: BEGIN
jump28:
      IF (H EQ 0) OR (K EQ 0) OR (ABS(H) EQ ABS(K)) OR (ABS(2*H) EQ ABS(K)) OR (ABS(H) EQ ABS(2*K)) THEN L=ABS(L)
      IF (H GE K) AND (K GE 0) THEN GOTO, jumpendcase
      X=H
      Y=K
      I=-H-K
      H=I
      K=X
      IF (H GE K) AND (K GE 0) THEN GOTO, jumpendcase
      H=Y
      K=I
      IF (H GE K) AND (K GE 0) THEN GOTO, jumpendcase
      H=-X
      K=-Y
      IF (H GE K) AND (K GE 0) THEN GOTO, jumpendcase
      H=-I
      K=-X
      IF (H GE K) AND (K GE 0) THEN GOTO, jumpendcase
      H=-Y
      K=-I
      IF (H GE K) AND (K GE 0) THEN GOTO, jumpendcase
      H=Y
      K=X
      L=-L
      GOTO, jump28
     END
;
;     6MM
;
 29: BEGIN
jump29:
      IF (H GE K) AND (K GE 0) THEN GOTO, jumpendcase
      X=H
      Y=K
      I=-H-K
      H=I
      K=X
      IF (H GE K) AND (K GE 0) THEN GOTO, jumpendcase
      H=Y
      K=I
      IF (H GE K) AND (K GE 0) THEN GOTO, jumpendcase
      H=-X
      K=-Y
      IF (H GE K) AND (K GE 0) THEN GOTO, jumpendcase
      H=-I
      K=-X
      IF (H GE K) AND (K GE 0) THEN GOTO, jumpendcase
      H=-Y
      K=-I
      IF (H GE K) AND (K GE 0) THEN GOTO, jumpendcase
      H=Y
      K=X
      GOTO, jump29
     END
;
;     -6M2
;
 30: BEGIN
      L=ABS(L)
      IF (H EQ 0) AND (K EQ 0) THEN GOTO, jumpendcase
 jump530: 
      IF (H GT 0) AND (K GE 0) THEN GOTO, jumpendcase
      X=H
      Y=K
      I=-H-K
      H=I
      K=X
      IF (H GT 0) AND (K GE 0) THEN GOTO, jumpendcase
      H=Y
      K=I
      IF (H GT 0) AND (K GE 0) THEN GOTO, jumpendcase
      H=-Y
      K=-X
      GOTO, jump530
     END
;
;     -62M
;
 31: BEGIN
      L=ABS(L)
      IF (H EQ 0) AND (K EQ 0) THEN GOTO, jumpendcase
 jump531: 
      IF (H GT 0) AND (-2*K LT H) AND (H LE K) THEN GOTO, jumpendcase
      X=H
      Y=K
      I=-H-K
      H=I
      K=X
      IF (H GT 0) AND (-2*K LT H) AND (H LE K) THEN GOTO, jumpendcase
      H=Y
      K=I
      IF (H GT 0) AND (-2*K LT H) AND (H LE K) THEN GOTO, jumpendcase
      H=Y
      K=X
      GOTO, jump531
     END
;
;     6/MMM
;
 32: BEGIN
      L=ABS(L)
      GOTO, jump29
     END
;
;     TRIGONAL - RHOMBOHEDRAL AXES
;
;     R3
;
 33: BEGIN
jump33:
      PRINT, " STOP ' RE-INDEX ON HEXAGONAL AXES'"
      STOP
     END
;
;     R-3
;
 34:  GOTO, jump33
;
;     R32
;
 35:  GOTO, jump33
;
;     R3M
;
 36:  GOTO, jump33
;
;     R-3M
;
 37:  GOTO, jump33
;
;     CUBIC
;
;     23
;
 38: BEGIN
      IF (H EQ K) AND (K EQ ABS(L)) THEN GOTO, jumpendcase
 jump538: 
      IF (H GT ABS(L)) AND (K GE ABS(L)) THEN GOTO, jumpendcase
      X=H
      Y=K
      Z=L
      H=-X
      K=-Y
      L=Z
      IF (H GT ABS(L)) AND (K GE ABS(L)) THEN GOTO, jumpendcase
      H=-X
      K=Y
      L=-Z
      IF (H GT ABS(L)) AND (K GE ABS(L)) THEN GOTO, jumpendcase
      H=X
      K=-Y
      L=-Z
      IF (H GT ABS(L)) AND (K GE ABS(L)) THEN GOTO, jumpendcase
      H=Z
      K=X
      L=Y
      GOTO, jump538
     END
;
;     M3
;
 39: BEGIN
      H=ABS(H)
      K=ABS(K)
      L=ABS(L)
      IF (H EQ K) AND (K EQ L) THEN GOTO, jumpendcase
 jump539: 
      IF (H GT L) AND (K GE L) THEN GOTO, jumpendcase
      X=H
      Y=K
      Z=L
      H=Z
      K=X
      L=Y
      IF (H GT L) AND (K GE L) THEN GOTO, jumpendcase
      H=Y
      K=Z
      L=X
      GOTO, jump539
     END
;
;     432
;
 40: BEGIN
      IF (ABS(H) NE ABS(K)) OR (ABS(K) NE ABS(L)) THEN GOTO, jump540
      H=ABS(H)
      K=ABS(K)
      L=ABS(L)
      GOTO, jumpendcase
 jump540: 
      IF (H GT L) AND (K GE L) AND (L GE 0) THEN GOTO, jumpendcase
      X=H
      Y=K
      Z=L
      H=Z
      K=X
      L=Y
      IF (H GT L) AND (K GE L) AND (L GE 0) THEN GOTO, jumpendcase
      H=Y
      K=Z
      L=X
      IF (H GT L) AND (K GE L) AND (L GE 0) THEN GOTO, jumpendcase
      H=-X
      K=-Y
      L=Z
      IF (H GT L) AND (K GE L) AND (L GE 0) THEN GOTO, jumpendcase
      H=Z
      K=-X
      L=-Y
      IF (H GT L) AND (K GE L) AND (L GE 0) THEN GOTO, jumpendcase
      H=-Y
      K=Z
      L=-X
      IF (H GT L) AND (K GE L) AND (L GE 0) THEN GOTO, jumpendcase
      H=-X
      K=Y
      L=-Z
      IF (H GT L) AND (K GE L) AND (L GE 0) THEN GOTO, jumpendcase
      H=-Z
      K=-X
      L=Y
      IF (H GT L) AND (K GE L) AND (L GE 0) THEN GOTO, jumpendcase
      H=Y
      K=-Z
      L=-X
      IF (H GT L) AND (K GE L) AND (L GE 0) THEN GOTO, jumpendcase
      H=X
      K=-Y
      L=-Z
      IF (H GT L) AND (K GE L) AND (L GE 0) THEN GOTO, jumpendcase
      H=-Z
      K=X
      L=-Y
      IF (H GT L) AND (K GE L) AND (L GE 0) THEN GOTO, jumpendcase
      H=-Y
      K=-Z
      L=X
      IF (H GT L) AND (K GE L) AND (L GE 0) THEN GOTO, jumpendcase
      H=Y
      K=X
      L=-Z
      GOTO, jump540
     END
;
;     -43M
;
 41: BEGIN
jump41: 
     IF (H GE K) AND (K GE ABS(L)) THEN GOTO, jumpendcase
      X=H
      Y=K
      Z=L
      H=Z
      K=X
      L=Y
      IF (H GE K) AND (K GE ABS(L)) THEN GOTO, jumpendcase
      H=Y
      K=Z
      L=X
      IF (H GE K) AND (K GE ABS(L)) THEN GOTO, jumpendcase
      H=-X
      K=-Y
      L=Z
      IF (H GE K) AND (K GE ABS(L)) THEN GOTO, jumpendcase
      H=Z
      K=-X
      L=-Y
      IF (H GE K) AND (K GE ABS(L)) THEN GOTO, jumpendcase
      H=-Y
      K=Z
      L=-X
      IF (H GE K) AND (K GE ABS(L)) THEN GOTO, jumpendcase
      H=-X
      K=Y
      L=-Z
      IF (H GE K) AND (K GE ABS(L)) THEN GOTO, jumpendcase
      H=-Z
      K=-X
      L=Y
      IF (H GE K) AND (K GE ABS(L)) THEN GOTO, jumpendcase
      H=Y
      K=-Z
      L=-X
      IF (H GE K) AND (K GE ABS(L)) THEN GOTO, jumpendcase
      H=X
      K=-Y
      L=-Z
      IF (H GE K) AND (K GE ABS(L)) THEN GOTO, jumpendcase
      H=-Z
      K=X
      L=-Y
      IF (H GE K) AND (K GE ABS(L)) THEN GOTO, jumpendcase
      H=-Y
      K=-Z
      L=X
      IF (H GE K) AND (K GE ABS(L)) THEN GOTO, jumpendcase
      H=Y
      K=X
      L=Z
      GOTO, jump41
     END
;
;     M3M
;
 42: BEGIN
      H=ABS(H)
      K=ABS(K)
      L=ABS(L)
      IF (H GE K) AND (K GE L) THEN GOTO, jumpendcase
      X=H
      Y=K
      Z=L
      H=Z
      K=X
      L=Y
      IF (H GE K) AND (K GE L) THEN GOTO, jumpendcase
      H=Y
      K=Z
      L=X
      IF (H GE K) AND (K GE L) THEN GOTO, jumpendcase
      H=Z
      K=Y
      L=X
      IF (H GE K) AND (K GE L) THEN GOTO, jumpendcase
      H=X
      K=Z
      L=Y
      IF (H GE K) AND (K GE L) THEN GOTO, jumpendcase
      H=Y
      K=X
      L=Z
      GOTO, jumpendcase
     END
;-----------------------------------------------------------------------
;
DEFAULT: BEGIN
jumpendcase:
    END

ENDCASE

IF arr_type EQ 1 THEN BEGIN
  HKLARR(0) = H
  HKLARR(1) = K
  HKLARR(2) = L
ENDIF ELSE BEGIN
  HKLARR(0,index) = H
  HKLARR(1,index) = K
  HKLARR(2,index) = L
ENDELSE
;
; End of loop over hkl_arr
;
ENDFOR

END
