pro completeness, rec, full, completeness, $
	histrec, histfull, histssq, histd, $
	minssq=minssq, maxssq=maxssq, no_bins=no_bins, par=par, $
	recsort=recsort, fullsort=fullsort, $
	drec=drec, ssqrec=ssqrec, dfull=dfull, ssqfull=ssqfull, $
	plot=plot, colors=colors, scale_colors=scale_colors, $
	barnames=barnames, baroffset=baroffset, $
	barwidth=barwidth, barspace=barspace, overplot=overplot, $
	title=title,xtitle=xtitle,ytitle=ytitle
;+
; NAME:
;	COMPLETENESS
;
; PURPOSE:
;	Calculate the completeness as a function of resolution and
;	possibly make a bar plot of it.
;
; CATEGORY:
;	Crystallography
;
; CALLING SEQUENCE:
; 	COMPLETENESS, rec, full, completeness, $
;	histrec, histfull, histssq, histd, $
;	minssq=minssq, maxssq=maxssq, no_bins=no_bins, par=par, $
;	recsort=recsort, fullsort=fullsort, $
;	drec=drec, ssqrec=ssqrec, dfull=dfull, ssqfull=ssqfull, $
;	plot=plot, colors=colors, scale_colors=scale_colors, $
;	barnames=barnames, baroffset=baroffset, $
;	barwidth=barwidth, barspace=barspace, overplot=overplot, $
;	title=title,xtitle=xtitle,ytitle=ytitle
;
; INPUTS:
;	REC: An array containing the recorded reflections. Should have
;		a tag M containing the Miller indices as an INTARR(3).
;	FULL: An array containing all the possible reflections. Should have
;		a tag M containing the Miller indices as an INTARR(3).
;	
;
; KEYWORDED PARAMETERS:
;	Optional input:
;	MINSSQ: Minimum ssq (=(1/2d)^2) of the histogram. Default: 0.
;	MAXSSQ: Maximum ssq (=(1/2d)^2) of the histogram. Default: max(Sfull).
;	NO_BINS: Number of bins in the histogram. Default: 10.
;	PAR: The crystal parameters. Structure containing the tags
;		a,b,c,alpha,beta,gamma. Angles in degrees.
;		Defaults to cubic with a=50.0 (Defaults defined in D_CALC.)
;	Output:
;	RECSORT,FULLSORT: REC and FULL sorted on ssq (=(1/2d)^2).
;	DREC,DFULL: The d-spacing for the reflections in RECSORT and FULLSORT
;		respectively.
;	SSQREC,SSQFULL: ssq (=(1/2d)^2) for the reflections in RECSORT and 
;		FULLSORT respectively.
;	Optional input:
;	PLOT: If set then a bar plot is made.
;	COLORS: If given the bars are colored according to this array of 
;		NO_BINS elements. Default all to 1.
;	SCALE_COLORS: The colours (as from COLORS) or are scaled from 0 to the 
;		maximum in COLORS. The maximum in COLORS will be given the 
;		colour index !D.N_COLORS-1.
;	BAROFFSET, BARWIDTH, BARSPACE, OVERPLOT, TITLE, XTITLE, YTITLE:
;		These keywords are passed on to BAR_PLOT which is used 
;		for doing the bar plot.
;
; OUTPUTS:
;	COMPLETENESS: The calculated completeness as a NO_BINS-element
;		vector. Expressed in percent. (100*HISTREC/HISTFULL)
;	HISTREC: The histogram of recorded reflections.
;	HISTFULL: The histogram of possible reflections.
;	HISTSSQ: The maximum value of ssq (=(1/2d)^2) of each bin.
;	HISTD: The minimum value of d of each bin. 
;	BARNAMES: The tickmarks that were used.
;
; COMMON BLOCKS:
;	None.
;
; SIDE EFFECTS:
;	If PLOT is set then a plot is made on the current graphical
;	device.
;
; RESTRICTIONS:
;	None
;
; PROCEDURE:
;	Straightforward
;
; MODIFICATION HISTORY:
;	Thomas Ursby, Sep 1995
;-

drec= D_CALC(rec.m, par=par)
ssqrec= 1/(2*drec)^2
dfull= D_CALC(full.m, par=par)
ssqfull= 1/(2*dfull)^2
i1= SORT(ssqrec)
i2= SORT(ssqfull)

IF NOT KEYWORD_SET(minssq) THEN minssq=0.0
IF NOT KEYWORD_SET(maxssq) THEN maxssq= MAX(ssqfull)
IF NOT KEYWORD_SET(no_bins) THEN no_bins=10 
binsize=(maxssq-minssq)/(no_bins-0.0001d)
IF NOT KEYWORD_SET(colors) THEN colors=FLTARR(no_bins)+1 
IF KEYWORD_SET(scale_colors) THEN $
	colors= fix((!d.n_colors-1.0)*float(colors)/max(colors))

histrec= HISTOGRAM(ssqrec(i1), binsize=binsize, min=minssq, max=maxssq)
histfull= HISTOGRAM(ssqfull(i2), binsize=binsize, min=minssq, max=maxssq)
completeness= 100*float(histrec)/(histfull>1)
histssq= minssq+binsize*(FINDGEN(no_bins)+1)
histd= 1/(2*sqrt(histssq))

recsort= rec(i1)
fullsort= full(i2)
drec= drec(i1)
dfull= dfull(i2)
ssqrec= ssqrec(i1)
ssqfull= ssqfull(i2)

IF KEYWORD_SET(plot) THEN BEGIN
  !y.range=[0.0,100.0]
  !x.range=[0,no_bins-1]
  barnames1=STRING(histd, format='(f4.2)')
  barnames= STRARR(no_bins) & barnames(*)=' '
  IF (no_bins gt 10) THEN $
	FOR i=0,no_bins-1,FIX(no_bins/10.0)+1 DO barnames(i)=barnames1(i) $
  ELSE barnames=barnames1
  IF NOT KEYWORD_SET(overplot) THEN $
    BAR_PLOT, completeness, barnames=barnames, $
	baroffset=baroffset, barwidth=barwidth, $
	barspace=barspace, title=title, xtitle=xtitle, ytitle=ytitle, $
	colors=colors $
  ELSE $
    BAR_PLOT, completeness, $
	baroffset=baroffset, barwidth=barwidth, $
	barspace=barspace, overplot=overplot, $
	colors=colors
  !y.range=[0,0]
  !x.range=[0,0]
ENDIF

END