;+ 
; 
;     ============================= XNLFit  ======================
; 
; 
; XNLfit is a graphical user interface to fit tabulated data with 
; a supplied (user-typed) model. It may use three non-linear minimization
; algorithms to minimize ChiSquare= Sum{ ([y-f(x;a)]^2)/(sigma^2)  }
; 
; The following methods are implemented: 
; 
;   i) MPFit: Performs bounded Levenberg-Marquardt least-squares
;     minimization (MINPACK-1) using the MPFit IDL routine written by
;     Craig B. Markwardt, NASA/GSFC Code 662, Greenbelt, MD 20770
;     http://astrog.physics.wisc.edu/~craigm/idl.html and distributed 
;     here (XOP/XNLFit) with permission of the author.
;
;   ii) CurveFit: Gradient-expansion algorithm (Marquardt) from the book 
;     Data Reduction and Error Analysis for the Physical Sciences, P.R. 
;     Bevington & D.K. Robinson, 2nd ed. McGraw-Hill, 1992 pag 162.
;
;   iii) Powell: The POWELL method is based on the routine 
;     powell described in section 10.5 of Numerical Recipes 
;     in C: The Art of Scientific Computing (Second Edition), published 
;     by Cambridge University Press.  
; 
; The user must define the fitting function in IDL syntax as an expression 
; of the independent variable x and a parameter array p.  For example a 
; straight line is : p[0]*x+p[1], a gaussian is 
; p[0]*exp( -((x-p[1])/p[2])^2/2) etc., etc.
; 
; The initial parameter guess must be provided in array notation:
; [p[0],p[1],p[2],...,p[n]]. It is also necessary to define the abscissas
; domain for the fit. By setting xmin=xmax, the program uses all the abscissas
; interval for the fit.
; 
; In the case that the  MPFIT or CURVEFIT are selected, it is possible to set
; flags that indicate which parameters are fixed (flag=0) or free (flag=1)
; during the fitting process. In the only case that MPFit is set, It is
; also possible to define boundaries for the parameter intervals. 
; 
; With CURVEFIT, the calculations are done in double precission if the
; model parameters are in double precision (i.e. a=[0.0D0,1,3]).  Remember
; to convert to double precision in subsequential runs, when wanted. Under
; Windows, the calculations are always done in double precision.
; 
; The sigma (error) values can be : 
;   i) Constant for all points
;   ii) Statistical noise (sigma = sqrt(y))
;   iii) rad from an external file (two column file or multicolumn file. 
;     In the last case the last column is considered.)
; 
; In addition, the user may select the option to calculate the covariance
; matrix on the model and fitted parameters.
; 
; The fitting equation and  current model parameters can be stored in a 
; file (XNLFit macro) which can be edit and reused. This macro defines 
; the following variables: equations, p. In addition a set of variables 
; a0,...,a10 can be used for intermediate calculations. 
; For example, the a file with the following text will be accepted:
;     ; this is a comment ',$
;     a0='p[0]*exp(-((x-p[1])/p[2])^2/2)'  ; gaussian
;     a1='+ p[3]*(p[5]/2)^2 /((x-p[4])^2+(p[5]/2)^2)' ; lorentzian
;     a2=[0.01,0,0.02]  ; gaussian model parameters
;     a3=[0.01,0,0.02]  ; lorentzian model parameters
;     equation = a0+a1
;     p = [a2,a3]
; 
; The current parameters (equation, guess, boundaries, etc.) can be saved
; (and then reloaded) in a parameters file. Optionally this parameter file
; can be used as default. 
; 
; Two option to read new data from a file and save the current data points
; and model in a file. 
; 
; Note on colors: XNLFit charges the XOP default color table when
; initialized. This color table consists in a black-and-white graded table
; (loadct,0) with 32 discrete colors in the first 32 indices (tek_color).
; Some color codes are used when for plots. If the user changes this colot
; table, the plots may apper blank. The default colors used are:
;
; Color  Index     Used for:
; 1     1-white    for the axes and data points outside the fitting interval.
; 2     2-red      for the data points in the interval.
; 3     3-green    for fitting results.
; 4     5-cyan     for model overplotted.
;       4-blue
;       6-magenta
;       8-orange, etc.
;
; These values can be changed using the "color sequence" entry in 
; the File/Preferences... menu.
; 
; 
; ====================Non-interactive use og XNLFit =====================
;
;	NAME:
;		XNLFIT
;
;	PURPOSE:
;		Widget interface to non-linear fits ( MPFit, CurveFit &
;		PowellFit).
;
;	CALLING SEQUENCE:
;		xnlfit,set (or xnlfit,x,y)
;
;	INPUT PARAMETERS:
;		x: it can either be 
;		   i) A data set (FltArr(2,npts))
;		   ii) A pointer to a data set 
;		   iii) An array (in this case define the optional y)
;
;	OPTIONAL INPUT PARAMETERS:
;		y: The ordinates array.
;
;	KEYWORD PARAMETERS:
;		GROUP = The widget ID of the widget that calls XPowellFit.
;		When this ID is specified, a death of the caller results 
;		in a death of PowellFit.
;		NODRAW = avoids creating a widget_draw for display.
;			It uses the current window as display.
;		XPLOTMAIN = Only for use from XPlot. This keyword
;			is the widget ID of the manin xplot (caller)
;			window.
;		PSYSVAR = a structure with the system variables to be 
;			applied when overplotting (to be used from xplot).
;		NO_BLOCK = When set, the NO_BLOCK keyword is passed to Xmanager.
;		INIT_PARS = Set this keyword to an array of strings
;			defining the initial values for the equation, model 
;			parameters, fix/free flags and min and max for the fit.
;			 For example:
;			tmp = ["equation = 'a(0)*exp(-((x-a(1))/a(2))^2/2)'",$
;				"a=[1.0,0,.2]",$
;				"free_flags=[1,1,1]",$
;				"xmin=0.0",$
;				"xmax=0.0"]
;			xnlfit,data,INIT_PARS=tmp
;			NOTE: this is not loaded in the case that a
;			default file xnlfit.xop exists containing 
;			customized inputs.
;		InputFile = a string cntaining the input file name.
;		
;	OUTPUTS:
;
;	COMMON BLOCKS:
;               COMMON powellfit,equation,x,y, sigma, xrange, chi2fit
;		(set PowellFit help for a description)
;
;	SIDE EFFECTS:
;		If not active, starts Xmanager
;
;	RESTRICTIONS:
;		The limitations of the provided methods.
;
;	PROCEDURE:
;		Uses the following non-linear fitting methods:
;		i) MPFit: Perform bounded Levenberg-Marquardt least-squares
;			minimization (MINPACK-1)
;		ii) CurveFit: Gradient-expansion algorithm (pages 237-239, 
;			Bevington, Data Reduction and Error Analysis for 
;			the Physical Sciences.)
;		iii) Powell: The POWELL method is based on the routine 
;			powell described in section 10.5 of Numerical Recipes 
;			in C: The Art of Scientific Computing (Second Edition), ;			published by Cambridge University Press.
;
;	MODIFICATION HISTORY:
;       	by  Manuel Sanchez del Rio. ESRF. 95-02-13
;		95-03-16 MSR adds statistic weight, covariance, external
;			input of sigma...
;		95-11-24 MSR creates XNLFIR based on XPOWELLFIT.
;		97-02-12 MSR changes the gaussian funct (it wasn't normalized)
;		97-09-08 MSR adds the NO_BLOCK keyword. Cosmetics in help.
;		97-10-22 MSR includes fix/free flags, load/save macro file, 
;			INIT_PARS kew and other changes. 
;		98-09-29 MSR includes psysvar kw.
;		99-04-08 MSR lots of improvements. Use of MPFit
;			(levenberg-Marqurdt). Allows boundaries. V 1.2.
;-
;
;========================================================================
;
FUNCTION XNLFit_Version
RETURN,'1.2'
END ; XNLFit_Version

;
;========================================================================
;
PRO XNLFit_clean,wid

; print,'In cleaning...'
Widget_Control,wid,GET_UVALUE=state,/No_Copy
IF Ptr_Valid(state.ptrSet) THEN Ptr_Free,state.ptrSet
IF Ptr_Valid(state.ptrExtendedSet) THEN Ptr_Free,state.ptrExtendedSet
END ; XNLFit_clean

;
;========================================================================
;

FUNCTION XNLFit_ReadSigma,sigmafile,Group=group
catch, error_status
if error_status ne 0 then begin
   ; commented because error managing is done by the caller if RET=0
   ;message,/info,'error caught: '+!err_string
   ;if sdep(/w) then itmp = Dialog_Message(/Error,$
   ;	Dialog_Parent=group,$
   ;	'XNLFIT_READSIGMA: error caught: '+!error_state.msg)
   ;catch, /cancel
   on_error,2
   RETURN,0
endif

if checkfile(sigmafile) NE 1 then Message,'File '+sigmafile+' does not exist.'

  	    print,'XNLFIT: Reading sigma from file: ',sigmafile
  	    ss = rascii(sigmafile)
if not(keyword_set(ss)) then Message,'File reading error'

nncol = n_elements(ss(*,0))-1
txt = ['XNLFIT: Reading sigma from last column ('+$
  StrCompress(nncol+1,/Rem)+') of the file: ',sigmafile]
Message,/Info,txt[0]+' '+txt[1]
itmp = Dialog_Message(/Info,Dialog_Parent=Group, txt)
ss=reform(ss(nncol,*))  ;last column

RETURN,ss
END ; XNLFit_ReadSigma


;
;========================================================================
;
PRO XNLFit_MapPanels,str,wids,Group=group

catch, error_status
if error_status ne 0 then begin
   message,/info,'error caught: '+!err_string
   if sdep(/w) then itmp = Dialog_Message(/Error,$
	'XNLFIT_MAPPANELS: error caught: '+!error_state.msg)
   catch, /cancel
   on_error,2
   RETURN
endif
;
; map for both MPFit and CurveFit
;
IF Fix(str.pref_method[0]) LE 1 THEN BEGIN
  widget_control,wids.FlagsBase,MAP=1 
  widget_control,wids.SigmaBase,MAP=1 
  IF Fix(str.SigmaValue[0]) EQ 2 THEN $
    widget_control,wids.SigmaFileBase,MAP=1 ELSE $
    widget_control,wids.SigmaFileBase,MAP=0
ENDIF ELSE BEGIN
  widget_control,wids.FlagsBase,MAP=0
  widget_control,wids.SigmaBase,MAP=0 
  widget_control,wids.SigmaFileBase,MAP=0
ENDELSE

;
; map for only MPFit
;
IF Fix(str.pref_method[0]) EQ 0 THEN $
  widget_control,wids.boundBase,MAP=1 ELSE $ 
  widget_control,wids.boundBase,MAP=0 

END ; XNLFit_MapPanels

;
;========================================================================
;
FUNCTION XNLFit_Defaults,Group=group,FromFile=fromFile 

catch, error_status
if error_status ne 0 then begin
   message,/info,'error caught: '+!err_string
   if sdep(/w) then itmp = Dialog_Message(/Error,$
	Dialog_Parent=group, $
	'XNLFIT_DEFAULTS: error caught: '+!err_string)
   catch, /cancel
   on_error,2
   RETURN,0
endif

fromFile=0

xnlfitstr = { SigmaValue:['0','1','Statistic','From file'], $
	SigmaFile:'xnlfit_sigma.dat',	$
	  pref_method:['0','Levenberg-Marquardt','CurveFit','Powell'],$
          pref_tol:1D-10, $
	  pref_maxIter:200L, $
	  pref_nPrint:5, $
	  pref_timeLimit:30.0, $
	  pref_color1:1, $
	  pref_color2:2, $
	  pref_color3:3, $
	  pref_color4:5, $
	  pref_color5:5, $
	equation:'a[0]*exp(-((x-a[1])/a[2])^2/2)',$
	guess:'[1.0,0,1]', $
	free_flags:'[1,1,1]',$
	limited1:'[0,0,0]', $
	limited2:'[0,0,0]', $
	limits1:'[0.0,0.0,0.0]', $
	limits2:'[0.0,0.0,0.0]'  }

;
;----------------------- WARNING !! -------------------------------
; If this section is modified, please check also dabax_defaults.pro
; If this section is modified, please check also xop_defaults.pro
;------------------------------------------------------------------
;

;
; Overwrite data with values from defaults files
; (same section in dabax_defaults, xop_defaults, xplot_defaults 
;  and xnlfit_defaults)
;
sep = sdep(/ds)
filedir = [Xop_GetEnv('XOP_HOME')+sep+'defaults'] 
filedir = [filedir]
IF Xop_GetEnv('XOP_DEFAULTS_DIR') NE '' THEN $
   filedir = [filedir,Xop_GetEnv('XOP_DEFAULTS_DIR')]

nfiles = n_elements(filedir)
if nfiles EQ 1 then filedir = [filedir]
tmp=0
for i=0,nfiles-1 do begin
  file = filedir[i]+sep+'xnlfit.xop'
  tmp = checkfile(file)
  if tmp EQ 1 then begin
    str_par = Xop_Input_Load(inputFile=file)
    ;IF N_Tags(out.parameters) NE 0 THEN BEGIN
    tmp = xnlfitstr
    Copy_Structure,str_par, tmp, /OnlyFirstField
    xnlfitstr = tmp
    message,/info,'Defaults from file: '+file+'
    fromFile=1
  endif
endfor

RETURN,xnlfitstr
END ; xnlfit_defaults

;
;========================================================================
;
FUNCTION XNLFit_UpdateStr,wids,str,Group=group
;
catch, error_status
if error_status ne 0 then begin
   message,/info,'error caught: '+!err_string
   if sdep(/w) then itmp = Dialog_Message(/Error,$
	'XNLFIT_UPDATESTR: error caught: '+!error_state.msg)
   catch, /cancel
   on_error,2
   RETURN,str
endif
newStr=str

tmp=''

Widget_Control,wids.SigmaValue,Get_Value=tmp
newStr.sigmaValue[0]=tmp[0]

Widget_Control,wids.sigmaFile,Get_Value=tmp
newStr.sigmaFile=tmp[0]

Widget_Control,wids.equation,Get_Value=tmp
newStr.equation=tmp[0]

Widget_Control,wids.guess,Get_Value=tmp
newStr.guess=tmp[0]

Widget_Control,wids.free_Flags,Get_Value=tmp
newStr.free_Flags=tmp[0]

Return,newStr
END ; XNLFit_UpdateStr

;
;========================================================================
;
PRO XNLFit_UpdatePanel,str,wids,Group=group
;
catch, error_status
if error_status ne 0 then begin
   message,/info,'error caught: '+!err_string
   if sdep(/w) then itmp = Dialog_Message(/Error,$
	'XNLFIT_UPDATEPANEL: error caught: '+!error_state.msg)
   catch, /cancel
   on_error,2
   RETURN
endif

Widget_Control,wids.SigmaValue,Set_Value=str.SigmaValue[0]
Widget_Control,wids.sigmaFile,Set_Value=str.sigmaFile
Widget_Control,wids.equation,Set_Value=str.equation
Widget_Control,wids.guess,Set_Value=str.guess
Widget_Control,wids.free_Flags,Set_Value=str.free_Flags

END ; XNLFit_UpdatePanel

;========================================================================
;
PRO xnlfit_plot,x,y,psysvar=psysvar,_extra=extra,over=over,set=set

IF Keyword_Set(set) THEN BEGIN
  x = set[0,*]
  y = set[N_Elements(set[*,0])-1,*]
ENDIF

if type(psysvar) EQ 8 then begin
  p_old=!p
  x_old=!x
  y_old=!y
  !p=psysvar.p
  !x=psysvar.x
  !y=psysvar.y
endif
IF Keyword_Set(over) THEN oplot,x,y,_extra=extra ELSE $
	plot,x,y,_extra=extra
if type(psysvar) EQ 8 then begin
  !p=p_old
  !x=x_old
  !y=y_old
endif
end ; xnlfit_plot

;
;============================================================================
;
PRO xnlfit_load,commands,wids,group=group
;
catch, error_status
if error_status ne 0 then begin
  message,/info,'error caught: '+!err_string
  itmp = Dialog_Message(/Error,Dialog_Parent=group,$
    'XNLFIT_LOAD: '+'error caught: '+!err_string)
  catch, /cancel
  return
endif

; initializa vars
equation = 0 & p=0 & free_flags=0 & xmin=0 & xmax=0
a0 = 0 & a1=0 & a2=0 & a3=0 & a4=0 & a5=0 
a6=0 & a7=0 & a8=0 & a9=0 & a10=0

; execute buffer
for i=0,n_elements(commands)-1 do begin
  itmp = execute(commands(i))
  if itmp NE 1 then begin
    itmp = Dialog_Message(/ERROR,widget_parent=$
    event.top,'Error executing '+commands(i))
  endif
endfor

;update panels
widget_control,wids.equation,set_value=equation
widget_control,wids.guess,set_value=vect2string(p)
; widget_control,wids.free_Flags,set_value=vect2string(free_flags)
; if xmin LT xmax then begin
;   widget_control,wids.xmin,set_value=xmin
;   widget_control,wids.xmax,set_value=xmax
; endif
END ; xnlfit_load

;
;============================================================================
;
PRO XNLFit_MPFit_Display, fcn, x, iter, FunctArgs=fcnargs, FMT=fmt, $
         Quiet=quiet, _Extra=iterargs

Forward_Function mpfit_enorm
; this routine is called by MPFit

Catch, error_status
IF error_status NE 0 THEN BEGIN
  Message,/Info,'error caught: '+!err_string
  itmp = Dialog_Message(/Error, $
    'XNLFit_MPFit_Display: error caught: '+!error_state.msg)
  Catch, /Cancel
  RETURN
ENDIF

IF Keyword_Set(quiet) THEN RETURN
fvec = Call_Function(fcn, x, _Extra=fcnargs)
fnorm = mpfit_enorm(fvec)

XDisplayfile1_Append, [String(iter, fnorm^2, $
  Format='("Iter ",I6,"   CHI-SQUARE = ",G20.8)'), $
  '           Timing [sec]:  '+$
	String(SysTime(1)-iterargs.t0,Format='(G6.2)')]
IF N_Elements(fmt) GT 0 THEN BEGIN
    XDisplayfile1_Append, String(p, format=fmt)
ENDIF ELSE BEGIN
    p = '  P('+StrTrim(LIndGen(N_Elements(x)),2)+') = ' $
      + StrTrim(String(x,Format='(G20.6)'),2) + '  '
    XDisplayfile1_Append, String('         '+p, Format='(A)')
ENDELSE

IF (SysTime(1)-iterargs.t0) GE iterargs.tmax THEN BEGIN
  itmp = Widget_Message(Dialog_Parent=iterargs.parent, $
	['Maximum time ('+StrCompress(Long(iterargs.tmax))+' sec) reached.',$
	 'Fitting process aborted.'])
  !err=-1
ENDIF
  
RETURN
END ; XNLFit_MPFit_Display

;
;============================================================================
;
PRO xnlfit_event, event
common powellfit, equation, x, y, sigma, xrange, chi2fit
common xplot_columnoper,str,commandhistory,commandhistoryi

catch, error_status
if error_status ne 0 then begin
  message,/info,'error caught: '+!err_string
  itmp = Dialog_Message(/Error,Dialog_Parent=event.top,$
    'XNLFIT_EVENT: '+'error caught: '+!err_string)
  catch, /cancel
  if n_elements(state) NE 0 then widget_control, event.top, $
	SET_UVALUE = state, /NO_COPY
  return
endif

; to correctly delete pointers when quitting 
IF Tag_Names(event, /Structure_Name) EQ 'WIDGET_KILL_REQUEST' THEN BEGIN
 Widget_Control,event.top,/Destroy
 RETURN
END

;   Widget_Control,event.id,GET_UVALUE=uval
Widget_Control,event.id,GET_UVALUE=uval

widget_control,event.top,GET_UVALUE=state,/NO_COPY

set = *(state.ptrSet)
x = set[0,*]
y = set[N_Elements(set[*,0])-1,*]

wset,state.window


IF keyword_set(uval) then BEGIN
  case uval of
    'Quit': begin
	;Ptrs are deleted using xnlfit_clean
	;IF Ptr_Valid(state.ptrSet) THEN Ptr_Free,state.ptrSet
	;
	; put state as UValue to be retrieved by xnlfit_clean
	Widget_Control, event.top, SET_UVALUE = state, /NO_COPY
	Widget_Control,event.top,/DESTROY
	RETURN
	end
    'Help': xhelp,'xnlfit',GROUP=event.top
    'Preferences': BEGIN
	str0=state.defs
	str = {pref_method:str0.pref_method, pref_tol:str0.pref_tol, $
	  pref_maxIter:str0.pref_maxIter, pref_nPrint:str0.pref_nPrint, $
	  pref_timeLimit:str0.pref_timeLimit,$
	  pref_color1:str0.pref_color1, pref_color2:str0.pref_color2, $
	  pref_color3:str0.pref_color3, pref_color4:str0.pref_color4, $
	  pref_color5:str0.pref_color5 }
	action=''
	flags=['1','1','1','w(0) EQ 0','w(0) EQ 0', $
		'1','1','1','1','0']
	titles=['Fitting method','Fitting tolerance', $
	  'Max of fitting iterations',$
	  'printing frequency','time limit [sec]',$
	  'Color 1 ','Color 2','Color 3','Color 4','Color 5']
	XScrmenu,str,/Interp,/NoType,Action=action,group=event.top, $
	  Flags=flags, Titles=titles, WTitle='XNLFit preferences', NCol=2
	IF action EQ 'DONT' THEN GoTo,out
	Copy_Structure,str,str0
	state.defs = str0
	XNLFit_MapPanels,str0,state.wids
	END
    'FileInput': BEGIN
	; update structure from widget values (for saving)
	tmp = XNLFit_Updatestr(state.wids,state.defs)
	state.defs=tmp

	action=''
	Widget_Control,event.id, Get_Value=action
	CASE action OF
	  'Load parameters from file...': BEGIN
	     filter='*.xop'
	     str_par = Xop_Input_Load(Title=$
	     'Select XNLFit input file...',$
	     /NoConf,Filter=filter,Group=event.top)
	     IF Type(str_par) EQ 8 THEN BEGIN
	       tmp = state.defs 
	       Copy_Structure,str_par, tmp,Group=event.top, /OnlyFirstField
	       state.defs = tmp
	       XNLFit_UpdatePanel,state.defs,state.wids
	       XNLFit_MapPanels,state.defs,state.wids
	     ENDIF
	     END
	  'Save parameters to file...': BEGIN
	     ; update structure from widget values
	     tmp = XNLFit_Updatestr(state.wids,state.defs)
	     state.defs=tmp

	     str_par = state.defs
	     Xop_Input_Save,str_par,File='xnlfit.xop',$
	       /Write, Group=event.top, Comment='; xop/xnlfit(v'+$
	       xnlfit_version()+') input file on '+SysTime()
	     END
	  'Save parameters as default': BEGIN
	     ; update structure from widget values
	     tmp = XNLFit_Updatestr(state.wids,state.defs)
	     state.defs=tmp

	     str_par = state.defs
	     Xop_Input_Save,str_par,Group=event.top, $
	       Default='xnlfit.xop',Comment='; xop/xnlfit(v'+$
	       xnlfit_version()+') input file on '+SysTime()
	     END
	  'Clean default parameters': BEGIN
	     file=Xop_GetEnv('XOP_DEFAULTS_DIR')+SDep(/ds)+'xnlfit.xop'
	     IF checkFile(file) NE 1 THEN GoTo,out
	     itmp = Dialog_Message(/Info,Dialog_Parent=event.top, $
		['Deleting file: '+file,'Please confirm'],/Cancel)
	     IF itmp EQ 'Cancel' THEN GoTo,out
             Delete_Files,[file],Group=event.top
	     END
	ENDCASE
	END
    'LoadData': BEGIN
	file = Dialog_PickFile(Dialog_Parent=event.top, /Read, $
	    file='xnlfit.dat')
	IF StrCompress(file,/Remove_All) EQ '' THEN GoTo,out
	a = RAscii(file)
	IF N_Elements(a) LE 1 THEN BEGIN
	  itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
	    ['File '+file+' does not contain valid data',$
	    'Operation aborted.'])
	    GoTo,out
	ENDIF
	x = Reform(a[0,*]) & y = Reform(a[N_Elements(a[*,0])-1,*])
	xmin = 0 & xmax = 0
	xmin = Min(x,Max=xmax)
	Widget_Control,state.wids.xmin, Set_Value=xmin
	Widget_Control,state.wids.xmax, Set_Value=xmax
	set = Make_Set(x,y)
	*(state.ptrset)=set
	xnlfit_plot,x,y,psysvar=state.psysvar,color=state.defs.pref_color1
	xnlfit_plot,x,y,psysvar=state.psysvar,color=state.defs.pref_color2,/Over
	END
    'SaveData': BEGIN
	file = ''
	WHILE file EQ '' DO BEGIN
	  file = Dialog_PickFile(Dialog_Parent=event.top, /Write, $
	    file='xnlfit.dat')
	  IF file EQ '' THEN GoTo,out
	  IF CheckFile(file) THEN BEGIN
	    itmp = Dialog_Message(/Question,Dialog_Parent=event.top, $
	      ['File '+file+' already exists.','Overwrite it?'])
	    IF itmp EQ 'No' THEN file = ''
	  ENDIF
	ENDWHILE

	; update structure from widget values
	tmp = XNLFit_Updatestr(state.wids,state.defs)
	state.defs=tmp

	p = 0 & y2 = 0
	command = 'p='+state.defs.guess
	Message,/Info,'Executing command: '+command  
	error = not( execute(command) )
	if error then begin
  	  tmp  = Dialog_Message(Dialog_Parent=event.top, $
		'Error defining initial parameters.',/ERROR)
	  goto,out
	endif
	command = 'y2='+equation
	Message,/Info,'Executing command: '+command  
	error = not( execute(command) )
	if error then begin
	  tmp  = Dialog_Message(Dialog_Parent=event.top, $
		'Error executing function.',/ERROR)
	  goto,out
	endif

	OpenW,unit,file,/get_Lun
	Printf,unit,'#F '+file
	Printf,unit,'#S 1 data from Xplot/XNLFit'
	Printf,unit,'#C '
	Printf,unit,'#C File created by XNLFit on '+SysTime()
	Printf,unit,'#C '
	Printf,unit,'#C Model function: '+state.defs.equation
	Printf,unit,'#C P parameters: '+state.defs.guess
	Printf,unit,'#N 3'
	Printf,unit,'#C '
	Printf,unit,'#L X  YModel  Y'
	FOR i=0L,N_Elements(x)-1 DO $
	  PrintF,unit,x[i],y2[i],y[i]
	Free_Lun,unit
	Message,/Info,'File '+file+' written to disk.'
	END
    'Reset': begin
	tmp0 = 0
	widget_control,state.wids.guess,get_value=tmp0
	nn = 0
	itmp = execute('nn = n_elements('+tmp0(0)+')')
	if itmp  NE 1 then begin
	   itmp = Dialog_Message(/ERROR,Dialog_Parent=event.top,$
	   'Error freeing all parameters. Set Model Parameters before.')
	   goto,out
	endif
	; free_flags
	tmp = vect2string(replicate(1,nn))
	widget_control,state.wids.free_Flags,set_value=tmp
	state.defs.free_flags=tmp
	; boundaries
	tmp = vect2string(replicate(0,nn))
        state.defs.limited1=tmp
        state.defs.limited2=tmp
        state.defs.limits1=Float(tmp)
        state.defs.limits2=Float(tmp)
	end

    'Bound': BEGIN

	str = {limited1:state.defs.limited1, limited2:state.defs.limited2, $
		limits1:state.defs.limits1, limits2:state.defs.limits2 }
	action=''
	flags=['1','1','1','1']
	titles=['Lower bound? (1=Yes, 0=No)',$
	  'Upper  bound? (1=Yes, 0=No)',$
	  'Lower boundary values',$
	  'Upper boundary values' ]
	XScrmenu,str,/Interp,/NoType,Action=action,group=event.top, $
	  Flags=flags, Titles=titles, WTitle='XNLFit boundaries', $
	  Dialog_Parent=event.top

	IF action EQ 'DONT' THEN GoTo,out

	str0 = state.defs
	Copy_Structure,str,str0
	state.defs = str0
	END


    'write': begin
	gfile = Dialog_Pickfile(Dialog_Parent=event.top, FILE='xnlfit.mac',$
	  Title='Select a file for writing XNLFit macro')
	if (gfile eq '') then goto,out
	if checkfile(gfile) then begin
	   itmp = Dialog_Message(/Question,dialog_parent=event.top,$
	    ['File exists: '+gfile,'Overwrite it?'])
	   if itmp EQ 'No' then begin
	     itmp = Dialog_Message(dialog_parent=event.top,$
	      'Writing file operation aborted.')
	     goto,out
	   endif
	endif
	openw,unit,gfile,/get_lun
	printf,unit,';File created by XOP/Xplot/XNLFit. '+$
		'Can be edited and reloaded.'
	printf,unit,"equation='"+state.defs.equation+"'"
	printf,unit,"p="+state.defs.guess
	; printf,unit,"free_flags="+state.defs.free_flags

	; widget_control,state.wids.xmin,get_value=tmp0
	; printf,unit,"xmin="+strcompress(tmp0(0),/rem)
	; widget_control,state.wids.xmax,get_value=tmp0
	; printf,unit,"xmax="+strcompress(tmp0(0),/rem)
	free_lun,unit
	end
    'load': begin
	if sdep() eq 'UNIX' then filter='*.mac' else filter=0
	gfile = Dialog_Pickfile(Dialog_Parent=event.top, FILTER=filter,$
	  Title='Select a file for reading XNLFit macro')
	if (gfile eq '') then goto,out
	commands = read_textfile(gfile)
	xnlfit_load,commands,state.wids,group=event.top
	tmp = XNLfit_UpdateStr(state.wids,state.defs)
	state.defs=tmp
	end
    'Xplotaddcolumn': begin
	; add column cased on the extended set
	set = *(state.ptrExtendedSet)
	x = set[0,*]
	y = set[N_Elements(set[*,0])-1,*]
	;
        widget_control,state.wids.guess,get_val = guess
        guess = guess(0)
        a = 0
        y2 = 0
        command = 'p='+guess
        Message,/Info,'Executing command: '+command  
	error = not( execute(command) )
        if error then begin
          tmp  = Dialog_Message(DIALOG_PARENT=event.top, $
	  'Error defining initial parameters.',/ERROR)
          goto,out
        endif
        widget_control,state.wids.equation,get_val = equation
        equation = equation(0)
        command = 'y2='+equation
        Message,/Info,'Executing command: '+command  
	error = not( execute(command) )
        if error then begin
          tmp  = Dialog_Message(DIALOG_PARENT=event.top, $
	  'Error executing function.',/ERROR)
          goto,out
        endif

	;widget_control,state.wids.xplot,get_uvalue = xplotstate,/no_copy
	xplot_addcol,state.wids.xplot,y2,group=event.top
	;widget_control,state.wids.xplot,set_uvalue = xplotstate,/no_copy

	end
    'Fit': begin
	; update structure from widget values
	tmp = XNLFit_Updatestr(state.wids,state.defs)
	state.defs=tmp

	guess = state.defs.guess
        widget_control,state.wids.xmin,GET_VALUE=xmin
        widget_control,state.wids.xmax,GET_VALUE=xmax
	equation = state.defs.equation
	if strcompress(guess,/rem) EQ '' or $
		strcompress(equation,/rem) EQ '' then begin
		tmp = Dialog_Message(DIALOG_PARENT=event.top, $
		/ERROR,'Error in inputs.')
		goto,out
	endif
	;
	; prepare Fit call
	;
	p = 0
	error = not( execute('p='+guess) )
	if error then begin
  	tmp  = Dialog_Message(DIALOG_PARENT=event.top, $
		['Error defining initial parameters.', $
		'Non-linear fit FAILED.  '],/ERROR)
  		goto,out
	endif 
	xrange=[xmin,xmax]
	;
	; sigma values
	;

	CASE Fix(state.defs.SigmaValue[0]) OF
	  0: BEGIN
	    sigma=reform(y*0.+1.)
	    errtxt = 'Using constant errors'
	    END
	  1: BEGIN
	    sigma=reform(sqrt(y))
	    errtxt = 'Using statistical errors'
	    END
	  2: BEGIN
  	    widget_control,state.wids.SigmaFile,get_value=sigmafile
	    sigmafile = sigmafile(0)
	    sigma = XNLFit_ReadSigma(sigmafile,Group=event.top)
  	    if n_elements(sigma) EQ 1 then begin
	      tmp  = Dialog_Message(/Error,DIALOG_PARENT=event.top, $
		'XNLFIT_EVENT: '+!error_state.msg)
	      goto,out
  	    endif
  	    if n_elements(sigma) NE n_elements(x) then begin
	      tmp  = Dialog_Message(/Error,DIALOG_PARENT=event.top, $
		'XNLFIT_EVENT: Fit aborted: Sigma File dimensions ('+$
		  StrCompress(n_elements(sigma),/Rem)+$
		  ') different from number of points ('+$
		  StrCompress(n_elements(x),/Rem)+')' )
	      goto,out
  	    endif
	    errtxt = 'Using errors (1*sigma) from file: '+sigmafile
	  END
	ENDCASE

	; call fit
	widget_control,event.id,get_value=vval,/hourglass
	iter = 0
	itmax = state.defs.pref_maxIter
	tol = state.defs.pref_tol
	tmax = state.defs.pref_timeLimit

	case Fix(state.defs.pref_method[0]) of 
	2: begin   ; POWELL
		fittype = 'Fitting method: Powell'
		message,/info,'XNLLFIT: calling powellfit...'
		powellfit,p,yfit=y2,error=error, iter=iter, $
		 itmax=itmax  ;,statsigma=statsigma,/show
		message,/info,'XNLFIT: back from powellfit.'
		if error then begin
		  tmp = Dialog_Message(DIALOG_PARENT=event.top, $
			'Powellfit error found',/ERROR)
		  goto,out
		endif
		chi2tmp = chi2fit
		end
	1: begin ; CURVEFIT1
        	widget_control,state.wids.free_Flags,GET_VALUE=flags
		fita = 0
		itmp = execute('fita = '+flags(0))
		if itmp NE 1 then begin	
		  itmp = Dialog_Message(/Error,dialog_parent=event.top,$
		  'Error in Free/Fix parameters defintion.')
		  goto,out
		endif
		if n_elements(fita) NE n_elements(p) then begin
		  itmp = Dialog_Message(/Error,dialog_parent=event.top,$
		  'Free/Fix parameters dimension ('+strcompress($
		  n_elements(fita),/rem)+$
		  ') must be the same as Model Parameters dimension ('+$
		  strcompress(n_elements(p),/rem)+').' )
		  goto,out
		endif
		; this is to speed the curvefit1 in case that all flags are 1
		if (where(fita NE 1))(0) EQ -1 then fita=0
		fittype = 'Fitting method: Curvefit'
		message,/info,'XNLFIT: calling curvefit1...'
		y2 = 0
		x = reform(x) & y = reform(y)
		w = 1./sigma^2
		; bug fixed 99=04=05 use w instead sigma
		y2 = curvefit1(x,y,w,p,sigmaai,chi2=chi2i, $
			Function_name=equation,iter=iter,tol=tol,$
			itmax=itmax-1,fita=fita)
		message,/info,'XNLFIT: back from curvefit1.'
		chi2tmp = chi2i
		Print,'XNLFIT_EVENT: Sigmaai: ',Sigmaai
		error = 0
		end
	0: BEGIN ; MPFit
        	widget_control,state.wids.free_Flags,GET_VALUE=flags
		fita = 0
		itmp = execute('fita = '+flags(0))
		if itmp NE 1 then begin	
		  itmp = Dialog_Message(/Error,dialog_parent=event.top,$
		  'Error in Free/Fix parameters defintion.')
		  goto,out
		endif
		if n_elements(fita) NE n_elements(p) then begin
		  itmp = Dialog_Message(/Error,dialog_parent=event.top,$
		  'Free/Fix parameters dimension ('+strcompress($
		  n_elements(fita),/rem)+$
		  ') must be the same as Model Parameters dimension ('+$
		  strcompress(n_elements(p),/rem)+').' )
		  goto,out
		endif

		fittype = 'Fitting method: '+$
		  'MPFit (Levenberg-Marquardt [MINPACK-1])'
		message,/info,'XNLFIT: calling MPFit...'
		y2 = 0
		x = reform(x) & y = reform(y)

		parinfo = replicate({value:0.D, fixed:0, limited:[0,0], $
		   limits:[0.D,0]}, N_Elements(p) )
		parinfo(*).value = p
		parinfo(*).fixed = not(fita)+2


		limited = IntArr(2,N_Elements(p))
		limits = FltArr(2,N_Elements(p))
		;IF N_Elements(p) NE Total(fita) THEN BEGIN
		  itmp1 = execute('limited[0,*] = '+state.defs.limited1) 
		  itmp2 = execute('limited[1,*] = '+state.defs.limited2) 
		  itmp3 = execute('limits[0,*] = '+state.defs.limits1) 
		  itmp4 = execute('limits[1,*] = '+state.defs.limits2) 
		  IF itmp1 NE 1 OR itmp2 NE 1 OR itmp3 NE 1 OR itmp4 NE 1 THEN $
			BEGIN	
		    itmp = Dialog_Message(/Error,dialog_parent=event.top,$
		    'Error in defintion of boundaries. Check dimensions.')
		    goto,out
		  ENDIf
		;ENDIF
		parinfo(*).limited = limited
		parinfo(*).limits = limits

		covar=0  &  status=0  &  pError = 0

		Widget_Control,/HourGlass
		if xrange(0) EQ xrange(1) then xrangetxt = 'all' else $
		  xrangetxt=vect2string(xrange)
		XDisplayfile1,Text=['',$
		  'Running MPFit (Levenberg-Marquardt [MINPACK-1]) on '+$
		    SysTime(),'',$
		  'Function Expression: '+equation, errtxt, $
		  'Initial parameters: '+guess,$
		  'Fitting range: '+xrangetxt,$
		  'Fixed params (0=free,1=fix):'+Vect2String(parinfo(*).fixed),$
		  'Lower bounds (0=No,1=Yes):'+$
			Vect2String(parinfo(*).limited[0]),$
		  'Upper bounds (0=No,1=Yes):'+$
			Vect2String(parinfo(*).limited[1]),$
		  'Lower limits:'+Vect2String(parinfo(*).limits[0]),$
		  'Upper limits:'+Vect2String(parinfo(*).limits[1]),$
		  '',''],$
		  Group=event.top, Title='Non linear fitting results'

		r = mpfitexpr(equation, x, y, sigma, ParInfo=parInfo, $
		  FTol=tol, XTol=tol, GTol=tol, $
		  NPrint=state.defs.pref_nprint, MaxIter=itMax, $
		  IterProc='xnlfit_mpfit_display',Status=status, $
		  NIter=iter, Covar=covar, PError=pError,  $
		  IterArgs={t0:systime(1), tmax:tmax, parent:event.top} )

		CASE status OF
		  0:  text='improper input parameters.'
		  1:  text=['both actual and predicted relative reductions',$
		    'in the sum of squares are at most FTOL(=TOL).']
		  2:  text=['relative error between two consecutive iterates',$
		    'is at most XTOL(=TOL)']
		  3:  text=['both actual and predicted relative reductions',$
		    'in the sum of squares are at most FTOL(=TOL). And, ',$
		    'relative error between two consecutive iterates',$
		    'is at most XTOL(=TOL)']
		  4:  text=['the cosine of the angle between fvec and any ',$
		    'column of the jacobian is at most GTOL(=TOL) in absolute',$
		    'value.']
		  5:  text='the maximum number of iterations has been reached'
		  6:  text=['FTOL(=TOL) is too small. no further reduction in',$
		    'the sum of squares is possible.']
		  7:  text=['XTOL(=TOL) is too small. no further improvement ',$
		    'in the approximate solution x is possible.']
		  8:  text=['GTOL(=TOL) is too small. fvec is orthogonal to ',$
		    'the columns of the jacobian to machine precision.']
		  else: text='Unknown.'
		ENDCASE


		XDisplayFile1_Append,['','MPFit termination status: ',text,$
		  '','Number of iterations done: '+StrCompress(iter),$
		  '','Resulting params: '+Vect2String(r),$
		  '','Errors on resulting params: '+Vect2String(pError),$
		  '', $
		  'Fitting result overplots the original '+$
		  'graph with triangular symbols.',' ' ]

		p=r
		y2=0
		command = 'y2='+equation
		Message,/Info,'Executing command: '+command
		error = not( execute(command) )
		if error then begin
		  tmp  = Dialog_Message(DIALOG_PARENT=event.top, $
		  'Error executing function.',/ERROR)
		  goto,out
		endif


		; chi2tmp='?????'
		message,/info,'XNLFIT: back from MPFit.'
		END
	else:
	endcase
	
	;
	; sort the outputs
	;
	if xrange(0) EQ xrange(1) then xrangetxt = 'all' else $
		xrangetxt=vect2string(xrange)

	if error then begin
	  tmp = Dialog_Message(DIALOG_PARENT=event.top, $
		' Non-linear fit FAILED. ',/ERROR)
	  goto,out
	endif

	xnlfit_plot,x,y2,psym=-5,psysvar=state.psysvar,/Over,color= $
	  state.defs.pref_color3

	IF Fix(state.defs.pref_method[0]) GT 0 THEN BEGIN
	  txt = ['Fitting result overplots the original '+$
	   'graph with triangular symbols.',' ', $
	   fittype,' ',$
	   'Fitting tolerance: '+ strcompress(tol,/rem), ' ', $
	   'Fitting iterations: '+ strcompress(iter,/rem), ' ', $
	   'Fitting iterations limit: '+ strcompress(itmax,/rem), ' ', $
	   'fitting equation: '+ equation, ' ', $
	   'fitting parms: '+ vect2string(p), ' ', $
	   'fitting range: '+ xrangetxt, ' ', $
	   'Chi2: '+strcompress(chi2tmp,/rem),' ' ]
	  XDisplayFile1,TITLE='Fitting results',TEXT=txt,GROUP=event.top
	ENDIF

	tmp = Dialog_Message(DIALOG_PARENT=event.top,/QUESTION, $
	  'Do you want to update parameters to fitting results?')
	IF tmp EQ 'Yes' then BEGIN
	  tmp = vect2string(p)
	  widget_control,state.wids.guess, set_value = tmp
	  state.defs.guess=tmp
	ENDIF

	END
    'PlotModel': begin
	widget_control,state.wids.guess,get_val = guess
	guess = guess(0)
	widget_control,state.wids.equation,get_val = equation
	equation = equation(0)

	p = 0 & y2 = 0

	command = 'p='+guess
	Message,/Info,'Executing command: '+command  
	error = not( execute(command) )
	if error then begin
  	  tmp  = Dialog_Message(DIALOG_PARENT=event.top, $
		'Error defining initial parameters.',/ERROR)
	  goto,out
	endif
	command = 'y2='+equation
	Message,/Info,'Executing command: '+command  
	error = not( execute(command) )
	if error then begin
	  tmp  = Dialog_Message(DIALOG_PARENT=event.top, $
		'Error executing function.',/ERROR)
	  goto,out
	endif

	widget_control,event.id,get_value = tmp
	case tmp of
	'Covariance': begin
	  ;
	  ; sigma values
	  ;
	  covar_fit,x,y,p,equation,covar=covar ,sigma=sigma, $
	  outsigma=outsigma ;,statsigma=statsigma
  	  nco = n_elements(covar(0,*))
  	  atext = strarr(nco)
	  for i=0,nco-1 do atext(i)=vect2string(covar(*,i))
	  txt = [ ' Equation: y = '+equation, $
	   'Model parameters: p = '+vect2string(p), $
	   'xrange = all', ' ',$
	   'Sigma of the model parameters: '+vect2string(outsigma), $
	   'covariance matrix:', atext]
	  ;ShowInfo1,TITLE='Covariance Calculations', $
	  ; INFOTEXT=txt, WIDTH=max(strlen(txt)), HEIGH=n_elements(txt), $
	  ; GROUP=event.top
	  XDisplayfile1,TITLE='Covariance Calculations', $
	   TEXT=txt,GROUP=event.top
	  end
	'(X)Plot model': xplot,x,y2,title=equation+', p='+vect2string(p),$
		No_Block=state.no_block
	'Overplot model': xnlfit_plot,x,y2,psym=-4,psysvar=state.psysvar, $
		/Over,Color=state.defs.pref_color4
	endcase
	end
    'Equation': begin
	widget_control,event.id,get_value = evval
	;widget_control,state.wids.Equation,get_value=Equation
	equation = state.defs.equation
	Equation = Equation(0)
	case evval of
	  'Clear': begin
		Equation = ''
		state.npar = 0
		end
	  'Add Gaussian': begin	
		npar = state.npar
		state.npar = npar + 3
		if npar eq 0 then sign = '' else sign = '  +'
		a0 = 'p['+strcompress(npar+0,/rem)+']'
		a1 = 'p['+strcompress(npar+1,/rem)+']'
		a2 = 'p['+strcompress(npar+2,/rem)+']'
		Equation = Equation + sign + a0+'*exp(' +$
		 ;'-((x-'+a1+')/'+a2+')^2/'+a2+')'
		 '-((x-'+a1+')/'+a2+')^2/2)'
		end
	  'Add Lorentzian': begin	
		npar = state.npar
		state.npar = npar + 3
		if npar eq 0 then sign = '' else sign = '  +'
		a0 = 'p['+strcompress(npar+0,/rem)+']'
		a1 = 'p['+strcompress(npar+1,/rem)+']'
		a2 = 'p['+strcompress(npar+2,/rem)+']'
		;Equation = Equation + sign + a0+'*'+a2+' /(' +$
		 ;'(x-'+a1+')^2+'+a2+')'
		Equation = Equation + sign + a0+'*('+a2+'/2)^2 /(' +$
		 '(x-'+a1+')^2+('+a2+'/2)^2)'
		end
	  'Add Polynomial': begin	
		strtmp = { n:1 }
		XscrMenu,strtmp,/NoType,Titles='Polynomial Degree', $
		  Action=action, Dialog_Parent=event.top, Wtitle='Polynomial'
		if action ne 'DO' then goto,out
        	deg = strtmp.n
		npar = state.npar
		state.npar = npar + deg + 1
		if npar eq 0 then sign = '' else sign = '  +'
		Equation = Equation + sign 
		for i=0,deg-1,+1 do Equation = Equation+ $
		  ' X^'+strcompress(i,/rem)+ $
		  '*p['+strcompress(npar+i,/rem)+']+'
		Equation = Equation +  $
		 'X^'+strcompress(deg,/rem) + $
		 '*p['+strcompress(npar+deg,/rem)+ ']'
		end
	  endcase
	  widget_control,state.wids.Equation, set_value = Equation
	  state.defs.equation=equation
	  ;state.fpar.Equ = Equation
	  end
    'Guess': begin
	widget_control,event.id,get_value = evval
	case evval(0) of
	  'Clear parameters': begin
		Guess = '[]'
		end
	  'Template': begin	
		strtmp = { n:state.npar }
		XscrMenu,strtmp,/NoType,Titles='Number of parameters:', $
			Action=action,Wtitle='Template',Dialog_Parent=event.top
		if action ne 'DO' then goto,out
		if strtmp.n eq 0 then Guess = '[]' else Guess = $
		  vect2string(lonarr(strtmp.n))
		end
	  else: guess = evval(0)
	  endcase
	  widget_control,state.wids.Guess,set_value=guess
	  state.defs.guess=guess
	end
    'PlotData': begin
	if (event.id eq state.wids.allinterval) then begin
	  widget_control,state.wids.xmin,SET_VALUE=0.0 ; x(0)
	  widget_control,state.wids.xmax,SET_VALUE=0.0 ; x(0)
	  xmin = 0.0  &  xmax=0.0
	endif
	widget_control,state.wids.xmin,GET_VALUE=xmin
	widget_control,state.wids.xmax,GET_VALUE=xmax

	if xmin NE xmax then begin
	    tmp1 = fltarr(2,n_elements(x))
	    tmp1(0,*) = reform(x)  &  tmp1(1,*) = reform(y)
	    if xmin LE xmax then xrange=[xmin,xmax] else xrange=[xmax,xmin]
	    cutset,tmp1,tmp2,xrange=xrange
	    if state.wids.xplot EQ 0 then $
	      plot,x,y,color=state.defs.pref_color1 else  begin
             widget_control,state.wids.xplot,get_uvalue = xplotstate
             xplot_plot,xplotstate
	    endelse
	    xnlfit_plot,tmp2(0,*),tmp2(1,*),color=state.defs.pref_color2,$
	      psysvar=state.psysvar,/Over
	endif else begin
	    xrange=[xmin,xmax]
	    if state.wids.xplot EQ 0 then plot,x,y,/nodata,$
	      color=state.defs.pref_color1 else begin
              widget_control,state.wids.xplot,get_uvalue = xplotstate
              xplot_plot,xplotstate
	    endelse
	    xnlfit_plot,x,y,color=state.defs.pref_color2,psysvar=state.psysvar,$
		/Over
	endelse
	end
    'ColorTable': BEGIN
	tmp=''
	Widget_Control,event.id,Get_value=tmp
	CASE tmp[0] OF
	  'Set Default': BEGIN
		LoadCT,0
		Tek_Color
		END
	  'Change table...': XLoadCT,Group=event.top
	ENDCASE
	END
    'SigmaFileBrowser': Begin
	tmp = Dialog_Pickfile(Dialog_Parent=event.top,/READ,/NOCO)
	if  strcompress(tmp,/REM) EQ '' then goto,out
	widget_control,state.wids.SigmaFile,set_val=tmp
	end
    'EXTRACT': Begin
        widget_control,state.wids.xmin,GET_VALUE=xmin
        widget_control,state.wids.xmax,GET_VALUE=xmax
        if xmin NE xmax then begin
            tmp1 = fltarr(2,n_elements(x))
            tmp1(0,*) = reform(x)  &  tmp1(1,*) = reform(y)
            if xmin LE xmax then xrange=[xmin,xmax] else xrange=[xmax,xmin]
            cutset,tmp1,tmp2,xrange=xrange
	    xplot,tmp2,wtitle='Fitting data in '+vect2string(xrange)
	endif else xplot,x,y,wtitle='Fitting data in all interval'
	end
    'SIGMAVALUE': begin
        tmp = XNLfit_UpdateStr(state.wids,state.defs)
        state.defs=tmp
	XNLFit_MapPanels,state.defs,state.wids
        end
    'UPDATE': begin
        tmp = XNLfit_UpdateStr(state.wids,state.defs)
        state.defs=tmp
        end
     else:
  endcase
ENDIF

out:
widget_control,event.top,SET_UVALUE=state,/NO_COPY

end  ; xnlfit_event

;
;============================================================================
;
PRO xnlfit, input, optinp, group = group, nodraw=nodraw, $
 xplotmain=xplotmain, no_block=no_block,init_pars=init_pars, $
 psysvar=psysvar, InputFile=inputFile, ExtendedSet=extendedSet


catch, error_status
if error_status ne 0 then begin
   message,/info,'error caught: '+!err_string
   if sdep(/w) then itmp = Dialog_Message(/Error,Dialog_Parent=group, $
	'XNLFIT: error caught: '+!err_string)
   catch, /cancel
   on_error,2
   RETURN
endif

loadct,0
tek_color

IF N_Elements(no_block) EQ 0 THEN no_block=0

CASE n_params() OF
  0: BEGIN  ; No inputs 
    x = FindGen(100)
    y = FindGen(100)^2
    set = Make_Set(x,y)
    ptrSet = Ptr_New(set)
    END
  1: BEGIN ; One input: either s set or a pointer to set
    IF Type(input) EQ 10 THEN BEGIN ; pointer
      ptrset = input
      set = *(ptrSet)
      x = Reform(set[0,*])
      y = Reform(set[N_Elements(set[*,0])-1,*])
    ENDIF ELSE BEGIN
      set = input
      ptrSet=Ptr_New(set)
      x = Reform(set[0,*])
      y = Reform(set[N_Elements(set[*,0])-1,*])
    ENDELSE
    END
  2: BEGIN ; Two inputs: x & y arrays
    x = reform(set)
    y = reform(optinp)
    set = Make_Set(x,y)
    ptrSet=Ptr_New(set)
    END
ENDCASE

xmax = max(x,min=xmin)

fromFile=0
defs = xnlfit_defaults(FromFile=fromFile)
;
IF KeyWord_Set(inputFile) THEN BEGIN
  str_par = Xop_Input_Load(InputFile=inputFile)
  IF Type(str_par) EQ 8 THEN BEGIN
    tmp = defs
    Copy_Structure,str_par, tmp, Group=group
    defs = tmp
  ENDIF
ENDIF


; 
; Widget definition
;
wWin = Widget_base(TITLE='XNLFit (Non-linear fitting) '+$
	xnlfit_version(),MBAR=wMenu, /TLB_KILL_REQUEST_EVENTS, UValue='')
;
; define menu bar
;
wtmp0 = WIDGET_BUTTON(wMenu, VALUE='File', /MENU)

 wtmp1 = widget_button(wtmp0,VALUE='Load parameters from file...', $
	UValue='FileInput')
 wtmp1 = widget_button(wtmp0,VALUE='Save parameters to file...', $
	UValue='FileInput')
 wtmp1 = widget_button(wtmp0,VALUE='Save parameters as default', $
	UValue='FileInput')
 wtmp1 = widget_button(wtmp0,VALUE='Clean default parameters', $
	UValue='FileInput')

 wtmp1 = WIDGET_BUTTON(wtmp0, VALUE='Load new data...', UVALUE='LoadData',$
	/Separator)
 wtmp1 = WIDGET_BUTTON(wtmp0, VALUE='Save data and model...', UVALUE='SaveData')

 wtmpb = WIDGET_BUTTON(wtmp0, VALUE='XNLFit Macro File', /Menu,/Separator)
   wtmp1 = WIDGET_BUTTON(wtmpb, VALUE='Load...', UValue='load')
   wtmp1 = WIDGET_BUTTON(wtmpb, VALUE='Write...', UValue='write')


 wtmp1 = WIDGET_BUTTON(wtmp0, VALUE='Preferences...', UVALUE='Preferences',$
	/Separator)
 wtmp1 = WIDGET_BUTTON(wtmp0, VALUE='Quit', UVALUE='Quit',/SEPARATOR)
wtmp0 = WIDGET_BUTTON(wMenu, VALUE='Tools', /MENU)
 wtmpb = Widget_Button(wtmp0, VALUE='Color table', /Menu)
   wtmp1 = WIDGET_BUTTON(wtmpb, VALUE='Set Default', UVALUE='ColorTable')
   wtmp1 = WIDGET_BUTTON(wtmpb, VALUE='Change table...', UVALUE='ColorTable')
wtmp0 = WIDGET_BUTTON(wMenu, VALUE='Help', /HELP)
 wtmp1 = WIDGET_BUTTON(wtmp0, VALUE='xnlfit', UVALUE='Help')



wBase = Widget_Base(wWin,/Column)

wtmp = widget_base(wBase,/ROW)
wRefresh = widget_Button(wtmp,VALUE='Quit',UVALUE='Quit')
wRefresh = widget_Button(wtmp,VALUE='Refresh Screen',UVALUE='PlotData')
wtmp0 = widget_Button(wtmp,VALUE='**Start Fit**',UVALUE='Fit')


if not(keyword_set(nodraw)) then $
wDr = widget_draw(wBase, XSIZE=550, YSIZE=300, RETAIN=2)

wEq = widget_base(wBase,/COLUMN,/FRAME)
wEquationLabel =  widget_label(wEq,VALUE=$
	'Model equation: y=f(x;p[0],p[1],...)', /ALIGN_LEFT)
wEquation =  widget_text(wEq,VALUE=defs.equation,/ALIGN_LEFT,/EDIT,$
	UVALUE='Equation',XSIZE=75)
wtmp = widget_base(wEq,/ROW)
wtmp1 = widget_button(wtmp,VALUE='Clear',UVALUE='Equation')
wtmp2 = widget_button(wtmp,VALUE='Add Gaussian',UVALUE='Equation')
wtmp3 = widget_button(wtmp,VALUE='Add Lorentzian',UVALUE='Equation')
wtmp4 = widget_button(wtmp,VALUE='Add Polynomial',UVALUE='Equation')

wGu = widget_base(wBase,/COLUMN,/FRAME)
wGuessLabel =  widget_label(wGu,VALUE='Model parameters:',/ALIGN_LEFT)
wGuess =  widget_text(wGu,VALUE=defs.guess,/ALIGN_LEFT,/EDIT,XSIZE=75, $
	UVALUE='Guess')
wtmp = widget_base(wGu,/ROW)
wtmp1 = widget_button(wtmp,VALUE='Clear parameters',Uvalue='Guess')
wtmp2 = widget_button(wtmp,VALUE='Template',Uvalue='Guess')
wtmp3 = widget_Button(wtmp,VALUE='(X)Plot model',UVALUE='PlotModel')
wtmp4 = widget_Button(wtmp,VALUE='Overplot model',UVALUE='PlotModel')
wtmp5 = widget_Button(wtmp,VALUE='Covariance',UVALUE='PlotModel')
if keyword_set(xplotmain) then wtmp6 = widget_Button(wGu, $
	VALUE='Put Model in a new column',UVALUE='Xplotaddcolumn')

wFlagsbase = widget_base(wBase,/COLUMN,/FRAME)
wFlagsLabel =  widget_label(wFlagsbase,VALUE=$
  'Free (1) or Fix (0) parameters for fit:',/ALIGN_LEFT)
wFlags =  widget_text(wFlagsbase,VALUE=defs.free_flags,/ALIGN_LEFT,$
	/EDIT,XSIZE=75, UVALUE='UPDATE')
wtmp = widget_base(wFlagsbase,/ROW)
  wtmpb = Widget_Base(wtmp,/Row)
  wtmp1 = widget_button(wtmpb,VALUE='Reset',Uvalue='Reset')
  wBoundBase = Widget_Base(wtmp,/Row)
  wtmp1 = widget_button(wBoundBase,VALUE='Boundaries...',Uvalue='Bound')

wRa = widget_base(wBase,/COL,/FRAME)
wLa =  widget_label(wRa,VALUE=$
   'Fitting Abscissas Domain: (Xmin=Xmax mean the whole interval)',/ALIGN_LEFT)
wtmp = widget_base(wRa,/ROW)
  wx1= cw_field(wtmp, TITLE='X minimum',UVALUE='PlotData',$
         VALUE=xmin,/Floating)
  wx2= cw_field(wtmp, TITLE='X maximum',UVALUE='PlotData',$
         VALUE=xmax,/Floating)
wtmp0 = widget_base(wRa,/ROW)
wAllinterval =  widget_button(wtmp0,VALUE='Set all interval',UVALUE='PlotData')
wtmp1 =  widget_button(wtmp0,VALUE='Xplot with data in domain',$
	UVALUE='EXTRACT')

wSigmaBase = widget_base(wBase,/ROW,/FRAME)
  wtmp1 = widget_label(wSigmaBase,VALUE='Sigma (error) value:',/ALIGN_LEFT)
  ; wSigmaValue = widget_droplist(wSigmaBase,VALUE=['1','Statistic','From file'],$
  wSigmaValue = CW_DropList(wSigmaBase,VALUE=defs.sigmaValue,$
	UValue='SIGMAVALUE')
  ;wSigmaFileBase = widget_base(wSigmaBase,/ROW)
  wSigmaFileBase = widget_base(wSigmaBase,/ROW)
    wtmp = widget_button(wSigmaFileBase,VALUE='Browser', $
	UVALUE='SigmaFileBrowser')
    wSigmaFile = widget_text(wSigmaFileBase,VALUE=defs.sigmaFile, $
	/ALIGN_LEFT,/EDIT,UValue='UPDATE')

wtmp = widget_base(wBase,/COL,/FRAME)

if keyword_set(nodraw) then begin
  wnum = !d.window
  if wnum LT 0 then begin
    window,/free
    wnum = !d.window
  endif
endif  else wnum=-1

if not(keyword_set(xplotmain)) then xplotmain=0L

if not(keyword_set(xplotmain)) then xplotmain=0L
if not(keyword_set(psysvar)) then psysvar=0L
IF KeyWord_Set(extendedSet) THEN $
   ptrExtendedSet=Ptr_New(extendedSet) ELSE $
   ptrExtendedSet=ptrSet

wids = { equation:wEquation, guess:wGuess, xmin:wx1, xmax:wx2, $
	Allinterval:wAllinterval,  $
	SigmaBase:wSigmaBase, SigmaValue:wSigmaValue, $
	SigmaFileBase:wSigmaFileBase, SigmaFile:wSigmaFile, $
	Xplot:xplotmain, FlagsBase:wFlagsBase, free_Flags:wFlags,$
	boundBase:wBoundBase }

state= { window:wnum, ptrset: ptrset, ptrExtendedSet:ptrExtendedSet, $
	 wids:wids, npar:0 , psysvar:psysvar, defs:defs, no_block:no_block }

widget_control,wWin,/REALIZE 

XNLFit_UpdatePanel,defs,wids
XNLFit_MapPanels,defs,wids

;IF Fix(defs.pref_method[0]) LE 1 THEN $
;  widget_control,wids.FlagsBase,MAP=1 ELSE $
;  widget_control,wids.FlagsBase,MAP=0

if not(keyword_set(nodraw)) then begin
  widget_control,wDr, get_val = wnum
  state.window=wnum
endif

IF Keyword_Set(init_pars) AND fromFile EQ 0 THEN BEGIN
  xnlfit_load,init_pars,wids,group=group
ENDIF

widget_control,wWin, SET_UVAL=state,/NO_COPY
if xplotmain EQ 0 then xnlfit_plot,x,y,color=defs.pref_color1
plotfile,set,ki=2,color=2,xrange=xrange
;xplot_changecol,xplotmain
xnlfit_plot,x,y,color=defs.pref_color2,xrange=xrange,/Over

xmanager, 'XNLFit', wWin, GROUP_LEADER=group,No_Block=no_block, $
  CleanUp='xnlfit_clean'
end
