function makearray1,npts,xmin,xmax, empty=empty, float=float, double=double, $
  long=long, fix=fix, string=string, log=log


;+
; NAME:
;	MAKEARRAY1
;
; PURPOSE:
;	This function creates a 1 dimensional array 
;
; CATEGORY:
;	Mathematics
;
; CALLING SEQUENCE:
;
;	result = makearray1(npoints [, xminimun, xmaximum])
;
; INPUTS:
;	npoints: the number of points
;
; OPTIONAL INPUTS:
;	xminimum: the minimum value for the array
;	xmaximum: the maximum value for the array
;	  If these values are not set, the resulting interval is [0,1] in 
;	  linear and [1,10] in logarithmic scale. 
;	
; KEYWORD PARAMETERS:
;	EMPTY: when set, create an empty array (all zeros)
;	LOG: when set, creates arrays equally spaced in log scale  
;	FLOAT: when set, create an array of float values 
;	FIX: when set, create an array of short integer values 
;	LONG: when set, create an array of short integer values 
;	DOUBLE: when set, create an array of double precission values (default)
;	STRING: when set, the array is converted to string
;
; OUTPUTS:
;	the resulting array
;
; SIDE EFFECTS:
;
; PROCEDURE:
;	the array is always created as float, and then converted 
;	to the desired type (when the corresponding keyword is
;	selected).
;
; EXAMPLES:
;
;	print,makearray1(5,-2,2)
;       -2.00000     -1.00000      0.00000      1.00000      2.00000
;
;	print,makearray1(5,-1,1,/fix)
;       -1       0       0       0       1
;
; MODIFICATION HISTORY:
; 	Written by:	M. Sanchez del Rio
;	August 6, 1998
;	2002/04/23 Makes calculations in double precision. Default set to 
;	  double. Added log kw.
;
;-

on_error,2
out = 0
if n_params() EQ 0 then message,'Usage: result=makearray1(npoints [,min,max])'
if keyword_set(empty) then begin
  out = dblarr(npts)
endif else begin
  IF Keyword_Set(log) THEN BEGIN
    IF n_params() NE 3 THEN BEGIN
      axmin = alog10(1.0D0)
      axmax = alog10(10.0D0)
   ENDIF ELSE BEGIN
     axmin = alog10(xmin)
     axmax = alog10(xmax)
   ENDELSE
   out = dindgen(npts)/(npts-1)
   out = 10.0D0^( axmin + out*(axmax-axmin) ) 
  ENDIF ELSE BEGIN
    out = dindgen(npts)/(npts-1)
    if n_params() EQ 3 then out = double(xmin) + out*(double(xmax)-double(xmin)) 
  ENDELSE
endelse
if keyword_set(fix) then out = fix(out)
if keyword_set(long) then out = long(out)
if keyword_set(float) then out = float(out)
if keyword_set(double) then out = double(out)
if keyword_set(string) then out = string(out)
return,out
end
