;______________________________________________________________________________
;+
; NAME:
;       SPEC_HEADERS()
;
; PURPOSE:
;       This function returns header lines from a particular scan in a SPEC
;       data file. Header lines begin by the '#' character and can be situated
;       before or after the block of data lines.
;
; CATEGORY:
;       Input/Output.
;
; CALLING SEQUENCE:
;       Result = SPEC_HEADERS(Handle, Scan_Id [, Selec_Array])
;
; INPUTS:
;       Handle   - Handle to the SPEC data file initialised by a previous call
;                  to SPEC_ACCESS().
;
;       Scan_Id  - Scan identifier. This parameter can be a numeric or string
;                  scalar value that accepts several syntax and keywords as
;                  it is explained in the `SCAN IDENTIFIERS' section at the
;                  beginning of this file). The simplest case is to use
;                  the SPEC scan number as Scan_Id.
;
; OPTIONAL INPUTS:
;       Selec_Array - Array that contains selectors to the header lines.
;                  A selector is a string with the first characters of the
;                  header lines that are wanted to be returned. A selector can
;                  include or not the first '#' character of the header lines.
;                  For instance, the string 'P' selects all the headers in the
;                  scan that contain motor positions (those that begin with 
;                  the '#P' characters) and the string '#G0' selects the 
;                  first line of geometry parameters.
;
; KEYWORDS:
;       INDEX:  Set this keyword to interpret a numeric value in Scan_Id
;               as the index of the scan instead of the scan number.
;
;       ALL:    Set this keyword to extend the scope of the search and include
;               the main block of header lines as well as comments embedded in
;               the data block.
;               In normal SPEC data files the main header block is placed at
;               the beginning of the file and includes the file name line ('#F')
;               the epoch line ('#E') and the motor names ('#O' lines). If 
;               there are more than one main header block in the file, the
;               function uses the last one before the given scan.
;               Comment lines appear embedded in the data block when a scan
;               is resumed after been aborted from SPEC.
;	LABEL_REMOVE: If this keyword is set, returns the string(s) with
;		the keyword label (i.e., name) removed. 
;	Fix: If this keyword is set, converts the result to Fix
;	Long: If this keyword is set, converts the result to Long
;	Float: If this keyword is set, converts the result to Float
;	Double: If this keyword is set, converts the result to Double
;
; OUTPUT: 
;       This function returns a string array that contains all the header lines
;       that match the selection criteria. If Selec_Array is not specified
;       all the header lines are returned.
;
; RESTRICTIONS:
;       This function requires a valid handle to a SPEC data file obtained by
;       a previous call to SPEC_ACCESS().
;       If Scan_Id does not represent a valid scan the function produces an
;       error.
;
; EXAMPLE:
;       To load the string array NewHeaders with the header lines that contain
;       the positions of the first eight motors ('#P0' line) and the user
;       defined values ('#U' lines), enter:
;
;       Dummy = SPEC_ACCESS(FileHandle, 'baddata')
;       Nscans = SPEC_HEADERS(FileHandle, BestScan, ['#P0','U'])
;
;       (the file in this example is 'baddata' and the scan number is in the
;       variable called BestScan)
;______________________________________________________________________________
;-

function spec_headers, handle, scan_id, headers, ALL=all, INDEX=idx, $
	Label_Remove=label_remove, Fix=kFix, Long=kLong, Float=kFloat, $
	Double = kDouble
   catch, error_status
   if error_status ne 0 then begin
      catch, /cancel
      on_error, 2
      message, __cleanmsg(!err_string);, /traceback
   endif

   IF N_Elements(label_remove) EQ 0 THEN label_remove = 0
   IF Keyword_Set(kFix) THEN label_remove=1 
   IF Keyword_Set(kLong) THEN label_remove=1 
   IF Keyword_Set(kFloat) THEN label_remove=1 
   IF Keyword_Set(kDouble) THEN label_remove=1 

   ; Check arguments
   if N_PARAMS() lt 2 then message, 'Incorrect number of arguments.'
   if N_PARAMS() eq 2 then begin
      headers = ''
   endif
   __speccheck, handle
   __specload, handle, scan_id, errmsg, INDEX=idx
   if !ERR then message, errmsg
   if keyword_set(all) then begin
      __headerload, handle
      nlh = handle.scan(handle.currscan).headerlines 
   endif else begin
      nlh = 0L
   endelse

   aux = size(headers)
   if aux(aux(0) + 1) ne 7 or aux(0) gt 1 then begin
       message, 'Argument is not a valid header specification.'
   endif
   nheaders = aux(aux(0) + 2)
   _headers = strarr(nheaders)

   for i = 0L, nheaders - 1 do begin
      if strmid(headers(i), 0, 1) ne '#' then begin
         _headers(i) = '#' + headers(i)
      endif else begin
         _headers(i) = headers(i)
      endelse
   endfor
   headlen = strlen(_headers)

   nl0 = nlh + handle.scan(handle.currscan).predatalines 
   nl1 = nl0 + handle.scan(handle.currscan).datalines
   nlt = nl1 + handle.scan(handle.currscan).postdatalines

   openr,Unit,handle.specfile,/get_lun
   if nlh gt 0 then begin
      point_lun, Unit, handle.scan(handle.currscan).headerpt
   endif 
   line = ''
   ct = 0L
   for i = 1L, nlt do begin
      if i eq nlh + 1 then begin
         point_lun, Unit, handle.scan(handle.currscan).scanpt
      endif
      __readline, Unit, line
      if i le nl0 or i gt nl1 or keyword_set(all) then begin
         if line ne '' and strmid(line, 0, 1) eq '#' then begin
            for j = 0L, nheaders - 1 do begin
               if strmid(line, 0, headlen(j)) eq _headers(j) then begin
		  IF Keyword_Set(Label_Remove) THEN $
			line=StrMid(line,headlen(j)+1)
;print,'*************'+line+'******************',headlen(j)
                  if ct eq 0 then begin
                     result = line
                  endif else begin
                     result = [result, line]
                  endelse
                  ct = ct + 1
                  goto, outfor
               endif
            endfor
outfor:
         endif
      endif
   endfor
   free_lun, Unit

   if ct eq 0 then begin
      return, ''
   endif else begin
      IF Keyword_Set(kFix) EQ 0 AND  Keyword_Set(kLong) EQ 0 AND $
         Keyword_Set(kFloat) EQ 0 AND  Keyword_Set(kDouble) EQ 0 THEN BEGIN
         return, result
      ENDIF ELSE BEGIN
	 IF ct GT 1 THEN BEGIN
	   MESSAGE,/Info, $
	      'Multiple keyword found. Not possible to convert to Integer/Real.'
          return, result
         ENDIF ELSE BEGIN
	   n = StrParse(StrCompress(result),' ,',list)
           IF Keyword_Set(kFix) EQ 1 THEN RETURN,Fix(list)
           IF Keyword_Set(kLong) EQ 1 THEN RETURN,Long(list)
           IF Keyword_Set(kFloat) EQ 1 THEN RETURN,Float(list)
           IF Keyword_Set(kDouble) EQ 1 THEN RETURN,Double(list)
           return, result ; just in case
	 ENDELSE
      ENDELSE
   endelse
end
