;+
; NAME:
;       SYNC_F
;
; PURPOSE:
;       Calculates the function used for calculating the angular 
;	dependence of synchrotron radiation. 
;
; CATEGORY:
;       Mathematics.
;
; CALLING SEQUENCE:
;       Result = sync_hi(rAngle [,rEnergy] )
;
; INPUTS:
;       rAngle:  the reduced angle, i.e., angle[rads]*Gamma. It can be a
;		scalar or a vector.
; OPTIONAL INPUTS:
;       rEnergy:  a value for the reduced photon energy, i.e., 
;		energy/critical_energy. It can be an scalar or a verctor. 
;		If this input is present, the calculation is done for this 
;		energy. Otherwise, the calculation results is the integration 
;		over all photon energies.
;
; KEYWORD PARAMETERS:
;	POLARIZATION: 0 Total 
;		      1 Parallel       (l2=1, l3=0, in Sokolov&Ternov notation)
;		      2 Perpendicular  (l2=0, l3=1)
;
;	L2: The polarization value of L2
;	L3: The polarization value of L3
;		Note: If using L2 and L3, both L2 and L3 must be defined.
;		      In this case, the Pol keyword is ignored.
;
;	GAUSS: When this keyword is set, the "Gaussian" approximaxion 
;		instead of the full calculation is used. 
;		Only valid for integrated flux aver all photon energies.
; OUTPUTS:
;       returns the value  of the sync_f function
;		It is a scalar if both inputs are scalar. If one input
;		is an array, the result is an array of the same dimension. 
;		If both inputs are arrays, the resulting array has dimension
;		NxM, N=Dim(rAngle) and M=Dim(rEnergy)
;
; PROCEDURE:
;       The number of emitted photons versus vertical angle Psi is
;	proportional to sync_f, which value is given by the formulas
;	in the references.
;
;	For angular distribution integrated over full photon energies (rEnergy 
;	optional input not present) we use the Formula 9, pag 4 in Green. 
;	For its gaussian approximation (in this case the polarization keyword 
;	has no effect) we use for 87 in pag 32 in Green.
;
;	For angular distribution at a given photon energy (rEnergy 
;	optional input not present) we use the Formula 11, pag 6 in Green. 
;
;
;	References: 
;		G K Green, "Spectra and optics of synchrotron radiation" 
;			BNL 50522 report (1976)
;		A A Sokolov and I M Ternov, Synchrotron Radiation, 
;			Akademik-Verlag, Berlin, 1968
;
; OUTPUTS:
;       returns the value  of the sync_hi function
;
; PROCEDURE:
;       Uses IDL's BeselK() function
;
; MODIFICATION HISTORY:
;       Written by:     M. Sanchez del Rio, srio@esrf.fr, 2002-05-23
;	2002-07-12 srio@esrf.fr adds circulat polarization term for 
;		wavelength integrated spectrum (S&T formula 5.25)
;
;-
FUNCTION SYNC_F,rAngle,rEnergy,Polarization=pol,gauss=gauss,l2=l2,l3=l3

IF N_Elements(pol) EQ 0 THEN pol=0
pol = Fix(pol)

IF N_Elements(rAngle) EQ 0 THEN $
  Message,'Define the reduced angle (Gamme*Psi)'

rAngle=Double(rAngle)
;
; angle distribution integrated over full energies
;
IF N_Elements(rEnergy) EQ 0 THEN BEGIN  ; angle distribution integrated over full energies
  IF Keyword_Set(gauss) THEN BEGIN
    ; Formula 87 in Pag 32 in Green 1975
    efe = 0.4375*exp(-0.5D0*rAngle*rAngle/0.608/0.608)
    RETURN,efe
  ENDIF


  IF N_Elements(l2)*N_Elements(l3) EQ 0 THEN BEGIN
    CASE pol OF
      0: BEGIN
	  RETURN,sync_f(rAngle,pol=1)+sync_f(rAngle,pol=2)
	 END
      1: BEGIN
	  l2=1.0D0
	  l3=0.0D0
	END
      2: BEGIN
	  l2=0.0D0
	  l3=1.0D0
	END
      else: Message,'Wrong polarization'
    ENDCASE
  ENDIF ELSE BEGIN
    l2=Double(l2)
    l3=Double(l3)
  ENDELSE
  ; For 9 in Pag 4 in Green 1975
  ; The two summands correspond to the par and per polarizations, as 
  ; shown in Sokolov&Ternov formulas (3.31) and 5.26)
  ; However, for circular polarization a third term (S&T 5.25) must also be used
  efe = (7.0D0/16.0D0)*l2*l2+ $
	(5.0D0/16.0D0)*(rAngle*rAngle/(1+rAngle*rAngle))*l3*l3 +$
	(64.0D0/16.0D0/!dpi/Sqrt(3.0D0))*(rAngle/(1+rAngle*rAngle)^(0.5D0))*l2*l3
  efe = efe * ( (1.0D0+rAngle*rAngle)^(-5.0D0/2.0D0) )
  RETURN,efe
ENDIF

;
; angle distribution for a single energy
;
IF N_Elements(l2)*N_Elements(l3) EQ 0 THEN BEGIN
  CASE pol OF
    0:RETURN,sync_f(rAngle,rEnergy,pol=1)+sync_f(rAngle,rEnergy,pol=2)
    1: BEGIN
	l2=1.0D0
	l3=0.0D0
	END
    2: BEGIN
	l2=0.0D0
	l3=1.0D0
	END
    else: Message,'Wrong polarization'
  ENDCASE
ENDIF ELSE BEGIN
  l2=Double(l2)
  l3=Double(l3)
ENDELSE

IF N_Elements(rEnergy) EQ 0 THEN $
  Message,'Define the reduced energy (E/Ec).'

rEnergy = Double(rEnergy) 

; For 11 in Pag 6 in Green 1975

ji = Sqrt( (1.0D0+rAngle*rAngle)^3 )
ji = ji#rEnergy/2.0D0
rAngle2 = rAngle#(rEnergy*0.0D0+1.0D0)


efe = l2*BeselK(ji,2.0D0/3.0D0)+ $
      l3* rAngle2*BeselK(ji,1.0D0/3.0D0)/Sqrt(1.0D0+rAngle2*rAngle2)
efe = efe* (1.0D0+rAngle2*rAngle2)
efe = efe*efe

RETURN,efe

END
