function parse_mixture,input1,s,f,z,fw,molwt=molwt,verbose=verbose
; 
; European Synchrotron Radiation Facility (ESRF)
;+
; NAME:
;       PARSE_MIXTURE
;
; PURPOSE:
;	Returns the number of elements minus one (and optionally their symbols,
;	weight fractions, and atomic numbers) from a chemical mixture.
;	Returns 0 (zero) if the input expression is not understood.
;	
; CATEGORY:
;       X-Ray optics. DABAX data base.
;
; CALLING SEQUENCE:
;	out = parse_mixture(input [,symbols,fractions,zetas])
; INPUTS:
;	input: a string with the formula to be processed.
;		The syntaxis is the following: string with the chemical
;		formula of the compounds with their weight fraction between 
;		parenthesis.
;		Examples:
;		SiO2(0.807)B2O3(0.129)Na2O(0.038)Al2O3(0.022)K2O(0.004)
;		H2O(0.99)Cl2(0.01)
;		H(0.11189834)O(0.88810166)
; OPTIONAL OUTPUTS:
;	symbols: a string array with the chamical symbols of the elements
;		in the mixture formula.
;	fractions: a double array with the formula index for
;		each respective element.
;	zetas: a integer array with the atomic number for each atom.
;	fractionsweight:  a double array with the fraction in weight for
;		each respective element.
;	
; KEYWORDS:
;	molwt = Set this keyword to a named variable which will contain
;		the molecular weight of the mixture (weighted sum of
;		the molecular weight of its components). 
;	VERBOSE = Flag to inform on elements and fractions used (default=1)
;
; OUTPUT:
;	out: the number of elements that constitute the mixture.
;
; PROCEDURE:
;	Uses parse_compound.
;
; EXAMPLES:
;	IDL> print,$
; parse_mixture('SiO2(0.807)B2O3(0.129)Na2O(0.038)Al2O3(0.022)K2O(0.004)',s,f,z)
; 	%PARSE_MIXTURE:-SiO2(0.807)B2O3(0.129)Na2O(0.038)Al2O3(0.022)K2O(0.004)
;	% PARSE_MIXTURE: ---Al(13)   0.011643515
;	% PARSE_MIXTURE: ---B (5)   0.040063629
;	% PARSE_MIXTURE: ---K (19)   0.0033205913
;	% PARSE_MIXTURE: ---Na(11)   0.028190583
;	% PARSE_MIXTURE: ---O (8)   0.53956168
;	% PARSE_MIXTURE: ---Si(14)   0.37721997
;	           5
;
; MODIFICATION HISTORY:
;       96-10-11 Written by M. Sanchez del Rio (srio@esrf.fr)
;       96-11-25 MSR fix a bug (return 2 when calling 
;	parse_mixtute('Si(0.5)Si(0.5)',s,f,z). Now it returns 1)
;	01-02-28 MSR added molwt keyword
;	2003-07-07 srio@esrf.fr fix a bug when passing mixtures
;		from Compounds.dat (NIST entries). 
;-
;
;on_error,2

;input = 'SiO2(0.807)B2O3(0.129)Na2O(0.038)Al2O3(0.022)K2O(0.004)'
;input = 'B(0.04006)O(0.53956)Na(0.02819)Al(0.01164)Si(0.37722)K(0.00332)'
;input = 'Si'

idebug = 0
input = StrCompress(input1,/Remove_All)

;Find where parenthesis are
w1 = where ( byte(input) eq (byte('('))(0) )
w2 = where ( byte(input) eq (byte(')'))(0) )
if n_elements(w1) ne n_elements(w2) then goto,error
IF n_elements(verbose) EQ 0 THEN verbose=1

ncomp =  n_elements(w1) > 1
if n_params() eq 1 then goto,fin
if idebug then message,/info,'Number of compounds: '+string(ncomp)


comp = strarr(ncomp)
frac = strarr(ncomp)

tmp = strsubstitute(input,'(','/')
tmp = strsubstitute(tmp,')','/')
if idebug  then print,tmp
;
; get the compounds that constitute the mixture
;
n = strparse(tmp,'/',nlist)
if n eq 0 then begin
  comp(0) = tmp 
  frac(0) = '1.0'
endif else begin
  for i=0,ncomp-1 do begin
    comp(i) = nlist(2*i)
    frac(i) = nlist(2*i+1)
    if idebug  then print,'#,comp : ',i+1,' ',comp(i)
    if idebug  then print,'#,frac : ',i+1,' ',frac(i)
  endfor
endelse
if total(float(frac)) le 0.9999 then $
  message,/info,'Fractions not normalized: '+string(total(float(frac)))

;
; for each compound, calls parse_compound to get its elements.
;
molwt = 0.0
for i=0,ncomp-1 do begin
  if idebug then print,'>> calling parse_compound with : ','***'+comp(i)+'***'
  molwti=0 ; initialize variable
  n = parse_compound(comp(i),s,f,fw,molwt=molwti)
  f = f * float(frac(i))
  fw = fw * float(frac(i))
  molwti = molwti * float(frac(i))
  molwt = molwt+molwti
  if i eq 0 then sall = s else sall = [sall,s]
  if i eq 0 then fall = f else fall = [fall,f]
  if i eq 0 then fwall = fw else fwall = [fwall,fw]
endfor
if idebug then print,'molwt: ',molwt
if idebug then print,'sall: ',sall
if idebug then print,'fall: ',fall
if idebug then print,'fwall: ',fwall

;
; prepare the optional outputs
;
if n_elements(sall) eq 1 then begin
  s=sall
  f=fall
  fw=fwall
endif else begin
  s = sall(uniq(sall,sort(sall)))
  f = dblarr(n_elements(s)) 
  fw = dblarr(n_elements(s)) 
  for i=0,n_elements(s)-1 do f(i) = total( fall(where(sall eq s(i))) )
  for i=0,n_elements(s)-1 do fw(i) = total( fwall(where(sall eq s(i))) )
endelse
z = atomic_constants(s,h,ret='Z')
s = s(sort(z))
f = f(sort(z))
fw = fw(sort(z))
z = z(sort(z))
ncomp = n_elements(s)

IF verbose THEN BEGIN
  Message,/Info,'---'+input
  FOR i=0,n_elements(s)-1 do BEGIN
     message,/info,'---'+s(i)+'(Z='+strcompress(z(i),/rem)+')   indx='+$
      strcompress(f(i),/rem)+'  weight='+strcompress(fw(i),/rem)
  ENDFOR
ENDIF

goto,fin
error:
message,/info,'Expression not understood: '+input
return,0

fin:
return,ncomp

end


