;+
; 
; NAME: 
;	OMPP__DEFINE
;
; PURPOSE:
;	Object class and methods to handle multiple arrays of data.
;	Used with Xop|Xaid|Exodus application. 
;
; CATEGORY:
;	Object tools
;
; CALLING SEQUENCE:
;	o1 = Obj_New('ompp',input)
;	o1-><method>
;
; F=Function, P=Procedure
;
; METHODS:
;	INIT(F)	called when the object is created with o1=Obj_New('ompp',input)
;		INPUTS:
;			input: either a data array or an array of pointers 
;				pointing to data. In the first case the 
;				data is copied, whereas in the second case 
;				the data is referenced. 
;		KEYWORDS:
;			NTIMES: the number of array slots to initialize.
;				It is not necessary to entry the data.
;			TITLE: A string or array of strings with the
;				array(s)' title(s).
;			FLAG: an integer or array of integer(s) with an
;				auxiliar flag.
;
;	CLEANUP(P)	called when the object is destroyed with Obj_destroy,o1
;
;	INFO(F)	returns a text with information of the stored arrays 
;		KEYWORDS:
;			N_GOOD_ELEMENTS: when set, returns the number of 
;				good elements (slots with data) stored in 
;				the object.
;			N_ELEMENTS: when set, returns the number of elements 
;				(slot arrays) stored in the object.
;			VERBOSE: if set, displays on the terminal the info
;
;	SET(P)	used to input data in the object
;		INPUTS:
;			index (optional): a string with the description of 
;			the array with the indices to fill. Default: '[*]'
;			Examples: '[*]', '[1:2]'
;		KEYWORDS:
;			TITLE: a string with the title
;			FLAG: a flag value
;			POINTER: a pointer to the data to be stored
;			   (warning: the heap variable asociated will be
;			   destroyed if the object is cleaned or destroyed).
;			VALUE: the array with the data to be stored
;			AUX: Some auxiliar data (space to store any 
;				additional variable associated to this
;				data set).
;			CLEAN: when set, cleans the corresponding data
;				(does not remove the slot).
;				It also cleans auxiliar data.
;			CLEANAUX: when set, cleans the corresponding auxiliar data
;			ADD: if set, the data is added to the bottom of the 
;				list.
;			OVERADD: if set, the data is added to the bottom of the 
;				list, however, if only one slot is
;				defined, ans ut is empy, this slot is used. 
;				(this is useful when adding elements
;				and wanted to fill the first slot). 
;	REMOVE(P)	used to remove data from the object. It frees the 
;			pointers (attention: if you entered pointers in the 
;			ompp object, they are always freed). 
;		INPUTS: the index or array with indices to delete. 
;			
;	VALUE(F)	used to return arrays and parameters
;		INPUTS:
;			INDEX: a string with the description of the array with 
;				the indices. Examples: '[*]', '[1:2]'
;		KEYWORDS:
;			*** With no keywords or with Aux keyword, the 
;			function returns the array with data corresponding 
;			to a single index. When using other keyword, it can 
;			also return a multiple data arrays ***
;
;			TITLE: if set, the function returns the title.
;			FLAG: if set, the function returns the flag.
;			POINTEr: Returns a pointer to the data.
;			AUX: Returns auxiliar value for a single index
;			ISAUXDEF: Returns 1 if auxiliat data is defined
;			ISDATADEF: Returns 1 if data data is defined
;			 	(this option is used for checking if the object 
;				contains some data or is empty. As the object must 
;				contain at least one line, one should check with this 
;				keyword to veryfy that this line is empty)
;
;	SORT(P)	used to sort the list of arrays in the object
;		INPUTS: the index or array with indices to delete. 
;	MPLOT(P)	used to plot arrays
;		INPUTS: the index or array with indices to plot. 
;		KEYWORDS:
;			XCol: the column to plot as abscissa
;			YCol: the ordinates column
;			XRange: a vector containing the X range ([xMin,xMax])
;			YRange: a vector containing the Y range ([yMin,yMax])
;			Legend: when this flag is set, a plot legend is created
;			ShiftValue: A factor to shift up the spectra 1,...N-1. 
;				The spectra are usually shifted up for better
;				visualization. The amount to shift each 
;				spectrum related to the one before is 
;				Abs(Max(y)-Min(y))*shiftValue, being y the 
;				ordinates of the first spectrum (index 0). 
;				Default = no shift
;			_Extra: any other keyword to be passed to "plot"
;	LCFUNC	used to return linear combination of arrays
;		INPUTS:
;			x: the abscissas array 
;			a: the array with the weights of the linear combination.
;		KEYWORDS:
;			index: the index or array with indices to combine. 
;	
;
; EXAMPLE:
;		See and run the file ompp_test.pro
;
; MODIFICATION HISTORY:
;	Initial version "MPP" by M. Sanchez del Rio, February 2000,
;	01-04-27 srio@esrf.fr creates the OO "OMPP" version.
;	01-11-28 srio@esrf.fr adds AUX keyword (auxiliary data)
;	01-12-07 srio@esrf.fr adapted for being used with FuFiFa.
;	03-03-29 srio@esrf.fr adds N_GOOD_ELEMENTS to ompp::info,
;				   OVERADD to ompp::set.
;
;-


;
;===============================================================================
;
FUNCTION __ompp_template
; only for internal use
RETURN,{title:'', ptrData:Ptr_New(), Flag:0L, ptrAux:Ptr_New()}
END


;
;===============================================================================
;
FUNCTION ompp::LCFunct,x,a, index=index

Catch, error_status
IF error_status NE 0 THEN BEGIN
   Message,/Info,'error caught: '+!err_string
   IF SDep(/w) THEN itmp = Dialog_Message(/Error,$
	'OMPP::LCFUNCT: error caught: '+!err_string)
   Catch, /Cancel
   On_Error,2
   RETURN,0
ENDIF

mppstr=*(self.list)

IF N_Elements(index) EQ 0 THEN ind=LIndGen(N_Elements(a)) ELSE ind=index

set = self->value(ind[0])
y = Interpolate(Reform(set[1,*]),FIndex(Reform(set[0,*]),x))
aa = a/Double(Total(a))
out = aa[0]*y
IF N_ELEMENTS(a) GT 1 THEN BEGIN
  FOR i=1,N_Elements(aa)-1 DO BEGIN
    set = self->value(ind[1])
    y = Interpolate(Reform(set[1,*]),FIndex(Reform(set[0,*]),x))
    out = out + Abs(aa[i])*y
  ENDFOR
ENDIF

RETURN,out
END

;
;===============================================================================
;
PRO ompp::MPlot,list,XCol=xCol,yCol=yCol,XRange=xRange,YRange=yRange,$
  Legend=iLegend,ShiftValue=shiftValue,_Extra=extra

Catch,error_status
IF error_status ne 0 THEN BEGIN
  Message,/Info,'error caught: '+!error_state.msg
  itmp=Dialog_Message(/Error, $
    'OMPP::MPLOT: error caught: '+!error_state.msg)
  Catch, /Cancel
  RETURN
ENDIF

IF N_Elements(list) EQ 0 THEN list=LIndGen(self.n)
IF N_Elements(xCol) EQ 0 THEN xCol=1
IF N_Elements(yCol) EQ 0 THEN yCol=2
IF N_Elements(ilegend) EQ 0 THEN ilegend=1

IF N_Elements(xRange) NE 2 THEN BEGIN
  calculate_xrange = 1
ENDIF ELSE BEGIN
  calculate_xrange = 0
  IF xRange[0] EQ xRange[1] THEN  calculate_xrange = 1 
ENDELSE

IF N_Elements(yRange) NE 2 THEN BEGIN
  calculate_yrange = 1
ENDIF ELSE BEGIN
  calculate_yrange = 0
  IF yRange[0] EQ yRange[1] THEN  calculate_yrange = 1 
ENDELSE


ptrs = self->Value(List,/Pointer)
nn = N_Elements(ptrs)

IF calculate_xrange EQ 1 THEN BEGIN
  FOR i=0,N_Elements(ptrs)-1 DO $
    IF i EQ 0 THEN a0 = Reform((*ptrs[i])[xCol-1,*]) ELSE $
    a0 = [Temporary(a0),Reform((*ptrs[i])[xCol-1,*])]
  xrange=[Min(a0),Max(a0)]
ENDIF

IF calculate_yrange EQ 1 THEN BEGIN
  FOR i=0,N_Elements(ptrs)-1 DO $
    IF i EQ 0 THEN a1 = Reform((*ptrs[i])[yCol-1,*]) ELSE $
    a1 = [Temporary(a1),Reform((*ptrs[i])[yCol-1,*])]
  yrange=[Min(a1),Max(a1)]
ENDIF

IF N_Elements(shiftValue) NE 0 THEN BEGIN
  shiftValue = Float(shiftValue)
  tmp = (*ptrs[0])[yCol-1,*]
  yDelta = Abs(max(tmp)-min(tmp)) * shiftValue
  yRange[1]=yRange[1]+(yDelta*(N_Elements(ptrs)-1))
  yShift = FIndGen(N_Elements(ptrs)) * yDelta
ENDIF ELSE BEGIN
  yShift = Replicate(0.0, N_Elements(ptrs))
ENDELSE

plot,[0,1], XRange=xrange,YRange=yRange,/NoData,_Extra=extra
FOR i=0,N_Elements(ptrs)-1 DO BEGIN
  oplot,(*ptrs[i])[xCol-1,*],((*ptrs[i])[yCol-1,*])+yShift[i],Color=i+2,_Extra=extra
ENDFOR

IF iLegend EQ 1 THEN $
  Legend,self->Value(List,/Title),/Fill, $
  PSym=Replicate(8,nn),Colors=SIndGen(nn)+2,Box=0

END

;
;===============================================================================
;
PRO ompp::Sort,indices
Catch, error_status
IF error_status NE 0 THEN BEGIN
   Message,/Info,'error caught: '+!err_string
   IF SDep(/w) THEN itmp = Dialog_Message(/Error,$
	'OMPP::SORT: error caught: '+!err_string)
   Catch, /Cancel
   On_Error,2
   RETURN
ENDIF

nn = N_Elements(indices) 
IF nn NE self.n THEN $
  Message,'Number of elements in sorting info ('+StrCompress(nn,/Rem)+ $
          ') is different from number of stored variables ('+$
          StrCompress(self.n,/Rem)+')'
mppstr=*(self.list)
mppstr = mppstr[indices]
*(self.list)=mppstr
END

;
;===============================================================================
;
FUNCTION ompp::Value,index,Title=title,Flag=flag, Pointer=pointer, $
  Aux=Aux,IsAuxDef=isAuxDef,IsDataDef=isDataDef

Catch, error_status
IF error_status NE 0 THEN BEGIN
   Message,/Info,'error caught: '+!err_string
   IF SDep(/w) THEN itmp = Dialog_Message(/Error,$
	'OMPP::VALUE: error caught: '+!err_string)
   Catch, /Cancel
   On_Error,2
   RETURN,0
ENDIF

mppstr=*(self.list)

IF N_Elements(index) EQ 0 THEN ind='[*]' ELSE ind=index
IF Type(ind) NE 7 THEN ind='['+Vect2String(ind)+']'

n = self.n
indices = LIndGen(n)
cmd = 'indices = indices'+ind
itmp = Execute(cmd)

out=0
IF Keyword_Set(Title) THEN BEGIN
  out = mppstr[indices].title
  RETURN,out
ENDIF
IF Keyword_Set(Flag) THEN BEGIN
  out = mppstr[indices].Flag
  RETURN,out
ENDIF

IF Keyword_Set(pointer) THEN BEGIN
  out = mppstr[indices].ptrData
  RETURN,out
ENDIF

IF Keyword_Set(IsAuxDef) THEN BEGIN
  ptrout = mppstr[indices].ptrAux
  IF Ptr_Valid(ptrout) EQ 1 THEN RETURN,1 ELSE RETURN,0
ENDIF

IF Keyword_Set(IsDataDef) THEN BEGIN
  ptrout = mppstr[indices].ptrData
  IF Ptr_Valid(ptrout) EQ 1 THEN RETURN,1 ELSE RETURN,0
ENDIF

IF Keyword_Set(aux) THEN BEGIN
  ptrout = mppstr[indices].ptrAux
  IF Ptr_Valid(ptrout) NE 1 THEN BEGIN
    Message,'No auxiliar information stored.'
  ENDIF ELSE out=*ptrout
  RETURN,out
ENDIF

ptrs = 0
ptrs = mppstr[indices].ptrData
IF N_Elements(ptrs) NE 1 THEN $
  Message,'Cannot return multiple data blocks'
out = *ptrs
RETURN,out
END

;
;===============================================================================
;
PRO ompp::remove,indices


Catch, error_status
IF error_status NE 0 THEN BEGIN
   Message,/Info,'error caught: '+!err_string
   IF SDep(/w) THEN itmp = Dialog_Message(/Error,$
	'OMPP::REMOVE: error caught: '+!err_string)
   Catch, /Cancel
   On_Error,2
   RETURN
ENDIF

remove_all = 0
IF N_Elements(indices) EQ 0 THEN BEGIN
  indices = LIndGen(self.n)
  remove_all = 1
ENDIF
indices=[indices]
mppstr = *(self.list)
ptrs = mppstr.ptrData
ptrsAux = mppstr.ptrAux
;
; empty arrays
;
FOR i=0,N_elements(indices)-1 DO BEGIN
   ii = indices[i]
   IF Ptr_Valid(ptrs[ii]) THEN Ptr_Free,ptrs[ii]
   IF Ptr_Valid(ptrsAux[ii]) THEN Ptr_Free,ptrsAux[ii]
ENDFOR
;
; empty slots
;
IF remove_all EQ 0 THEN BEGIN
  ; srio bug fixed 01/12/07
  ;tmp = Where(IndGen(self.n) NE indices[0])
  tmp = IndGen(self.n)
  iii = IntArr(self.n)
  FOR ii=0L,N_Elements(indices)-1 DO BEGIN
    iii = iii + (tmp EQ indices[ii])
  ENDFOR
  tmp = Where(iii EQ 0)
  IF tmp[0] EQ -1 THEN BEGIN
    Message,/info,'At least one item must exist'
    RETURN
  ENDIF
  mppstr=mppstr[tmp]
ENDIF ELSE BEGIN
  tmp=__ompp_template() ; {title:'', ptrData:Ptr_New(), Flag:0L}
  mppstr = Replicate(tmp,1)
ENDELSE
*(self.list) = mppstr
self.n = N_Elements(mppstr)
END

;
;===============================================================================
;
PRO ompp::set,index,Title=title,Flag=flag, $
  Pointer=pointer,Value=value,Clean=Clean,ClAux=clAux,$
  Add=add,Aux=aux,OverAdd=overAdd

Catch, error_status
IF error_status NE 0 THEN BEGIN
   Message,/Info,'error caught: '+!err_string
   IF SDep(/w) THEN itmp = Dialog_Message(/Error,$
	'OMPP::SET: error caught: '+!err_string)
   Catch, /Cancel
   On_Error,2
   RETURN
ENDIF
mppstr = *(self.list)

; 
; add a new data structure 
;
IF Keyword_Set(add) THEN BEGIN
   ; mppstr = [Temporary(mppstr),{title:'',ptrData:Ptr_New(),Flag:0L}]
   ;mppstr = [Temporary(mppstr),__ompp_template()]
   ; artificial way using copy_structure to avoid problems with
   ; concatenation of structures
   tmp = mppstr[0]
   copy_structure,__ompp_template(),tmp
   mppstr = [mppstr,tmp]
   nn = N_Elements(mppstr) 
   index = N_Elements(mppstr)-1
   *(self.list) = mppstr
   self.n = N_Elements(mppstr)
ENDIF



IF Keyword_Set(overAdd) THEN BEGIN
   ; mppstr = [Temporary(mppstr),{title:'',ptrData:Ptr_New(),Flag:0L}]
   ;mppstr = [Temporary(mppstr),__ompp_template()]
   ; artificial way using copy_structure to avoid problems with
   ; concatenation of structures
   IF self.n EQ 1 AND Ptr_Valid( (mppstr.ptrData)[0] ) EQ 0 THEN BEGIN ; empty defined slot
     nn = N_Elements(mppstr) 
     index = N_Elements(mppstr)-1
   ENDIF ELSE BEGIN
     tmp = mppstr[0]
     copy_structure,__ompp_template(),tmp
     mppstr = [mppstr,tmp]
     nn = N_Elements(mppstr) 
     index = N_Elements(mppstr)-1
     *(self.list) = mppstr
     self.n = N_Elements(mppstr)
  ENDELSE
ENDIF


IF N_Elements(index) EQ 0 THEN ind='[*]' ELSE ind=index
IF Type(ind) NE 7 THEN ind='['+Vect2String(ind)+']'

n = self.n                 ; MPP_Info(mppstr,/N_Elements)
indices = LIndGen(n)
cmd = 'indices = indices'+ind
itmp = Execute(cmd)


IF  Keyword_Set(title) OR  Keyword_Set(flag) OR  Keyword_Set(pointer) THEN BEGIN
  FOR i=0,N_Elements(indices)-1 DO BEGIN
    tmp=mppstr[indices[i]]
    IF Keyword_Set(title) THEN tmp.title = title[i]
    IF Keyword_Set(flag) THEN tmp.flag = flag[i]
    IF Keyword_Set(pointer) THEN tmp.pointer = pointer[i]
    mppstr[indices[i]]=tmp
  ENDFOR
ENDIF

ptrs = mppstr.ptrData
ptrsAux = mppstr.ptrAux

IF Keyword_Set(Clean) THEN BEGIN
  FOR i=0,N_elements(indices)-1 DO BEGIN
     ii = indices[i]
     IF Ptr_Valid(ptrs[ii]) THEN Ptr_Free,ptrs[ii]
     IF Ptr_Valid(ptrsAux[ii]) THEN Ptr_Free,ptrsAux[ii]
  ENDFOR
ENDIF

IF Keyword_Set(ClAux) THEN BEGIN
  FOR i=0,N_elements(indices)-1 DO BEGIN
     ii = indices[i]
     IF Ptr_Valid(ptrsAux[ii]) THEN Ptr_Free,ptrsAux[ii]
  ENDFOR
ENDIF

IF Keyword_Set(Value) THEN BEGIN
  ptrs = mppstr[indices].ptrData
  FOR i=0,N_elements(ptrs)-1 DO BEGIN
     ptrs[i] = Ptr_New(value)
  ENDFOR
  mppstr[indices].ptrData = ptrs
ENDIF


IF Keyword_Set(Aux) THEN BEGIN
  ptrs = mppstr[indices].ptrAux
  FOR i=0,N_elements(ptrs)-1 DO BEGIN
     IF Ptr_Valid(ptrs[i]) THEN Ptr_Free,ptrs[i]
     ptrs[i] = Ptr_New(aux)
  ENDFOR
  mppstr[indices].ptrAux = ptrs
ENDIF

*(self.list)=mppstr 

END

;
;===============================================================================
;
FUNCTION ompp::Info,N_Elements=nelem,Verbose=verbose,N_Good_Elements=ngood

Catch, error_status
IF error_status NE 0 THEN BEGIN
   Message,/Info,'error caught: '+!err_string
   IF SDep(/w) THEN itmp = Dialog_Message(/Error,$
	'OMPP::INFO: error caught: '+!err_string)
   Catch, /Cancel
   On_Error,2
   RETURN,0
ENDIF

mppstr = *(self.list)
IF Keyword_Set(ngood) THEN BEGIN
  tmp = mppstr.ptrdata
  out = Long(Total(Ptr_Valid(tmp)))
  RETURN,out
ENDIF

IF Keyword_Set(nelem) THEN BEGIN
  out = N_Elements(mppstr)
  RETURN,out
ENDIF

out = ''
txt = StrArr(N_Elements(mppstr)+1)
txt[0] = String('Index',Format='(a5)') + $
      String('Flag',Format='(a6)') + $
      String('Type',Format='(a20)') + $
      String('Dim',Format='(a10)') + $
      String('TypeAux',Format='(a20)') + $
      String('DimAux',Format='(a10)') + $
      ' Title'
FOR i=0,N_Elements(mppstr)-1 DO BEGIN
  title = (mppstr.title)[i]
  flag = (mppstr.flag)[i]
  pi = (mppstr.ptrData)[i] 
  IF Ptr_Valid( (mppstr.ptrData)[i] ) THEN BEGIN
    data = *( (mppstr[i]).ptrData )
    s = Size(data)
    IF s[0] GT 0 THEN dim = Vect2String(s[1:1+(s[0]-1)]) ELSE $
      dim=Vect2String(s[0]+1)
    td = Type(data,/Verbose)
  ENDIF ELSE BEGIN
    dim = '<None>'
    td = '<None>'
  ENDELSE


  IF Ptr_Valid( (mppstr.ptrAux)[i] ) THEN BEGIN
    aux = *( (mppstr[i]).ptrAux )
    s = Size(aux)
    IF s[0] GT 0 THEN dimAux = Vect2String(s[1:1+(s[0]-1)]) ELSE $
      dim=Vect2String(s[0]+1)
    tdAux = Type(aux,/Verbose)
  ENDIF ELSE BEGIN
    dimAux = '<None>'
    tdAux = '<None>'
  ENDELSE

  txt[i+1] = String(i,Format='(I5)')+$
        String(flag,Format='(I6)')+$
        ' '+String(td,Format='(a19)') + $
        String(dim,Format='(a10)') + $
        String(tdAux,Format='(a20)') + $
        String(dimAux,Format='(a10)') + $
        ' "'+title+'"'
ENDFOR
IF Keyword_Set(verbose) THEN BEGIN
  Print,'************************************************************'
  Print,'              MPP_INFO (Info on MultiPlotPointer)           '
  Print,''
  FOR i=0,N_Elements(txt)-1 DO Print,txt[i]
  Print,'************************************************************'
ENDIF

out=txt
RETURN,out
END


;
;===============================================================================
;
PRO ompp::cleanup

Catch, error_status
IF error_status NE 0 THEN BEGIN
   Message,/Info,'error caught: '+!err_string
   IF SDep(/w) THEN itmp = Dialog_Message(/Error,$
	'OMPP::CLEANUP: error caught: '+!err_string)
   Catch, /Cancel
   On_Error,2
   RETURN
ENDIF

ptrList = self.list

ptrs = (*ptrList).ptrData
FOR i=0L,N_Elements(ptrs)-1 DO $
  IF Ptr_Valid(ptrs[i]) THEN Ptr_Free,ptrs[i]

ptrs = (*ptrList).ptrAux
FOR i=0L,N_Elements(ptrs)-1 DO $
  IF Ptr_Valid(ptrs[i]) THEN Ptr_Free,ptrs[i]

Ptr_Free,self.list
END


;
;===============================================================================
;
FUNCTION ompp::init, input , NTimes=nTimes, Title=title,Flag=flag



Catch, error_status
IF error_status NE 0 THEN BEGIN
  Message,/Info,'error caught: '+!err_string
  IF SDep(/w) THEN itmp = Dialog_Message(/Error,$
    'OMPP::INIT: error caught: '+!err_string)
  Catch, /Cancel
  On_Error,2
  RETURN,0
ENDIF

IF Not(KeyWord_Set(nTimes)) THEN nTimes = 1
IF Type(input) EQ 10 THEN nTimes=N_Elements(input) ; input is pointer


; create the structure array
tmp = __ompp_template() ; {title:'', ptrData:Ptr_New(), Flag:0L}  
tmp1 = Replicate(tmp,nTimes)
self.list = Ptr_New(tmp1)

; fill it
CASE Type(input) OF 
 0: BEGIN       ; no input
  self.n = nTimes
  END
10: BEGIN ; pointer
  self.n = nTimes 
  ptrList = self.list 
  tmp = (*ptrList).ptrData 
  tmp[0,N_Elements(input)-1]=input
  (*ptrList).ptrData = tmp
  END
else: BEGIN ; data
  self.n = nTimes 
  ptrList = self.list 
  tmp = (*ptrList).ptrData 
  FOR i=0,nTimes-1 DO BEGIN
    tmp[i]=Ptr_New(input)
  ENDFOR
  (*ptrList).ptrData = tmp
  END
ENDCASE

IF KeyWord_Set(title) THEN BEGIN
  ptrList = self.list 
  tmp = (*ptrList).title
  tmp[0:N_Elements(title)-1]=title
  (*ptrList).title = tmp
ENDIF

IF KeyWord_Set(flag) THEN BEGIN
  ptrList = self.list 
  tmp = (*ptrList).flag
  tmp[0:N_Elements(flag)-1]=flag
  (*ptrList).flag = tmp
ENDIF
  
RETURN,1
END

;
;===============================================================================
;
PRO ompp__define
  tmp = {ompp, list:Ptr_New(), n:0L }
END

