;+
;
;==============================  Xwiggler   ====================================
;
; Xwiggler is a widget based graphical interface to calculate
; the spectrum of a Wiggler Synchrotron Radiation source.
;
; The Flux and Spectral Power values calculated with XWiggler
; correspond to the integration in FULL vertical and horizontal 
; emission angles. 
;
; XWiggler calculations are done using the IDL program wiggler_spectrum. 
; This program is an IDL version of the Fortran program with the same 
; name available in the SHADOW ray-tracing package. 
;
;
; DESCRIPTION OF THE CONTROLS IN THE MAIN WINDOW:
;
;  File:
;    XWiggler input parameters: This option allows to save the current
;		parameters to a file for later loading. It also allows
;		to save the current parameters as defaults for being
;		used when the application is initialized. In the last
;		case, the file is named "application".xop (where 
;		"application " is the name of the current XOP
;		application) and is written in the directory pointed
;		by the XOP_DEFAULTS_DIR environment variable (which
;		must be set). The parameter file is ASCII and can be
;		read and edited with care.
;    Write Files for XOP/Optics: Saves the current undulator
;		spectrum in the SRCOMPE (flux) and SRCOMPW (Power) files
;		to be used for the other XOP application from Optics
;		menu.
;  Quit: to exit from the program
;
; Set_Parameters:
;  K(gap) Tool: This tool permits to calculate the deflection
;		parameter K as a function of the magnet gap.
;  Set Parameters: to define the parameters for the calculation.
;		The same result is obtained pressing the "Set Parameters"
;		button in the main XWiggler window. 
;               Please refer to the information under the HELP
;		button for a complete description of the parameters. After 
;		pressing the ACCEPT button, wiggler_spectrum starts running.
;  Set Defaults: Sets the default parameters.
;
;  Show: Display results
;    Spectra: Plots the calculated wiggler spectra.
;    Trajectories: Plots the calculated electron trajectories (individual paths
;			and 3D view) 
;    Parameters: Show the input parameters and some calculated values.
;
;  Help:   Shows the XWiggler help (this text).
;
;
; COPYRIGHT:
;	xwiggler  belongs to XOP package and it is distributed within XOP.
;	PLEASE REFER TO THE XOP COPYRIGHT NOTICE BEFORE USING IT.
;
; CREDITS:
;	Published calculations made with XOP should refer:
;
;	  M. Sanchez del Rio and R. J. Dejus "XOP: Recent Developments"
;	  SPIE proceedings vol. 3448, pp.340-345, 1998.
;
;	In addition, published calculations using the XWIGGLER aplication 
;	could also cite:
;
;	  C. Welnak, G.J. Chen and F. Cerrina, "SHADOW: a synchrotron 
;	  radiation X-ray Optics simulation tool",
;	  Nucl. Instr. and Meth. in Phys. Res. A347 91994) 344
;	or
;	  SHADOW WEB Page: http://www.xraylith.wisc.edu/shadow/shadow.html
;
; LAST MODIFICATION: msr/msr/99-03-16
;
;-
;
;       Author: M. Sanchez del Rio (srio@esrf.fr) , ESRF, Jan 1994
;       Modification history:
;		94-02-17 MSR includes the asymmetrical wiggler
;		94-10-27 MSR Removes common blocks. General cosmetics.
;		95-11-07 MSR adds file in/out. Adapts to xop 1.3b
;       	96/01/19 MSR changes labels Xtransmit -> XOP/Optics
;       	97/01/23 MSR adapts for Windows95. Cosmetics,version 1.1
;       	97/10/01 MSR renames read_ascii by rascii, uses sdep(),
;			 uses coltitles in xplot.
;       	97/10/02 MSR makes use of xop_wd,/default 
;       	97/10/27 MSR updates doc.
;       	97/11/10 MSR adds K(GAP) Tool.
;		98/11/05 MSR uses xop_spawn. Removes idl4 dependencies.
;		98/11/24 MSR changes input files for xop2.0. Version 1.2
;		99/03/16 MSR updates doc.
;		2002/07/19 srio@esrf.fr version 1.3 for XOP 2.1. Shadow-free
;
;
;========================================================================
;
FUNCTION XWIGGLER_VERSION
return,'1.3'
end
;
;==========================================================================
;
PRO xwiggler_event,event


Forward_Function xop_defaults
Catch, error_status
IF error_status NE 0 THEN BEGIN
   Message,/Info,'error caught: '+!err_string
   itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
     'XWIGGLER: error caught: '+!err_string)
   If Type(stateid) EQ 3 THEN $
     IF Widget_Info(stateid,/Valid_Id) AND N_Elements(state) NE 0 THEN $
     Widget_control, stateid,  Set_uvalue=state ; ,/No_Copy
   Catch, /Cancel
   On_Error,2
   RETURN
ENDIF
;
; register the events
;
Widget_control, event.id, Get_uvalue=eventuval

stateid = Widget_Info(event.handler,/Child)
Widget_control, stateid,  Get_uvalue=state ; ,/No_Copy

CASE eventuval OF
	'QUIT': BEGIN
		files = ['xwiggler.trj', 'xwiggler.par','xwiggler.out'] 
		Delete_Files,files,Group=event.top
		WIDGET_CONTROL,event.top,/DESTROY
		Return
		END
	'FILEINPUT': BEGIN
		action=''
		Widget_Control,event.id, Get_Value=action
		CASE action OF
		'Load from file...': BEGIN
		  ;if sdep() EQ 'UNIX' then filter='*.xop' else filter=0
		  filter='*.xop'
		  str_par = Xop_Input_Load(Title=$
		    'Select Xwiggler input file...',$
		    /NoConf,Filter=filter,Group=event.top)
		  IF Type(str_par) EQ 8 THEN BEGIN
		    tmp = state.str.parameters
		    Copy_Structure,str_par, tmp,Group=event.top, /OnlyFirstField
		    state.str.parameters = tmp
		  ENDIF
		  END
		'Save to file...': BEGIN
		  str_par = state.str.parameters
		  Xop_Input_Save,str_par,File='xwiggler.xop',$
		    /Write, Group=event.top, Comment='; xop/xwiggler(v'+$
		    xwiggler_version()+') input file on '+SysTime()
		  END
		'Save as default': BEGIN
		  str_par = state.str.parameters
		  Xop_Input_Save,str_par,Group=event.top, $
		    Default='xwiggler.xop',Comment='; xop/xwiggler(v'+$
		    xwiggler_version()+') input file on '+SysTime()
		  END
		ENDCASE
		END
	'SETDEF': BEGIN
		itmp = Dialog_Message(Dialog_Parent=event.top, $
		  /Question,['This option initializes the',$
		  'xwiggler parameters to their default values.',$
		  'Then you must click Set_parameters to run the program.',$
		  'Please confirm'],Title='xwiggler: set defaults')
		IF itmp EQ 'No' THEN GoTo,out
		tmp = Xop_Defaults('xwiggler')
		state.str.parameters = tmp.parameters
		END
	'KGAP_TOOL': BEGIN
		K=state.str.parameters.K
		Period=state.str.parameters.Ulambda
		Br=state.str.kgap_tool.Br
		Bp=state.str.kgap_tool.Bp
		kgap_tool,K=K,Period=Period,group=event.top,Br=Br,Bp=Bp
		state.str.parameters.K=K
		state.str.parameters.Ulambda=Period
		state.str.kgap_tool.Br=Br
		state.str.kgap_tool.Bp=Bp
		END
	'SETPAR': BEGIN
		inp = state.str.parameters

		ds = SDep(/ds)
                helpcmd = "xdisplayfileNative,'"+getenv('XOP_HOME')+$
                  ds+"doc"+ds+"xwiggler_par.txt'"

		XSCRMENU,inp,GROUP=base,NCOLUMN=2,/NOTYPE,$
		  TITLES=state.str.titles, FLAGS=state.str.flags,  $
		  ACTION=action, WTITLE='Xwiggler input values',/INTER, $
		  HELP=helpcmd

		if action eq 'DONT' then GoTo,out

		state.str.parameters = inp
		widget_control,/hourglass

		Delete_Files,['xwiggler.trj','xwiggler.par','xwiggler.out']
		Wiggler_Trajectory,inp,TrajFile='xwiggler.trj', $
		  ParFile='xwiggler.par'
		wiggler_spectrum,inp,outFile='xwiggler.out'
		END
        'HELP': xhelp,'xwiggler',Group=event.top
        'WRITE': BEGIN
		;
		; Prepare file with Power, needed for XOP/Optics
		;
		tmp = rascii('xwiggler.out')
		IF not(keyword_set(tmp) ) THEN BEGIN
		  print,'Xwiggler: Data in  xwiggler.out not found. '
		  itmp = Dialog_Message(dialog_parent=event.top, $
		    /ERROR,['wiggler data file [xwiggler.out] not found...',$ 
			' Set Parameters before...  '])
		  GoTo,out
		ENDIF ELSE BEGIN
		  tmp2 = tmp
		  tmp2(1,*)=tmp(1,*)*1.602189E-16
		  openw,Unit,'SRCOMPW',/GET_LUN
		  printf,Unit,tmp2
		  FREE_LUN,Unit
		  if sdep() EQ 'WINDOWS' then command = $
		    'copy xwiggler.out SRCOMPE' else command = $
		    '/bin/cp xwiggler.out SRCOMPE'
		  message,/info,'Executing: '+command
		  spawn,command
		  itmp = Dialog_Message(dialog_parent=event.top, $
			/INFO,'files SRCOMPE and SRCOMPW'+ $
			' written to disk.')
		ENDELSE
		END
	'PARAMETERS':	BEGIN
		WIDGET_CONTROL,event.id,GET_VALUE=eventval
		IF CheckFile('xwiggler.par') NE 1 THEN BEGIN
		 message,/info,'Data in file xwiggler.par not found. '
		 itmp =  Dialog_Message(dialog_parent=event.top, $
			/ERROR,['wiggler data file [xwiggler.par] not found...',$ 
			' Set Parameters before...  '])
		 goto,out
		ENDIF 
		XDISPLAYFILE1,'xwiggler.par',GROUP=event.top,title=$
		'xwiggler.par'
		END
	'SHOW':	BEGIN
		WIDGET_CONTROL,event.id,GET_VALUE=eventval
		IF CheckFile('xwiggler.out') NE 1 THEN BEGIN
		 message,/info,'Data in file xwiggler.out not found. '
		 itmp =  Dialog_Message(dialog_parent=event.top, $
			/ERROR,['wiggler data file [xwiggler.out] not found...',$ 
			' Set Parameters before...  '])
		 goto,out
		ENDIF 
		widget_control,/hourglass
		logFlag = Fix(state.str.parameters.logPlot[0])
		XPLOT,SPEC="xwiggler.out",XCOL=1,YCOL=2,title="#s",no_Block=1, $
		    xtitle='-1',ytitle='-1',WTITLE='xwiggler results: xwiggler.out', $
		    XLOG=logFlag,YLOG=logFlag
		END
	'TRAJ':	BEGIN
		WIDGET_CONTROL,event.id,GET_VALUE=eventval
		IF CheckFile('xwiggler.out') NE 1 THEN BEGIN
		 message,/info,'Data in file xwiggler.out not found. '
		 itmp =  Dialog_Message(dialog_parent=event.top, $
			/ERROR,['wiggler data file [xwiggler.out] not found...',$ 
			' Set Parameters before...  '])
		 goto,out
		ENDIF 
		widget_control,/hourglass
		XPLOT,SPEC="xwiggler.trj",XCOL=2,YCOL=1,title="#s",no_Block=1, $
		    xtitle='-1',ytitle='-1',WTITLE='xwiggler trajectories: xwiggler.out'
		END
	'TRAJ3D':	BEGIN
		WIDGET_CONTROL,event.id,GET_VALUE=eventval
		IF CheckFile('xwiggler.out') NE 1 THEN BEGIN
		 message,/info,'Data in file xwiggler.out not found. '
		 itmp =  Dialog_Message(dialog_parent=event.top, $
			/ERROR,['wiggler data file [xwiggler.out] not found...',$ 
			' Set Parameters before...  '])
		 goto,out
		ENDIF 
		widget_control,/hourglass
		buffer = ["file = 'xwiggler.trj'", $
			"pold = !p.charsize", $
			"!p.charsize=1.7", $
			"!p.multi=[0,1,2]", $
			"a=rascii(file)", $
			"x=Reform(a[0,*])", $
			" y=Reform(a[1,*])", $
			"  z=Reform(a[2,*])", $
			"vx=Reform(a[3,*])", $
			" vy=Reform(a[4,*])", $
			"  vz=Reform(a[5,*])", $
			"a1 = Min([x,z])", $
			"a2 = Max([x,z])", $
			"surface,y#x,y,x,/nodata,/save,xtitle='Y',zrange=[a1,a2],yrange=[a1,a2],zstyle=4,xstyle=4,ystyle=4", $
			"axis,0,0,xaxis=0,/t3d,XTitle='Y'", $
			"axis,0,0,yaxis=0,/t3d", $
			"axis,0,0,zaxis=0,/t3d", $
			"plots,y,x,z,/t3d", $
			"a1 = Min([vx,vz])", $
			"a2 = Max([vx,vz])", $
			"surface,y#vx,y,vz,/nodata,/save,xtitle='Y',zrange=[a1,a2],yrange=[a1,a2],zstyle=4,xstyle=4,ystyle=4", $
			"axis,0,0,xaxis=0,/t3d,XTitle='Y'", $
			"axis,0,0,yaxis=0,/t3d", $
			"axis,0,0,zaxis=0,/t3d", $
			"plots,y,vx,vz,/t3d", $
			"!p.multi=0", $
			"!p.charsize=pold"]
			XWindow,Buffer=buffer,Edit=2,Wtitle="Electron trajectory and velocity vs Y"

		END
ENDCASE
out:
Widget_Control, stateid, Set_Uvalue=state ; , /No_Copy
END
;
;==============================================================================
;
PRO XWIGGLER, GROUP=group, INPUTFILE=inputfile, No_Block=no_Block
;
;
Forward_Function xop_defaults
Catch, error_status
IF error_status NE 0 THEN BEGIN
   Message,/Info,'error caught: '+!err_string
   itmp = Dialog_Message(/Error,Dialog_Parent=group, $
     'XWIGGLER: error caught: '+!err_string)
   Catch, /Cancel
   On_Error,2
   RETURN
ENDIF

if xregistered('xwiggler') then return

IF N_Elements(no_Block) EQ 0 THEN BEGIN
  IF Xop_Getenv('XOP_MODE') EQ 'DEVELOPER' THEN no_block=1 ELSE no_block=0
ENDIF

;go to the default XOP working directory when xrgent is not called
;from the main XOP window
;if not(keyword_set(group)) then xop_wd,/default

;
; widget definition
;

base=WIDGET_BASE(/COLUMN,TITLE='Xwiggler '+xwiggler_version(),$
	MBAR=wMenuBar)

wButtons = widget_base(base,/Column) ; also to store state

wFile = widget_button(wMenuBar,VALUE='File',/MENU)
  wtmp0 = widget_button(wFile,VALUE='XWiggler input parameters', /Menu)
  wtmp = widget_button(wtmp0,VALUE='Load from file...',UValue='FILEINPUT')
  wtmp = widget_button(wtmp0,VALUE='Save to file...',UValue='FILEINPUT')
  wtmp = widget_button(wtmp0,VALUE='Save as default',UValue='FILEINPUT')
  wtmp = widget_button(wFile,VALUE='Write Files for XOP/Optics',UValue='WRITE')
  wtmp = widget_button(wFile,VALUE='Quit', UVALUE='QUIT',/SEPARATOR)

wSetParameters = widget_button(wMenuBar,VALUE='Set_Parameters', /MENU)
  wtmp = widget_button(wSetParameters,VALUE='K(Gap) Tool', UVALUE='KGAP_TOOL')
  wtmp = widget_button(wSetParameters,VALUE='Set Parameters', UVALUE='SETPAR')
  wtmp = widget_button(wSetParameters,VALUE='Set Defaults', UVALUE='SETDEF')

wResults = widget_button(wMenuBar,VALUE='Show',/MENU)
  wSpec = widget_button(wResults,VALUE='Spectra',UVALUE='SHOW')
  wTraj = widget_button(wResults,VALUE='Trajectories',/Menu)
    wtmp = widget_button(wTraj,VALUE='All',UVALUE='TRAJ')
    wtmp = widget_button(wTraj,VALUE='3D View',UVALUE='TRAJ3D')
  wtmp = widget_button(wResults,VALUE='Parameters',UVALUE='PARAMETERS')

wHelp = widget_button(wMenuBar,VALUE='Help', /Help)
  wtmp = widget_button(wHelp,VALUE='xwiggler', UVALUE='HELP')
;
wtmp = widget_button(BASE,VALUE='Set Parameters', UVALUE='SETPAR')

if sdep() EQ 'WINDOWS' then font = 'VERDANA*BOLD*ITALIC*24' else $
  font = '-adobe-helvetica-bold-o-normal--18-180-75-75-p-104-iso8859-1'

;junk = WIDGET_LABEL( BASE, FONT=font, VALUE=' SHADOW utilities:')
junk = WIDGET_LABEL( BASE, FONT=font, VALUE='   XWIGGLER   ')
junk = WIDGET_LABEL( BASE, FONT=font, VALUE='      Wiggler   Radiation      ')

;
;state
;
str = xop_defaults('xwiggler')
IF KeyWord_Set(inputFile) THEN BEGIN
  str_par = Xop_Input_Load(InputFile=inputFile)
  IF Type(str_par) EQ 8 THEN BEGIN
    tmp = str.parameters
    Copy_Structure,str_par, tmp, Group=group
    str.parameters = tmp
  ENDIF
ENDIF
wids = {dummy:0L}
IF N_Elements(no_Block) EQ 0 THEN no_Block=0
state = { str:str, wids:wids, no_Block:no_Block}
;
;actions
;
Widget_control,Widget_Info(base,/Child),Set_uvalue=state ; ,/No_Copy
Widget_control,base,/Realize

Xmanager,'xwiggler',base,Group_leader=group, No_Block=no_Block
END
