FUNCTION sync_ene,f_psi,energy_ev,ec_ev=ec_ev,Pol=pol,  $
  E_GeV=e_gev,I_A=i_a,HDiv_mrad=hdiv_mrad, $
  Psi_Min=psi_min, Psi_Max=psi_max, Psi_NPoints=psi_npoints, $
  FMatrix=fMatrix,Angle_mrad=angle_mrad

;+
; NAME:
;       SYNC_ENE
;
; PURPOSE:
;       Calculates the synchrotron radiation energy spectrum
;
; CATEGORY:
;       Mathematics.
;
; CALLING SEQUENCE:
;       Result = sync_ene(flag, Energy )
;
; INPUTS:
;	flag: 	0 Flux fully integrated in angle (Psi)
;		1 Flux at Psi=0
;		2 Flux integrated in the angular interval [Psi_Min,Psi_Max]
;		3 Flux at Psi=Psi_Min
;
;       energy:  the energy array [in eV]
;
; KEYWORD PARAMETERS:
;	POLARIZATION: 0 Total 
;		      1 Parallel       (l2=1, l3=0, in Sokolov&Ternov notation)
;		      2 Perpendicular  (l2=0, l3=1)
;
;	If FLAG=0 or FLAG=1 the following keywords MUST BE ENTERED
;
;		EC_eV= The critical energy [eV]
;		E_GeV= The electron energy [in GeV] 
;		I_A= the electron beam intensity [in A]
;		HDiv_mrad= the horizontal divergence [in mrad]
;
;	If FLAG=2, in addition to the mentioned keywords, the following 
;		ones must be present:
;
;		Psi_Min the minimum integration angle [in mrad]
;		Psi_Max the maximum integration angle [in mrad]
;		Psi_NPoints the number of points in psi for integration
;
;	If FLAG=3, in addition to the mentioned keywords for FLAG=0 OR 
;
;		FLAG=1, the following kewford must be defined: 
;		Psi_Min the Psi angular value [in mrad]
;
; KEYWORD PARAMETERS (OUTPUT):
;
;	IF FLAG=2, the following keywords can be used to obtain additional info:
;
;		FMATRIX=A two dimensional variable containing the matrix of 
;			flux as a function of angle [first index] and energy 
;			[second index]
;		ANGLE_MRAD= A one-dim array with the angular points [in mrad]
;
; OUTPUTS:
;       returns the array with the flux [photons/sec/0.1%bw] for FLAG=0,2
;       and the flux [photons/sec/0.1%bw/mrad] for FLAG=1,3
;
; PROCEDURE:
;
;	References: 
;		G K Green, "Spectra and optics of synchrotron radiation" 
;			BNL 50522 report (1976)
;		A A Sokolov and I M Ternov, Synchrotron Radiation, 
;			Akademik-Verlag, Berlin, 1968
;
; EXAMPLE:
;	The following program was used for testing sync_ene
;	
;	
;	e=makearray1(10,20000.0,30000.0) ; create 10-points energy array in [20,30] keV
;	
;	;
;	; test of spectra at Psi=0
;	;
;	; at psi=0 (i.e., FLAG=1)
;	y1=sync_ene(1,e,ec_ev=19166.0,e_gev=6,i_a=0.1,hdiv=1)
;	; at psi_min (FLAG=3)
;	y2=sync_ene(3,e,ec_ev=19166.0,e_gev=6,i_a=0.1,hdiv=1,psi_min=0.0)
;	
;	plot,e,y1
;	oplot,e,y2,Psym=2
;	
;	
;	;
;	; test of integrated spectra 
;	;
;	
;	; Integrating (by hand) using FLAG=3
;	y3 = y1*0.0D0
;	a = makearray1(50,-0.2,0.2)  ; it is large enough to cover the full radiation fan
;	
;	FOR i=0,N_Elements(a)-1 DO BEGIN
;	  y2=sync_ene(3,e,ec_ev=19166.0,e_gev=6,i_a=0.1,hdiv=1,psi_min=a[i])
;	  y3 = y3 + y2
;	  ;plot,e,y2,title='Psi='+StrCompress(a[i])
;	  ;wait,0.1
;	ENDFOR
;	y3=y3*(a[1]-a[0])
;	
;	; Integrating (automatically) using FLAG=2
;	y4 = sync_ene(2,e,ec_ev=19166.0,e_gev=6,i_a=0.1,hdiv=1,psi_min=-0.2,psi_max=0.2,psi_npoints=50)
;	
;	; Integrated (over all angles) using FLAG=0
;	y5 = sync_ene(0,e,ec_ev=19166.0,e_gev=6,i_a=0.1,hdiv=1)
;	
;	pause
;	plot,e,y4
;	pause
;	oplot,e,y4,psym=2
;	pause
;	oplot,e,y5,psym=4
;	
;	print,'   energy    int_by_hand     int_num    int'
;	FOR i=0,N_Elements(e)-1 DO BEGIN
;	  print,e[i],y3[i],y4[i],y5[i]
;	ENDFOR
;	
;	end
;
;	The results obtained are: 
;		  energy    int_by_hand     int_num       int
;	       20000.000   9.3255428e+12   9.3255428e+12   9.3319989e+12
;	       21111.111   8.9528668e+12   8.9528668e+12   8.9590648e+12
;	       22222.222   8.5885671e+12   8.5885671e+12   8.5945128e+12
;	       23333.333   8.2334350e+12   8.2334350e+12   8.2391347e+12
;	       24444.444   7.8880553e+12   7.8880553e+12   7.8935159e+12
;	       25555.556   7.5528452e+12   7.5528452e+12   7.5580737e+12
;	       26666.667   7.2280844e+12   7.2280844e+12   7.2330880e+12
;	       27777.778   6.9139400e+12   6.9139400e+12   6.9187260e+12
;	       28888.889   6.6104867e+12   6.6104867e+12   6.6150626e+12
;	       30000.000   6.3177237e+12   6.3177237e+12   6.3220968e+12
;
;
;	EXAMPLE 2
;		Surface plot of flux versus angle ane energy
;		e = makeArray1(20,20000,30000)
;		tmp = sync_ene(2,e,ec_ev=19166.0,e_gev=6,i_a=0.1,hdiv=1,$
;			psi_min=-0.2,psi_max=0.2,psi_npoints=50,fm=fm,an=a)
;		surface,fm,a,e, ztitle='Flux[phot/sec/0.1%bw/mradPsi', $
;			xtitle='Angle [mrad]',ytitle='Energy [eV]'
; MODIFICATION HISTORY:
;       Written by:     M. Sanchez del Rio, srio@esrf.fr, 2002-06-03
;
;-

CASE Fix(f_psi) OF
  0: BEGIN ; fully integrated in Psi
	; numerical cte for integrated flux
        a8 = sqrt(3d0)*9d6*physical_constants('ec')/physical_constants('h')/ $
	physical_constants('c')/physical_constants('mee') 
	a5 = a8*e_GeV*i_a*hdiv_mrad*Sync_G1(energy_ev/ec_ev,Pol=pol)
	END
  1: BEGIN ; at Psi = 0
	a8 = physical_constants('ec')/(physical_constants('mee'))^2/ $
	physical_constants('h')*(9d-2/2/!dpi) ; 1.3264d13
	a5 = a8*(e_gev)^2*i_a*hdiv_mrad*sync_hi(energy_ev/ec_ev,Pol=pol)
	END

  2: BEGIN ; between PsiMin and PsiMax
        a8 = physical_constants('ec')/(physical_constants('mee'))^2/ $
          physical_constants('h')*(9d-2/2/!dpi) ; 1.3264d13
        eene = energy_ev/ec_ev
        gamma = E_GEV*1d3/Physical_Constants('MEE')
	angle_mrad = MakeArray1(psi_npoints,psi_min,psi_max)
	angle_step = (Double(psi_Max)-Double(Psi_Min))/Double(Psi_NPoints-1)
        ;;a5=Total(sync_f(angle_mrad*gamma/1d3,eene,Pol=pol),1) * angle_step
	;;a5 = a5*(eene)^2*a8*i_A*hDiv_mrad*(e_gev)^2
	eene2 = (angle_mrad*0.0D0+1)#eene
        a5=sync_f(angle_mrad*gamma/1d3,eene,Pol=pol) 
	a5 = a5*(eene2)^2*a8*i_A*hDiv_mrad*(e_gev)^2
	fMatrix = a5*angle_step
	a5 = Total(fMatrix,1) 
	END
  3: BEGIN ; at PsiMin
        a8 = physical_constants('ec')/(physical_constants('mee'))^2/ $
          physical_constants('h')*(9d-2/2/!dpi) ; 1.3264d13
        eene = energy_ev/ec_ev
        gamma = E_GEV*1d3/Physical_Constants('MEE')
	angle_mrad = psi_min
        a5=sync_f(angle_mrad*gamma/1d3,eene,Pol=pol)
	a5 = a5*(eene)^2*a8*i_A*hDiv_mrad*(e_gev)^2
	END
ENDCASE

RETURN, a5
END
