;+
;
;==================================  xfh      ==================================
;
; XFH is a widget based graphical interface to calculate structure
; factor of crystals.
;
; The data are taken from the DABAX data base.
;
; It access the following DABAX files:
;	CrystalStructures.dat
;	CrystalCell.dat
;	f1f2_*.dat
;	f0_CromerMann.dat,f0_CromerMann_old1968.dat,f0_WaasKirf.dat
;
;
; CUSTOMIZATION OF XFH INPUT PARAMETERS:
;  -If you want to add(remove) another f1f2* file, just add(remove) it
;   in any(all) directory of $DABAX_PATH. 
;  -If you want to add(remove) another f0 file, follow the inscructions
;	given before.
;  -If you want to modify the crystal list, do the following:
;        a) copy the CrystalStructures.dat and CrystalCell.dat files to
;           a given directory (i.e. current directory ".")
;        b) Modify these files to add/remove/change the entries. Both
;	    files must have the same entries (or "scans").
;        c) Redefine $DABAX_PATH to include your new directoty:
;           setenv  DABAX_PATH = .:$DABAX_PATH
;        d) Restart the application.
;
; DESCRIPTION OF THE CONTROLS IN THE MAIN WINDOW:
;
;  File:
;    Xfh input parameters: This option allows to save the current
;		parameters to a file for later loading. It also allows
;		to save the current parameters as defaults for being
;		used when the application is initialized. In the last
;		case, the file is named "application".xop (where 
;		"application " is the name of the current XOP
;		application) and is written in the directory pointed
;		by the XOP_DEFAULTS_DIR environment variable (which
;		must be set). The parameter file is ASCII and can be
;		read and edited with care.
;  Quit: to exit from the program
;
; Set_Parameters:
;  Set Parameters: to define the parameters for the calculation.
;		The same result is obtained pressing the "Set Parameters"
;		button in the main Xfh window. 
;               Please refer to the information under the HELP
;		button for a complete description of the parameters. After 
;		pressing the ACCEPT button, xfh starts running and
;		creates a graphical window with the results.
;  Set Defaults: Sets the default parameters.
;
;  Show: Diaplay results
;    Plot crystal parameters: performs a plot of the energy-dependent parameters
;		(this option os only available for a number of energy points 
;		 nonequal to one)
;    Show crystal parameters: Show text information of the crystal parameters
;    Show crystallographic parameters: Show parameters used to build the 
;		structure factor. 
;  Help:   Shows the xfh help (this text).
;
;
; COPYRIGHT:
;	xfh  belongs to XOP package and it is distributed within XOP.
;	PLEASE REFER TO THE XOP COPYRIGHT NOTICE BEFORE USING IT.
;
; CREDITS:
;	Published calculations made with XOP should refer:
;
;	  M. Sanchez del Rio and R. J. Dejus "XOP: Recent Developments"
;	  SPIE proceedings vol. 3448, pp.340-345, 1998.
;
;
; LAST MODIFICATION: msr/msr/2003-01-10
;
;-
;
; =========================================================================
;
;	MODIFICATION HISTORY:
;       by  Manuel Sanchez del Rio. ESRF. July 1996.
;       96/12/04 srio@esrf.fr Small fixes.
;       97/01/15 srio@esrf.fr adapts top Windows.
;       97/10/15 srio@esrf.fr makes some changes for xop 1.9
;       98/12/14 srio@esrf.fr adapts for XOP 2.0. Version 1.1
;       02/08/29 srio@esrf.fr adapted for XOP 2.1. 
;			      Added energy dependence Version 1.2
;
; =========================================================================
;
Function xfh_version
return,'1.2'
end
;
;=====================================================================
;
PRO xfh_event,event


Catch, error_status
IF error_status NE 0 THEN BEGIN
   Message,/Info,'error caught: '+!err_string
   itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
     'XFH_EVENT: error caught: '+!err_string)
   Catch, /Cancel
   If Type(stateid) EQ 3 THEN $
     If Widget_Info(stateid,/Valid_Id) AND N_Elements(state) NE 0 THEN $
     Widget_Control,stateid,Set_UValue=state ; ,/No_Copy
   On_Error,2
   RETURN
ENDIF

Widget_Control, event.id, get_UValue=eventUValue

if n_elements(eventuvalue) EQ 0 then eventuvalue = ''
if not(keyword_set(eventuvalue)) then eventuvalue = ''

stateid = Widget_Info(event.handler,/Child)
Widget_Control, stateid, get_UValue=state ; , /No_Copy

case eventuvalue of

  'FILEINPUT': BEGIN
    action=''
    Widget_Control,event.id, Get_Value=action
    CASE action OF
      'Load from file...': BEGIN
        if sdep() EQ 'UNIX' then filter='*.xop' else filter=0
        str_par = Xop_Input_Load(Title=$
        'Select xFH input file...',$
        /NoConf,Filter=filter,Group=event.top)
        IF Type(str_par) EQ 8 THEN BEGIN
          tmp = state.str.parameters
          Copy_Structure,str_par, tmp, Group=event.top , /OnlyFirstField
          state.str.parameters = tmp
        ENDIF
      END
      'Save to file...': BEGIN
        str_par = state.str.parameters
        Xop_Input_Save,str_par,File='xfh.xop',$
          /Write, Group=event.top, Comment='; xop/xfh(v'+$
        xfh_version()+') input file on '+SysTime()
      END
      'Save as default': BEGIN
        str_par = state.str.parameters
        Xop_Input_Save,str_par,Group=event.top, $
          Default='xfh.xop',Comment='; xop/xfh(v'+$
        xfh_version()+') input file on '+SysTime()
      END
    ENDCASE
  END
  'QUIT':begin	
	delete_files,['xfh.bra','xfh.out','xfh.inf']
	widget_control,/destroy,event.top
	return
	end
  'HELP': Xhelp,'xfh',GROUP=event.top

  'SETDEF': BEGIN
		itmp = Dialog_Message(Dialog_Parent=event.top,$
		/Question,['This option initializes the',$
		'xFH parameters to their default values.',$
		'Then you must click Set_parameters to run the program.',$
		'Please confirm.'],title='xFH')
		if itmp eq 'No' then goto,out
		state.str.parameters = state.str_defaults
	END
  'SET':begin	
	delete_files,['xfh.bra','xfh.out','xfh.inf']
	str1 = state.str.parameters
        helpcmd="xdisplayfileNative,'"+Xop_GetEnv('XOP_HOME')+$
                 sdep(/ds)+'doc'+sdep(/ds)+"xfh_par.txt'"
  	XScrMenu,str1,/Interp,/NoType,action=action,Ncol=3, $
		titles=state.str.titles,flags=state.str.flags, $
		help=helpcmd,wtitle='xFh input parameters', $
		Dialog_Parent=event.top
  	if  action EQ 'DONT' then goto,out
  	widget_control,/hourglass

	;
	; run bragg_calc
	;
	str = dabax_defaults('bragg',group=group)
  	for i=0,7 do str.parameters.(i) = str1.(i)
  	str.parameters.i_absorp(0) = '1'
  	str.parameters.temper = str1.temper
	IF str1.npoints EQ 1 THEN BEGIN
  	  str.parameters.emin = str1.energy-1000.0
  	  str.parameters.emax = str1.energy+1000.0
  	  str.parameters.estep = 50.0
	ENDIF ELSE BEGIN
  	  str.parameters.emin = min([str1.energy,str1.energy_end])-100.0
  	  str.parameters.emax = max([str1.energy,str1.energy_end])+100.0
  	  str.parameters.estep = 50.0
	ENDELSE
  	str.parameters.outfil = 'xfh.bra'
	state.str.parameters = str1

  	out= bragg_calc(str,group=event.top)
    	bragg_out,out ; writes file xfh.bra

	;
	; run crystal_fh
	;
  	text = ''
	IF str1.npoints EQ 1 THEN ener = str1.energy ELSE BEGIN
	  IF str1.npoints LT 0 THEN log=1 else log=0
	  ener = MakeArray1(Abs(str1.npoints),str1.energy,str1.energy_end,Log=log)
	ENDELSE
  	crystal_fh,out,ener,TEXT=TEXT,out=result
	;
	; write outputs to files
	;
	OpenW,unit,'xfh.inf',/Get_Lun
	FOR i=0L,N_Elements(text)-1 DO PrintF,unit,text[i]
	Free_Lun,unit

	IF Abs(str1.npoints) GT 1 THEN BEGIN
	tmp = Make_Set(result.phot, $
	    Physical_Constants('hc')/result.phot, $
	    result.theta, $
	    double(result.f_0), $
	    imaginary(result.f_0),  $
	    double(result.fh), $
	    imaginary(result.fh),  $
	    double(result.fh_bar), $
	    imaginary(result.struct),  $
	    double(result.struct), $
	    imaginary(result.fh_bar),  $
	    result.delta_ref, $
	    double(result.refrac), $
	    imaginary(result.refrac),  $
	    result.absorp, $
	    2.0d6*result.ssr, $
	    2.0d6*result.spr, $
	    result.ratio, $
	    result.psi_over_f)

	ncol = '19'
	OpenW,unit,'xfh.out',/Get_Lun
	PrintF,unit,"#F xfh.out"
	PrintF,unit,"#D "+SysTime()
        PrintF,unit,"#S 1 xop/xfh run"
        PrintF,unit,"#C results of xop/xfh (crystal structure factor) "
        PrintF,unit,"#N "+ncol
        ;PrintF,unit,"#UT0 Crystal_Bent 1.2 {cryst_ml} Si 111 E=33144.0eV, t=0.70mm, R=12.9m, !4a!X=63.7deg
        PrintF,unit,"#L Photon energy [eV]  Wavelength [A]  Bragg angle [rad]  Re(f_0)  Im(f_0)  "+ $
		"Re(FH)  Im(FH)  Re(FH_BAR)  Im(FH_BAR)  Re(F(h,k,l))  Im(F(h,k,l))  delta (1-Re(refrac))  "+ $
		"Re(refrac index)  Im(refrac index)  absorption coeff  "+$
		"s-pol Darwin width [microrad]  p-pol Darwin width [microrad]  Sin(Bragg angle)/Lambda  "+$
		"psi_over_f"
	FOR i=0L,(size(tmp))[2]-1 DO BEGIN
	  tmp1 = Reform(tmp[*,i])
	  PrintF,unit,tmp1,Format='('+ncol+'(G20.13," "))'
	ENDFOR
	Free_Lun,unit
	ENDIF

	state.str.parameters = str1
	end
  'SHOW': BEGIN
          WIDGET_CONTROL,event.id,GET_VALUE=eventval
          ffile = 'xfh.inf'
          IF (checkfile(ffile) EQ 0) THEN BEGIN
             message,/info,'File not found. '
             itmp = Dialog_Message(dialog_parent=event.top,$
                        /ERROR,['xFh file not found...',$
                        'Set Parameters before...  '])
             Goto,Out
          ENDIF
	  CASE eventval OF
	    'Plot crystal parameters': BEGIN
		IF state.str.parameters.npoints EQ 1 THEN BEGIN
		  itmp = Dialog_message(/Info,'Option only available for more than one energy point',$
			Dialog_Parent=event.top)
		  GoTo,out
		ENDIF
		Xplot,Parent=p,/No_Block,Title="xFh"
		Xplot_LoadFile,p,spec='xfh.out',/NoRefresh, WTitle='xFh results'
		Xplot_Controls_Action,p,PSymbol=1,/NoRefresh
		Xplot_ChangeCol,p,YCol=10,/NoRefresh
		Xplot_setLimits,p,ystyle=16
		END
	    'Show crystal parameters': BEGIN
		XDisplayfile1,'xfh.inf',Dialog_parent=event.top
		END
	    'Show crystallographic parameters': BEGIN
		XDisplayfile1,'xfh.bra',Dialog_parent=event.top
		END
	    else: message,'case not found: '+eventval
	  ENDCASE
	END
  else:
endcase

out:
Widget_Control, stateid, set_UValue=state ; , /No_Copy
end
;
;=======================================================================
;
PRO xfh,GROUP=group, InputFile=inputFile, No_Block=no_Block
;
Forward_Function dabax_defaults
;
Catch, error_status
IF error_status NE 0 THEN BEGIN
   Message,/Info,'error caught: '+!err_string
   itmp = Dialog_Message(/Error,Dialog_Parent=group, $
     'XOP_IFC_READ: error caught: '+!err_string)
   Catch, /Cancel
   On_Error,2
   RETURN
ENDIF

IF xregistered('XFH') THEN RETURN

IF N_Elements(no_block) EQ 0 THEN no_block=1

wbase = widget_base(/COLUMN,TITLE='XFH '+xfh_version(),$
	MBAR=wMenuBar)
;
;
; the blocks box
;
Bbox=widget_base(wbase,/Column) ; also to store state
 

;Menu bar
wFile =  WIDGET_BUTTON(wMenuBar, VALUE='File', /MENU)
  wtmp0 = widget_button(wFile,VALUE='xFH input parameters', /Menu)
    wtmp = widget_button(wtmp0,VALUE='Load from file...',UValue='FILEINPUT')
    wtmp = widget_button(wtmp0,VALUE='Save to file...',UValue='FILEINPUT')
    wtmp = widget_button(wtmp0,VALUE='Save as default',UValue='FILEINPUT')
  wtmp = widget_button(wFile,VALUE='Quit', UVALUE='QUIT',/SEPARATOR)

wSetParameters = widget_button(wMenuBar,VALUE='Set_Parameters', /MENU)
  wtmp = widget_button(wSetParameters,VALUE='Set Parameters', UVALUE='SET')
  wtmp = widget_button(wSetParameters,VALUE='Set Defaults', UVALUE='SETDEF')

wResults = widget_button(wMenuBar,VALUE='Show',/MENU)
  wtmp = widget_button(wResults,VALUE='Plot crystal parameters', UVALUE='SHOW')
  wtmp = widget_button(wResults,VALUE='Show crystal parameters', UVALUE='SHOW')
  wtmp = widget_button(wResults,VALUE='Show crystallographic parameters', $
        UVALUE='SHOW')


wHelpMenu = WIDGET_BUTTON(wMenuBar, VALUE='Help', /HELP)
  wtmp = WIDGET_BUTTON(wHelpMenu, VALUE='xFh', UVALUE='HELP')

tmp = widget_button(Bbox,VALUE='Set Parameters',UVALUE='SET')



if sdep() EQ 'WINDOWS' then $
font = 'VERDANA*BOLD*ITALIC*24' else $
font = '-adobe-helvetica-bold-o-normal--18-180-75-75-p-104-iso8859-1'

junk = WIDGET_LABEL( Bbox, FONT=font, VALUE=' xFh')
junk = WIDGET_LABEL( Bbox, FONT=font, VALUE='Crystal Structure Factors.')


str = dabax_defaults('xfh',group=wbase)
str_defaults = str.parameters


IF KeyWord_Set(inputFile) THEN BEGIN
  str_par = Xop_Input_Load(InputFile=inputFile)
  IF Type(str_par) EQ 8 THEN BEGIN
    tmp = str.parameters
    Copy_Structure,str_par, tmp, Group=group, /OnlyFirstField
    str.parameters = tmp
  ENDIF
ENDIF
state = { str:str, str_defaults:str_defaults }

widget_control,Widget_Info(wbase,/Child),set_uvalue=state
widget_control,wbase,/REALIZE
xmanager,'XFH',wbase,GROUP=group, No_Block=no_Block
;
end
