;+
;
;==================================  XCrossSec      ============================
;
; XCrossSec is a widget based graphical interface to calculate Photon-Atom
; Cross Sections, Mass Absorption Coefficients amd Liner Absorption Coeff.
; for elements, compounds and mixtures.
; The data are taken from the DABAX data base. 
; It uses the following DABAX files:
;    CrossSec_*.dat  (for tabulated Cross Sections. 
;                       Presently *={EPDL,XCOM,McMaster}
;    Compounds.dat   (for mixture and compounds list)
;    AtomicConstants.dat (for atomic Mass and Density)
;
; Possible calculations:
;	Cross Section [barn/atom]
;	Cross Section [cm^2]
;	Mass Absorption Coefficient [cm^2/g]
;	Linear Absorption Coefficient [cm^-1]
;	Energy Transfer (Linear and Mass) Coeff [cm^2/g,cm^-1]: This 
;	 option is only allowed by using the DABAX file 
;	 CrossSec_EDPL.dat, where these values are tabulated. 
;	 See doc on this file for additional information.
;
;
; CUSTOMIZATION OF XCROSSSEC INPUT PARAMETERS:
;  -If you want to add(remove) another CrossSec_* file, just add(remove) it
;	in any(all) directory of $DABAX_PATH. Note that the CrossSec
;	files containing all the partial cross section (the ones used
;	here) only have a single underscore (_). 
; -If you want to modify the mixture table  list, do the following:
;     > a) copy the Compounds.dat DABAX file to a given directory 
;	(e.g., current directory ".")
;     > b) Modify this file to add/remove/change the entries.
;     > c) Redefine $DABAX_PATH to include your new directoty:
;	setenv  DABAX_PATH = .:$DABAX_PATH
;     > d) Restart the application.
;
;
; DESCRIPTION OF THE CONTROLS IN THE MAIN WINDOW:
;
;  File:
;    XCrossSec input parameters: This option allows to save the current
;		parameters to a file for later loading. It also allows
;		to save the current parameters as defaults for being
;		used when the application is initialized. In the last
;		case, the file is named "application".xop (where 
;		"application " is the name of the current XOP
;		application) and is written in the directory pointed
;		by the XOP_DEFAULTS_DIR environment variable (which
;		must be set). The parameter file is ASCII and can be
;		read and edited with care.
;  Quit: to exit from the program
;
; Set_Parameters:
;  Set Parameters: to define the parameters for the calculation.
;		The same result is obtained pressing the "Set Parameters"
;		button in the main XCrossSec window. 
;               Please refer to the information under the HELP
;		button for a complete description of the parameters. After 
;		pressing the ACCEPT button, xcrosssec starts running and
;		creates a graphic window with the results.
;  Set Defaults: Sets the default parameters.
;
;  Help:   Shows the XCrossSec help (this text).
;
;
; COPYRIGHT:
;	xcrosssec  belongs to XOP package and it is distributed within XOP.
;	PLEASE REFER TO THE XOP COPYRIGHT NOTICE BEFORE USING IT.
;
; CREDITS:
;	Published calculations made with XOP should refer:
;
;	  M. Sanchez del Rio and R. J. Dejus "XOP: Recent Developments"
;	  SPIE proceedings vol. 3448, pp.340-345, 1998.
;
;
; LAST MODIFICATION: msr/msr/03-01-10
;
;-
; =========================================================================
;
;	MODIFICATION HISTORY:
;       by  Manuel Sanchez del Rio. ESRF. December 3 1996.
;	97/01/15 srio@esrf.fr adapts to Windows.
;	97/02/20 srio@esrf.fr corrects a bug when using compounds from
;		table. Bug noticed by D. Abernathy.
;	97/10/15 srio@esrf.fr makes some changes for xop 1.9
;	98/12/14 srio@esrf.fr adapts for XOP2,0. version 1.1.
;
;-
Function xcrosssec_version
return,'1.1'
end
;
;=====================================================================
;
PRO xcrosssec_event,event


Catch, error_status
IF error_status NE 0 THEN BEGIN
   Message,/Info,'error caught: '+!err_string
   itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
     'XCROSSSEC_EVENT: error caught: '+!err_string)
   Catch, /Cancel
   If Type(stateid) EQ 3 THEN $
     If Widget_Info(stateid,/Valid_Id) AND N_Elements(state) NE 0 THEN $
     Widget_Control,stateid,Set_UValue=state,/No_Copy
   On_Error,2
   RETURN
ENDIF
Widget_Control, event.id, get_UValue=eventUValue

if n_elements(eventuvalue) EQ 0 then eventuvalue = ''
if not(keyword_set(eventuvalue)) then eventuvalue = ''

stateid = Widget_Info(event.handler,/Child)
Widget_Control, stateid, get_UValue=state, /No_Copy

case eventuvalue of

  'FILEINPUT': BEGIN
    action=''
    Widget_Control,event.id, Get_Value=action
    CASE action OF
      'Load from file...': BEGIN
        if sdep() EQ 'UNIX' then filter='*.xop' else filter=0
        str_par = Xop_Input_Load(Title=$
        'Select xCrossSec input file...',$
        /NoConf,Filter=filter,Group=event.top)
        IF Type(str_par) EQ 8 THEN BEGIN
          tmp = state.str.parameters
          Copy_Structure,str_par, tmp, Group=event.top , /OnlyFirstField
          state.str.parameters = tmp
        ENDIF
      END
      'Save to file...': BEGIN
        str_par = state.str.parameters
        Xop_Input_Save,str_par,File='xcrosssec.xop',$
          /Write, Group=event.top, Comment='; xop/xcrosssec(v'+$
        xcrosssec_version()+') input file on '+SysTime()
      END
      'Save as default': BEGIN
        str_par = state.str.parameters
        Xop_Input_Save,str_par,Group=event.top, $
          Default='xcrosssec.xop',Comment='; xop/xcrosssec(v'+$
        xcrosssec_version()+') input file on '+SysTime()
      END
    ENDCASE
  END

  'SETDEF': BEGIN
		itmp = Dialog_Message(Dialog_Parent=event.top,$
		/Question,['This option initializes the',$
		'xCrossSec parameters to their default values.',$
		'Then you must click Set_parameters to run the program.',$
		'Please confirm.'],title='xCrossSec')
		if itmp eq 'No' then goto,out
		state.str.parameters = state.str_defaults
	END
  'QUIT':begin	
	widget_control,/destroy,event.top
	return
	end
  'HELP': xhelp,'xcrosssec',GROUP=event.top
  'SET':begin	
	str = state.str.parameters
        helpdir = Xop_GetEnv('XOP_HOME')+SDep(/ds)+'doc'
	helpcmd="xdisplayfileNative,'"+helpdir+sdep(/ds)+"xcrosssec_par.txt'
  	XScrMenu,str,/Interp,/NoType,action=action,Ncol=2, $
		titles=state.str.titles, $
		flags=state.str.flags,help=helpcmd,$
		wtitle='xCrossSec input parameters'
  	if  action EQ 'DONT' then goto,out
	state.str.parameters = str
	; mat_flag : 0=element, 1=mixture(form) 2=mixture(table)
	mat_flag = fix(str.mat_flag(0))
	case mat_flag of
	0: begin
	    iname = 0 
            descriptor = strcompress(str.descriptor,/rem)
            if strlen(descriptor) EQ 1 then descriptor=descriptor+' '
	    imix = 0
	   end
	1: begin
	    iname = 0 
            descriptor = strcompress(str.descriptor,/rem)
            if strlen(descriptor) EQ 1 then descriptor=descriptor+' '
	    density = str.density
	    imix = 1
	   end
	2: begin
	    iname = 1 & descriptor = str.mat_list(fix(str.mat_list(0))+1)
	    density = 0
	    imix = 1
	   end
	endcase
	
  	widget_control,/hourglass

        ; define an array with the selected datasets
        if strcompress(str.datasets(0),/rem) EQ '0' then $
                datasets = str.datasets(2:N_elements(str.datasets)-1) else $
                datasets = str.datasets(fix(str.datasets(0))+1)

	case strcompress(str.grid(0),/rem) of
	  '0':
	  '1': BEGIN
		if str.gridn EQ 1 then energy = [str.gridstart] else $
		energy = findgen(str.gridn)/(float(str.gridn)-1) * $
		(str.gridend-str.gridstart) + str.gridstart
		END
	  '2': energy = [str.gridstart]
	endcase
	; icalc is the magnitude text for calculations
	; put in f1f2_calc,...,f=icalc
	icalc = str.calculate(0)

	iunit = fix(str.unit(0))

	FOR I = 0,n_elements(datasets)-1 DO BEGIN ; starts main loop
	    fcol_tit = 0
	    if imix then $
	      fout = cross_calc_mix(datasets(i),descriptor,energy,fcol_tit,$
		partial=icalc,unit=iunit,name=iname,density=density,$
	        group=event.top) else $
	      fout = cross_calc(datasets(i),descriptor,energy,fcol_tit,$
		partial=icalc,unit=iunit,group=event.top)
	    fcol_tit = strmid(datasets(i),9,strpos(datasets(i),'.')-9)+$
		':'+fcol_tit
	    if I eq 0 then begin
	      out = fltarr(1,n_elements(energy))
	      out(0,*) = reform(energy)
	      col_tit = ['PhotonEnergy [eV]']
	    endif
	    ncol = n_elements(out(*,0))
	    npoints = n_elements(out(0,*))
	    out_old = out
	    out = fltarr(ncol+n_elements(fout(*,0)),npoints)
	    out(0:ncol-1,*) = out_old(0:ncol-1,*)
	    out(ncol:ncol+n_elements(fout(*,0))-1,*) = fout
	    col_tit = [col_tit,fcol_tit]
	ENDFOR
	title = 'Material: '+descriptor+' '
        if (size(out))(0) EQ 1 then begin
          tmp = strarr(n_elements(out))
          for ii=0,n_elements(out)-1 do tmp(ii)=col_tit(ii)+'  = '+$
            strcompress(out(ii))
          xdisplayfile1,text=tmp,group=event.top,title=title
        endif else begin
	  xplot,out,coltitles=col_tit,group = event.top,/xlog,/ylog, $
	    title=title,ycol=2,wtitle='xCrossSec results'
        endelse
	end
  else:
endcase

out:
Widget_Control, stateid, set_UValue=state, /No_Copy
END
;
;=======================================================================
;
PRO xcrosssec,GROUP=group, InputFile=inputFile, No_Block=no_Block
;
;
Forward_Function dabax_defaults
Catch, error_status
IF error_status NE 0 THEN BEGIN
   Message,/Info,'error caught: '+!err_string
   itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
     'XCROSSSEC: error caught: '+!err_string)
   Catch, /Cancel
   On_Error,2
   RETURN
ENDIF
wbase = widget_base(/COLUMN,TITLE='xCrossSec '+xcrosssec_version(),$
        MBAR=wMenuBar)

;
;
; the blocks box
;
Bbox=widget_base(wbase,/Column) ; also to store state
 

;Menu bar
wFile =  WIDGET_BUTTON(wMenuBar, VALUE='File', /MENU)

  wtmp0 = widget_button(wFile,VALUE='xCrossSec input parameters', /Menu)
    wtmp = widget_button(wtmp0,VALUE='Load from file...',UValue='FILEINPUT')
    wtmp = widget_button(wtmp0,VALUE='Save to file...',UValue='FILEINPUT')
    wtmp = widget_button(wtmp0,VALUE='Save as default',UValue='FILEINPUT')
  wtmp = widget_button(wFile,VALUE='Quit', UVALUE='QUIT',/SEPARATOR)

wSetParameters = widget_button(wMenuBar,VALUE='Set_Parameters', /MENU)
  wtmp = widget_button(wSetParameters,VALUE='Set Parameters', UVALUE='SET')
  wtmp = widget_button(wSetParameters,VALUE='Set Defaults', UVALUE='SETDEF')


wHelpMenu = WIDGET_BUTTON(wMenuBar, VALUE='Help', /HELP)
  wtmp = WIDGET_BUTTON(wHelpMenu, VALUE='XCrossSec', UVALUE='HELP')

tmp = widget_button(Bbox,VALUE='Set Parameters',UVALUE='SET')


if sdep() EQ 'WINDOWS' then $
  font = 'VERDANA*BOLD*ITALIC*24' else $
  font = '-adobe-helvetica-bold-o-normal--18-180-75-75-p-104-iso8859-1'

junk = WIDGET_LABEL( Bbox, FONT=font, VALUE=' XCrossSec')
junk = WIDGET_LABEL( Bbox, FONT=font, $
  VALUE='Photon-Atom Interaction Cross Sections')


str = dabax_defaults('xcrosssec',group=wbase)
str_defaults = str.parameters


IF KeyWord_Set(inputFile) THEN BEGIN
  str_par = Xop_Input_Load(InputFile=inputFile)
  IF Type(str_par) EQ 8 THEN BEGIN
    tmp = str.parameters
    Copy_Structure,str_par, tmp, Group=group, /OnlyFirstField
    str.parameters = tmp
  ENDIF
ENDIF
wids = {dummy:0L}
state = { wids:wids, str:str, str_defaults:str_defaults }

widget_control,Widget_Info(wbase,/Child),set_uvalue=state
widget_control,wbase,/REALIZE
xmanager,'xcrosssec',wbase,GROUP=group, No_Block=no_Block
;
end
