Function f0_calc,input,descriptor,ratio,group=group,npoints=npoints, $
  xmax=xmax,z=z, verbose=verbose

; 
; European Synchrotron Radiation Facility (ESRF)
;+
; NAME:
;       F0_CALC
;
; PURPOSE:
;	calculate the elastic Photon-Atom F0 coefficient
;	as a function of ratio = sin(theta)/lambda[Angs].
;	It uses data from DABAX data-base.
;	
; CATEGORY:
;       X-Ray optics. DABAX data base.
;
; CALLING SEQUENCE:
;	f0 = f0_calc(input,descriptor[,ratio])
; INPUTS:
;	input: a dabax input file as accepted by dabax_access().
;	descriptor: a string  with a description of the material.
;		The string is searched in the scan titles #S in order
;		to find the appropiated data.
; OPTIONAL INPUTS:
;	ratio: if undefined, it uses the standard ratio grid,
;		which is the grid in the data file for the tabulated
;		cases, and  npoints in [0,xmax] for the parametrized
;		data. When ratio is defined, this value is used as a grid
;		for the results (it interpolates in the case of 
;		tabulated data)
;	
; KEYWORDS:
;       group = The widget ID of the widget that calls the routine.
;               This is passed to widget_message in its dialog_parent
;               keyword to center the window message.
;	npoints = When ratio is undefined, the number of points used
;	xmax = When ratio is undefined, the max value of X
;	z = A named variable where the atomic mumber is written
;       VERBOSE= If set (default is verbose=1) prints some info.
;
; OUTPUT:
;	out: an array with the values of f0.
;
; PROCEDURE:
;	Depends on the value of the #UF0TYPE keyword in the f0_*.dat file:
;	UF0TYPE = PARAMETRIZATION:
;	  Applies the selected parametrization for the case of parametrization.
;	UF0TYPE = TABLE_NOR 
;	  Interpolates the tabulated data and multiplies by Z
;	UF0TYPE = TABLE
;	  Interpolates the tabulated data
;
; EXAMPLE:
;	delvar,ratio ; just for the case it was defined
;	f0 = f0_calc('f0_WaasKirf.dat','Si',ratio)
;	plot,ratio,f0
;
; MODIFICATION HISTORY:
;       96-07-10 Written by M. Sanchez del Rio (srio@esrf.fr)
;       97-10-16 srio@esrf.fr adds GROUP keyword.
;	02-09-30 srio@esrf.fr introduces the UF0TYPE keyword in the
;		DABAX f0_*.dat data to handle automatically the
;		parametrized or tabulated data.  Added npoints and xmax kw.
;	02-10-16 srio@esrf.fr Added the Z keyword to be used with ish_calc
;-
on_error,2

h = dabax_access(input,group=group)

IF N_Elements(verbose) EQ 0 THEN verbose=1
IF N_Elements(npoints) EQ 0 THEN npoints=515
IF N_Elements(xmax) EQ 0 THEN xmax=8.0D0

dataset = spec_headers(h,-1,'#F',/index,/all)
dataset = strcompress(strsubstitute(dataset,'#F',''),/rem)

igood = dabax_select(h,descriptor,/casematch,group=group)
IF verbose THEN print,'Using scan: ',spec_name(h,igood,/index)
z = double(strsubstitute(spec_headers(h,igood,'#S',/index),'#S',''))

data = spec_data(h,igood,/index,/double)

;print,data

f0type = StrSplit(spec_headers(h,iGood,'UF0TYPE',/index,/all),/Extract)
IF N_Elements(f0type) LT 2 THEN Message, $
	'F0 keyword UF0TYPE must be defined to either TABLE or PARAMETRIZATION'

case StrCompress(f0type[1],/Rem) of
  'TABLE': BEGIN
	ratio1 = data(0,*)
	f0 = data(1,*)
	if n_elements(ratio) EQ 0 then ratio =  ratio1 else begin
	  f0 = interpol(f0,ratio1,ratio)
	endelse
	end
  'TABLE_NOR': BEGIN
	ratio1 = data(0,*)
	f0 = double(z)*data(1,*)
	if n_elements(ratio) EQ 0 then ratio =  ratio1 else begin
	  f0 = interpol(f0,ratio1,ratio)
	endelse
	end
  'PARAMETRIZATION':  begin
	if ((n_elements(data) NE 9) AND (n_elements(data) NE 11)) then begin
	  message,/info,'I do not know how to process data in :'+dataset
	  message,/info,'Plese update function f0_calc with the new data set.'
	  if n_elements(ratio) EQ 0 then ratio = MakeArray1(npoints,0.0,xmax,/Double)
	  f0 = ratio*0.
	endif else begin
	  if n_elements(ratio) EQ 0 then ratio = MakeArray1(npoints,0.0,xmax,/Double)
	  icentral = n_elements(data)
	  icentral = fix(icentral/2)
	  f0 = data(icentral)
	  for i=0,icentral-1 do f0 = f0 + data(i) * exp(-1.0d0*data(i+icentral+1)*ratio^2)
	endelse
	end
  else:  Message, $
	'F0 keyword UF0TYPE must be defined to TABLE/TABLE_NOR/PARAMETRIZATION'
endcase
return,f0

end
