;______________________________________________________________________________
;+
; NAME:
;       SPEC_SAVEMCA()
;
; PURPOSE:
;       This function saves the data contained in a 2D numeric array into a file
;       following the SPEC/MCA data file format. If the file does not exist it 
;	is created. I the file exits, it must be a valid SPEC data file and 
;	the data is appended as a new scan. 
;
; CATEGORY:
;       Input/Output.
;
; CALLING SEQUENCE:
;       Result = SPEC_SAVEMCA(Handle, Data [, File])
;
; INPUTS:
;       Handle - Handle to the SPEC data file. It is initialized, if necessary,
;		 and updated after the data is saved.
;
;       Data   - 1-D Numeric array containing the data to be saved. 
;
; OPTIONAL INPUT/OUTPUT
;       File   - If Handle is not a valid SPEC file handle, this is parameter 
;		is the name of the file where the data will be saved. 
;		Otherwise File specifies a named variable where the actual 
;		name of the file will be returned.
;
; KEYWORDS:
;
;
;       COMMENTS: String vector that contains comment lines that will be 
;		included in the header section of the scan.
;
;       NAME:   String that will appear as scan name in the `#S' line.
;
;       OVERWRITE: Set this keyword to overwrite the file if it exists. 
;		By default data is appended to existing files.
;
;       NO_CONFIRM: Set this keyword to not ask for confirmation when 
;		writting data on existing files.
;
;       NO_ERROR: Set this keyword to not produce an IDL error if an error 
;		is found when writting the data. 
;
;	DIALOG_PARENT: Set this keyword equal to a string that specifies 
;		the name of the parent widget (to be passed to Dialog_Message)
;
;	CALIB: A 3-elements array with the calibration coefficients 
;		Energy=A[0]+channel*a[1]+channel^2 *a[2]
;
;	FORMAT: A string with the format for the output (default: format='I7')
;
;	NSPLIT: Number of channels per line (default: nsplit=16)
;
; OUTPUT: 
;       This function returns the scan number under which the data is saved 
;	in the file. If the operation is cancelled, the function returns 
;	zero. If an error happens and NO_ERROR is set the function 
;	returns -1 and the error message is stored in the system variable 
;	!ERR_STRING.
;
; EXAMPLE:
;	; import some data
;       tmp  = read_aglae('10avr027.x1',h=h,calib=[-10.41,54.26])
;	; write it
;	print,spec_savemca(hspec,h.channels,'10avr027x1.mca',/over, $
;		calib=[h.e0,h.gain,0])
;
; AUTHOR:
;	M. Sanchez del Rio, srio@esrf.fr, 2003/09/09 based on spec_save
;
;______________________________________________________________________________
;-
function spec_savemca, handle, data, file, $
	COMMENTS=comm, NAME=name, NO_ERROR=noerror,             $
                     OVERWRITE=overwrite, NO_CONFIRM=no_conf,                $
		     DIALOG_PARENT=dialog_parent, $
	calib=calib,format=format,nsplit=nsplit
   catch, error_status
   if error_status ne 0 then begin
      catch, /cancel
      on_error, 2
      if keyword_set(noerror) then return, -1
      message, __cleanmsg(!err_string);, /traceback
   endif

   aux = size(data)
   if aux(0) ne 1 then begin
      message, 'Data is not a 1D array.'
   end


   if not keyword_set(name) then name = ''

   if not keyword_set(file) then begin
      if __isaspechandle(handle) then begin
         file = handle.specfile
      endif else begin
         ;Select a filename
         file = Dialog_Pickfile(/write)
         if file eq '' then begin
            return, 0
         endif
      endelse
   endif

   ; Check if writable and if new or SPEC file 

   openw, Unit, file, /get_lun, /append, error=err
   if err then begin
      message, 'Cannot open "'+file+'" for writting.'
   endif
   status = fstat(Unit)
   if status.size gt 0 then begin
      if spec_access(handle, file) eq 0 then begin
         close, Unit
         message, 'Exists and is not a SPEC file'
      endif else begin
         scan = spec_scan(handle, 'last') + 1
      endelse
   endif else begin
      scan = 1
   endelse

   if keyword_set(overwrite) then begin
      if not keyword_set(no_conf) and scan gt 1 then begin
         tmp = Dialog_Message('SPEC file exists, overwrite data ?', /default_no, $
                              title='Writting data', /question, $
			      DIALOG_PARENT=dialog_parent)
         if tmp eq 'No' then begin
            close, Unit
            return, 0
         endif
      endif
      close, Unit
      openw, Unit, file, /get_lun
      scan = 1
   endif else begin
      if not keyword_set(no_conf) and scan gt 1 then begin
         tmp = Dialog_Message('SPEC file exists, append data ?', /question, $
                               title='Writting data',		       $
			       DIALOG_PARENT=dialog_parent)
         if tmp eq 'No' then begin
            close, Unit
            return, 0
         endif
      endif
   endelse

   ; Write the data

   if scan eq 1 then begin
      printf, Unit, '#F ',file
      printf, Unit, '#E',strcompress(long(systime(1)))
      printf, Unit, '#D ',systime(0)
   endif 
   printf, Unit, ''
   printf, Unit, '#S',strcompress(scan),' ',name
   printf, Unit, '#D ',systime(0)
   if keyword_set(comm) then printf, Unit, transpose('#C ' + comm)
   ;printf, Unit, format='("#N ", I2)' ,ncol
   ;printf, Unit, format= '("#L",'+string(ncol)+'("  ",A))', labels
   ;printf, Unit, format='('+string(ncol)+'(G0.6," "))', data(columns, *)
   printf, Unit, '#@MCA'
   nchannels  = N_Elements(data)
   dataout = Reform(data)
   printf, Unit, '#@CHANN '+StrCompress(nchannels,/Rem)+' 0 '+$
	StrCompress(nchannels-1,/Rem)+' 1'
   IF Keyword_Set(calib) THEN BEGIN
     printf, Unit, '#@CALIB '+String(calib,Format='(3G10.4)')
   ENDIF ELSE BEGIN
     printf, Unit, '#@CALIB 0 1 0'
   ENDELSE
   IF N_Elements(format) EQ 0 THEN format='I7'
   IF N_Elements(nsplit) EQ 0 THEN nsplit=16
   snsplit=StrCompress(nsplit,/Rem)
   ; first line
   printf, Unit, '@A '+String(dataout[0:(nsplit-1)],Format='('+snsplit+format+')')+'\'
   ; body of data
   FOR i=1,nchannels/nsplit-2 DO BEGIN
     printf, Unit, String(dataout[nsplit*i:(nsplit*i+nsplit-1)], $
	Format='('+snsplit+format+')')+'\'
   ENDFOR
   ; last line
   nelem = StrCompress(nchannels - nsplit*i,/Rem)
   printf, Unit, String(dataout[nsplit*i:(nchannels-1)],Format='('+nelem+format+')')
   free_lun, Unit

   return, scan
end
