;______________________________________________________________________________
;+
; NAME:
;       SPEC_MOTORS()
;
; PURPOSE:
;       This function returns the names or position of motors in a particular
;       scan in a SPEC file.
;
; CATEGORY:
;       Input/Output.
;
; CALLING SEQUENCE:
;       Result = SPEC_MOTORS(Handle, Scan_Id [, MotorList [, BadValue]])
;
; INPUTS:
;       Handle   - Handle to the SPEC data file initialised by a previous call
;                  to SPEC_ACCESS().
;
;       Scan_Id  - Scan identifier. This parameter can be a numeric or string
;                  scalar value that accepts several syntax and keywords as
;                  it is explained in the `SCAN IDENTIFIERS' section at the
;                  beginning of this file). The simplest case is to use
;                  the SPEC scan number as Scan_Id.
;
; OPTIONAL INPUTS:
;       MotorList - String array that contains the names of the motors which
;                  position is requested.
;                  In addition to motor names, it is also possible to specify
;                  a motor by using a string with the format 'line,motor' where
;                  `line' in the number of the `#P' header line that includes
;                  the motor information, and `motor' is the relative position
;                  of the motor in that line. For instance, '2,1' refers to
;                  the first motor in the `#P2' line.
;
;       BadValue - Dummy value to be returned as a motor position if the
;                  corresponding motor is not found or when the headers are
;                  corrupted and the NOERROR ketword is set. This value is
;                  defaulted to 1E20.
;
; KEYWORDS:
;       INDEX:  Set this keyword to interpret a numeric value in Scan_Id
;               as the index of the scan instead of the scan number.
;
;       NOERROR: Set this keyword to not return an error when the file headers
;               are corrupted. 
;
; OUTPUT: 
;       If MotorList is specified this function returns the position of the
;       motors included in the list. Otherwise it returns a string array
;       that contains the names of all the motors associated to the given scan.
;
; RESTRICTIONS:
;       This function requires a valid handle to a SPEC data file obtained by
;       a previous call to SPEC_ACCESS().
;       If Scan_Id does not represent a valid scan the function produces an
;       error.
;       If the file was created with an old version of SPEC and a motor name
;       has more than 8 characters this function will not work. It will not
;       work if a motor name includes two or more consecutive white spaces.
;
; EXAMPLE:
;       To print the position of the motors 'Theta', 'Piezo 1' and the second
;       motor of the first `#P' line in the fifth scan of a file called
;       'baddata', enter:
;
;       Dummy = SPEC_ACCESS(HHH, 'baddata')
;       PRINT, SPEC_MOTORS(HHH, -5, ['Theta', 'Piezo 1', '0,2'])
;______________________________________________________________________________
;-
function spec_motors, handle, scan_id, motorlist, badvalue, INDEX=idx,   $
                                                           NOERROR=noerror
   catch, error_status
   if error_status ne 0 then begin
      catch, /cancel
      on_error, 2
      message, __cleanmsg(!err_string);, /traceback
   endif

   ; Check arguments
   if N_PARAMS() lt 2 or N_PARAMS() gt 4 then       $
                                    message, 'Incorrect number of arguments.'

   __speccheck, handle
   __specload, handle, scan_id, errmsg, INDEX=idx
   if !ERR then message, errmsg
   __headerload, handle
 
   if N_PARAMS() ge 3 then begin
      aux = size(motorlist)
      if aux(aux(0) + 1) eq 7 then begin
         list = strtrim(motorlist, 2)
      endif else begin
         message, 'Argument is not a valid motor name list.'
      endelse
   endif

   if N_PARAMS() lt 4 then begin
      returnbadvalue = 0
      badvalue = 1E20
   endif else begin
      returnbadvalue = 1
   endelse

   headersP = spec_headers(handle, 'current', '#P')
   motorlines = n_elements(headersP)

   if motorlines eq 0 then begin
      rows = 0
   endif else begin
      for i = 0, motorlines - 1 do begin
         auxline = headersP(i)
         aux = __linecut(auxline)
         headersP(i) = auxline
         reads, aux, format='(2X,I)', motline
         mot = __linesplit(auxline, 1, 0, 0)
         if i eq 0 then begin
            linelist = motline
            motlist = n_elements(mot)
         endif else begin
            linelist = [linelist, motline]
            motlist = [motlist, n_elements(mot)]
         endelse
      endfor
      rows = max(linelist) + 1
      cols = intarr(rows)
      allmotpos = fltarr(rows, max(motlist))

      for i = 0, motorlines - 1 do begin
         motline = linelist(i)
         cols(motline) = motlist(i)
         mot = __linesplit(headersP(i), 1, 0, 0)
         allmotpos(motline, 0:cols(motline)-1) = float(mot(0:cols(motline)-1))
      endfor
   endelse

   if handle.nmotorlines ne rows then begin
      corrupted = 1
   endif else begin
      corrupted = 0
      for i = 0, rows - 1 do begin
         if cols(i) ne handle.motor_no(i) then begin
            corrupted = 1
         endif
      endfor
   endelse

   if corrupted and not keyword_set(noerror) then begin
      message, 'File headers corrupted'
   endif

   if N_PARAMS() eq 2 then begin
      motornames = ''
      if not corrupted then begin
         for i = 0, handle.nmotorlines - 1 do begin
            for j = 0, handle.motor_no(i) - 1 do begin
               if motornames(0) eq '' then begin
                  motornames = handle.motornames(i, j)
               endif else begin
                  motornames = [motornames, handle.motornames(i, j)]
               endelse
            endfor
         endfor
      endif
      return, motornames
   endif else begin
      for k = 0, n_elements(list) - 1 do begin
         pos = badvalue
         motfound = 0
         for i = 0, rows - 1 do begin
            for j = 0, cols(i) - 1 do begin
               if not corrupted then begin
                  if handle.motornames(i, j) eq list(k) then begin
                     pos = allmotpos(i, j)
                     motfound = 1
                     goto, found
                  endif
               endif
            endfor
         endfor
         if not motfound then begin
            p = strpos(list(k), ',')
            if p gt 0 then begin
               on_ioerror, arestrings
               x = fix(strmid(list(k), 0, p))
               y = fix(strmid(list(k), p+1, strlen(list(k))-p-1)) - 1
               if x ge 0 and x lt rows then begin
                  if y ge 0 and y le cols(x) then begin
                     pos = allmotpos(x, y)
                     motfound = 1
                  endif
               endif
arestrings:    on_ioerror, null
            endif
         endif
found:
         if motfound or returnbadvalue or keyword_set(noerror) then begin
            if k eq 0 then begin
               motorpos = pos
            endif else begin
               motorpos = [motorpos, pos]
            endelse
         endif else begin
            message, 'Motor "'+ list(k) +'" not found.'
         endelse
      endfor
      return, motorpos
   endelse

end
