;______________________________________________________________________________
;+
; NAME:
;       SPEC_DATA()
;
; PURPOSE:
;       This function returns an array with data from a scan in a SPEC file.
;
; CATEGORY:
;       Input/Output.
;
; CALLING SEQUENCE:
;       Result = SPEC_DATA(Handle, Scan_Id [, Columns])
;
; INPUTS:
;       Handle   - Handle to the SPEC data file initialised by a previous call
;                  to SPEC_ACCESS().
;
;       Scan_Id  - Scan identifier. This parameter can be a numeric or string
;                  scalar value that accepts several syntax and keywords as
;                  it is explained in the `SCAN IDENTIFIERS' section at the
;                  beginning of this file). The simplest case is to use
;                  the SPEC scan number as Scan_Id.
;
; OPTIONAL INPUTS:
;       Columns  - Vector that describes the columns to be loaded in the output
;                  array. This vector can be of either numeric or string type.
;                  When numbers are used columns are selected by their relative
;                  position in the data file. When non-numeric strings are used
;                  they must match the labels of the columns to select. By
;                  using a string array it is possible to combine both
;                  selection methods.
;                  If negative numbers are used it is assumed that the column
;                  position is obtained by counting backwards from the last
;                  one: i.e. -1 means the last column, -2 the one before the
;                  last, and so on.
;
; KEYWORDS:
;       INDEX:  Set this keyword to interpret a numeric value in Scan_Id
;               as the index of the scan instead of the scan number.
;
;       MCDATA: Set this keyword to return data from the multichannel data
;               lines instead of the standard data columns.
;
;       LABELS: String vector to be filled with the labels of the returned
;               columns. This option can only be used if the MCDATA keyword
;               is not set.
;
;       OFFSET: Number of points that will be skipped before extracting the
;               required data. This value is defaulted to 0.
;
;       STEP:   Interval used to extract data points. Only the first one of
;               each group of STEP consecutive data points will be extracted.
;               The default value is 1.
;
;       NPOINTS: Maximum number of points that will be returned. If it is
;               zero or greater than the actual number of available data
;               points, this keyword has not any effect.
;
;       CHANNELS: Float vector to be filled with the channel numbers of the
;               multichannel data. It is not used if the MCDATA keyword is
;               not set.
;
;       CALIBRATED: Float vector to be filled with the calibrated values
;               (usually energy) of the channels in the multichannel data set.
;               It is not used if the MCDATA keyword is not set.
;
;       STRING: Set this keyword to force the output to be a string array. 
;
;       DOUBLE: Set this keyword to force the output to be a double-precision
;               array.
;
;       FLOAT:  Set this keyword to force the output to be a single-precision
;               floating-point array.
;
;       LONG:   Set this keyword to force the output to be a longword integer
;               array.
;
;       INT:    Set this keyword to force the output to be an integer array.
;
; OUTPUT: 
;       This function returns an array containing the data columns selected by
;       the Columns parameter. If no Columns parameter is given, all the
;       columns in the selected scan are returned. By default the output 
;       is a double-precision array for standard data and a long array for
;       multichannel data, but the output array can be forced to any type by
;       setting the proper keyword.
;       If a particular column does not exist in the SPEC file, the
;       corresponding column in the output array is filled with zeros.
;
; RESTRICTIONS:
;       This function requires a valid handle to a SPEC data file obtained by
;       a previous call to SPEC_ACCESS().
;       If Scan_Id does not represent a valid scan the function produces an
;       error.
;
; EXAMPLE:
;       To load the array GoodData with the position of the first column of the
;       scan 5, the monitor counts and the detector counts (that are assumed to
;       be in the last column of the file), enter:
;
;       Dummy = SPEC_ACCESS(File_handle, 'weakdata')
;       GoodData = SPEC_DATA(File_handle, 27, ['1', 'Monitor', '-1'])
;
;______________________________________________________________________________
;-

function spec_data, handle, scan_id, columns, INDEX=idx, MCDATA=mcdata,        $
                           LABELS=labels, OFFSET=offset, STEP=step,NPOINTS=np,$
                           CHANNELS=chann, CALIBRATED=calib,                  $
                           DOUBLE=dtype, FLOAT=ftype, LONG=ltype, INT=itype,  $
                           STRING=stype
   catch, error_status
   if error_status ne 0 then begin
      catch, /cancel
      on_error, 2
      message, __cleanmsg(!err_string);, /traceback
   endif

   ; Check arguments
   if N_PARAMS() lt 2 or N_PARAMS() gt 3 then  $
                                   message, 'Incorrect number of arguments.'
   __speccheck, handle
   __specload, handle, scan_id, errmsg, INDEX=idx
   if !ERR then message, errmsg

   if keyword_set(mcdata) and keyword_set(labels) then  $
                                   message, 'No labels in multichannel data.'
   if not keyword_set(offset) then offset = 0
   if not keyword_set(step) then step = 1
   if not keyword_set(np) then np = 0
   offset = long(offset)
   step = long(step)
   np = long(np)
   if offset lt 0 then message, 'OFFSET must be a positive value.'
   if step lt 1 then message, 'STEP must be >= 1.'
   if step lt 0 then message, 'NPOINTS must be >= 0.'

   if keyword_set(mcdata) then begin
      n = handle.scan(handle.currscan).n_chan
      npoints = handle.scan(handle.currscan).a_datapoints
   endif else begin
      n = handle.scan(handle.currscan).n_col
      npoints = handle.scan(handle.currscan).n_datapoints
   endelse
   if npoints le 0 then message, 'No data points.'
   if npoints le offset then message, 'Offset too big.'
   npoints = long((npoints - offset - 1)/step) + 1
   if np gt 0 and npoints gt np then npoints = np

   if N_PARAMS() eq 2 then begin
      ncols = n
      index = indgen(ncols)
   endif else begin
      aux = size(columns)
      if aux(0) gt 1 then message, 'Too many dimensions.'
      ncols= aux(aux(0) + 2)
      index=intarr(ncols)
      aux = aux(aux(0) + 1)
      if aux gt 0 and aux lt 6 then begin
         index = fix(columns)  
      endif else if aux eq 7 then begin
         for i=0, ncols-1 do begin
            c=0
            on_ioerror, isastring
            c=fix(columns(i)) 
isastring:  on_ioerror, null
            if c eq 0 then begin
               if keyword_set(mcdata) then begin
                  message, "Not labels allowed with multichannel data."
               endif
               for j=1, n do begin
                  if columns(i) eq handle.label(j-1) then begin
                     c=j 
                     goto, set
                  endif
               endfor
            endif 
set:        index(i)=c
         endfor
      endif else begin
         message, "Bad data type."
      endelse
      index = index * (index ge -n and index le n) - 1
      index = index + (n+1)*(index lt -1) 
   endelse

   aux = where(index lt 0)
   if aux(0) ne -1 then begin
      message, "Wrong column specification [" +                  $
                                    strtrim(string(columns(aux(0))),2) + "]."
   endif

   if keyword_set(stype) then begin
      data = strarr(ncols, npoints)
      datal = strarr(n)
      datatype = 2
   endif else if keyword_set(mcdata) and not keyword_set(ftype) and  $
                                      not keyword_set(dtype) then begin
      data = lonarr(ncols, npoints)
      datal = lonarr(n)
      datatype = 1
   endif else begin
      data = dblarr(ncols, npoints)
      datal = dblarr(n)
      datatype = 0
   endelse

   openr,Unit,handle.specfile,/get_lun
   point_lun, Unit, handle.scan(handle.currscan).datapt
   line=''
   point = -offset
   pt = 0L
   for i = 0L, handle.scan(handle.currscan).datalines -1 do begin
      if pt eq npoints then goto, done
      __readline, Unit, line
      ok = point ge 0 and (point mod step) eq 0
      found = 0
      case strmid(line, 0, 1) of
         '':  ; Empty line. Do nothing

         '#': ; Header or comment. Do nothing

         '@': begin
            if keyword_set(mcdata) and strmid(line, 1, 1) eq 'A' then begin
               found = 1
               if ok then begin
                  aux0 = __linecut(line)
                  __reads, line, datal
               endif
            endif
         end

         else: begin
            if not keyword_set(mcdata) then begin
               found = 1
               if ok then begin
                  __reads, line, datal
               endif
            endif
         end
      endcase
      if found then begin
         if ok then begin
            data(*, pt) = datal(index)
            pt = pt + 1
         endif
         point = point + 1
      endif
   endfor
done:
   if keyword_set(mcdata) then begin
      nchan = 0      ; integer
      chmin = 0.     ; float
      chmax = 0.     ; float
      Chreduc = 0    ; integer
      calA = 0.      ; float
      calB = 1.      ; float
      calC = 0.      ; float
      point_lun, Unit, handle.scan(handle.currscan).scanpt
      for i = 1, handle.scan(handle.currscan).predatalines do begin
         __readline, Unit, line
         if strmid(line, 0, 7) eq '#@CHANN' then begin
            line = strmid(line, 7, strlen(line)-7) + ' 0 0 0 0'
            reads, line, nchan, chmin, chmax, chreduc
            if chreduc eq 0 then chreduc = 1
            if chmin eq chmax and nchan gt 1 then begin
               chmax = chmin + (nchan - 1) * chreduc
            endif
         endif else if strmid(line, 0, 7) eq '#@CALIB' then begin
            line = strmid(line, 7, strlen(line)-7) + ' 0 0 0'
            reads, line, calA, calB, calC 
         endif
      endfor
      if nchan gt 1 then begin
         chann = chmin + findgen(nchan)*(chmax-chmin)/(nchan - 1)
      endif else begin
         chann = chmin
      endelse
      calib = calA + chann * (calB + chann * calC)
   endif
   free_lun, Unit

   if not keyword_set(mcdata) then begin
      aux = [handle.label, '']
      labels = aux(index)
   endif

   if keyword_set(stype) then return, data
   if keyword_set(ftype) then return, float(data)
   if keyword_set(itype) then return, fix(data)
   if keyword_set(ltype) then return, long(data)
   return, reform(data)
end
