;+
;===============================  Xus   ========================================
;
;   Xus is an widget based graphical user interface to calculate 
;   spectral properties of ideal undulator insertion devices.
;
; DESCRIPTION OF THE CONTROLS IN THE MAIN WINDOW:
;
;  File:
;    Xus input parameters: This option allows to save the current
;		parameters to a file for later loading. It also allows
;		to save the current parameters as defaults for being
;		used when the application is initialized. In the last
;		case, the file is named "application".xop (where 
;		"application " is the name of the current XOP
;		application) and is written in the directory pointed
;		by the XOP_DEFAULTS_DIR environment variable (which
;		must be set). The parameter file is ASCII and can be
;		read and edited with care.
;    US file: This option allows to load/write the current parameter
;		and result files for further calculations. The used file
;		formats are those used by US. Therefore, they
;		can be used to run US without the interface or to
;		import US files created outside the XUS application.
;    Write Files for XOP/Optics: Saves the current undulator
;		spectrum in the SRCOMPE (flux) and SRCOMPW (Power) files
;		to be used for the other XOP application from Optics
;		menu.
;  Quit: to exit from the program
;
; Set_Parameters:
;  Set Parameters: to define the parameters for the calculation.
;		The same result is obtained pressing the "Set Parameters"
;		button in the main XUS window. 
;		After pressing the ACCEPT button, US starts running.
;  Set Defaults: Sets the default parameters.
;
;  Show: Display results
;    Plot ...: Plots the US results.
;    View Results: Displays the US result file.
;
;  Help:   Shows the US help and the XUS help (this text).
;
;
; COPYRIGHT:
;	XUS  belongs to XOP package and it is distributed within XOP.
;	PLEASE REFER TO THE XOP COPYRIGHT NOTICE BEFORE USING IT.
;
; CREDITS:
;	Published calculations made with XOP should refer:
;
;	  M. Sanchez del Rio and R. J. Dejus "XOP: Recent Developments"
;	  SPIE proceedings vol. 3448, pp.340-345, 1998.
;
;	In addition, published calculations using the US aplication 
;	may also cite:
;
;	  
;	  P. Ilinski, R.J. Dejus, E. Gluskun and T. I. Morrison
;	  SPIE proceedings vol. 2856, pp.15-25, 1996.
;
;  KNOWN BUGS:
;       When a us calculation producing 3D results is run, the Show/Plot 2D
;	menu gives wrong results (MSR 98-12-04).
;
;
; LAST MODIFICATION: rjd/msr/00-07-12
;
;-
;
; -----------------------------------------------------------------------
;	Author: Roger J. Dejus (dejus@aps.anl.gov), XFD/APS, May, 1994.
;	MODIFICATION HISTORY:
;
;        Date     | Name  | Description
; ----------------+-------+-----------------------------------------------------
; 06-OCT-1994     | RJD   | Modified to display data for US v.1.6.
;                 |       | Added options for the surface, show3, and 2d-
;                 |       | plots to be able to display the polarized
;                 |       | intensities. Modified default parameters Kx, Ky, 
;                 |       | and IHARM.
; ----------------+-------+-----------------------------------------------------
; 12-APR-1995     | RJD   | Rewrote the complete graphical user interface.
;                 |       | Removed all common blocks and introduced anonymous
;                 |       | structures.  All data are now passed using anonymous
;                 |       | structures and IDL handles. The compound pull-down
;                 |       | menu now replaces the XPDMENU. The pop-up dialog
;                 |       | widget no longer requires the user to press the
;                 |       | carriage return after each entry. The input fields
;                 |       | are still checked for valid entries and valid ranges
;                 |       | Added many new options under the FILE menu including
;                 |       | the selection of loading an input default file and
;                 |       | the option to save files for XTRANSMIT.
; ----------------+-------+-----------------------------------------------------
; 04-JAN-1996     | RJD   | Modified the number of decimal places for the sigx1
;                 |       | and sigy1 variables to four and the number of
;                 |       | decimals to three for the distance (d variable) in
;                 |       | the us.inp file.
; ----------------+-------+-----------------------------------------------------
; 17-JAN-1996     | MSR   | Modified for xop1.3. Changed xtext -> Widget_Message
;                 |       | and xsurface -> xsurface1. Changed labels Xtransmit
;                 |       | -> XOP/Optics.
; ----------------+-------+-----------------------------------------------------
; 22-JAN-1997     | MSR   | Started porting to Windows95 (not checked).
; ----------------+-------+-----------------------------------------------------
; 05-FEB-1997     | MSR   | Removed "print" options from menus, now done from
;                 |       | the display windows menu.
; ----------------+-------+-----------------------------------------------------
; 20-MAR-1997     | MSR   | Introduced colfiles in xplot and adapted for
;                 |       | idl5.0b5.
; ----------------+-------+-----------------------------------------------------
; 29-SEP-1997     | RJD   | Modified for the IDL 5.0.2. Introduced system
;                 |       | dependencies via the function sdep.pro. Currently 
;                 |       | UNIX and WINDOWS are supported; (MACOS) is
;                 |       | predefined. Renamed the functions read_dat,
;                 |       | read2d_plt, write_dat, and read_ascii to
;                 |       | xus_read_dat, xus_read2d_plt, xus_write_dat, and
;                 |       | rascii, respectiveley. Made several clean-ups
;                 |       | consistent with changes suggested by MSR.
;                 |       | Introduced version number for the interface: v1.9.
; ----------------+-------+-----------------------------------------------------
; 18-NOV-1997     | RJD   | Renamed calls to read3d_plt -> us_read3d_plt.
;                 |       | Changed order of plotted functions for 2d-plots
;                 |       | to be same as for 3d-plots. Improved error checking
;                 |       | using widget_message. Energy converted to keV for
;                 |       | plots. Changed FILTER keyword in PICKFILE so that
;                 |       | it is undefined for operating systems other than
;                 |       | UNIX. Added the keyword DIALOG_PARENT to
;                 |       | WIDGET_MESSAGE. Sets working directory if the GROUP
;                 |       | keyword is not set.
;                 |       | Current release is v1.9.
; ----------------+-------+-----------------------------------------------------
; 04-DEC-1998     | MSR   | Using the new (xop2.0) input file system.
;                 |       | Changed the menu look to adapt to xop "standards".
;                 |       | Added Xus_Version function. Use of Xop_Spawn.
;                 |       | Backgroud task in a new window.
;                 |       | SHOW3 plots now available from SURFACE1 window 
;                 |       | (supressed from Show Menu). Changed label font.
;                 |       | Addes Spectral Power in results menus.
;                 |       | Current release is v2.0Beta1.
; ----------------+-------+-----------------------------------------------------
; 12-JUL-2000     | MSR   | Foregroung run as default. Version 2.0
; ----------------+-------+-----------------------------------------------------
; 17-JUL-2003     | MSR   | Added x and y axes in 3d plots
; ----------------+-------+-----------------------------------------------------
;
;

FORWARD_FUNCTION sdep, us_read3d_plt

;
;===============================================================================
;
FUNCTION Xus_Version
RETURN,'2.0'
END ; Xus_Version

;
;===============================================================================
;
FUNCTION xus_read_dat, struct, FNAME=fname, Group=group
Catch, error_status
IF error_status NE 0 THEN BEGIN
   Message,/Info,'error caught: Error reading US input data file'
   itmp = Dialog_Message(/Error,Dialog_Parent=group, $
     'XUS_READ_DAT: error caught: Error reading US input data file')
   Catch, /Cancel
   On_Error,2
   RETURN,error_status
ENDIF

;catch, error_status
;if error_status ne 0 then begin
;  print, !error_state.msg
;  print, 'XUS_READ_DAT: Error reading US input data file'
;  return, error_status
;endif

if n_elements(fname) eq 0 then fname = 'us.inp'

inp = XOP_DEFAULTS('xus') ; define structure
openr, unit, fname, /get_lun
readf, unit, inp
free_lun, unit
struct = inp ; make assignment

return, error_status
END ; xus_read_dat

;
;===============================================================================
;
FUNCTION xus_read2d_plt, a, nd, FNAME=fname, Group=group

Catch, error_status
IF error_status NE 0 THEN BEGIN
   Message,/Info,'error caught: '+!error_state.msg
   ;itmp = Dialog_Message(/Error,Dialog_Parent=group, $
   ;  'XUS_READ2D_PLT: error caught: '+!error_state.msg)
   ;Catch, /Cancel
   ;On_Error,2
   RETURN,error_status
ENDIF

if n_elements(fname) eq 0 then fname = 'us.out'

a = rascii(fname, npoints=nd, error_status=error_status)
if error_status ne 0 then begin
  Message, 'Error reading US plot file'
  ;print, 'XUS_READ2D_PLT: Error reading US plot file'
endif

return, error_status
END ; xus_read2d_plt

;
;===============================================================================
;
FUNCTION xus_write_dat, struct, FNAME=fname, Group=group


Catch, error_status
IF error_status NE 0 THEN BEGIN
   Message,/Info,'error caught: '+!error_state.msg
   itmp = Dialog_Message(/Error,Dialog_Parent=group, $
     'XUS_WRITE_DAT: error caught: '+!error_state.msg)
   Catch, /Cancel
   On_Error,2
   RETURN,error_status
ENDIF

if n_elements(fname) eq 0 then fname = 'us.inp'

;catch, error_status
;if error_status ne 0 then begin
;  print, !error_state.msg
;  print, 'XUS_WRITE_DAT: Error writing US input data file'
;  return, error_status
;endif

;  Define text strings
text_machine   = '                               Ring-Energy Current'
text_beam      = '               Sx Sy Sxp Syp'
text_undulator = '               Period N Kx Ky'
text_energy    = '                   Emin Emax Ne'
text_pinhole   = '   D Xpc Ypc Xps Yps Nxp Nyp'
text_mode      = '                       Mode Method Iharm'
text_calc      = ' Nphi Nalpha Dalpha2 Nomega Domega Nsigma'

openw, unit, fname, /get_lun

printf, unit, struct.title, format='(a)'
printf, unit, struct.energy, struct.cur, text_machine, format='(2f8.2,a)'
printf, unit, struct.sigx, struct.sigy, struct.sigx1, struct.sigy1, text_beam, $
	      format='(2f8.3,2f8.4,a)'
printf, unit, struct.period, struct.np, struct.kx, struct.ky, text_undulator, $
	      format='(f8.2,i8,f8.3,f8.3,a)'
printf, unit, struct.emin, struct.emax, struct.n, text_energy, $
	      format='(2f10.1,i8,a)'
printf, unit, struct.d, struct.xpc, struct.ypc, struct.xps, struct.yps, $
	      struct.nxp, struct.nyp, text_pinhole, $
              format='(f8.3,2f8.3,2f8.3,2i6,a)'
printf, unit, struct.mode, struct.method, struct.iharm, text_mode, $
	      format='(3i8,a)'
printf, unit, struct.nphi, struct.nalpha, struct.calpha2, struct.nomega, $
	      struct.comega, struct.nsigma, text_calc, $
              format='(2i8,f8.1,i8,f8.1,i6,a)'
printf, unit, struct.run_mode_name, $
	      format='(a)'
free_lun, unit
END ; xus_write_dat

;
;===============================================================================
;
PRO xus_event, event

Catch, error_status
IF error_status NE 0 THEN BEGIN
   Message,/Info,'error caught: '+!error_state.msg
   itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
     'XUS_EVENT: error caught: '+!error_state.msg)
   IF N_Elements(info) NE 0 THEN Widget_Control, event.top, Set_UValue=info
   Catch, /Cancel
   On_Error,2
   RETURN
ENDIF

;  Retrieve anonymous structure
WIDGET_CONTROL, event.top, GET_UVALUE=info
hc = info.hc ; pointer to control
hd = info.hd ; pointer to input data
hp = info.hp ; pointer to output results
;print,'xus_event',hc,hd,hp
;help,/str,event
;  Get structures
;HANDLE_VALUE, hc, control    ; srio comments: , /NO_COPY
;HANDLE_VALUE, hd, data    ; srio comments: , /NO_COPY 
;HANDLE_VALUE, hp, plt    ; srio comments: , /NO_COPY
HANDLE_VALUE, hc, control, /NO_COPY
HANDLE_VALUE, hd, data, /NO_COPY 
HANDLE_VALUE, hp, plt, /NO_COPY

delimiter = control.delimiter
fname_res = control.fname_res
osversion = control.osversion
ds        = control.ds

IF N_ELEMENTS(data) eq 0 THEN BEGIN
  l1d = 0b
ENDIF ELSE BEGIN
  IF STRMATCH1('dat1',TAG_NAMES(data)) ne -1 THEN l1d = 1b ELSE l1d = 0b
ENDELSE
IF N_ELEMENTS(plt) eq 0 THEN BEGIN
  l2p = 0b & l3p = 0b
ENDIF ELSE BEGIN
  IF STRMATCH1('plt2',TAG_NAMES(plt)) ne -1 THEN l2p = 1b ELSE l2p = 0b
  IF STRMATCH1('plt3',TAG_NAMES(plt)) ne -1 THEN l3p = 1b ELSE l3p = 0b
ENDELSE
;print,'l1d l2p l3p',l1d,l2p,l3p

;  Definitions
C_EV_KEV = 1.0e-3		; conversion eV -> keV
ptxt = ['',' data not found...', $
	' No polarized components for Brilliance mode ']


; srio 
IF (Where(StrUpCase(Tag_Names(event)) EQ 'VALUE'))(0) NE -1 THEN BEGIN
  eVal = event.value
ENDIF ELSE BEGIN
  uVal=''
  Widget_Control,event.id, Get_UValue=uVal
  eVal=uVal
ENDELSE

nv = strparse(eVal, delimiter, elist)
;print,'nv elist',nv,' ',elist
;print, '**************** eVal:***'+eVal+'************************'



CASE eVal OF

  'FILEINPUT': BEGIN
    action=''
    Widget_Control,event.id, Get_Value=action
    CASE action OF
      'Load from file...': BEGIN
        ;if sdep() EQ 'UNIX' then filter='*.xop' else filter=0
	filter='*.xop'
        str_par = Xop_Input_Load(Title=$
          'Select XUS input file...',$
          /NoConf,Filter=filter,Group=event.top)
        IF Type(str_par) EQ 8 THEN BEGIN
          tmp = data.dat1
          Copy_Structure,str_par,tmp,Group=event.top ; ,/OnlyFirstField
          data.dat1 = tmp
        ENDIF
      END
     'Save to file...': BEGIN
        str_par = data.dat1
        Xop_Input_Save,str_par,File='xus.xop',$
          /Write, Group=event.top, Comment='; xop/xus(v'+$
          xus_version()+') input file on '+SysTime()
      END
     'Save as default': BEGIN
        str_par = data.dat1
        Xop_Input_Save,str_par,Group=event.top, $
          Default='xus.xop',Comment='; xop/xus(v'+$
          xus_version()+') input file on '+SysTime()
      END
    ENDCASE
  END

  'File:Load  US:Input      File': BEGIN
  CD, CURRENT=cwd & cwd = cwd +ds
  CASE osversion OF
    'UNIX':    filter = '*.inp'
    'WINDOWS': filter = 0
    'MACOS':   filter = 0 ; 'MACOS command here'
  ENDCASE
  fp = Dialog_PICKFILE(PATH=cwd, FILTER=filter, GET_PATH=path, GROUP=event.top)
  IF fp eq '' or fp eq cwd THEN GOTO, LBL900 ; cancel
  ff = FINDFILE(fp)
  fname = ff(0)
  IF fname eq '' THEN BEGIN
    stxt = STRMID(fp, STRLEN(path), STRLEN(fp) -STRLEN(path)) ; get filename
    junk = WIDGET_MESSAGE(/ERROR, DIALOG_PARENT=event.top, $
      ['',stxt,'File not found'])
    GOTO, LBL900
  ENDIF
  error_status = xus_read_dat(dat1, FNAME=fname,group=event.top)
  IF error_status NE 0 THEN BEGIN
    stxt = STRMID(fname, STRLEN(path), STRLEN(fname) -STRLEN(path)) ; get filename
    junk = WIDGET_MESSAGE(/ERROR, DIALOG_PARENT=event.top, $
	['','Error reading file', stxt, $
	'This seems not an US ','input file  '])
    GOTO, LBL900
  ENDIF
  data = {dat1:dat1}
  END
  'File:Load  US:Output  2d-File': BEGIN
  CD, CURRENT=cwd & cwd = cwd +ds
  CASE osversion OF
    'UNIX':    filter = '*.out'
    'WINDOWS': filter = 0
    'MACOS':   filter = 0 ; 'MACOS command here'
  ENDCASE
  fp = Dialog_PICKFILE(PATH=cwd, FILTER=filter, GET_PATH=path, GROUP=event.top)
  IF fp eq '' or fp eq cwd THEN GOTO, LBL900 ; cancel
  ff = FINDFILE(fp)
  fname = ff(0)
  IF fname eq '' THEN BEGIN
    stxt = STRMID(fp, STRLEN(path), STRLEN(fp) -STRLEN(path)) ; get filename
    junk = WIDGET_MESSAGE(/ERROR, DIALOG_PARENT=event.top, $
      ['',stxt,'File not found'])
    GOTO, LBL900
  ENDIF
  error_status = xus_read2d_plt(a2, nd, FNAME=fname, Group=event.top)
  IF error_status NE 0 THEN BEGIN
    stxt = STRMID(fname, STRLEN(path), STRLEN(fname) -STRLEN(path)) ; get filename
    junk = WIDGET_MESSAGE(/ERROR, DIALOG_PARENT=event.top, $
        ['','Error reading file', stxt, $
	'This seems not an US ','output 2d-file  '])
    GOTO, LBL900
  ENDIF
  control = {delimiter:delimiter, fname_res:fname}
  nc = nd(0)
  plt2 = {a2:a2, nc:nc}
  plt  = {plt2:plt2}
  END
  'File:Load  US:Output  3d-File': BEGIN
  CD, CURRENT=cwd & cwd = cwd +ds
  CASE osversion OF
    'UNIX':    filter = '*.out'
    'WINDOWS': filter = 0
    'MACOS':   filter = 0 ; 'MACOS command here'
  ENDCASE
  fp = Dialog_PICKFILE(PATH=cwd, FILTER=filter, GET_PATH=path, GROUP=event.top)
  IF fp eq '' or fp eq cwd THEN GOTO, LBL900 ; cancel
  ff = FINDFILE(fp)
  fname = ff(0)
  IF fname eq '' THEN BEGIN
    stxt = STRMID(fp, STRLEN(path), STRLEN(fp) -STRLEN(path)) ; get filename
    junk = WIDGET_MESSAGE(/ERROR, DIALOG_PARENT=event.top, $
      ['',stxt,'File not found'])
    GOTO, LBL900
  ENDIF
  error_status = us_read3d_plt(z3, x3, y3, p1, p2, p3, p4, FNAME=fname) 
  IF error_status NE 0 THEN BEGIN
    stxt = STRMID(fname, STRLEN(path), STRLEN(fname) -STRLEN(path)) ; get filename
    junk = WIDGET_MESSAGE(/ERROR, DIALOG_PARENT=event.top, $
      ['','Error reading file', stxt])
    GOTO, LBL900
  ENDIF
  control = {delimiter:delimiter, fname_res:fname}
  plt3 = {z3:z3, x3:x3, y3:y3, p1:p1, p2:p2, p3:p3, p4:p4}
  plt  = {plt3:plt3}
  END
  'File:Load  US:Default    File': BEGIN

  itmp = Dialog_Message(Dialog_Parent=event.top, $
    /Question,['This option initializes the',$
    'xus parameters to their default values.',$
    'Then you must click Set_parameters to run the program.',$
    'Please confirm'],Title='xus: set defaults')
    IF itmp EQ 'No' THEN GoTo,LBL900
  dat1 = XOP_DEFAULTS('xus')
  data = {dat1:dat1}
  END
  'File:Write US:Input  File': BEGIN
  CD, CURRENT=cwd & cwd = cwd +ds
  CASE osversion OF
    'UNIX':    filter = '*.inp'
    'WINDOWS': filter = 0
    'MACOS':   filter = 0 ; 'MACOS command here'
  ENDCASE
  fp = Dialog_PICKFILE(PATH=cwd, FILTER=filter, GET_PATH=path, $
    TITLE='Save File As', GROUP=event.top)
  IF fp eq '' or fp eq cwd THEN GOTO, LBL900 ; cancel
  stxt = 'us.inp'		; check for existence of us.inp
  ff = FINDFILE(path +stxt)
  fname = ff(0)
  IF fname eq '' THEN BEGIN ; file does not exist
    junk = WIDGET_MESSAGE(/ERROR, DIALOG_PARENT=event.top, $
	['','Error copying file ' +stxt, $
	'File not found','Run US first and try again!'])
    GOTO, LBL900
  ENDIF
  ff = FINDFILE(fp)
  fname = ff(0)
  IF fname ne '' THEN BEGIN ; file already exist
    stxt = STRMID(fp, STRLEN(path), STRLEN(fp) -STRLEN(path)) ; get filename
    junk = WIDGET_MESSAGE(/QUESTION, DIALOG_PARENT=event.top, $
      ['',stxt,'Already exist; Overwrite File ?'])
    IF junk eq 'No' THEN GOTO, LBL900
  ENDIF
  CASE osversion OF
    'UNIX':    command = '/bin/cp ' +path +'us.inp ' +fp
    'WINDOWS': command = 'copy '    +path +'us.inp ' +fp
    'MACOS':   command = 'MACOS command here'
  ENDCASE
  message,/info,'Executing: '+command
  SPAWN, command
  END
  'File:Write US:Output File': BEGIN
  CD, CURRENT=cwd & cwd = cwd +ds
  CASE osversion OF
    'UNIX':    filter = '*.out'
    'WINDOWS': filter = 0
    'MACOS':   filter = 0 ; 'MACOS command here'
  ENDCASE
  fp = Dialog_PICKFILE(PATH=cwd, FILTER=filter, GET_PATH=path, $
    TITLE='Save File As', GROUP=event.top)
  IF fp eq '' or fp eq cwd THEN GOTO, LBL900 ; cancel
  stxt = 'us.out'		; check for existence of us.out
  ff = FINDFILE(path +stxt)
  fname = ff(0)
  IF fname eq '' THEN BEGIN ; file does not exist
    junk = WIDGET_MESSAGE(/ERROR, DIALOG_PARENT=event.top, $
	['','Error copying file ' +stxt, $
	'File not found','Run US first and try again!'])
    GOTO, LBL900
  ENDIF
  ff = FINDFILE(fp)
  fname = ff(0)
  IF fname ne '' THEN BEGIN ; file already exist
    stxt = STRMID(fp, STRLEN(path), STRLEN(fp) -STRLEN(path)) ; get filename
    junk = WIDGET_MESSAGE(/QUESTION, DIALOG_PARENT=event.top, $
      ['',stxt,'Already exist; Overwrite File ?'])
    IF junk eq 'No' THEN GOTO, LBL900
  ENDIF
  CASE osversion OF
    'UNIX':    command = '/bin/cp ' +path +'us.out ' +fp
    'WINDOWS': command = 'copy '    +path +'us.out ' +fp
    'MACOS':   command = 'MACOS command here'
  ENDCASE
  message,/info,'Executing: '+command
  SPAWN, command
  END
  'File:Display File': BEGIN
  CD, CURRENT=cwd & cwd = cwd +ds
  fp = Dialog_PICKFILE(PATH=cwd, GET_PATH=path, GROUP=event.top)
  IF fp eq '' or fp eq cwd THEN GOTO, LBL900 ; cancel
  XDISPLAYFILE1, fp
  END
  'File:Write Files for XOP/Optics': BEGIN
  ff = findfile(fname_res)
  fname = ff(0)
  if fname eq '' THEN BEGIN
    CD, CURRENT=path & path = path +ds
    stxt = STRMID(fname_res, STRLEN(path), STRLEN(fname_res) -STRLEN(path)) ; get filename
    junk = WIDGET_MESSAGE(/ERROR, DIALOG_PARENT=event.top, $
	['','Error reading file ' +stxt, $
	'File not found','Rerun US and try again!'])
    GOTO, LBL900
  ENDIF

  dum  = ''
  ncol = 6
  lines_to_skip = 23 ; skip number of header lines for us.out
  CASE osversion OF
    'UNIX': BEGIN
      SPAWN, 'wc -l ' +fname, wc & wc = wc(0) ; get number of lines
      nv = STRPARSE(wc, ' ', wlist)
      nrow = FIX(wlist(0)) -lines_to_skip
    END
    ;'WINDOWS': BEGIN		; read file and count number of lines
    else: BEGIN
    nrow = 0
    openr,Unit,fname,/get_lun
    tmp = ''
    while not eof(Unit) do begin
      readf,Unit,tmp
      nrow = nrow +1
    endwhile
    free_lun,unit
    nrow = nrow -lines_to_skip
    END
    ;'MACOS': BEGIN		; read file and count number of lines
    ;print,'MACOS command here'
    ;END
  ENDCASE

  a2 = DBLARR(ncol, nrow)
  OPENR, UNIT, fname, /GET_LUN
  FOR i=1,lines_to_skip DO BEGIN
    READF, UNIT, dum
    IF i eq 10 THEN mline = dum ; save mode line
  ENDFOR
  nv = STRPARSE(mline, ' ', mlist)
  mode = FIX(mlist(1))
  IF (mode ne 2) and (mode ne 4) and (mode ne 5) THEN BEGIN
    junk = WIDGET_MESSAGE(/INFO, DIALOG_PARENT=event.top, $
	['','Incorrect MODE', $
	'Choose one of the following modes:', $
	'Angular/spatial flux density spectrum', $
	'Flux spectrum through a pinhole', $
	'Flux spectrum integrated over all angles', $
	'Rerun US and try again!'])
    FREE_LUN, UNIT
    GOTO, LBL900
  ENDIF

  PRINT
  PRINT, ' Reading ' +fname +'; PLEASE WAIT!'
  READF, UNIT, a2 ; read data
  FREE_LUN, UNIT

;  Conversion factors
  BW = 1.0D-3 ; 0.1%
  EC = 1.60217733D-19

  a2w = a2(0:1,*)
  OPENW, UNITE, 'SRCOMPE', /GET_LUN & OPENW, UNITW, 'SRCOMPW', /GET_LUN
;  Keep ph/s/0.1%bw/unit for now
;  a2(1,*) = a2(1,*)/BW/a2(0,*) ; convert to ph/s/eV/unit from ph/s/0.1%bw/unit
  PRINTF, UNITE, a2(0:1,*), FORMAT="(' ',f10.3,tr2,e13.6)" 
  a2w(1,*) = a2w(1,*)/BW*EC ; convert to W/eV/unit from ph/s/0.1%bw/unit
  PRINTF, UNITW, a2w(0:1,*), FORMAT="(' ',f10.3,tr2,e13.6)"
  FREE_LUN, UNITE & FREE_LUN, UNITW

  PRINT, ' Files SRCOMPE and SRCOMPW successfully written in the current directory.'
  ; srio added:
  itmp = widget_message(dialog_parent=event.top, /INFO, $
    'files SRCOMPE (flux) and SRCOMPW (spectral power) written to disk.')
  bell = STRING(7b)
  FOR i=0,1 DO BEGIN
    PRINT, bell
    WAIT, 0.2
  ENDFOR
  itmptmp = WIDGET_MESSAGE(/INFO, DIALOG_PARENT=event.top, $
   ' Files SRCOMPE and SRCOMPW successfully written in the current directory.')
  END
  'File:Quit': BEGIN
  HANDLE_FREE, hc 		; Note free handle here
  WIDGET_CONTROL, event.top, /DESTROY
  Delete_Files,['us.inp','us.out'],Group=event.top
  END

  'Set & Run': BEGIN
  Widget_Control,/HourGlass
;  IF NOT l1d THEN BEGIN
;    ff = findfile('us.inp')
;    fname = ff(0)
;    if fname eq '' THEN BEGIN
;      dat1 = XOP_DEFAULTS('xus') ; not found, define new structure
;    ENDIF ELSE BEGIN ; use file us.inp
;      error_status = xus_read_dat(dat1,group=event.top)
;      IF error_status NE 0 THEN BEGIN
;	junk = WIDGET_MESSAGE(/ERROR, DIALOG_PARENT=event.top, $
;	  ['','Error reading file', $
;	  'us.inp','This seems not an US ','input file  '])
;	GOTO, LBL900
;      ENDIF
;    ENDELSE
;    data = {dat1:dat1}
;  END

  XUS_SET, data, ACTION=action, GROUP=event.top
  IF action eq 'DONT' THEN GOTO, LBL900

;  Save us.inp
  error_status = XUS_WRITE_DAT(data.dat1, Group=event.top)
  IF error_status NE 0 THEN BEGIN
    junk = WIDGET_MESSAGE(/ERROR, DIALOG_PARENT=event.top, $
      'XUS_EVENT: Error writing file us.inp')
    GOTO, LBL900
  ENDIF

;  Run US
  IF data.dat1.run_mode_name eq 'foreground' THEN BEGIN ; foreground execution
    PRINT, ' Running US in the foreground; PLEASE WAIT!'
    CASE osversion OF
      ; 'UNIX':    command = 'source $XOP_RC ; us' 
      'UNIX':    command = 'us' 
      'WINDOWS': command = 'us'
      'MACOS':   command = 'us' ; 'MACOS command here'
    ENDCASE
    Widget_Control,/HourGlass
    Xop_Spawn,command,CleanFiles=['us.out']
    bell = STRING(7b)
    FOR i=0,1 DO BEGIN
      PRINT, bell
      WAIT, 0.2
    ENDFOR
  ENDIF ELSE BEGIN ; background execution
    CASE osversion OF
      'UNIX': BEGIN
	PRINT, ' Running US in the background'
        command=''
        Xop_Spawn,'us',onlyGetCommand=command
	; command = 'source $XOP_RC ; us &'
        command = 'xterm -e '+command+' &'
        message,/info,'Executing: '+command
        SPAWN, command
      END
      'WINDOWS': BEGIN
	itmp = WIDGET_MESSAGE(/INFO, DIALOG_PARENT=event.top, $
	  ['Background execution ','not implemented under Windows'])
      END
      'MACOS': BEGIN
	itmp = WIDGET_MESSAGE(/INFO, DIALOG_PARENT=event.top, $
	  ['Background execution ','not implemented under MacOS'])
        ; print,'MACOS command here'
      END
    ENDCASE
  ENDELSE
  END

  'Help:US': BEGIN
  hfile = Xop_GetEnv('XOP_HOME') +ds +'doc' +ds +'us.txt'
  XDISPLAYFILE1, hfile
  END
  'Help:Xus': Xhelp,'xus',Group=event.top

ELSE: BEGIN ; Show
  widget_control,/hourglass ; added by srio
  CASE elist(1) OF 
    'Plot 2d Results': BEGIN
    IF l2p THEN BEGIN ; display 2d data saved in anonymous structure
      a2 = plt.plt2.a2 ; 2d data
      nc = plt.plt2.nc ; # of columns
    ENDIF ELSE BEGIN ; read 2d data and save in anonymous structure
      error_status = xus_read2d_plt(a2, nd, Group=event.top)
      IF error_status NE 0 THEN BEGIN
        stxt = 'us.out'
        junk = WIDGET_MESSAGE(/ERROR, DIALOG_PARENT=event.top, $
	  ['','Error reading file', stxt])
        GOTO, LBL900
      ENDIF
      nc = nd(0)
      plt2 = {a2:a2, nc:nc}
      plt  = {plt2:plt2}
    ENDELSE
    CASE elist(2) OF 
      'Total Intensity': BEGIN
	ap = a2(0:1,*)
	ap(0,*) = ap(0,*)*C_EV_KEV		; convert eV -> keV
	xplot,ap,group=event.top,coltitles=['Energy(keV)','Total Intensity'], $
	  xtitle='-1',ytitle='Intensity(ph/s/mm^2/0.1%bw)',$
	  WTitle='Xus results: Total Intensity'
        END
      'Spectral Power': BEGIN
        ;  Conversion factors
        BW = 1.0D-3 ; 0.1%
        EC = 1.60217733D-19
	a2w = a2(0:1,*)
	a2w(0,*) = a2w(0,*)*C_EV_KEV		; convert eV -> keV
	a2w(1,*) = a2w(1,*)/BW*EC ; convert to W/eV/unit from ph/s/0.1%bw/unit
	xplot,a2w,group=event.top,coltitles=['Energy(keV)','Spectral Power'], $
	  xtitle='-1',ytitle='Spectral Power(Watts/eV)', $
	  WTitle='Xus results: Spectral Power'
        END
      'All': BEGIN
	IF nc NE 6 THEN BEGIN
	  junk = WIDGET_MESSAGE(/INFO, DIALOG_PARENT=event.top, ptxt)
	  GOTO, LBL900
	ENDIF
        ; ap = fltarr(12,n_elements(a2(0,*)))
        ap = fltarr(13,n_elements(a2(0,*)))
        ap(0:1,*) = a2(0:1,*)			; Energy, Total Intensity
	ap(0,*) = ap(0,*)*C_EV_KEV		; convert eV -> keV
        ap(2,*) = a2(1,*)*(1.0+a2(2,*))/2.0	; Linear Hor.  Intensity
        ap(3,*) = a2(1,*)*(1.0-a2(2,*))/2.0	; Linear Vert. Intensity
        ap(4,*) = a2(1,*)*a2(5,*)		; Unpolarized  Intensity
        ap(5,*) = 1.0-a2(5,*)			; Degree of Polarization
        ;ellipticity
        chi= 0.5*asin(a2(4,*)/(1.0-a2(5,*))) ; 1/2 asin(p3/p)
        ep = tan(chi) ; -pi/4.<=chi<=+pi/4., -1.<=ep<=+1.
        ap(6,*) = ep
        ;Azimuthal Angle
        psi= 0.5*atan2(a2(3,*),a2(2,*),/zero,/deg); 1/2 atan2(p2/p1)
        ap(7,*) = psi
        ap(8,*) = a2(2,*)			; Parameter p1
        ap(9,*) = a2(3,*)			; Parameter p2
        ap(10,*)= a2(4,*)			; Parameter p3
        ap(11,*)= a2(5,*)			; Parameter p4
        ;  Conversion factors
        BW = 1.0D-3 ; 0.1%
        EC = 1.60217733D-19
        ap(12,*)= a2(1,*)/BW*EC			; Spectral Power

        titles = ['Energy(keV)','Total Intensity',$
          'Linear Hor. Intensity','Linear Vert. Intensity', $
          'Unpolarized Intensity', 'Degree of Polarization', $
          'Ellipticity','Azimuthal Angle', $
          'Parameter p1','Parameter p2','Parameter p3','Parameter p4', $
	  'Spectral Power']
        xplot, ap, GROUP=event.top, coltitles=titles, xtitle='-1',ytitle='-1',$
          YCOL=2, Wtitle='Xus results: All parameters'
	END
    ENDCASE
    END ; Plot 2d Results

    'Plot 3d Results': BEGIN
    IF l3p THEN BEGIN ; display 3d data saved in anonymous structure
      z3 = plt.plt3.z3
      x3 = plt.plt3.x3
      y3 = plt.plt3.y3
      xx3=Reform(x3[*,0])
      yy3=Reform(y3[0,*])

      p1 = plt.plt3.p1
      p2 = plt.plt3.p2
      p3 = plt.plt3.p3
      p4 = plt.plt3.p4
    ENDIF ELSE BEGIN ; read 3d data and save in anonymous structure
      error_status = us_read3d_plt(z3, x3, y3, p1, p2, p3, p4)
      xx3=Reform(x3[*,0])
      yy3=Reform(y3[0,*])
      IF error_status NE 0 THEN BEGIN
        stxt = 'us.out'
        junk = WIDGET_MESSAGE(/ERROR, DIALOG_PARENT=event.top, $
	  ['','Error reading file', stxt])
        GOTO, LBL900
      ENDIF
      plt3 = {z3:z3, x3:x3, y3:y3, p1:p1, p2:p2, p3:p3, p4:p4}
      plt  = {plt3:plt3}
    ENDELSE
      CASE elist(2) OF 
        'Surface Plot': BEGIN
        CASE elist(3) OF 
	  'Total Intensity': BEGIN
	  sf = z3
	  END
	  'Linear Hor.  Intensity': BEGIN  ; Incl. unpolarized intensity
	  sf = z3*(1.0 +p1)/2.0
	  ;sf = z3*(1.0 -p4 +p1)/2.0 ; w/o unpol. intensity
	  END
	  'Linear Vert. Intensity': BEGIN
	  sf = z3*(1.0 -p1)/2.0
	  ;sf = z3*(1.0 -p4 -p1)/2.0 ; w/o unpol. intensity
	  END
	  'Unpolarized  Intensity': BEGIN
	  sf = z3*p4
	  END
	  'Degree of Polarization': BEGIN
	  sf = 1.0-p4
	  END
	  'Ellipticity': BEGIN
	  chi = 0.5*asin(p3/(1.0-p4)) ; 1/2 asin(p3/p)
	  ep  = tan(chi)
	  sf  = ep
	  END
	  'Azimuthal Angle': BEGIN
	  CASE elist(4) OF
	    'Surface': BEGIN
	    psi = 0.5*atan2(p2,p1,/zero,/deg); 1/2 atan2(p2/p1)
	    XSURFACE1, psi, xx3, yy3, Group=event.top, WTitle=elist[3]
	    END
	    'Vector Field': BEGIN
	    psi = 0.5*atan2(p2,p1,/zero); 1/2 atan2(p2/p1)
	    up  = cos(psi) ; length of vector is unity
	    vp  = sin(psi)
	    up1 = x3(*,0)
	    vp1 = y3(0,*)
	    command = 'VeloVect,data.up,data.vp,data.up1,data.vp1,'+$
		'length=0.8,xtitle="X",ytitle="Y"'
	    XWindow,Buffer=command,Data={up:up, vp:vp, up1:up1, vp1:vp1},$
		WTitle='Vector Field',/Edit, Group=event.top
	    ;velovect, up, vp, up1, vp1, length=0.8, $
	    ;  xtitle='X', ytitle='Y', title = 'Vector Field'
	    END
	  ENDCASE ; elist(4)
	  END ; Azimuthal Angle
	  'Parameter p1': BEGIN
	  sf = p1
	  END
	  'Parameter p2': BEGIN
	  sf = p2
	  END
	  'Parameter p3': BEGIN
	  sf = p3
	  END
	  'Parameter p4': BEGIN
	  sf = p4
	  END
	ENDCASE ; elist(3)
	IF elist(3) ne 'Azimuthal Angle' THEN $
	  XSURFACE1 , sf, xx3, yy3,  Group=event.top, WTitle=elist[3]
        END ; Surface Plot

;        'Show-3d Plot': BEGIN
;        CASE elist(3) OF 
;          'Total Intensity': BEGIN
;	  sf = z3
;	  END
;	  'Linear Hor.  Intensity': BEGIN
;	  sf = z3*(1.0 +p1)/2.0
;	  END
;	  'Linear Vert. Intensity': BEGIN
;	  sf = z3*(1.0 -p1)/2.0
;	  END
;	  'Unpolarized  Intensity': BEGIN
;	  sf = z3*p4
;	  END
;	  'Degree of Polarization': BEGIN
;	  sf = 1.0-p4
;	  END
;	  'Ellipticity': BEGIN
;	  chi = 0.5*asin(p3/(1.0-p4)) ; 1/2 asin(p3/p)
;	  ep  = tan(chi)
;	  sf  = ep
;	  END
;	  'Azimuthal Angle': BEGIN
;	  psi = 0.5*atan2(p2,p1,/zero,/deg); 1/2 atan2(p2/p1)
;	  sf  = psi
;	  END
;	  'Parameter p1': BEGIN
;	  sf = p1
;	  END
;	  'Parameter p2': BEGIN
;	  sf = p2
;	  END
;	  'Parameter p3': BEGIN
;	  sf = p3
;	  END
;	  'Parameter p4': BEGIN
;	  sf = p4
;	  END
;        ENDCASE ; elist(3)
;	xwindow,buffer='SHOW3, data',data=sf,/edit
;        END ; Show-3d Plot

      ENDCASE ; elist(2)
    END ; Plot 3d Results

    'View  Results': BEGIN
    XDISPLAYFILE1, fname_res, GROUP=event.top
    END

  ENDCASE ; elist(1)

ENDELSE ; eVal
ENDCASE ; eVal

LBL900:
IF WIDGET_INFO(event.top, /VALID) THEN BEGIN $
  HANDLE_VALUE, hc, control, /SET, /NO_COPY
  HANDLE_VALUE, hd, data, /SET, /NO_COPY
;  save structure plt if not a new run
  IF eVal ne 'Set & Run' THEN HANDLE_VALUE, hp, plt, /SET, /NO_COPY
; WIDGET_CONTROL, event.top, SET_UVALUE=info; save syntax for future use
ENDIF
RETURN

END ; xus_event
 
;
;===============================================================================
;
PRO xus, GROUP=group, InputFile=inputFile, No_Block=no_block

Catch, error_status
IF error_status NE 0 THEN BEGIN
   Message,/Info,'error caught: '+!error_state.msg
   itmp = Dialog_Message(/Error,Dialog_Parent=group, $
     'XUS: error caught: '+!error_state.msg)
   Catch, /Cancel
   On_Error,2
   RETURN
ENDIF

;  Definitions and labels
XUS_TITLE1 = 'Xus '+Xus_version()
; XUS_LBL1   = 'US: Undulator Spectrum'
XUS_LBL1   = 'US'
XUS_LBL2   = '   Undulator Spectrum   '


delimiter = ':'
osversion = sdep()	; system dependent operating system
ds = sdep(/ds)		; system dependent directory separator

;  set working directory
if not keyword_set(group) then xop_wd,/default

CD, CURRENT=cwd & cwd = cwd +ds
fname_res = cwd +'us.out'

usbase = WIDGET_BASE(COLUMN=1,TITLE=XUS_TITLE1, MBar=wMenuBar)

wFile = widget_button(wMenuBar,VALUE='File',/MENU)
  wtmp0 = widget_button(wFile,VALUE='Xus input parameters', /Menu)
  wtmp = widget_button(wtmp0,VALUE='Load from file...',UValue='FILEINPUT')
  wtmp = widget_button(wtmp0,VALUE='Save to file...',UValue='FILEINPUT')
  wtmp = widget_button(wtmp0,VALUE='Save as default',UValue='FILEINPUT')

  wtmp0 = widget_button(wFile,VALUE='US file', /Menu)
  wtmp = widget_button(wtmp0,VALUE='Load input file...',$
	UValue='File:Load  US:Input      File')
  wtmp = widget_button(wtmp0,VALUE='Load output 2D file...',$
	UValue='File:Load  US:Output  2d-File')
  wtmp = widget_button(wtmp0,VALUE='Load output 3D file...',$
	UValue='File:Load  US:Output  3d-File')
  wtmp = widget_button(wtmp0,VALUE='Write input file...',$
	UValue='File:Write US:Input  File')
  wtmp = widget_button(wtmp0,VALUE='Write output file...',$
	UValue='File:Write US:Output File')

  wtmp = widget_button(wFile,VALUE='Save files for XOP/Optics', $
	UVALUE='File:Write Files for XOP/Optics')

  wtmp = widget_button(wFile,VALUE='Quit', UVALUE='File:Quit',/Separator)


wSetParameters = widget_button(wMenuBar,VALUE='Set_Parameters', /MENU)
  wtmp = widget_button(wSetParameters,VALUE='Set Parameters', $
	UVALUE='Set & Run')
  wtmp = widget_button(wSetParameters,VALUE='Set Defaults', $
	UVALUE='File:Load  US:Default    File')

wResults = widget_button(wMenuBar,VALUE='Show',/MENU)
  wtmp0 = widget_button(wResults,VALUE='Plot 2D',/Menu) 
    wtmp = widget_button(wtmp0,VALUE='Total Intensity',$
	UValue='Show:Plot 2d Results:Total Intensity')
    wtmp = widget_button(wtmp0,VALUE='Spectral Power',$
	UValue='Show:Plot 2d Results:Spectral Power')
    wtmp = widget_button(wtmp0,VALUE='All',$
	UValue='Show:Plot 2d Results:All')
  wtmp0 = widget_button(wResults,VALUE='Plot 3D Surface ',/Menu)
    wtmp = widget_button(wtmp0,VALUE='Total Intensity',$
	UValue='Show:Plot 3d Results:Surface Plot:Total Intensity')
    wtmp = widget_button(wtmp0,VALUE='Linear Hor.  Intensity',$
	UValue='Show:Plot 3d Results:Surface Plot:Linear Hor.  Intensity')
    wtmp = widget_button(wtmp0,VALUE='Linear Vert. Intensity',$
	UValue='Show:Plot 3d Results:Surface Plot:Linear Vert. Intensity')
    wtmp = widget_button(wtmp0,VALUE='Unpolarized  Intensity',$
	UValue='Show:Plot 3d Results:Surface Plot:Unpolarized  Intensity')
    wtmp = widget_button(wtmp0,VALUE='Degree of Polarization',$
	UValue='Show:Plot 3d Results:Surface Plot:Degree of Polarization')
    wtmp = widget_button(wtmp0,VALUE='Ellipticity',$
	UValue='Show:Plot 3d Results:Surface Plot:Ellipticity')
    wtmp = widget_button(wtmp0,VALUE='Azimuthal Angle (Surface)',$
	UValue='Show:Plot 3d Results:Surface Plot:Azimuthal Angle:Surface')
    wtmp = widget_button(wtmp0,VALUE='Azimuthal Angle (Vector Field)',$
	UValue='Show:Plot 3d Results:Surface Plot:Azimuthal Angle:Vector Field')
    wtmp = widget_button(wtmp0,VALUE='Parameter p1',$
	UValue='Show:Plot 3d Results:Surface Plot:Parameter p1')
    wtmp = widget_button(wtmp0,VALUE='Parameter p2',$
	UValue='Show:Plot 3d Results:Surface Plot:Parameter p2')
    wtmp = widget_button(wtmp0,VALUE='Parameter p3',$
	UValue='Show:Plot 3d Results:Surface Plot:Parameter p3')
    wtmp = widget_button(wtmp0,VALUE='Parameter p4',$
	UValue='Show:Plot 3d Results:Surface Plot:Parameter p4')
  wtmp = widget_button(wResults,VALUE='View  Results',$
	UValue='Show:View  Results')

wHelp = widget_button(wMenuBar,VALUE='Help', /Help)
  wtmp = widget_button(wHelp,VALUE='US', UValue='Help:US')
  wtmp = widget_button(wHelp,VALUE='Xus', UValue='Help:Xus')
;

wtmp = widget_button(usbase,VALUE='Set Parameters', UVALUE='Set & Run')



; ;  Structure for pull-down menu
; junk = {CW_PDMENU_S, flags:0, name:''}
; 
; md = [ $
;   {CW_PDMENU_S, 1, 'File'}, $
;     {CW_PDMENU_S, 1, 'Load  US'}, $
;       {CW_PDMENU_S, 0, 'Input      File'}, $
;       {CW_PDMENU_S, 0, 'Output  2d-File'}, $
;       {CW_PDMENU_S, 0, 'Output  3d-File'}, $
;       {CW_PDMENU_S, 2, 'Default    File'}, $
;     {CW_PDMENU_S, 1, 'Write US'}, $
;       {CW_PDMENU_S, 0, 'Input  File'}, $
;       {CW_PDMENU_S, 2, 'Output File'}, $
;     {CW_PDMENU_S, 0, 'Display File'}, $
;     {CW_PDMENU_S, 0, 'Write Files for XOP/Optics'}, $
;     {CW_PDMENU_S, 2, 'Quit'}, $
; 
;   {CW_PDMENU_S, 0, 'Set & Run'}, $
; 
;   {CW_PDMENU_S, 1, 'Show'}, $
;     {CW_PDMENU_S, 1, 'Plot 2d Results'}, $
;       {CW_PDMENU_S, 0, 'Total Intensity'}, $
;       {CW_PDMENU_S, 2, 'All'}, $
;     {CW_PDMENU_S, 1, 'Plot 3d Results'}, $
;       {CW_PDMENU_S, 1, 'Surface Plot'}, $
; 	{CW_PDMENU_S, 0, 'Total Intensity'}, $
; 	{CW_PDMENU_S, 0, 'Linear Hor.  Intensity'}, $
; 	{CW_PDMENU_S, 0, 'Linear Vert. Intensity'}, $
; 	{CW_PDMENU_S, 0, 'Unpolarized  Intensity'}, $
; 	{CW_PDMENU_S, 0, 'Degree of Polarization'}, $
; 	{CW_PDMENU_S, 0, 'Ellipticity'}, $
; 	{CW_PDMENU_S, 1, 'Azimuthal Angle'}, $
; 	  {CW_PDMENU_S, 0, 'Surface'}, $
; 	  {CW_PDMENU_S, 2, 'Vector Field'}, $
; 	{CW_PDMENU_S, 0, 'Parameter p1'}, $
; 	{CW_PDMENU_S, 0, 'Parameter p2'}, $
; 	{CW_PDMENU_S, 0, 'Parameter p3'}, $
; 	{CW_PDMENU_S, 2, 'Parameter p4'}, $
;       {CW_PDMENU_S, 3, 'Show-3d Plot'}, $
; 	{CW_PDMENU_S, 0, 'Total Intensity'}, $
; 	{CW_PDMENU_S, 0, 'Linear Hor.  Intensity'}, $
; 	{CW_PDMENU_S, 0, 'Linear Vert. Intensity'}, $
; 	{CW_PDMENU_S, 0, 'Unpolarized  Intensity'}, $
; 	{CW_PDMENU_S, 0, 'Degree of Polarization'}, $
; 	{CW_PDMENU_S, 0, 'Ellipticity'}, $
; 	{CW_PDMENU_S, 0, 'Azimuthal Angle'}, $
; 	{CW_PDMENU_S, 0, 'Parameter p1'}, $
; 	{CW_PDMENU_S, 0, 'Parameter p2'}, $
; 	{CW_PDMENU_S, 0, 'Parameter p3'}, $
; 	{CW_PDMENU_S, 2, 'Parameter p4'}, $
;     {CW_PDMENU_S, 2, 'View  Results'}, $
; 
;   {CW_PDMENU_S, 3, 'Help'}, $
;     {CW_PDMENU_S, 0, 'US'}, $
;     {CW_PDMENU_S, 2, 'Xus'} ]
; 
; pd = CW_PDMENU(usbase, md, DELIMITER=delimiter, /RETURN_FULL_NAME)

CASE osversion OF
  'UNIX': BEGIN
    ; font='-b&h-lucida-bold-i-normal-sans-24-240-75-75-p-151-iso8859-1'
    font = '-adobe-helvetica-bold-o-normal--18-180-75-75-p-104-iso8859-1'
  END
  'WINDOWS': BEGIN
    font = 'VERDANA*BOLD*ITALIC*24'
  END
  'MACOS': BEGIN
    font = 'VERDANA*BOLD*ITALIC*24'  ; 'MACOS command here'
  END
ENDCASE

lb = WIDGET_LABEL(usbase, FONT=font, VALUE=XUS_LBL1)
lb = WIDGET_LABEL(usbase, FONT=font, VALUE=XUS_LBL2)

control = {delimiter:delimiter, fname_res:fname_res, osversion:osversion, $
	   ds:ds}
hc = HANDLE_CREATE(VALUE=control)
hd = HANDLE_CREATE(hc)
hp = HANDLE_CREATE(hc)




xus_str = Xop_Defaults('xus')

IF KeyWord_Set(inputFile) THEN BEGIN
  str_par = Xop_Input_Load(InputFile=inputFile)
  IF Type(str_par) EQ 8 THEN BEGIN
    tmp = xus_str
    Copy_Structure,str_par, tmp, Group=group
    xus_str = tmp
  ENDIF
ENDIF


Handle_Value,hd,{dat1:xus_str},/Set


info = {hc:hc, hd:hd, hp:hp}
;print,'main',hc,hd,hp,usbase

WIDGET_CONTROL, usbase, SET_UVALUE=info

WIDGET_CONTROL, usbase, /REALIZE
IF N_Elements(no_block) EQ 0 THEN no_block=0
XMANAGER, 'xus', usbase, GROUP_LEADER=group,No_Block=no_block

END ; xus
