;+
; ============================== XPLOT2D ======================================================
; 
; XPLOT2D  
;
; A software tool for analysing scientific images.
;
; Version 1.0 Beta 1  (March 15 2004)
; Version 1.0 Beta 2  (June 11 2004)
;
; Functionality: 
;    - Load and visualize several images
;    - Several file types: EDF, MAR, BINARY, XRD, etc.
;    - Automatically handles gzip MAR compressed files
;    - Loads multiple files 
;    - When an image is load, it can be added to the image stack or
;	substitute the first image in the stack.
;    - Visualize images in frames. 
;    - Apply several color tables and adjust their limits to data.
;		button: "c f" color full range
;		button: "c s" color stretched range
;    - Define ROIS (rectangle, circle, cake, polygon)
;    - Mask data to ROI or complement of ROI (1-ROI)
;    - Applies mask from another image
;    - Calculations: radial integration, histograms, horizontal or
;	vertical integration, etc.
;    - Simulations: Overplot rings from external x-ray diffraction profiles,
;	defined in with either "tewotheta" or "d-spacing" files,
;	containing columns with :
;	    twotheta [deg], intensity [,color index]  or
;	    d-spacing [A], intensity [,color indexa] , respectively.
;    - Makes a generic operation with one image. 
;    - Define/edit image titles
;
; To be done:
;   - Load more file formats
;   - Non-interactive use
;   - Zoom
;   - Improve profiles
;   - Undo levels
;   - New calculations and simulations
;   - Mask algebra
;   - Printing
;   - Documentation
;
; Authors: M. Sanchez del Rio, F. Matulic and V. Favre-Nicolin
;
; Modifications history: 
;   - 04-03-15 First beta version.
;   - 04-05-11 2nd beta version.
;
;-
;========================== Non interactive use of xplot2d ====================================
;
; NAME:
;   XPLOT2D
;
; PURPOSE:
;   Display image(s), print them, manage them.
; CATEGORY:
;   image display
;
; CALLING SEQUENCE:
;   xplot2d [,inData]  [,keywords]
;
; OPTIONAL INPUT PARAMETERS:
;      inData: the 2d image array or the filename of the image data
;             to import (currently understood:edf only)
;
;F. Matulic 24.09.02 completed development of the main prototype
;M. Sanchez del Rio, 07/2002 (painted skeleton)
;V. Favre-Nicolin, 01/2002 (skeleton)
;
;
; Remote Control example: 
;
;xplot2d,iq_pick('vfn06.gel'),/no_block,parent=p
;xplot2d_add_image,p,iq_pick('vfn14.gel')
;xplot2d_add_image,p,iq_pick('vfn17.gel')
;xplot2d_replace_image,p,iq_pick('vfn06.gel')
;xplot2d_set_scale,p,min=1,max=100,/log
;xplot2d_replace_image,p,iq_pick('vfn19.gel')
;xplot2d_change_image,p,0,min=10,max=1000,/log
;xplot2d_next_image,p
;xplot2d_rotate_image,p,2
;xplot2d_quit,p
;
;

; NAME:
;   XPLOT2D_SET_COMMENT
;
; PURPOSE:
;
; CATEGORY:
;   image display
;
; CALLING SEQUENCE:
;   xplot2d_Set_Comment,parent,comment
;
;...
;
;===============================================================================
;
PRO XPLOT2D_SET_COMMENT,wXplot2d,comment
;Set the comment line
   widget_control,wXplot2d,get_uvalue=uvalue
   widget_control,uvalue.wComment,set_value=comment
END


;
;===============================================================================
;
PRO XPLOT2D_DISPLAY_IMAGE,wXplot2d,print=print1
; Displays the image in the widget_draw window,
; or creates the command to be executed for printing

COMMON Xplot2D_common_variables, max_acceptable_screen_size_factor, scroll_step, margin_space, fileDir

widget_control,wXplot2d,get_uvalue=info
pData=info.data->Value([info.image_num],/pointer)
title=info.data->Value([info.image_num],/title)
img = *pData
datasize=size(img)
;print,"image size : ",datasize

IF keyword_set(print1) THEN info.scale=info.scale*10
;
; display data
;
XPLOT2D_SET_COMMENT,wXplot2d,'displaying image...'
;resize window

;new_?size is the total frame size including both margins
new_xsize=dataSize(1)+margin_space*2
new_ysize=dataSize(2)+margin_space*2

widget_control,info.wGraph,get_value=graphwin

IF Not(Keyword_Set(print1)) THEN BEGIN
  wset,graphwin
  erase
ENDIF
IF Not(Keyword_Set(print1)) THEN $
   device,get_screen_size=screen_size ELSE $
   screen_size=200*datasize[[1,2]]   ; srio TO BE CHECKED

; compute the maximum acceptable screen size dimensions using the current display settings
max_acceptable_screen_size=screen_size*max_acceptable_screen_size_factor

ncolors = !d.table_size - 32;total number of colours available for the image (32 system colours)

if info.ScaleType EQ 'log' then begin
        IF (info.min EQ info.max) THEN BEGIN
	      good_indices = where(img GE 0) ;get the indexes of the unmasked values
	      min_img = min(img[good_indices])
	      max_img = max(img)
	ENDIF ELSE BEGIN
	      min_img = info.min
	      max_img = info.max
	ENDELSE
	    img2 = bytscl(alog10(img),min=alog10(min_img),max=alog10(max_img),top=(ncolors - 1))+byte(32)
	    bad_indices = where(img LT 0)
	    IF bad_indices[0] NE -1 THEN img2[bad_indices] = byte(3)
endif else begin
	    IF (info.min EQ info.max) THEN BEGIN
	      good_indices = where(img GE 0)
	      min_img = min(img[good_indices])
	      max_img = max(img)
	    ENDIF ELSE BEGIN
	      min_img = info.min
	      max_img = info.max
	    ENDELSE
	    img2 = bytscl(img,min=min_img,max=max_img,top=(ncolors - 1))+byte(32)
            bad_indices = where(img LT 0)
	    IF bad_indices[0] NE -1 THEN img2[bad_indices] = byte(3)
endelse

IF info.scale NE 1 THEN BEGIN
  s1 = size(img2)
  img2 = ConGrid(img2,s1[1]*info.scale,s1[2]*info.scale)
ENDIF

IF Not(Keyword_Set(print1)) THEN BEGIN
    ;
    ; DISPLAY COLOR BAR
    ;
    device, window_state=window_state
    IF window_state(info.colorbarWindowId) EQ 1 THEN BEGIN ; window exists
	xplot2d_displayct,wXplot2d,MINRANGE=min_img,MAXRANGE=max_img
      ;IF window_state(info.colorbarWindowId) EQ 0 THEN BEGIN ; window not existing - Create it!
      ;   window, /free, xsize=110, ysize=250
      ;   info.colorbarWindowId = !d.window
      ;   Widget_Control,wXplot2d, Set_UValue=info
      ;ENDIF
      ;wset, info.colorbarWindowId ; should check whether the window is still available (how the heck can this be done in IDL ?)
      ;erase
;
;      if info.ScaleType EQ 'log' then $
;        ;using fanning's widget for the colourbar here
;        fcolorbar, bottom=32, color=1, ncolors=ncolors, position=[0.6, 0.05, 0.8, 0.95], /vertical, MINRANGE=(min_img), MAXRANGE=(max_img), format='(g0.2)', /ylog $
;      else $
;        fcolorbar, bottom=32, color=1, ncolors=ncolors, position=[0.6, 0.05, 0.8, 0.95], /vertical, MINRANGE=min_img, MAXRANGE=max_img, format='(g0.2)'
    ENDIF
ENDIF

; 
; DISPLAY IMAGE
;
IF Not(Keyword_Set(print1)) THEN wset,graphwin

s1 = size(img2)
new_xsize=s1(1)+margin_space*2
new_ysize=s1(2)+margin_space*2
IF (new_xsize le max_acceptable_screen_size[0]) and $
         (new_ysize le max_acceptable_screen_size[1]) then BEGIN ; check whether image size exceeds maximum acceptable screen size

         widget_control,info.wGraph,draw_xsize=new_xsize,draw_ysize=new_ysize ; not sure whether the draw_xsize is used anywhere
         info.image_position=[0,0] ; image can be fit in the current window so set the image position to [0,0]

         tv, img2, margin_space, margin_space, xsize=s1[1], ysize=s1[2]

		 ;display the axes around the image by means of the plot function
         plot, s1[1:2],/nodata, /device, /noerase, position=[margin_space, margin_space,$
		new_xsize-(margin_space),new_ysize-(margin_space)], color=1, $
		xrange=[info.axis_properties.x_origin,dataSize(1)/float(info.axis_properties.x_pixel_unit_ratio)],$
		yrange=[info.axis_properties.y_origin,dataSize(2)/float(info.axis_properties.y_pixel_unit_ratio)],$
		xstyle=1,ystyle=1,title=title

         info.scroll_vec=[0,0] ; reset scroll vector

ENDIF ELSE BEGIN ; one of the image dimension is bigger than max_acceptable_sreen_size

      IF (info.scroll_vec[0] eq 0 and info.scroll_vec[1] eq 0) then BEGIN ; no scrolling, display image in lower left corner (default behaviour)

        widget_control,info.wGraph,draw_xsize=max_acceptable_screen_size[0],$
                                      draw_ysize=max_acceptable_screen_size[1]

		window, 20, xsize=new_xsize, ysize=new_ysize, /pixmap ;draw the entire image in a virtual window

        tv, img2, margin_space, margin_space, xsize=s1[1], ysize=s1[2]
		plot, s1[1:2],/nodata, /device, /noerase, position=[margin_space, margin_space,$
		      new_xsize-(margin_space),new_ysize-(margin_space)], color=1,$
		      xrange=[info.axis_properties.x_origin,dataSize(1)/float(info.axis_properties.x_pixel_unit_ratio)],$
		      yrange=[info.axis_properties.y_origin,dataSize(2)/float(info.axis_properties.y_pixel_unit_ratio)],$
		      xstyle=1,ystyle=1,title=title
        IF Not(Keyword_Set(print1)) THEN wset,graphwin
        IF Not(Keyword_Set(print1)) THEN erase

		device, copy=[-info.image_position[0],-info.image_position[1],max_acceptable_screen_size[0],$
		                max_acceptable_screen_size[1],0,0,20]


ENDIF ELSE BEGIN ; scrolling

   		info.image_position=info.image_position+info.scroll_vec

		widget_control,info.wGraph,TLB_GET_SIZE=window_size

		;the following is to check wether the image has reached one of its bounds and therefore
		;whether scrolling any further should be denied
		if (info.scroll_vec[0] gt 0 and -info.image_position[0] lt 0) or $
		   (info.scroll_vec[0] lt 0 and (-info.image_position[0]+max_acceptable_screen_size[0]-(margin_space)) ge new_xsize) or $
		   (info.scroll_vec[1] gt 0 and -info.image_position[1] lt 0) or $
		   (info.scroll_vec[1] lt 0 and (-info.image_position[1]+max_acceptable_screen_size[1]-(margin_space)) ge new_ysize) then begin ;scroll out of bounds attempted

		   info.image_position=info.image_position-info.scroll_vec ; reset info
           info.scroll_vec=[0,0]
           widget_control,wXplot2d,set_uvalue=info

   		 endif

		 window, 20, xsize=new_xsize, ysize=new_ysize, /pixmap ;draw the entire image in a virtual window

         tv, img2, margin_space, margin_space, xsize=s1[1], ysize=s1[2]
		 plot, s1[1:2],/nodata, /device, /noerase, position=[margin_space, margin_space,$
			  						new_xsize-(margin_space),new_ysize-(margin_space)], color=1,$
			  						xrange=[info.axis_properties.x_origin,dataSize(1)/float(info.axis_properties.x_pixel_unit_ratio)],$
									yrange=[info.axis_properties.y_origin,dataSize(2)/float(info.axis_properties.y_pixel_unit_ratio)],$
									xstyle=1,ystyle=1

         IF Not(Keyword_Set(print1)) THEN wset,graphwin
         IF Not(Keyword_Set(print1)) THEN erase

		 device, copy=[-info.image_position[0],-info.image_position[1],max_acceptable_screen_size[0],$
			                max_acceptable_screen_size[1],0,0,20]

ENDELSE ; end scrolling section

ENDELSE ; end else image bigger than frame size section

; 
; overplot rings
; 
IF info.overplotPeaks EQ 1 OR info.overplotDSpacing THEN xplot2d_overplotPeaks,wXplot2d

IF keyword_set(print1) THEN info.scale=info.scale/10  ; back to original value

widget_control,info.wImageNum,set_value=info.image_num+1
widget_control,wXplot2d,set_uvalue=info
XPLOT2D_SET_COMMENT,wXplot2d,'displaying image done: '+title
END


;
;===============================================================================
;
PRO XPLOT2D_CHANGE_IMAGE,wXplot2d,index,min=min,max=max,log=log,linear=linear
;Display another image
; can also change the min/max/scale by the way
   widget_control,wXplot2d,get_uvalue=uvalue
	if(uvalue.image_num EQ index) $
	   OR (index LT 0) OR (index GE uvalue.data->Info(/N_Elements))$
	then begin
		widget_control,uvalue.wImageNum,set_value=uvalue.image_num+1
		return

	endif

   uvalue.image_num=index
   if(keyword_set(min)) then uvalue.min=min
   if(keyword_set(max)) then uvalue.min=max
   if(keyword_set(log)) then uvalue.ScaleType='log'
   if(keyword_set(linear)) then uvalue.ScaleType='linear'
   widget_control,wXplot2d,set_uvalue=uvalue
   xplot2d_display_image,wXplot2d
END

;
;===============================================================================
;
PRO XPLOT2D_LAST_IMAGE,wXplot2d
;Change the image currently displayed
   widget_control,wXplot2d,get_uvalue=uvalue  
   image_num=uvalue.data->INFO(/N_ELEMENTS)-1
   XPLOT2D_CHANGE_IMAGE,wXplot2d,image_num
END

;
;===============================================================================
;
PRO XPLOT2D_FIRST_IMAGE,wXplot2d
;Change the image currently displayed
   ;widget_control,wXplot2d,get_uvalue=uvalue
   XPLOT2D_CHANGE_IMAGE,wXplot2d,0
END

;
;===============================================================================
;
PRO XPLOT2D_NEXT_IMAGE,wXplot2d
;Change the image currently displayed
   widget_control,wXplot2d,get_uvalue=uvalue
   XPLOT2D_CHANGE_IMAGE,wXplot2d,uvalue.image_num+1
END

;
;===============================================================================
;
PRO XPLOT2D_PREVIOUS_IMAGE,wXplot2d
;Change the image currently displayed
   widget_control,wXplot2d,get_uvalue=uvalue
   XPLOT2D_CHANGE_IMAGE,wXplot2d,uvalue.image_num-1
END

;
;===============================================================================
;
PRO XPLOT2D_ADD_IMAGE,wXplot2d,indata,title=title,substitute=substitute
widget_control,wXplot2d,get_uvalue=uvalue
IF Keyword_Set(substitute) THEN BEGIN
   ;Substitute first image
   uvalue.data->set,0,/clean
   uvalue.data->set,0,value=indata,aux=indata,title=title
   uvalue.image_num=0
ENDIF ELSE BEGIN
   ;Add an image
   uvalue.data->set,value=indata,aux=indata,title=title,/add
   uvalue.image_num=uvalue.data->INFO(/N_ELEMENTS)-1
ENDELSE
widget_control,wXplot2d,set_uvalue=uvalue
xplot2d_display_image,wXplot2d
END

;
;===============================================================================
;
PRO XPLOT2D_REPLACE_IMAGE,wXplot2d,indata
;replace the current image by another
   widget_control,wXplot2d,get_uvalue=uvalue
   ;print,uvalue.data->Info()
   uvalue.data->set,[uvalue.image_num],value=indata,/clean
   ;print,uvalue.data->Info()
   widget_control,wXplot2d,set_uvalue=uvalue
   xplot2d_display_image,wXplot2d
END

;
;===============================================================================
;
PRO XPLOT2D_SET_SCALE,wXplot2d,min=min,max=max,log=log,linear=linear
;Change the scaling of the display
   widget_control,wXplot2d,get_uvalue=uvalue
   if(keyword_set(min)) then uvalue.min=min
   if(keyword_set(max)) then uvalue.max=max
   if(keyword_set(log)) then uvalue.ScaleType='log'
   if(keyword_set(linear)) then uvalue.ScaleType='linear'
   widget_control,wXplot2d,set_uvalue=uvalue
   xplot2d_display_image,wXplot2d
END


;
;===============================================================================
;
FUNCTION XPLOT2D_GET_DATAPTR,wXplot2d,index=index
;Access to the data
   widget_control,wXplot2d,get_uvalue=uvalue
   if not(keyword_set(index)) then index=uvalue.image_num
   return,uvalue.data->Value(index,/pointer)
END

;
;===============================================================================
;
PRO XPLOT2D_ROTATE_IMAGE,wXplot2d,direction
;Rotates the currently displayed image, with an optionnal mirror:
;
;Depending on the value of 'direction'
; dir mirror  rotation
; 0   No       None         X0    Y0
; 1   No       90         -Y0    X0
; 2   No       180        -X0   -Y0
; 3   No       270         Y0   -X0
; 4   Yes      None         Y0    X0
; 5   Yes      90         -X0    Y0
; 6   Yes      180        -Y0   -X0
; 7   Yes      270         X0   -Y0

	Widget_Control,/HourGlass
	if direction eq 0 then return
   widget_control,wXplot2d,get_uvalue=uvalue
   pData=XPLOT2D_GET_DATAPTR(wXplot2d)
	*pData=rotate(*pData,direction)
   xplot2d_display_image,wXplot2d
END

;
;===============================================================================
;
PRO XPLOT2D_QUIT,wXplot2d
;Quit xplot2d, erasing all data
   widget_control,wXplot2d,get_uvalue=info
   Obj_Destroy,info.data
   IF Ptr_Valid(info.roi_pixels) EQ 1 THEN Ptr_Free,info.roi_pixels
   widget_control,wXplot2d,/destroy
END


;
;===============================================================================
;
PRO XPLOT2D_SCROLL_IMAGE,wXplot2d,x_dir,y_dir
;Scroll image inside the drawing window
widget_control,wXplot2d,get_uvalue=info
info.scroll_vec=[x_dir,y_dir];set the scroll vector to the specified parameters
widget_control,wXplot2d,set_uvalue=info
xplot2d_display_image,wXplot2d
END

; function which returns the points of a cake outline (in device coordinates)

FUNCTION XPLOT2D_DRAW_CAKE,Xcent,Ycent,rho1,rho2,phi1,phi2
  x_roi=[0] ; need to create an empty array here but since this is impossible, array is initialized with a bogus value
  y_roi=[0] ; ditto
  step_grain=100.0 ; determines the fineness of the cake outline
  				   ; reducing the step amount increases the number of computed values for the roi
  if (rho1 ne 0.0) or ((phi1 ne 0.0) and (phi2 ne 360.0)) then begin ; check whether this is a cake and not a circle
    ;draw lower arc of the cake (whose radius is rho1)
	if phi1 gt phi2 then begin ;cake arcs cross polar origin axis => split the point creation process in 2 loops

	  ;first loop goes from phi1 to 360
      for phi=float(phi1),2*!pi, (2*!pi-phi1)/step_grain do begin
        x_roi=[x_roi,Xcent+rho1*cos(phi)]
        y_roi=[y_roi,Ycent+rho1*sin(phi)]
      endfor
      ;second loop goes from phi1 to 360
      for phi=0.0,float(phi2), phi1/step_grain do begin
        x_roi=[x_roi,Xcent+rho1*cos(phi)]
        y_roi=[y_roi,Ycent+rho1*sin(phi)]
      endfor

    endif else begin ; normal loop range from phi1 to phi2
      for phi=float(phi1),float(phi2), (phi2-phi1)/step_grain do begin
        x_roi=[x_roi,Xcent+rho1*cos(phi)]
        y_roi=[y_roi,Ycent+rho1*sin(phi)]
      endfor
    endelse
  endif

  ;draw upper arc (whose radius is rho2) starting from phi2
  if phi1 gt phi2 then begin ;same dichotomy as above

    for phi=float(phi2),0.0, -(2*!pi-phi1)/step_grain do begin
      x_roi=[x_roi,Xcent+rho2*cos(phi)]
      y_roi=[y_roi,Ycent+rho2*sin(phi)]
    endfor
    for phi=2*!pi,float(phi1), -phi1/step_grain do begin
      x_roi=[x_roi,Xcent+rho2*cos(phi)]
      y_roi=[y_roi,Ycent+rho2*sin(phi)]
    endfor

  endif else begin

    for phi=float(phi2),float(phi1), -(phi2-phi1)/step_grain do begin
      x_roi=[x_roi,Xcent+rho2*cos(phi)]
      y_roi=[y_roi,Ycent+rho2*sin(phi)]
    endfor

  endelse

  x_roi=x_roi[1:*] ; get rid of the first bogus value
  y_roi=y_roi[1:*]

  points = make_array(2,n_elements(x_roi),/float);create the array for return
  points[0,*] = x_roi
  points[1,*] = y_roi

  return, points

END

;
;===============================================================================
;
; Sets the current selection to act as a mask on the image (sets its pixels to -1)

PRO XPLOT2D_APPLY_SELECTION_TO_MASK,wXplot2d,Complementary=complementary

widget_control,wXplot2d,get_uvalue=info
COMMON Xplot2D_common_variables, max_acceptable_screen_size_factor, scroll_step, margin_space, fileDir
pData=info.data->Value([info.image_num],/pointer)
img = *pData
datasize=size(img)

IF ptr_valid(info.roi_pixels) THEN BEGIN
endif else begin
  blah=dialog_message(['No selection points found!', '(Make sure the ROI is entirely within the', 'image bounds to convert it to a mask)'],/ERROR)
  RETURN
endelse

roi_pixels = *info.roi_pixels
IF Keyword_Set(complementary) THEN BEGIN
  indices=lonarr(datasize[1]*datasize[2])
  indices[roi_pixels]=1
  roi_pixels=Where(indices EQ 0)
ENDIF

if (roi_pixels)[0] gt -1 then begin ; perform calculations only if the region is not empty
  img_old=img
  img[roi_pixels]=-1.0
  *pData=img
  ;write back image data to info structure
  info.data->set,[info.image_num],value=img,aux=img_old
  widget_control,wXplot2d,set_uvalue=info
  XPLOT2D_DISPLAY_IMAGE,wXplot2d
endif else begin
  blah=dialog_message(['No selection points found!', '(Make sure the ROI is entirely within the', 'image bounds to convert it to a mask)'],/ERROR)
endelse

END

;
;===============================================================================
;
PRO XPLOT2D_AZIMUTH_INTEGRAL,wXplot2d
widget_control,wXplot2d,get_uvalue=info
pData=info.data->Value([info.image_num],/pointer)
img = *pData
datasize=size(img)

centre_of_integration = [info.center[0],info.center[1]]
Widget_Control,/Hourglass

tmp = where(img LT 0)
IF tmp[0] EQ -1 THEN BEGIN
  Print,'Masked values: None' 
  radial_AT,img,center=centre_of_integration,INCREMENT=1.0,DIAGRAM=intensity,X=xx,ERROR=error
ENDIF ELSE BEGIN
  Print,'Masked values: ',N_Elements(tmp),' over',N_Elements(img)
  mask=img*0
  mask[where(img LT 0)]=1
  radial_AT,img,center=centre_of_integration,INCREMENT=1.0,DIAGRAM=intensity,X=xx,ERROR=error,$
	mask=mask
ENDELSE
;
; express profile in correct units (intensity versus angle)
;
dist = info.detector_dist  ; 11.0 ; cm
pixelsize = info.pixelsize ; 78.94e-4
e1 = info.e1 
print,'Using detector_dist [cm]: ',dist
print,'Using pixelSize [microns]: ',pixelsize
print,'Using photon energy [eV]: ',e1

twotheta = 180./!pi*atan(xx*pixelsize*1e-4/dist)
dspacing = physical_constants('hc')/2.0/e1/sin(twotheta/2.*!pi/180.)

tmp = make_set(findgen(N_Elements(xx)),xx,twotheta ,dspacing,intensity)

title=info.data->Value(info.image_num,/title)

xplot,tmp,/No_Block,Window_Size=[600,400],xcol=1,coltitles=$
  ['pixel','x','2theta[deg]','dspacing','intensity'],WTitle='Azimuth integration',$
  title=title

XPLOT2D_SET_COMMENT,wXplot2d,'Azimuth integration: center='+Vect2String(centre_of_integration)+$
   ', E='+String(e1,Format='(F7.1)')+', D='+String(dist,Format='(F4.1)')+$
   ', Pixel: '+String(pixelsize,Format='(F4.1)')
END


;
;===============================================================================
;
PRO XPLOT2D_ROI_POLYGON_MOUSE,wXplot2d
widget_control,wXplot2d,get_uvalue=info
COMMON Xplot2D_common_variables, max_acceptable_screen_size_factor, scroll_step, margin_space, fileDir

widget_control,info.wGraph,get_value=graphwin
wset, graphwin
XPLOT2D_SET_COMMENT,wXplot2d,'Click inside the image to insert points and click outside when finished'
pData=info.data->Value([info.image_num],/pointer)
img = *pData
datasize=size(img)

CURSOR, xCl, yCl, /DOWN, /DEVICE

polygon_x_values=[xCl]
polygon_y_values=[yCl]


while ((xCl ge margin_space) and (yCl ge margin_space) and $
      (xCl lt (dataSize(1)*info.scale)+info.image_position[0]+margin_space) and $
      (yCl lt (dataSize(2)*info.scale)+info.image_position[1]+margin_space)) do begin ;mouse within image bounds

CURSOR, xCl, yCl, /DOWN, /DEVICE

polygon_x_values=[polygon_x_values,xCl]
polygon_y_values=[polygon_y_values,yCl]

endwhile

xplot2d_display_image,wXplot2d

polygon_x_values=polygon_x_values[0:n_elements(polygon_x_values)-2] ;discard last element outside image range
polygon_y_values=polygon_y_values[0:n_elements(polygon_y_values)-2]

Plots, polygon_x_values, polygon_y_values, color=2, /device
plots, [polygon_x_values[0]], [polygon_y_values[0]], color=2, /device, /continue ;connect last point to the first

pData=info.data->Value([info.image_num],/pointer)
img = *pData
datasize=size(img)
scaled_down_polygon_x_values=(polygon_x_values-margin_space-info.image_position[0])/info.scale ;coordinates scaled back according to the original image
scaled_down_polygon_y_values=(polygon_y_values-margin_space-info.image_position[1])/info.scale
info.roi_pixels=ptr_new(polyfillv(scaled_down_polygon_x_values,scaled_down_polygon_y_values,$
							      datasize[1],datasize[2]))
;set the roi centre to centre of gravity of the polygon
info.roi_centre_pixel_coord=[total(scaled_down_polygon_x_values),total(scaled_down_polygon_y_values)]/$
                            float(n_elements(scaled_down_polygon_x_values))

widget_control,wXplot2d,set_uvalue=info

END

;
;===============================================================================
;
PRO XPLOT2D_ROI_RECTANGLE_MOUSE,wXplot2d
widget_control,wXplot2d,get_uvalue=info
COMMON Xplot2D_common_variables, max_acceptable_screen_size_factor, scroll_step, margin_space, fileDir

widget_control,info.wGraph,get_value=graphwin
wset, graphwin
XPLOT2D_SET_COMMENT,wXplot2d,'Click on the lower left corner of the rectangle...'

CURSOR, Xll, Yll, /DOWN, /DEVICE
XPLOT2D_SET_COMMENT,wXplot2d,'Click again to determine the upper right corner of the rectangle'
CURSOR, Xur, Yur, /DOWN, /DEVICE

info.roi_rectangle_info.lower_left_corner=[(Xll-margin_space-info.image_position[0])/info.Scale,$
								(Yll-margin_space-info.image_position[1])/info.Scale]
info.roi_rectangle_info.upper_right_corner=[(Xur-margin_space-info.image_position[0])/info.Scale,$
								(Yur-margin_space-info.image_position[1])/info.Scale]

xplot2d_display_image,wXplot2d

Plots, [Xll,Xll,Xur,Xur], [Yll,Yur,Yur,Yll], color=2, /device
plots, [Xll], [Yll], color=2, /device, /continue ;connect last point to the first

pData=info.data->Value([info.image_num],/pointer)
img = *pData
datasize=size(img)
info.roi_pixels=ptr_new(polyfillv([info.roi_rectangle_info.lower_left_corner[0],$
								   info.roi_rectangle_info.lower_left_corner[0],$
								   info.roi_rectangle_info.upper_right_corner[0],$
								   info.roi_rectangle_info.upper_right_corner[0]],$
								  [info.roi_rectangle_info.lower_left_corner[1],$
								   info.roi_rectangle_info.upper_right_corner[1],$
								   info.roi_rectangle_info.upper_right_corner[1],$
								   info.roi_rectangle_info.lower_left_corner[1]],$
								   datasize[1],datasize[2]))
info.roi_centre_pixel_coord=(info.roi_rectangle_info.lower_left_corner+info.roi_rectangle_info.upper_right_corner)/2

widget_control,wXplot2d,set_uvalue=info

END

;
;===============================================================================
;
PRO XPLOT2D_ROI_RECTANGLE_EDIT_EVENT, event
COMMON Xplot2D_common_variables, max_acceptable_screen_size_factor, scroll_step, margin_space, fileDir
widget_control,event.top,get_uvalue=info2
widget_control,info2.wXplot2d,get_uvalue=info
widget_control,event.id,get_uvalue=widget
if (widget eq 'UPDATE') or (widget eq 'OK') then begin

  widget_control,info2.lower_left_xID,get_value=lower_left_x
  info.roi_rectangle_info.lower_left_corner[0]=lower_left_x
  widget_control,info2.lower_left_yID,get_value=lower_left_y
  info.roi_rectangle_info.lower_left_corner[1]=lower_left_y
  widget_control,info2.upper_right_xID,get_value=upper_right_x
  info.roi_rectangle_info.upper_right_corner[0]=upper_right_x
  widget_control,info2.upper_right_yID,get_value=upper_right_y
  info.roi_rectangle_info.upper_right_corner[1]=upper_right_y

  xplot2d_display_image,info2.wXplot2d

  Xll=(lower_left_x*info.scale)+info.image_position[0]+margin_space
  Yll=(lower_left_y*info.scale)+info.image_position[1]+margin_space
  Xur=(upper_right_x*info.scale)+info.image_position[0]+margin_space
  Yur=(upper_right_y*info.scale)+info.image_position[1]+margin_space

  Plots, [Xll,Xll,Xur,Xur], [Yll,Yur,Yur,Yll], color=2, /device
  plots, [Xll], [Yll], color=2, /device, /continue ;connect last point to the first

  pData=info.data->Value([info.image_num],/pointer)
  img = *pData
  datasize=size(img)
  info.roi_pixels=ptr_new(polyfillv([lower_left_x,lower_left_x,upper_right_x,upper_right_x]$
        ,[lower_left_y,upper_right_y,upper_right_y,lower_left_y],datasize[1],datasize[2]))

  widget_control,info2.wXplot2d,set_uvalue=info

endif
if (widget eq 'OK') then begin
 info.roi_centre_pixel_coord=(info.roi_rectangle_info.lower_left_corner+info.roi_rectangle_info.upper_right_corner)/2
 widget_control,info2.wXplot2d,set_uvalue=info
 widget_control, event.top, /destroy
endif

if (widget eq 'Cancel') then begin
 info.roi_rectangle_info=info2.previousParams ; restore previous values
 widget_control,info2.wXplot2d,set_uvalue=info
 widget_control, event.top, /destroy

endif

END

PRO XPLOT2D_ROI_RECTANGLE_EDIT,wXplot2d
widget_control,wXplot2d,get_uvalue=info
COMMON Xplot2D_common_variables, max_acceptable_screen_size_factor, scroll_step, margin_space, fileDir

    previousParams=info.roi_rectangle_info ; save previous rectangle parameters in case user cancels modifications
    Rectangle_info_request_dialog = Widget_Base(Column=1,Title='Rectangle Parameters', group_leader=wXplot2d, $
    /Modal, /Floating, /Base_Align_Center)
    Rectangle_info_base = Widget_Base(Rectangle_info_request_dialog, Column=1, Frame=1)

    labelID = widget_label(Rectangle_info_base,value='Enter the values relative to the dimensions of the original image')
    lower_left_xID = CW_Field(Rectangle_info_base, Title='lower left corner''s x coordinate:', Value=info.roi_rectangle_info.lower_left_corner[0], /floating, /return_events, uvalue='UPDATE')
    lower_left_yID = CW_Field(Rectangle_info_base, Title='lower left corner''s y coordinate:', Value=info.roi_rectangle_info.lower_left_corner[1], /floating, /return_events, uvalue='UPDATE')
    upper_right_xID = CW_Field(Rectangle_info_base, Title='upper right corner''s x coordinate:', Value=info.roi_rectangle_info.upper_right_corner[0], /floating, /return_events, uvalue='UPDATE')
    upper_right_yID = CW_Field(Rectangle_info_base, Title='upper right corner''s y coordinate:', Value=info.roi_rectangle_info.upper_right_corner[1], /floating, /return_events, uvalue='UPDATE')

    but_base = Widget_Base(Rectangle_info_request_dialog,Row=1)
    cancel = Widget_Button(but_base, Value='Cancel',uvalue='Cancel')
    accept = Widget_Button(but_base, Value='OK',uvalue='OK')
    centertlb, Rectangle_info_request_dialog
    Widget_Control, Rectangle_info_request_dialog, /Realize
    info2 = {wXplot2d:wXplot2d,lower_left_xID:lower_left_xID,lower_left_yID:lower_left_yID,$
                            upper_right_xID:upper_right_xID,upper_right_yID:upper_right_yID,$
                            previousParams:previousParams}
    Widget_Control,Rectangle_info_request_dialog,set_uvalue=info2
    xmanager, 'XPLOT2D_ROI_RECTANGLE_EDIT', Rectangle_info_request_dialog

END


;
;===============================================================================
;
PRO XPLOT2D_ROI_CIRCLE_MOUSE,wXplot2d
widget_control,wXplot2d,get_uvalue=info
COMMON Xplot2D_common_variables, max_acceptable_screen_size_factor, scroll_step, margin_space, fileDir

widget_control,info.wGraph,get_value=graphwin
wset, graphwin
XPLOT2D_SET_COMMENT,wXplot2d,'Click on the centre of the circle...'

CURSOR, Xcent, Ycent, /DOWN, /DEVICE
XPLOT2D_SET_COMMENT,wXplot2d,'Click again to determine the circle''s radius'
CURSOR, Xr, Yr, /DOWN, /DEVICE

;print, "circle dimensions", Xcent, Ycent, Xr, Yr
radius = SQRT((Yr-Ycent)^2+(Xr-Xcent)^2)

xplot2d_display_image,wXplot2d

;points =Circle(Xcent, Ycent, radius) ; calculate points for the circle scaled back to match the original image size
points=XPLOT2D_DRAW_CAKE(Xcent,Ycent,0.0,radius,0.0,2*!Pi)
Plots, points[0,*], points[1,*], color=2, /device
plots, points[0,0], points[1,0], color=2, /device, /continue ;connect last point to the first
;print, "points[0,*] : ",points[0,*],"; points[1,*]", points[1,*]

pData=info.data->Value([info.image_num],/pointer)
img = *pData
datasize=size(img)
;print, "image size", datasize
info.roi_pixels=ptr_new(polyfillv((points[0,*]-info.image_position[0]-margin_space)/info.Scale,$
                (points[1,*]-info.image_position[1]-margin_space)/info.Scale,datasize[1],datasize[2]))
info.roi_circle_info.centre_coord=[(Xcent-info.image_position[0]-margin_space)/info.Scale,$
                                   (Ycent-info.image_position[1]-margin_space)/info.Scale]
info.roi_centre_pixel_coord=info.roi_circle_info.centre_coord
info.roi_circle_info.phi1=0
info.roi_circle_info.phi2=2*!Pi
info.roi_circle_info.rho1=0
info.roi_circle_info.rho2=radius/info.Scale

widget_control,wXplot2d,set_uvalue=info
;print, "roi points stored", *info.roi_pixels

;x=*info.roi_pixels mod datasize[1]
;y=*info.roi_pixels / datasize[2]
;plots, x+margin_space,y+margin_space, /device

;xplot2d_integrate,wXplot2d

END

;
;===============================================================================
;
PRO XPLOT2D_ROI_CIRCLE_EDIT_EVENT, event
COMMON Xplot2D_common_variables, max_acceptable_screen_size_factor, scroll_step, margin_space, fileDir
widget_control,event.top,get_uvalue=info2
widget_control,info2.wXplot2d,get_uvalue=info
widget_control,event.id,get_uvalue=widget
if (widget eq 'UPDATE') or (widget eq 'OK') then begin

  widget_control,info2.centre_xID,get_value=centre_x
  info.roi_circle_info.centre_coord[0]=centre_x
  widget_control,info2.centre_yID,get_value=centre_y
  info.roi_circle_info.centre_coord[1]=centre_y
  widget_control,info2.radiusID,get_value=radius
  if radius lt 0.0 then begin
   blah=dialog_message('Positive values only!',/ERROR)
   return
  endif
  info.roi_circle_info.phi1=0.0
  info.roi_circle_info.phi2=2*!Pi
  info.roi_circle_info.rho1=0.0
  info.roi_circle_info.rho2=radius

  ;widget_control,info2.wXplot2d,set_uvalue=info
  ;xplot2d_integrate,info2.wXplot2d

  pData=info.data->Value([info.image_num],/pointer)
  img = *pData
  datasize=size(img)
  points=XPLOT2D_DRAW_CAKE(info.roi_circle_info.centre_coord[0]*info.Scale+info.image_position[0]+margin_space,$
      info.roi_circle_info.centre_coord[1]*info.Scale+info.image_position[1]+margin_space, $
      info.roi_circle_info.rho1*info.Scale,info.roi_circle_info.rho2*info.Scale,info.roi_circle_info.phi1,info.roi_circle_info.phi2)
  info.roi_pixels=ptr_new(polyfillv((points[0,*]-info.image_position[0]-margin_space)/info.Scale,$
                (points[1,*]-info.image_position[1]-margin_space)/info.Scale,datasize[1],datasize[2]))
  widget_control,info2.wXplot2d,set_uvalue=info
  XPLOT2D_DISPLAY_IMAGE,info2.wXplot2d
  Plots, points[0,*], points[1,*], color=2, /device
  plots, points[0,0], points[1,0], color=2, /device, /continue ;connect last point to the first

endif
if (widget eq 'OK') then begin
info.roi_centre_pixel_coord=info.roi_circle_info.centre_coord
widget_control,info2.wXplot2d,set_uvalue=info
 widget_control, event.top, /destroy
endif

if (widget eq 'Cancel') then begin
 info.roi_circle_info=info2.previousParams ; restore previous values
 widget_control,info2.wXplot2d,set_uvalue=info
 widget_control, event.top, /destroy

endif

END

PRO XPLOT2D_ROI_CIRCLE_EDIT,wXplot2d
widget_control,wXplot2d,get_uvalue=info
COMMON Xplot2D_common_variables, max_acceptable_screen_size_factor, scroll_step, margin_space, fileDir

    previousParams=info.roi_circle_info ; save previous circle parameters in case user cancels modifications
    Circle_info_request_dialog = Widget_Base(Column=1,Title='Circle Parameters', group_leader=wXplot2d, $
    /Modal, /Floating, /Base_Align_Center)
    circle_info_base = Widget_Base(Circle_info_request_dialog, Column=1, Frame=1)

    labelID = widget_label(circle_info_base,value='Enter the values relative to the dimensions of the original image')
    centre_xID = CW_Field(circle_info_base, Title='centre x coordinate:', Value=info.roi_circle_info.centre_coord[0], /floating, /return_events, uvalue='UPDATE')
    centre_yID = CW_Field(circle_info_base, Title='centre y coordinate:', Value=info.roi_circle_info.centre_coord[1], /floating, /return_events, uvalue='UPDATE')

    ;labelID = widget_label(circle_info_base,value='Enter the value for the radius')
    radiusID = CW_Field(circle_info_base, Title='radius:', Value=info.roi_circle_info.rho2-info.roi_circle_info.rho1, /floating, /return_events, uvalue='UPDATE')

    but_base = Widget_Base(Circle_info_request_dialog,Row=1)
    cancel = Widget_Button(but_base, Value='Cancel',uvalue='Cancel')
    accept = Widget_Button(but_base, Value='OK',uvalue='OK')
    centertlb, Circle_info_request_dialog
    Widget_Control, Circle_info_request_dialog, /Realize
    info2 = {wXplot2d:wXplot2d,centre_xID:centre_xID,centre_yID:centre_yID,$
                            radiusID:radiusID,previousParams:previousParams}
    Widget_Control,Circle_info_request_dialog,set_uvalue=info2
    xmanager, 'XPLOT2D_ROI_CIRCLE_EDIT', Circle_info_request_dialog

END

;
;===============================================================================
;
PRO XPLOT2D_ROI_CAKE_MOUSE,wXplot2d
widget_control,wXplot2d,get_uvalue=info
COMMON Xplot2D_common_variables, max_acceptable_screen_size_factor, scroll_step, margin_space, fileDir

widget_control,info.wGraph,get_value=graphwin
wset, graphwin
XPLOT2D_SET_COMMENT,wXplot2d,'Click on the centre of the cake...'

CURSOR, Xcent, Ycent, /DOWN, /DEVICE

XPLOT2D_SET_COMMENT,wXplot2d,'Click again to determine the cake''s first lower corner'
CURSOR, Xl, Yl, /DOWN, /DEVICE

XPLOT2D_SET_COMMENT,wXplot2d,'Click again to determine the cake''s second upper corner'
CURSOR, Xu, Yu, /DOWN, /DEVICE

;calculate the cake parameters given these values
rho1=sqrt(float(Ycent-Yl)^2+float(Xcent-Xl)^2)
rho2=sqrt(float(Ycent-Yu)^2+float(Xcent-Xu)^2)

if rho2 lt rho2 then begin
   blah=dialog_message(['rho2 lower than rho1!','Please make sure your third click is farther','from the centre (first click) than the second'],/ERROR)
   return
endif

Yl_diff=float(Yl-Ycent)
Xl_diff=float(Xl-Xcent)
Yu_diff=float(Yu-Ycent)
Xu_diff=float(Xu-Xcent)
;since the atan function's image values are all contained in the interval [-Pi/2;Pi/2]
;different cases need to be considered in order to allow cake ROI with wider angles
if (Yl_diff ge 0) and (Xl_diff ge 0) then phi1=Atan(Yl_diff/Xl_diff) $
else if (Yl_diff ge 0) and (Xl_diff lt 0) then phi1=!pi-Atan(Yl_diff/(-Xl_diff)) $
else if (Yl_diff lt 0) and (Xl_diff lt 0) then phi1=!pi+Atan((-Yl_diff)/(-Xl_diff)) $
else if (Yl_diff lt 0) and (Xl_diff ge 0) then phi1=2*!pi+Atan(Yl_diff/Xl_diff)

if (Yu_diff ge 0) and (Xu_diff ge 0) then phi2=Atan(Yu_diff/Xu_diff) $
else if (Yu_diff ge 0) and (Xu_diff lt 0) then phi2=!pi-Atan(Yu_diff/(-Xu_diff)) $
else if (Yu_diff lt 0) and (Xu_diff lt 0) then phi2=!pi+Atan((-Yu_diff)/(-Xu_diff)) $
else if (Yu_diff lt 0) and (Xu_diff ge 0) then phi2=2*!pi+Atan(Yu_diff/Xu_diff)

;phi2=Atan(float(Yu-Ycent)/float(Xu-Ycent))

xplot2d_display_image,wXplot2d

;print, "rho1 : ", rho1, "; rho2 : ", rho2, "; phi1 : ", phi1, "; phi2 : ", phi2

if phi1 eq phi2 then begin
   blah=dialog_message('Identical values for phi!',/ERROR)
   return
endif
points=XPLOT2D_DRAW_CAKE(Xcent,Ycent,rho1,rho2,phi1,phi2)
Plots, points[0,*], points[1,*], color=2, /device
plots, points[0,0], points[1,0], color=2, /device, /continue ;connect last point to the first

pData=info.data->Value([info.image_num],/pointer)
img = *pData
datasize=size(img)

info.roi_pixels=ptr_new(polyfillv((points[0,*]-info.image_position[0]-margin_space)/info.Scale,$
                (points[1,*]-info.image_position[1]-margin_space)/info.Scale,datasize[1],datasize[2]))
info.roi_circle_info.centre_coord=[(Xcent-info.image_position[0]-margin_space)/info.Scale,$
                                   (Ycent-info.image_position[1]-margin_space)/info.Scale]
info.roi_centre_pixel_coord=info.roi_circle_info.centre_coord

info.roi_circle_info.phi1=phi1
info.roi_circle_info.phi2=phi2
info.roi_circle_info.rho1=rho1/info.scale
info.roi_circle_info.rho2=rho2/info.scale

widget_control,wXplot2d,set_uvalue=info

END

;
;===============================================================================
;
PRO XPLOT2D_ROI_CAKE_EDIT_EVENT, event
COMMON Xplot2D_common_variables, max_acceptable_screen_size_factor, scroll_step, margin_space, fileDir
widget_control,event.top,get_uvalue=info2
widget_control,info2.wXplot2d,get_uvalue=info
widget_control,event.id,get_uvalue=widget
if (widget eq 'UPDATE') or (widget eq 'OK') then begin

  widget_control,info2.centre_xID,get_value=centre_x
  info.roi_circle_info.centre_coord[0]=centre_x
  widget_control,info2.centre_yID,get_value=centre_y
  info.roi_circle_info.centre_coord[1]=centre_y
  widget_control,info2.rho1ID,get_value=rho1
  if rho1 lt 0.0 then begin
   blah=dialog_message('Positive values only!',/ERROR)
   return
  endif
  info.roi_circle_info.rho1=rho1
  widget_control,info2.rho2ID,get_value=rho2
  if rho2 lt 0.0 then begin
   blah=dialog_message('Positive values only!',/ERROR)
   return
  endif
  if rho2 le rho1 then begin
   blah=dialog_message('rho2 has to be greater than rho1!',/ERROR)
   return
  endif
  info.roi_circle_info.rho2=rho2
  widget_control,info2.phi1ID,get_value=phi1
  if (phi1 lt 0) or (phi1 gt 360) then begin
   blah=dialog_message('Values between 0 and 360 accepted only!',/ERROR)
   return
  endif
  phi1=phi1*2*!pi/360 ; convert to radians
  info.roi_circle_info.phi1=phi1
  widget_control,info2.phi2ID,get_value=phi2
  if (phi2 lt 0) or (phi2 gt 360) then begin
   blah=dialog_message('Values between 0 and 360 accepted only!',/ERROR)
   return
  endif
  phi2=phi2*2*!pi/360 ;
  info.roi_circle_info.phi2=phi2
  pData=info.data->Value([info.image_num],/pointer)
  img = *pData
  datasize=size(img)

  points=XPLOT2D_DRAW_CAKE(info.roi_circle_info.centre_coord[0]*info.Scale+info.image_position[0]+margin_space,$
      info.roi_circle_info.centre_coord[1]*info.Scale+info.image_position[1]+margin_space, $
      info.roi_circle_info.rho1*info.Scale,info.roi_circle_info.rho2*info.Scale,info.roi_circle_info.phi1,info.roi_circle_info.phi2)
  info.roi_pixels=ptr_new(polyfillv((points[0,*]-info.image_position[0]-margin_space)/info.Scale,$
                (points[1,*]-info.image_position[1]-margin_space)/info.Scale,datasize[1],datasize[2]))

  ;points=XPLOT2D_DRAW_CAKE(info.roi_circle_info.centre_coord[0]+margin_space,info.roi_circle_info.centre_coord[1]+margin_space, $
  ;info.roi_circle_info.rho1,info.roi_circle_info.rho2,info.roi_circle_info.phi1,info.roi_circle_info.phi2)
  ;info.roi_pixels=ptr_new(polyfillv(points[0,*]-margin_space,points[1,*]-margin_space,datasize[1],datasize[2]))
  widget_control,info2.wXplot2d,set_uvalue=info
  ;xplot2d_integrate,info2.wXplot2d
  XPLOT2D_DISPLAY_IMAGE,info2.wXplot2d
  Plots, points[0,*], points[1,*], color=2, /device
  plots, points[0,0], points[1,0], color=2, /device, /continue ;connect last point to the first

endif
if (widget eq 'OK') then begin
 info.roi_centre_pixel_coord=info.roi_circle_info.centre_coord
 widget_control,info2.wXplot2d,set_uvalue=info
 widget_control, event.top, /destroy

endif

if (widget eq 'Cancel') then begin
 info.roi_circle_info=info2.previousParams ; restore previous values
 widget_control,info2.wXplot2d,set_uvalue=info
 widget_control, event.top, /destroy

endif

END


PRO XPLOT2D_ROI_CAKE_EDIT,wXplot2d
widget_control,wXplot2d,get_uvalue=info
COMMON Xplot2D_common_variables, max_acceptable_screen_size_factor, scroll_step, margin_space, fileDir

    previousParams=info.roi_circle_info ; save previous circle parameters in case user cancels modifications
    Cake_info_request_dialog = Widget_Base(Column=1,Title='Cake Parameters', group_leader=wXplot2d, $
    /Modal, /Floating, /Base_Align_Center)
    cake_info_base = Widget_Base(Cake_info_request_dialog, Column=1, Frame=1)

    labelID = widget_label(cake_info_base,value='Enter the values relative to the dimensions of the original image')
    centre_xID = CW_Field(cake_info_base, Title='centre x coordinate:', Value=info.roi_circle_info.centre_coord[0], /floating, /return_events, uvalue='UPDATE')
    centre_yID = CW_Field(cake_info_base, Title='centre y coordinate:', Value=info.roi_circle_info.centre_coord[1], /floating, /return_events, uvalue='UPDATE')

    ;labelID = widget_label(cake_info_base,value='Enter the values for rho and phi')
    rho1ID = CW_Field(cake_info_base, Title='rho1:', Value=info.roi_circle_info.rho1, /floating, /return_events, uvalue='UPDATE')

    rho2ID = CW_Field(cake_info_base, Title='rho2:', Value=info.roi_circle_info.rho2, /floating, /return_events, uvalue='UPDATE')

    phi1ID = CW_Field(cake_info_base, Title='phi1 (in degrees):', Value=info.roi_circle_info.phi1*180/!pi, /floating, /return_events, uvalue='UPDATE')
    phi2ID = CW_Field(cake_info_base, Title='phi2 (in degrees):', Value=info.roi_circle_info.phi2*180/!pi, /floating, /return_events, uvalue='UPDATE')

    but_base = Widget_Base(Cake_info_request_dialog,Row=1)
    cancel = Widget_Button(but_base, Value='Cancel',uvalue='Cancel')
    accept = Widget_Button(but_base, Value='OK',uvalue='OK')
    centertlb, Cake_info_request_dialog
    Widget_Control, Cake_info_request_dialog, /Realize
    info2 = {wXplot2d:wXplot2d,centre_xID:centre_xID,centre_yID:centre_yID,$
              rho1ID:rho1ID,rho2ID:rho2ID,phi1ID:phi1ID,phi2ID:phi2ID,previousParams:previousParams}
    Widget_Control,Cake_info_request_dialog,set_uvalue=info2
    xmanager, 'XPLOT2D_ROI_CAKE_EDIT', Cake_info_request_dialog

END

;
;===============================================================================
;
PRO XPLOT2D_AXIS_PROPERTIES_EVENT, event
widget_control,event.top,get_uvalue=info2
widget_control,info2.wXplot2d,get_uvalue=info
widget_control,event.id,get_uvalue=widget
if (widget eq 'UPDATE') or (widget eq 'OK') then begin

  widget_control,info2.x_originID,get_value=x_origin
  info.axis_properties.x_origin=x_origin
  widget_control,info2.y_originID,get_value=y_origin
  info.axis_properties.y_origin=y_origin
  widget_control,info2.x_pixel_unit_ratioID,get_value=x_pixel_unit_ratio
  info.axis_properties.x_pixel_unit_ratio=x_pixel_unit_ratio
  widget_control,info2.y_pixel_unit_ratioID,get_value=y_pixel_unit_ratio
  info.axis_properties.y_pixel_unit_ratio=y_pixel_unit_ratio

  pData=info.data->Value([info.image_num],/pointer)
  img = *pData
  datasize=size(img)

  widget_control,info2.wXplot2d,set_uvalue=info

endif
if (widget eq 'OK') then begin
 widget_control, event.top, /destroy

endif

if (widget eq 'Cancel') then begin
 info.axis_properties=info2.previousParams ; restore previous values
 widget_control,info2.wXplot2d,set_uvalue=info
 widget_control, event.top, /destroy

endif
xplot2d_display_image,info2.wXplot2d
END

;
;===============================================================================
;
PRO XPLOT2D_AXIS_PROPERTIES,wXplot2d

widget_control,wXplot2d,get_uvalue=info

previousParams=info.axis_properties ; save previous circle parameters in case user cancels modifications

axis_info_request_dialog = Widget_Base(Column=1,Title='Axis Properties', group_leader=wXplot2d, $
    /Modal, /Floating, /Base_Align_Center)
axis_info_base = Widget_Base(axis_info_request_dialog, Column=1, Frame=1)

x_originID = CW_Field(axis_info_base, Title='x origin value:', Value=info.axis_properties.x_origin, /floating, /return_events, uvalue='UPDATE')
x_pixel_unit_ratioID = CW_Field(axis_info_base, Title='x pixel/unit ratio:', Value=info.axis_properties.x_pixel_unit_ratio, /floating, /return_events, uvalue='UPDATE')
y_originID = CW_Field(axis_info_base, Title='y origin value:', Value=info.axis_properties.y_origin, /floating, /return_events, uvalue='UPDATE')
y_pixel_unit_ratioID = CW_Field(axis_info_base, Title='y pixel/unit ratio:', Value=info.axis_properties.y_pixel_unit_ratio, /floating, /return_events, uvalue='UPDATE')

but_base = Widget_Base(axis_info_request_dialog,Row=1)
cancel = Widget_Button(but_base, Value='Cancel',uvalue='Cancel')
accept = Widget_Button(but_base, Value='OK',uvalue='OK')
centertlb, axis_info_request_dialog
    Widget_Control, axis_info_request_dialog, /Realize
    info2 = {wXplot2d:wXplot2d,x_originID:x_originID,y_originID:y_originID,$
             x_pixel_unit_ratioID:x_pixel_unit_ratioID,y_pixel_unit_ratioID:y_pixel_unit_ratioID,previousParams:previousParams}
    Widget_Control,axis_info_request_dialog,set_uvalue=info2
    xmanager, 'XPLOT2D_AXIS_PROPERTIES', axis_info_request_dialog

END
;
;===============================================================================
;
PRO XPLOT2D_EVENT,event

   COMMON Xplot2D_common_variables, max_acceptable_screen_size_factor, scroll_step, margin_space, fileDir

;   catch, error_status
;   if error_status ne 0 then begin
;      message,/info,'error caught: '+!err_string
;      if sdep(/w) then itmp = Dialog_Message(/Error,$
;           'XPLOT2D_EVENT: error caught: '+!err_string)
;      catch, /cancel
;      on_error,2
;      RETURN
;   endif


   IF Tag_Names(event,/Structure_Name) EQ 'WIDGET_KILL_REQUEST' THEN BEGIN
     xplot2d_quit,event.top
     RETURN
   ENDIF


   ;
   ; restore the info structure
   ;
   widget_control,event.top,get_uvalue=info

   ;
   ; Define some actions from widgets with no UValue
   ;
   if tag_names(event,/structure_name) EQ 'WIDGET_BASE' then begin
      action = 'RESIZE'
   endif else begin
      if tag_names(event,/structure_name) EQ 'WIDGET_DRAW' then begin
         action = 'MOUSE'
      endif else begin
			junk=where(tag_names(event) EQ 'VALUE',ct)
			if ct eq 0 then $
				Widget_Control, event.id, get_UValue=action $
			else action=event.value
      endelse
   endelse

   ;widget_control,event.id,get_uvalue=widget
   ;print, "widget : ", widget
   ;
   ; Actions
   ;

   ;
   ;change image to be displayed
   ;
   if event.id eq info.wImageNum then begin
	XPLOT2D_CHANGE_IMAGE,event.top,event.value-1
	return
   endif

   ; debugging messages
   ;print,'Action:','**'+string(action)+'**'
   ;print, "scroll_step : ",scroll_step

   CASE action of
      'RESIZE': begin
      ; widget_control,info.wGraph,draw_xsize=event.x,draw_ysize=event.y
      end
      ;
      ; Buttons and others
      ;
      'MOUSE':begin
         ;if event.press ne 0 then print,event
       pData=info.data->Value([info.image_num],/pointer)
	   img = *pData
	   datasize=size(img)
	   xVal=(event.x-margin_space-info.image_position[0])/info.scale
	   yVal=(event.y-margin_space-info.image_position[1])/info.scale
       Widget_Control,info.wXCoordBox,set_value=xVal
       Widget_Control,info.wYCoordBox,set_value=yVal
       if (xVal ge 0) and (yVal ge 0) and (xVal lt dataSize(1)) and (yVal lt dataSize(2)) then $ ;mouse within image bounds
           Widget_Control,info.wIntensityValueBox,set_value=float(img[xVal,yVal]) $
       else Widget_Control,info.wIntensityValueBox,set_value=-1
       widget_control,event.top,set_uvalue=info

	 END
      'FIRST IMAGE':   XPLOT2D_FIRST_IMAGE,event.top
      'LAST IMAGE':    XPLOT2D_LAST_IMAGE,event.top
      'NEXT IMAGE':    XPLOT2D_NEXT_IMAGE,event.top
      'PREVIOUS IMAGE':XPLOT2D_PREVIOUS_IMAGE,event.top
      'DELETE IMAGE': BEGIN
         numberOfImages=info.data->INFO(/N_ELEMENTS)

	 IF numberOfImages EQ 1 THEN BEGIN
           itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
            'At least one image must exist. Aborted')
	   Return
	 ENDIF


         itmp = Dialog_Message(/Info,Dialog_Parent=event.top, $
            'Delete current image (cannot undo)? Please confirm',/Cancel)
         IF itmp EQ 'Cancel' THEN RETURN

	 current_index = info.image_num
         title=(info.data)->value([current_index],/title)
         (info.data)->remove,[current_index]

	 ; display new image
	 Widget_Control,info.wimageNum,Get_Value=imgnum
         numberOfImages=info.data->INFO(/N_ELEMENTS)
         imgnum = imgnum < numberOfImages  ; number 1,2..

	 info.image_num=imgnum-1           ; index 0,1,...
	 Widget_Control,event.top,Set_UValue=info
	 ; refresh
         Widget_Control,/HourGlass
         xplot2d_display_image,event.top
	 XPLOT2D_SET_COMMENT,event.top,'Image removed: '+title

	END



      'DELETE ALL': BEGIN
         numberOfImages=info.data->INFO(/N_ELEMENTS)

	 IF numberOfImages EQ 1 THEN BEGIN
           itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
            'At least one image must exist. Aborted')
	   Return
	 ENDIF


         itmp = Dialog_Message(/Info,Dialog_Parent=event.top, $
            'Delete ALL images (cannot undo)? Please confirm',/Cancel)
         IF itmp EQ 'Cancel' THEN RETURN

	 current_index = info.image_num
         indices = (indgen(numberOfImages))[1:numberOfImages-1]
         (info.data)->remove,indices

	 ; display new image
         numberOfImages=info.data->INFO(/N_ELEMENTS)

	 info.image_num=0           ; index 0,1,...
	 Widget_Control,event.top,Set_UValue=info
	 ; refresh
         Widget_Control,/HourGlass
         xplot2d_display_image,event.top
	 XPLOT2D_SET_COMMENT,event.top,'All images (except the first which must always exist) removed.'

	END


      ;Scroll image events
      'SCROLL_LEFT': XPLOT2D_SCROLL_IMAGE,event.top,-scroll_step,0
      'SCROLL_RIGHT': XPLOT2D_SCROLL_IMAGE,event.top,scroll_step,0
      'SCROLL_UP': XPLOT2D_SCROLL_IMAGE,event.top,0,scroll_step
      'SCROLL_DOWN':XPLOT2D_SCROLL_IMAGE,event.top,0,-scroll_step
      'SCROLL_STEP': BEGIN
	widget_control,event.id,get_value=val,get_uvalue=uval
	scroll_step=Fix(val[0])
        XPLOT2D_SCROLL_IMAGE,event.top,0,0
	END

      ;
      ; File menu
      ;
      'LOAD IMAGE':BEGIN
   		; restore the widget value
   		Widget_Control,event.id,Get_Value=wValue
		fileFormat=info.imageFormat.format[1+Fix(info.imageFormat.format[0])]
		filter=info.imageFormat.filter
		convertToFloat = Fix(info.imageFormat.float[0])
		substitute = Fix(info.imageFormat.substitute[0])

		CASE fileFormat OF
		  'EDF': BEGIN
			file = dialog_Pickfile(/Must_Exist,filter=filter,path=filedir,get_path=filedir1, $
				/Multiple_Files,Dialog_Parent=event.top)
			Widget_Control,/HourGlass
			IF file[0] EQ '' THEN RETURN
			fileDir=fileDir1
			FOR i=0,N_Elements(file)-1 DO BEGIN
			  img = read_edf(file[i])
			  IF convertToFloat EQ 1 THEN img = float(img)
			  XPLOT2D_ADD_IMAGE,event.top,img,title=file[i],substitute=substitute
			  Wait,info.imageFormat.wait
			END
			END
		  'XRD': BEGIN 
			file = dialog_Pickfile(/Must_Exist,filter=filter,path=filedir,get_path=filedir1, $
				/Multiple_Files,Dialog_Parent=event.top)
			Widget_Control,/HourGlass
			IF file[0] EQ '' THEN RETURN
			fileDir=fileDir1
			FOR i=0,N_Elements(file)-1 DO BEGIN
			  restore,file=file[i],/verbose
			  IF N_Elements(xplot2d_image) EQ 0 THEN BEGIN
			    itmp = Dialog_Message('xplot2d_image not restored. Aborted')
			    Return
			  ENDIF
			  IF convertToFloat EQ 1 THEN xplot2d_image = float(xplot2d_image)
			  XPLOT2D_ADD_IMAGE,event.top,xplot2d_image,title=file[i]
			  Wait,info.imageFormat.wait
			ENDFOR
			END
		  'MAR': BEGIN 
			file = dialog_Pickfile(/Must_Exist,filter=filter,path=filedir,get_path=filedir1, $
				/Multiple_Files,Dialog_Parent=event.top)
			Widget_Control,/HourGlass
			IF file[0] EQ '' THEN RETURN
			fileDir=fileDir1
			FOR i=0,N_Elements(file)-1 DO BEGIN
			  im = read_mar(file[i]) 
			  print,'Extrema of the image: '+file[i],min(im), max(im)
			  IF convertToFloat EQ 1 THEN im = float(im)
			  XPLOT2D_ADD_IMAGE,event.top,im,title=file[i],substitute=substitute ; modifies info
			  Wait,info.imageFormat.wait
			ENDFOR
			END
		  'BINARY': BEGIN
			load_bin=info.load_bin
			Titles=['File (? for launching browser)','Header [bytes]', 'X size [pixels]',$
			  'Y size [pixels]','Data type']
			XScrMenu,load_bin,Action=act,Titles=titles, /NoType,/Interp, $
			WTitle='Load binary file',dialog_Parent=event.top,FieldLen=40
			IF act EQ 'DONT' THEN RETURN
			IF StrCompress(load_bin.file,/Rem) EQ '?' OR CheckFile(load_bin.file) EQ 0 THEN BEGIN
			  file = dialog_Pickfile(/Must_Exist,filter=filter,path=filedir,get_path=filedir1)
			  IF file EQ '' THEN RETURN
			  fileDir=fileDir1
			  load_bin.file=file
			ENDIF
			Widget_Control,/HourGlass
			;
			; get image
			;
			CASE Fix((load_bin.type)[0])  OF
			  0:	;Undefined	
			  1:	b=BytArr(load_bin.xsize,load_bin.ysize) ; Byte
			  2:	b=IntArr(load_bin.xsize,load_bin.ysize) ;Integer	
			  3:	b=LonArr(load_bin.xsize,load_bin.ysize) ;Longword integer	
			  4:	b=FltArr(load_bin.xsize,load_bin.ysize) ;Floating point	
			  5:	b=DblArr(load_bin.xsize,load_bin.ysize) ;Double-precision floating
			  6:	b=ComplexArr(load_bin.xsize,load_bin.ysize) ;Complex floating	
			  7:	;String	
			  8:	;Structure	
			  9:	b=DComplexBytArr(load_bin.xsize,load_bin.ysize) ;Double-precision complex	
			  10:	;Pointer	
			  11:	;Object reference	
			  12:	b=UIntArr(load_bin.xsize,load_bin.ysize) ;Unsigned Integer	
			  13:	b=ULonArr(load_bin.xsize,load_bin.ysize) ;Unsigned Longword Integer	
			  14:	b=Lon64Arr(load_bin.xsize,load_bin.ysize) ;64-bit Integer	
			  15:	b=ULon64Arr(load_bin.xsize,load_bin.ysize) ;Unsigned 64-bit Integer	
			  else: Message,'Data type not implemented.'
			ENDCASE

			openr,unit,load_bin.file,/get_lun
			  a=bytarr(load_bin.header)
			  readu,unit,a
			  readu,unit,b
			close,unit
			;
			; store image
			;
			IF convertToFloat EQ 1 THEN b = float(b)
			XPLOT2D_ADD_IMAGE,event.top,b,title=file,substitute=substitute
			END
		  'PNG/TIFF/etc': BEGIN
			itmp = dialog_read_Image(dialog_parent=event.top, Image=img, $
			  Query=q)
			IF itmp EQ 0 THEN RETURN
			IF type(img) EQ 1 THEN img=Fix(img)
			IF convertToFloat EQ 1 THEN b = float(b)
			XPLOT2D_ADD_IMAGE,event.top,img,substitute=substitute
			pData=info.data->Value([info.image_num],/pointer)
			print,min(*pData),max(*pData),total(*pData)
			END
		  else: Message,'Case option not found: '+action
		ENDCASE
		END
      'SET LOAD PROPERTIES':BEGIN
		tmp = info.imageFormat
		XScrMenu,tmp,Action=act,Titles= $
		  ['Format','Filter','Convert to float [needed for masks and operations]','Store image', $
			'Wait time [sec]'], /NoType,/Interp, WTitle='File format',dialog_Parent=event.top
		IF act EQ 'DONT' THEN RETURN
		info.imageFormat = tmp
		Widget_Control,info.wLoadImage,Set_Value='Load Image '+$
			tmp.format[1+Fix(tmp.format[0])]
	 	Widget_Control,event.top, Set_UValue=info ; register the changes
		END
      'SAVE IMAGE':BEGIN
   		; restore the widget value
   		Widget_Control,event.id,Get_Value=wValue

		CASE wValue OF
		  'EDF...': BEGIN
			filename=''
			filename=dialog_Pickfile(filter = '*.edf')
			IF filename EQ '' THEN RETURN
			pData=info.data->Value([info.image_num],/pointer)
			Print,Min(*pData),Max(*pData),Total(*pData)
			Write_edf,*pData,file=filename
			END
		  'XRD...': BEGIN
			filename=''
			filename=dialog_Pickfile(filter = '*.xrd', File='xplot2d.xrd')
			IF filename EQ '' THEN RETURN
			pData=info.data->Value([info.image_num],/pointer)
			Print,Min(*pData),Max(*pData),Total(*pData)
			xplot2d_image = *pData
			Save,xplot2d_image,file=filename
			END
		  'Other...': BEGIN
			pData=info.data->Value([info.image_num],/pointer)
			Print,Min(*pData),Max(*pData),Total(*pData)
			itmp = dialog_Write_Image(*pData,dialog_Parent=event.top,/Warn_Exist)
			END
;		  '**Selection...': BEGIN
;   			XPLOT2D_SET_COMMENT,event.top,'Click to create bounding box'
;			widget_control,info.wGraph,get_value=graphwin
;   			wset,graphwin
;			cursor,xmin,ymin,/device,/down
;			nx=40	; default box size
;			ny=40
;   			XPLOT2D_SET_COMMENT,event.top,'left to move, middle to extend, right to finish'
;   			box_cursor,xmin,ymin,nx,ny,/init
;			xmin=xmin>0
;			ymin=ymin>0
;			tmp=tvrd(xmin,ymin,nx,ny)
;			itmp = dialog_Write_Image(tmp,dialog_Parent=event.top,/Warn_Exist)
;			END
		  else: Message,'Case option not found: '+wValue
		ENDCASE
		END
;      'PRINT':BEGIN
;	 printbuffer='xplot2d_display_image,'+StrCompress(event.top,/Rem)+'L,/print'
;	 print,printbuffer
;         xprint,BUFFER=printbuffer,GROUP=event.top,/COLOR
;      	END
      'EXIT':BEGIN
         ;obj_destroy,info.data
         ;widget_control,event.top,/destroy
         ;While !D.Window NE -1 DO WDelete, !D.Window ;delete all open windows
         ;Widget_Control, /Reset
         Xplot2d_Quit, event.top
      	END

      ;
      ; Edit menu
      ;

      'UNDO' : begin
        if (info.data->Value([info.image_num],/isauxdef)) eq 1 then begin
          img=info.data->Value([info.image_num],/Aux)
          info.data->Set,[info.image_num],Value=img
          xplot2d_display_image,event.top
        endif else $
          blah=dialog_message('No modifications to be undone!',/ERROR)
      end
      'EDIT IMAGE TITLES' : BEGIN
	      txt_old = info.data->value(/title)
              txt_new = txt_old


              action = ''
              XDisplayFile1,Text=txt_new,Title='Edit image labels', $
                Group=event.top,/Modal,Action=action, $
                Dialog_Parent=event.top
              IF action EQ 'DONT' THEN RETURN
              IF N_Elements(txt_new) NE N_Elements(txt_old) THEN BEGIN
                itmp = Dialog_Message(/Error, Dialog_Parent=event.top, $
                 'Number of label lines has changed. Aborted')
                RETURN
              ENDIF
              (info.data)->set,Title=txt_new
	      Widget_Control,/HourGlass
              xplot2d_display_image,event.top
	END
      'AXIS PROP' : xplot2d_axis_properties, event.top

      ;
      ; View menu
      ;
      'REFRESH': BEGIN
	 Widget_Control,/HourGlass
         xplot2d_display_image,event.top
	 END
      'VIEW INFO': BEGIN
	 txt = info.data->info()
	 XDisplayFile1,text=txt,Title='Info on stored images'
	 END
      'VIEW SCALE': BEGIN
	 sValue = info.scale
	 xEdit,sValue,Text='Scale factor: ',Title='Scale',Action=act	, $
	   dialog_Parent=event.top,InfoText='Use View->Scroll Palette if image is larger than window'
	 IF act EQ 'CANCEL' THEN RETURN
	 info.scale=sValue
	 Widget_Control,event.top, Set_UValue=info ; register the changes
	 Widget_Control,/HourGlass
         xplot2d_display_image,event.top
	 END
      'VIEW SCROLL PALETTE': BEGIN
	 IF Widget_Info(info.wScroll,/Valid_ID) EQ 0 THEN BEGIN
	   xplot2d_scroll_palette, event.top, wScroll, no_block=info.no_block
	   info.wScroll=wScroll
	 ENDIF
         widget_control,event.top,set_uvalue=info
	 END

      'CLR FULLRANGE':    xplot2d_setcolorlimits,event.top,/fullrange,/refresh
      'CLR STRETCH':      xplot2d_setcolorlimits,event.top,/stretch,/refresh
      'CLR STRETCHVALUE': xplot2d_setcolorlimits,event.top,/stretch,/changeStretch,/refresh
      'CLR MINMAX':       xplot2d_setcolorlimits,event.top,/interactive,/refresh


      'LOGLINEARSCALE':begin
         if info.ScaleType eq 'log' then $
            info.ScaleType='linear' $
         else info.ScaleType='log'
         widget_control,event.top,set_uvalue=info
         xplot2d_display_image,event.top
         END
      'XLOADCT': BEGIN
	 xloadct, bottom=32,/Modal,Group=event.top
	 Widget_Control,/HourGlass
         xplot2d_display_image,event.top
	 END
      'SHOW COLOR TABLE': BEGIN
	 xplot2d_displayct,event.top,minrange=info.min,maxrange=info.max
	 END
;      'DEFAULT_COLOR_TABLE': BEGIN
;	 XOP_LOAD_IMAGE,/Set_Default
;	 Widget_Control,/HourGlass
;         xplot2d_display_image,event.top
;	 END
      'CALC XYPROFILE':begin

   	 XPLOT2D_SET_COMMENT,event.top,'Left button toggles Horiz/Vert profile. Right button exits'
	 pData=info.data->Value([info.image_num],/pointer)
	 img = *pData
	 IF info.scale NE 1 THEN BEGIN
	   sf = Size(img)
	   img = ConGrid(img,info.scale*sf[1],info.scale*sf[2])
	 ENDIF
	 profiles,img,SX=margin_space,SY=margin_space
   	 XPLOT2D_SET_COMMENT,event.top,'Done profiles.'
	 END

	  ;
	  ; ROI items
      ;

	  'ROI_CIRCLE':begin
	  Widget_Control,event.id,Get_Value=wValue

	  CASE wValue OF
		  'Mouse': xplot2d_roi_circle_mouse,event.top
          'Edit Parameters': xplot2d_roi_circle_edit,event.top

	  ENDCASE
	  END

	  'ROI_CAKE':begin
	  Widget_Control,event.id,Get_Value=wValue

	  CASE wValue OF
		  'Mouse': xplot2d_roi_cake_mouse,event.top
          'Edit Parameters': xplot2d_roi_cake_edit,event.top

	  ENDCASE
	  END

      'ROI_RECTANGLE':begin
	  Widget_Control,event.id,Get_Value=wValue

	  CASE wValue OF
		  'Mouse': xplot2d_roi_rectangle_mouse,event.top
          'Edit Parameters': xplot2d_roi_rectangle_edit,event.top

	  ENDCASE
	  END

      'ROI_POLYGON': xplot2d_roi_polygon_mouse,event.top

      'MASK' : XPLOT2D_APPLY_SELECTION_TO_MASK,event.top
      'MASK2' : XPLOT2D_APPLY_SELECTION_TO_MASK,event.top,/complementary
      'MASKFROMIMAGE' : BEGIN
          pData=info.data->Value([info.image_num],/pointer)
          img = *pData
          datasize=size(img)
          num=info.image_mask
          xedit,num,Text='Image # (i.e., index+1) from where mask is read: ',$
		Title='Set mask', Action=action, Dialog_Parent=event.top

          IF action EQ 'CANCEL' THEN Return
	  info.image_mask=num

          pMask=info.data->Value([num-1],/pointer)
          mask = *pMask
          masksize=size(mask)

          IF total(masksize) NE total(datasize) THEN BEGIN
            blah=Dialog_message( $
		['Image and mask have incompatible dimensions'],/Error,$
          	Dialog_Parent=event.top)
            Return
          ENDIF

          mask_pixels = Where(mask LT 0)

          if (mask_pixels)[0] gt -1 then begin 
            ; perform calc only if region is not empty
            img_old=img
            img[mask_pixels]=mask[mask_pixels]
            *pData=img
            ;write back image data to info structure
            info.data->set,[info.image_num],value=img,aux=img_old
            widget_control,event.top,set_uvalue=info
            XPLOT2D_DISPLAY_IMAGE,event.top
          endif else begin
            blah=dialog_message(['No masked pixels found!'],/ERROR,$
          	Dialog_Parent=event.top)
          endelse
	  END

      ;
      ; CALCULATIONS menu
      ;
      'CALC ROT': BEGIN
	; restore the widget value
   	Widget_Control,event.id,Get_Value=wValue
	CASE wValue OF
	  'Rotate 90 Clockwise':xplot2d_rotate_image,event.top,3
	  'Rotate 90 CounterClockwise':xplot2d_rotate_image,event.top,1
	  'Rotate 180':xplot2d_rotate_image,event.top,2
	  'Mirror up/down':xplot2d_rotate_image,event.top,7
	  'Mirror left/right':xplot2d_rotate_image,event.top,5
	  else: Message,'CASE action not found: '+wValue
 	ENDCASE
	END

      'CALC INTEGRATE': xplot2d_azimuth_integral,event.top

      'CALC INTEGRATE_MACRO': BEGIN
	    CASE !version.os OF
	       'sunos': bindir='sun'
		else:  bindir=!version.os
	    ENDCASE
	    path = '~boesecke/saxs/bin-'+bindir+'/'


	    title=info.data->Value(info.image_num,/title)
	    txt = [$
	    "; ",$
	    "; xop macro for azimuthal integration --", $
	    "; ",$
	    "spawn,'rm output.edf'", $
	    "spawn,'"+path+"saxs_angle input.edf output.edf 1 1 1 "+StrCompress(info.center[0],/Rem)+" "+StrCompress(info.center[1],/Rem)+" -1'", $
	    "img = read_edf('output.edf')", $
	    "tmp = where(img LT 0) ; masked indices", $
	    " ",$
	    "IF tmp[0] NE -1 THEN BEGIN ; there are masked values ",$
	    "  img2 = img*0 ",$
	    "  img2[ where(img GE 0) ] = 1  ",$
	    "  ht = total(img2,2) ",$
	    "  img[tmp] = 0 ",$
	    "ENDIF ELSE BEGIN ; not masked values ", $
	    "  ht=total(img*0+1,2) ", $
	    "ENDELSE", $
	    " ", $
	    "h = total(img,2)/ht", $
	    "hx = findgen(n_elements(h))", $
	    "; xplot2d,img ; display polar image", $
;	    "xplot,hx,h,/no_block,Window_Size=[600,400]", $
	    "  ",$
	    "; ", $
	    "; express profile in twotheta and angle units  ", $
	    "; ", $
	    " ", $
	    "dist = "+StrCompress(info.detector_dist), $
	    "pixelsize = "+StrCompress(info.pixelsize), $
	    "e1 = "+StrCompress(info.e1), $
	    "twotheta = 180./!pi*atan(hx*pixelsize*1e-4/dist)", $
	    "dspacing = physical_constants('hc')/2.0/e1/sin(twotheta/2.*!pi/180.)", $
	    "title = '"+title+"'", $
	    " ", $
	    "tmp = make_set(hx,twotheta ,dspacing,h)", $
	    " ", $
	    "xplot,tmp,/No_Block,Window_Size=[600,400],xcol=1,coltitles=['pixel','2theta[deg]','dspacing','intensity'],Title=title"]

	  img = info.data->Value(info.image_num)
	  write_edf,img,File='input.edf'
	  xop_macro,txt
	    
	  END

      'CALC PROFILE':begin
   	XPLOT2D_SET_COMMENT,event.top,'Click on the two extreme points'
	lineData=xplot2d_get_line(event.top)
	xplot,indgen(n_elements(lineData)),lineData,/No_Block,Window_Size=[600,400]
	END
      'CALC HISTOGRAM':begin
	img = info.data->Value(info.image_num)
	tmp = where(img LT 0)
	IF tmp[0] EQ -1 THEN BEGIN
	  Print,'Masked values: None' 
	  data=histogram(img)
	ENDIF ELSE BEGIN
	  Print,'Masked values: ',N_Elements(tmp),' over',N_Elements(img)
	  tmp=img[where(img GE 0)]
	  data=histogram(tmp)
	ENDELSE
	xplot,lindgen(n_elements(data)),data,/No_Block,Window_Size=[600,400]
	END


      'CALC CDF(HISTOGRAM)':begin
	img = info.data->Value(info.image_num)
	tmp = where(img LT 0)
	IF tmp[0] EQ -1 THEN BEGIN
	  Print,'Masked values: None' 
	  data=hist_equal(img,/hist)
	ENDIF ELSE BEGIN
	  Print,'Masked values: ',N_Elements(tmp),' over',N_Elements(img)
	  tmp=img[where(img GE 0)]
	  data=hist_equal(tmp,/hist)
	ENDELSE
	xplot,lindgen(n_elements(data)),data,/No_Block,Window_Size=[600,400]
	END

      'CALC INTV':BEGIN
	img = info.data->Value(info.image_num)
	tmp = where(img LT 0)
	IF tmp[0] EQ -1 THEN BEGIN
	  Print,'Masked values: None' 
	ENDIF ELSE BEGIN
	  Print,'Masked values: ',N_Elements(tmp),' over',N_Elements(img)
	  img[tmp]=0
	ENDELSE
	intv = total(img,2)
	xplot,lindgen(n_elements(intv)),intv,/No_Block,Window_Size=[600,400]
	END

      'CALC INTH':BEGIN
	img = info.data->Value(info.image_num)
	tmp = where(img LT 0)
	IF tmp[0] EQ -1 THEN BEGIN
	  Print,'Masked values: None' 
	ENDIF ELSE BEGIN
	  Print,'Masked values: ',N_Elements(tmp),' over',N_Elements(img)
	  img[tmp]=0
	ENDELSE
	inth = total(img,1)
	xplot,lindgen(n_elements(inth)),inth,/No_Block,Window_Size=[600,400]
	END

      'CALC SUBSTRACT':begin
          pData=info.data->Value([info.image_num],/pointer)
          img = *pData
          datasize=size(img)
          num=info.image_substract
          xedit,num,Text='Image # (i.e., index+1) of image to substract (noise): ',$
		Title='Substract image', Action=action, Dialog_Parent=event.top

          IF action EQ 'CANCEL' THEN Return
	  info.image_substract=num

          pNoise=info.data->Value([num-1],/pointer)
          noise = *pNoise
          noisesize=size(noise)

          IF total(noisesize) NE total(datasize) THEN BEGIN
            blah=Dialog_message( $
		['Data and noise images have incompatible dimensions'],/Error,$
          	Dialog_Parent=event.top)
            Return
          ENDIF

        img_old=img
        img=img-noise
        *pData=img
        ;write back image data to info structure
        info.data->set,[info.image_num],value=img,aux=img_old
        widget_control,event.top,set_uvalue=info
        XPLOT2D_DISPLAY_IMAGE,event.top

	END

      'CALC GENERIC':begin
	pData=info.data->Value([info.image_num],/pointer)
	img = *pData
	tmp = {oper:'img = Max(img)-img', res:['0','new window','this window']}
	XScrMenu,tmp,Action=act,Titles=['Operation','Result to'], /NoType,/Interp, $
	  WTitle='Generic image operation',dialog_Parent=event.top,FieldLen=40
	IF act EQ 'DONT' THEN RETURN
	itmp = Execute(tmp.oper)
	IF itmp NE 1 THEN BEGIN
	  jtmp = dialog_Message(/Error,['Operation not understood: ',tmp.oper],$
		dialog_Parent=event.top)
	ENDIF
	CASE Fix(tmp.res[0]) OF
	  0: xplot2d,img
	  1: XPLOT2D_ADD_IMAGE,event.top,img
	  else:
	ENDCASE
	END
      ;
      ; Simulations menu
      ;
      'SIMUL IMAGE':BEGIN
	pData=info.data->Value([info.image_num],/pointer)
	imageTitle=info.data->Value([info.image_num],/title)
	img = *pData
	ssize=size(img)

	tools_profile={file_profile:info.file_profile, detector_dist:info.detector_dist,  $
          dummy_pixels:info.dummy_pixels, pixelsize:info.pixelsize,  $
	  xsize:ssize[1],ysize:ssize[2], e0:info.e0, e1:info.e1, $
	  x:(info.center)[0], y:(info.center)[1], output:['0','New Image','New Window']}

	titles=['File (? for browser)','Distance sample-image [cm]','dummy pixels','pixelsize [microns]', $
	   	  'Pixels in X','Pixels in Y','Energy of XRD profile [eV]','Energy of current image [eV]', $
	   	  'Center in X [pixel]', 'Center in Y [pixel]', 'Output to']

	XScrMenu,tools_profile,Action=act,Titles=titles, /NoType,/Interp, $
	  	  WTitle='Simulate image',dialog_Parent=event.top,FieldLen=40, NCol=2

	IF act EQ 'DONT' THEN RETURN

	IF StrCompress(tools_profile.file_profile,/Rem) EQ '?' OR checkFile(tools_profile.file_profile) EQ 0 THEN BEGIN
	  file = Dialog_Pickfile(Title='Get XRD prfile')
	  IF file EQ '' THEN RETURN
	  tools_profile.file_profile=file
	ENDIF
        info.file_profile=tools_profile.file_profile 
        info.detector_dist=tools_profile.detector_dist 
        info.dummy_pixels=tools_profile.dummy_pixels
        info.pixelsize=tools_profile.pixelsize 
        info.e0=tools_profile.e0 
        info.e1=tools_profile.e1 
        info.center = [tools_profile.x,tools_profile.y]

        widget_control,event.top,set_uvalue=info

	Widget_Control,/HourGlass
        img = xplot2d_prof2image(info.file_profile,Plot=0, Dummy_Pixels=info.dummy_pixels, $
        size=[tools_profile.xsize,tools_profile.ysize], center=info.center, $
	Dist=info.detector_dist, pixelSize=info.pixelsize, e0=info.e0, e1=info.e1)

	CASE Fix( (tools_profile.output)[0]) OF
	 0: XPLOT2D_ADD_IMAGE,event.top,img,title='Simulated image from: '+info.file_profile
	 1: XPLOT2D,img,scale=info.scale,imageTitle='Simulated image from: '+info.file_profile
	ENDCASE
	END
      ;
      'SIMUL PEAKS':BEGIN
          ; 
          ; get/modify input data
          ; 
	  tools_profile={overplotPeaks:[StrCompress(info.overplotPeaks,/Rem),'No','Yes'], $
            file_peaks:info.file_peaks, detector_dist:info.detector_dist,  $
            pixelsize:info.pixelsize,  e0:info.e0, e1:info.e1, $
	    x:(info.center)[0], y:(info.center)[1]}


	  titles=['Overplot peaks', 'File (? for browser)','Distance sample-image [cm]','pixelsize [microns]', $
	   	  'Energy of theta-2theta peaks list [eV]','Energy of current image [eV]', $
	   	  'Center in X [pixel]', 'Center in Y [pixel]']

          flags=Replicate('w(0) EQ 1',N_Elements(titles))  & flags[0]='1'

	  XScrMenu,tools_profile,Action=act,Titles=titles, /NoType,/Interp, Flags=flags, $
	  	  WTitle='Overplot peaks',dialog_Parent=event.top,FieldLen=40

	  IF act EQ 'DONT' THEN RETURN

          info.overplotPeaks=Fix( (tools_profile.overplotPeaks)[0])
          info.detector_dist=tools_profile.detector_dist 
          info.pixelsize=tools_profile.pixelsize 
          info.e0=tools_profile.e0 
          info.e1=tools_profile.e1 
          info.center=[tools_profile.x,tools_profile.y]

	  IF info.overplotPeaks EQ 1 THEN BEGIN
	    IF StrCompress(tools_profile.file_peaks,/Rem) EQ '?' OR $
                  CheckFile(tools_profile.file_peaks) EQ 0 THEN BEGIN
	  	  file = Dialog_Pickfile(Title='Get file with theta-2theta peaks')
	  	  IF file EQ '' THEN RETURN
	  	  tools_profile.file_peaks=file
	    ENDIF
            info.file_peaks=tools_profile.file_peaks 
	  ENDIF

          widget_control,event.top,set_uvalue=info
          ; 
          ; refresh
          ; 
	  xplot2d_display_image,event.top
          ; xplot2d_overplotpeaks,event.top
	END

      'SIMUL D':BEGIN
          ; 
          ; get/modify input data
          ; 
	  tools_profile={overplotDSpacing:[StrCompress(info.overplotDSpacing,/Rem),'No','Yes'], $
            file_dspacing:info.file_dspacing, detector_dist:info.detector_dist,  $
            pixelsize:info.pixelsize,  e1:info.e1, $
	    x:(info.center)[0], y:(info.center)[1]}


	  titles=['Overplot rings of known d', 'File (? for browser)','Distance sample-image [cm]','pixelsize [microns]', $
	   	  'Energy of photon beam [eV]', $
	   	  'Center in X [pixel]', 'Center in Y [pixel]']

          flags=Replicate('w(0) EQ 1',N_Elements(titles))  & flags[0]='1'

	  XScrMenu,tools_profile,Action=act,Titles=titles, /NoType,/Interp, Flags=flags, $
	  	  WTitle='Overplot rings',dialog_Parent=event.top,FieldLen=40

	  IF act EQ 'DONT' THEN RETURN

          info.overplotDspacing=Fix( (tools_profile.overplotDspacing)[0])
          info.detector_dist=tools_profile.detector_dist 
          info.pixelsize=tools_profile.pixelsize 
          info.e1=tools_profile.e1 
          (info.center)=[tools_profile.x,tools_profile.y]

	  IF info.overplotDspacing EQ 1 THEN BEGIN
	    IF StrCompress(tools_profile.file_Dspacing,/Rem) EQ '?' OR $
                  CheckFile(tools_profile.file_Dspacing) EQ 0 THEN BEGIN
	  	  file = Dialog_Pickfile(Title='Get file with D-spacing Intensity')
	  	  IF file EQ '' THEN RETURN
	  	  tools_profile.file_dspacing=file
	    ENDIF
            info.file_dspacing=tools_profile.file_dspacing 
	  ENDIF

          widget_control,event.top,set_uvalue=info
          ; 
          ; refresh
          ; 
	  xplot2d_display_image,event.top
	END

      'SIMUL SCAN':BEGIN
	tmp = {var:['0','detector_distance','center_x','center_y','e0','e1'], $
	       min:0.0,max:0.0,npoints:20L} 
	print,'Current values ---------------------------------'
	print,'detector_distance: ',info.detector_dist
	print,'center_x: ',(info.center)[0]
	print,'center_y: ',(info.center)[1]
	print,'e0: ',info.e0
	print,'e1: ',info.e1
	print,'------------------------------------------------'
	XScrMenu,tmp,Action=act,Titles=['variable','min','max','number of points'],  $
		/NoType,/Interp, Flags=flags, WTitle='Scan parameter', $
		dialog_Parent=event.top,FieldLen=40
	  
	IF act EQ 'DONT' THEN RETURN
        CASE Fix((tmp.var)[0]) OF 
	  0: var_old = info.detector_dist
	  1: var_old = (info.center)[0]
	  2: var_old = (info.center)[1]
	  3: var_old = info.e0
	  4: var_old = info.e1
        ENDCASE

	FOR i=0,tmp.npoints-1 DO BEGIN
          var = tmp.min+(tmp.max-tmp.min)/(tmp.npoints-1)*i
 	  print,'Using variable: ',var

          CASE Fix((tmp.var)[0]) OF 
	    0: info.detector_dist = var
	    1: (info.center)[0] = var
	    2: (info.center)[1] = var
	    3: info.e0 = var
	    4: info.e1 = var
          ENDCASE
	  Widget_Control,event.top,Set_UValue=info
	  xplot2d_display_image,event.top
          itmp = Dialog_Message(/Question,'Continue?')
	  IF itmp EQ 'No' THEN GoTo,salimos
	ENDFOR

        salimos:
	
        itmp = Dialog_Message(/Question,'Keep last value of the scan?')
	IF itmp EQ 'No' THEN BEGIN
          CASE Fix((tmp.var)[0]) OF 
	    0: info.detector_dist = var_old
	    1: (info.center)[0] = var_old
	    2: (info.center)[1] = var_old
	    3: info.e0 = var_old
	    4: info.e1 = var_old
          ENDCASE
	  Widget_Control,event.top,Set_UValue=info
	  xplot2d_display_image,event.top
	ENDIF

	END
      'SIMUL DETECTOR':BEGIN

	tmp = {detector_dist:info.detector_dist, pixelsize:info.pixelsize, $
		center_x:(info.center)[0], center_y:(info.center)[1], $
		e1:info.e1}
	XScrMenu,tmp,Action=act,Titles=['detector diatance [cm]','pixelsize [microns]', $
		'Beam center H [pixel]','Beam center V [pixel]','Beam energy (e1) [eV]'],  $
		/NoType,/Interp, Flags=flags, WTitle='Detector and beam pars', $
		dialog_Parent=event.top,FieldLen=40
	  
	IF act EQ 'DONT' THEN RETURN

        info.detector_dist = tmp.detector_dist
	info.pixelsize     = tmp.pixelsize
	info.center        = [tmp.center_x,tmp.center_y]
	info.e1            = tmp.e1

	Widget_Control,event.top,Set_UValue=info
	xplot2d_display_image,event.top

	END
      ;
      ; Tools menu
      ;
      'TOOLS NEW WINDOW':BEGIN
	pData=info.data->Value([info.image_num],/pointer)
	imageTitle=info.data->Value([info.image_num],/title)
	img = *pData
	xplot2d,img,scale=info.scale,minColor=info.min,maxColor=info.max,imageTitle=imageTitle
	END
      'TOOLS XPLOT':BEGIN
	Xplot,no_block=info.no_block
	END
      ;
      ; Help menu
      ;
      'Help': XHelp,'xplot2d'
      else: Print,'Action not understood:',action
   ENDCASE
END

;
;===============================================================================
;
PRO XPLOT2D,indata,group=group,window_size=window_size,$
        wtitle=wtitle,parent=wXplot2d,no_block=no_block, $
	scale=scale,mincolor=mincolor,maxcolor=maxcolor,imagetitle=imagetitle
;XPLOT2D
   COMMON Xplot2D_common_variables, max_acceptable_screen_size_factor, scroll_step, margin_space, fileDir

   margin_space=25 ; space left for the axes to be drawn to the left and the right of the image
   max_acceptable_screen_size_factor=0.75 ;the value which determines how big the display window can be relative to the screen size
   scroll_step=20 ;scrolling step. Each time a scroll action is performed the image will be scrolled by an amount of step pixels

   ;Catch, error_status
   ;if error_status ne 0 then begin
   ;   message,/info,'error caught: '+!error_state.msg
   ;   if sdep(/w) then itmp = dialog_Message(/Error,$
   ;   'XPLOT: error caught: '+!error_state.msg)
   ;   catch, /cancel
   ;   on_error,2
   ;endif

   if keyword_set(group) then widget_control,/hourglass


   ; by default : no_block
   IF N_Elements(no_block) EQ 0 THEN no_block=1
   IF N_Elements(wtitle) EQ 0 THEN wTitle='xplot2d 1.0 Beta 2'
   IF N_Elements(scale) EQ 0 THEN scale=1.0
   IF N_Elements(scale) EQ 0 THEN scale=1.0
   IF N_Elements(mincolor) EQ 0 THEN mincolor=0L
   IF N_Elements(maxcolor) EQ 0 THEN maxcolor=0L
   IF N_Elements(imageformat) EQ 0 THEN  $
	imageformat={format:['2','EDF','XRD','MAR','BINARY','PNG/TIFF/etc'],$
		filter:'*.mccd.gz',float:['1','No','Yes'], $
		substitute:['0','add image','substitute first image'],wait:0.0}

   ;
   ;The ompp object which will keep all data
   ;
   CASE type(indata) OF
     0: BEGIN   ; no data input, create junk data
         indata = Dist(400)
         dataptr=ptr_new(indata,/no_copy)
         objData=Obj_New('ompp',dataptr,title='test image')
       END
     7:BEGIN  ; input is a file name (string)
        indata=read_edf(indata)
        dataptr=ptr_new(indata,/no_copy)
        objData=Obj_New('ompp',dataptr,title=imagetitle)
       END
     else: BEGIN  ; input is a variable
        ; TODO check size
        dataptr=ptr_new(indata,/no_copy)
        objData=Obj_New('ompp',dataptr,title=imagetitle)
      END
   ENDCASE

   ;
   ; use non-decomposed colors to be able to change colormap (...)
   ;
   Device,Decomposed=0



   ;
   ; Create widgets
   ;

   ;main window
   ;wXplot2d=Widget_base(title=Wtitle,MBAR=wMenuBar,/TLB_SIZE_EVENTS,/column)
   wXplot2d=Widget_base(title=Wtitle,MBAR=wMenuBar,/column,/TLB_KILL_REQUEST_EVENTS, Event_Pro='Xplot2D_Quit')

   if n_elements(Xplot2dWinList) eq 0 then Xplot2dWinList=[wXplot2d] $
   else Xplot2dWinList=[Xplot2dWinList,wXplot2d]

   ;Buttons
      wButtons=widget_base(wXplot2d,/Row)
      wImageNum=cw_field(wButtons,value=1,title='Image #',/integer,/return_events,xsize=4)
      wButtonList=cw_bgroup(wButtons,/row,['|<','<','x','>','>|','c f','c s'],button_uvalue=$
	['FIRST IMAGE','PREVIOUS IMAGE','DELETE IMAGE','NEXT IMAGE','LAST IMAGE', $
		'CLR FULLRANGE','CLR STRETCH'])
      wXCoordBox = CW_Field(wButtons, Title='X:', xsize=4,/integer)
      wYCoordBox = CW_Field(wButtons, Title='Y:', xsize=4,/integer)
      wIntensityValueBox = CW_Field(wButtons, Title='I:', xsize=8,/floating)

   ;graph window,
      ;device,get_screen_size=screen_size

      dataSize=size(indata)
      wGraph=widget_draw(wXplot2d,xsize=dataSize(1),ysize=dataSize(2),$
                          ;x_scroll_size=screen_size(0)*2/3,$
                          ;y_scroll_size=screen_size(1)*2/3,$
                          /button_events,/motion_events, event_pro='XPLOT2D_EVENT',$
                         RETAIN=2)
	;Comment window, to give instructions
		wComment=widget_text(wXplot2d,xsize=60,ysize=2)
   ;menu
      wFileMenu =  WIDGET_BUTTON(wMenuBar, VALUE='File', /MENU)
	tmpIndex = 1+fix( (imageFormat.format)[0] )
        tmpStr = (imageFormat.format)[tmpIndex]
        wLoadImage=WIDGET_BUTTON(wFileMenu, VALUE='Load Image...'+tmpStr,UVALUE='LOAD IMAGE')
        wtmp=WIDGET_BUTTON(wFileMenu, VALUE='Set load image properties...',UVALUE='SET LOAD PROPERTIES')
        ;wTmpBase=WIDGET_BUTTON(wFileMenu,VALUE='Load Image from File',/Menu)
        ;  wtmp=WIDGET_BUTTON(wTmpBase, VALUE='Load EDF...',UVALUE='LOAD IMAGE')
        ;  wtmp=WIDGET_BUTTON(wTmpBase, VALUE='Load XRD...',UVALUE='LOAD IMAGE')
        ;  wtmp=WIDGET_BUTTON(wTmpBase, VALUE='Load MAR...',UVALUE='LOAD IMAGE')
        ;  wtmp=WIDGET_BUTTON(wTmpBase, VALUE='Load BINARY...',UVALUE='LOAD IMAGE')
        ;  wtmp=WIDGET_BUTTON(wTmpBase, VALUE='Load PNG/TIFF/etc...',UVALUE='LOAD IMAGE')
;       ; wTmpBase=WIDGET_BUTTON(wFileMenu, VALUE='**Load List of Images',/Menu)
        wTmpBase=WIDGET_BUTTON(wFileMenu, VALUE='Save Image',/Menu)
          wtmp=WIDGET_BUTTON(wTmpBase, VALUE='EDF...',UVALUE='SAVE IMAGE')
          wtmp=WIDGET_BUTTON(wTmpBase, VALUE='XRD...',UVALUE='SAVE IMAGE')
          wtmp=WIDGET_BUTTON(wTmpBase, VALUE='Other...',UVALUE='SAVE IMAGE')
;          wtmp=WIDGET_BUTTON(wTmpBase, VALUE='**Selection...',UVALUE='SAVE IMAGE')
;        wtmp=WIDGET_BUTTON(wFileMenu, VALUE='Print...', UVALUE='PRINT',/Sepa)
        wtmp=WIDGET_BUTTON(wFileMenu, VALUE='Quit', UVALUE='EXIT',/Sepa)

      wEditMenu =  WIDGET_BUTTON(wMenuBar, VALUE='Edit', /MENU)
         wtmp=WIDGET_BUTTON(wEditMenu, VALUE='Undo last image modification',UVALUE='UNDO')
         wtmp=WIDGET_BUTTON(wEditMenu, VALUE='Edit image titles',UVALUE='EDIT IMAGE TITLES')
         wtmp=WIDGET_BUTTON(wEditMenu, VALUE='Delete current image',UVALUE='DELETE IMAGE',/Sepa)
         wtmp=WIDGET_BUTTON(wEditMenu, VALUE='Delete ALL image',UVALUE='DELETE ALL')
         wtmp=WIDGET_BUTTON(wEditMenu, VALUE='Set axis properties...',UVALUE='AXIS PROP',/SEPA)
      wViewMenu =  WIDGET_BUTTON(wMenuBar, VALUE='View', /MENU)
        wtmp=WIDGET_BUTTON(wViewMenu, VALUE='Refresh',UVALUE='REFRESH')
        wtmp=WIDGET_BUTTON(wViewMenu, VALUE='Scale on images',UVALUE='VIEW SCALE',/SEPA)
        wtmp=WIDGET_BUTTON(wViewMenu, VALUE='Scroll Palette',UVALUE='VIEW SCROLL PALETTE')
        wtmp=WIDGET_BUTTON(wViewMenu, VALUE='Display Color Table',UVALUE='SHOW COLOR TABLE',/SEPA)
        wtmp=WIDGET_BUTTON(wViewMenu, VALUE='Change Color Table',UVALUE='XLOADCT')
        wcolorlimits=WIDGET_BUTTON(wViewMenu, VALUE='Intensity <->Color Table',/Menu)
          wtmp=WIDGET_BUTTON(wColorLimits, VALUE='Full range',UVALUE='CLR FULLRANGE')
          wtmp=WIDGET_BUTTON(wColorLimits, VALUE='Stretch min&max intensity',UVALUE='CLR STRETCH')
          wtmp=WIDGET_BUTTON(wColorLimits, VALUE='Change stretch value...',UVALUE='CLR STRETCHVALUE')
          wtmp=WIDGET_BUTTON(wColorLimits, VALUE='Set min&max intensity...',UVALUE='CLR MINMAX',/SEPA)
        wtmp=WIDGET_BUTTON(wViewMenu, VALUE='Toggle Log/Linear Scale',UVALUE='LOGLINEARSCALE',/SEPA)
        wtmp=WIDGET_BUTTON(wViewMenu, VALUE='XYProfiles',UVALUE='CALC XYPROFILE',/SEPA)
        wtmp=WIDGET_BUTTON(wViewMenu, VALUE='Trans Profile',UVALUE='CALC PROFILE')
        wtmp=WIDGET_BUTTON(wViewMenu, VALUE='Info on images',UVALUE='VIEW INFO',/SEPA)


      wROIMenu =  WIDGET_BUTTON(wMenuBar, VALUE='ROI/Mask', /MENU)
        wtmpBase=WIDGET_BUTTON(wROIMenu, VALUE='Circle',/menu)
          wtmp=WIDGET_BUTTON(wTmpBase, VALUE='Mouse',UVALUE='ROI_CIRCLE')
          wtmp=WIDGET_BUTTON(wTmpBase, VALUE='Edit Parameters',UVALUE='ROI_CIRCLE')
        wtmpBase=WIDGET_BUTTON(wROIMenu, VALUE='Cake',/menu)
          wtmp=WIDGET_BUTTON(wTmpBase, VALUE='Mouse',UVALUE='ROI_CAKE')
          wtmp=WIDGET_BUTTON(wTmpBase, VALUE='Edit Parameters',UVALUE='ROI_CAKE')
        wtmpBase=WIDGET_BUTTON(wROIMenu, VALUE='Rectangle',/menu)
          wtmp=WIDGET_BUTTON(wTmpBase, VALUE='Mouse',UVALUE='ROI_RECTANGLE')
          wtmp=WIDGET_BUTTON(wTmpBase, VALUE='Edit Parameters',UVALUE='ROI_RECTANGLE')
        wtmpBase=WIDGET_BUTTON(wROIMenu, VALUE='Polygon',/menu)
          wtmp=WIDGET_BUTTON(wTmpBase, VALUE='Mouse',UVALUE='ROI_POLYGON')
        wtmp=WIDGET_BUTTON(wROIMenu, VALUE='Set MASK as ROI',UVALUE='MASK')
        wtmp=WIDGET_BUTTON(wROIMenu, VALUE='Set MASK as (1-ROI)',UVALUE='MASK2')

        ;wtmp=WIDGET_BUTTON(wROIMenu, VALUE='Rectangle',UVALUE='ROI')
        ;wtmp=WIDGET_BUTTON(wROIMenu, VALUE='Elliptical',UVALUE='ROI')
        ;wtmp=WIDGET_BUTTON(wROIMenu, VALUE='Polygonal',UVALUE='ROI')
        ;wtmp=WIDGET_BUTTON(wROIMenu, VALUE='Mask ROI',UVALUE='ROI')
        ;wtmp=WIDGET_BUTTON(wROIMenu, VALUE='Write mask to file',UVALUE='ROI')
        ;wtmp=WIDGET_BUTTON(wROIMenu, VALUE='Read+apply mask from file',UVALUE='ROI')
        wtmp=WIDGET_BUTTON(wROIMenu, VALUE='Apply mask from image...',UVALUE='MASKFROMIMAGE')


      wCalcMenu =  WIDGET_BUTTON(wMenuBar, VALUE='Calculations', /MENU)

        wTmpBase=WIDGET_BUTTON(wCalcMenu, VALUE='Rotate/Mirror',/Menu)
          wtmp=WIDGET_BUTTON(wTmpBase, VALUE='Rotate 90 Clockwise',UVALUE='CALC ROT')
          wtmp=WIDGET_BUTTON(wTmpBase, VALUE='Rotate 90 CounterClockwise',UVALUE='CALC ROT')
          wtmp=WIDGET_BUTTON(wTmpBase, VALUE='Rotate 180',UVALUE='CALC ROT')
          wtmp=WIDGET_BUTTON(wTmpBase, VALUE='Mirror up/down',UVALUE='CALC ROT',/SEPA)
          wtmp=WIDGET_BUTTON(wTmpBase, VALUE='Mirror left/right',UVALUE='CALC ROT')

	  wTmpBase=WIDGET_BUTTON(wCalcMenu, VALUE='Azimuth Integration',UVALUE='CALC INTEGRATE')
	  wTmpBase=WIDGET_BUTTON(wCalcMenu, VALUE='Azimuth Integration - Use Macro',UVALUE='CALC INTEGRATE_MACRO')


        wtmp=WIDGET_BUTTON(wCalcMenu, VALUE='Histogram',UVALUE='CALC HISTOGRAM')
        wtmp=WIDGET_BUTTON(wCalcMenu, VALUE='CDF(Histogram)',UVALUE='CALC CDF(HISTOGRAM)')
        wtmp=WIDGET_BUTTON(wCalcMenu, VALUE='Integral (vertical)',UVALUE='CALC INTV')
        wtmp=WIDGET_BUTTON(wCalcMenu, VALUE='Integral (horizontal)',UVALUE='CALC INTH')
        wtmp=WIDGET_BUTTON(wCalcMenu, VALUE='Substract image',UVALUE='CALC SUBSTRACT')
        wtmp=WIDGET_BUTTON(wCalcMenu, VALUE='Generic Operation',UVALUE='CALC GENERIC')

      wSimulMenu =  WIDGET_BUTTON(wMenuBar, VALUE='Simulations', /MENU)
        wtmp=WIDGET_BUTTON(wSimulMenu, VALUE='Create image from XRD profile...',UVALUE='SIMUL IMAGE')
        wtmp=WIDGET_BUTTON(wSimulMenu, VALUE='Overplot theta-2theta peaks...',UVALUE='SIMUL PEAKS')
        wtmp=WIDGET_BUTTON(wSimulMenu, VALUE='Overplot peaks from d-spacing...',UVALUE='SIMUL D')
        wtmp=WIDGET_BUTTON(wSimulMenu, VALUE='Scan parameter...',UVALUE='SIMUL SCAN')
        wtmp=WIDGET_BUTTON(wSimulMenu, VALUE='Define detector and beam pars...',UVALUE='SIMUL DETECTOR',/Sepa)

      wToolsMenu =  WIDGET_BUTTON(wMenuBar, VALUE='Tools', /MENU)
         wtmp=WIDGET_BUTTON(wToolsMenu, VALUE='Image in new window',UVALUE='TOOLS NEW WINDOW')
         wtmp=WIDGET_BUTTON(wToolsMenu, VALUE='Plotting tool (xplot)',UVALUE='TOOLS XPLOT')

      wHelpMenu =  WIDGET_BUTTON(wMenuBar, VALUE='Help', /Help)
        wtmp=WIDGET_BUTTON(wHelpMenu, VALUE='xplot2d',UVALUE='Help')

      ;window, /free, xsize=110, ysize=250
      ;colorbarWindowId = !d.window
	colorbarWindowId = 0L

   roi_circle_info = {centre_coord:[-1,-1],phi1:-1.0,phi2:-1.0,rho1:-1.0,rho2:-1.0}
   roi_rectangle_info = {lower_left_corner:[-1,-1],upper_right_corner:[-1,-1]}
   axis_properties = {x_origin:0,x_pixel_unit_ratio:1.0,y_origin:0,y_pixel_unit_ratio:1.0}
   load_bin = { file:'?', header:4096L, xsize:2048L, ysize:2048L, type:['12', $
	'Undefined (NOT IMPLEMENTED)', 'Byte', 'Integer', 'Longword integer', $
	'Floating point', 'Double-precision floating', 'Complex floating', $
	'String (NOT IMPLEMENTED)', 'Structure (NOT IMPLEMENTED)', 'Double-precision complex', $
	'Pointer (NOT IMPLEMENTED)', 'Object reference (NOT IMPLEMENTED)', 'Unsigned Integer', $
	'Unsigned Longword Integer', '64-bit Integer', 'Unsigned 64-bit Integer'] }


   ; xplot2d_scroll_palette, wXplot2d, wScroll, no_block=no_block
   wScroll=0L
   IF mincolor EQ maxcolor THEN BEGIN
	mincolor=min(*dataptr)
	maxcolor=max(*dataptr)
   ENDIF



   ; 
   ; define the "info" structure
   ; 
   info={ data: objData,$       ;pointer to the data, as an ompp object
               image_num:0,$         ;if multiple images, the one currently displayed. Note that this in
				     ; index starting from zero, whereas the displayed number is index+1
               ;
               ; view variables
               ;
               ScaleType:'linear',$      ;logarithmic ('log') or linear ('linear')
               scroll_vec:[0,0],$		;scrolling vector
               image_position:[0,0], $ ;image position relative to the visible lower left corner of the displaying window
               Min:minColor,$             ;min value displayed
               Max:maxColor,$               ;max value displayed
               Scale:scale, $  ;entry scale zoom factor
               Stretch:0.01, $  ;entry histogram strech value
               rebin_factor:1.,$      ;rebin image for display ? Used in transversal profiles ????
               ;
               ; roi variables
               ;
	       roi_pixels:ptr_new(), $		; pointer to the roi region pixels
	       roi_rectangle_info:roi_rectangle_info,$		; information about the circle or cake roi
	       roi_circle_info:roi_circle_info,$		; information about the circle or cake roi
	       roi_centre_pixel_coord:[0,0],$	;pointer to the coordinates of the point in the centre of the ROI
               ;
               ; physical and detector information
               ;
               detector_dist:11.0, $ ; distance detector-sample in cm 
               pixelsize:78.94,  $   ; pixel size in microns
               center:[0,0],     $   ; center of the image in pixels
               e1:10000.0,       $   ; photon energy for the image in eV
               ; 
               ; simulations
               ; 
               dummy_pixels:2,  $    ; number of pixels to eliminate close the center for simulations
               e0:8047.78,      $    ; energy of the loaded profile  
               file_profile:'profile.dat',      $    ; file name with XRD profile (theta-2theta)
               file_peaks:'peaks.dat',          $    ; file name with XRD peaks (theta-2theta)
               file_dspacing:'dspacing.dat',          $    ; file name with XRD peaks (theta-2theta)
               overplotPeaks:0,                 $    ; flag to overplot peaks
               overplotDSpacing:0,                 $    ; flag to overplot peaks
               ;
               ; structures with local information
               ;
               axis_properties:axis_properties,$	;axis properties defining origins and pixel/user unit ratios
	       load_bin:load_bin,$		; str with info on bin files
               ;
               ; widget and window id's
               ;
               wGraph:wGraph, $       ;graph window
               colorbarWindowId:colorbarWindowId, $ ;colorbar window ID
               wXCoordBox:wXCoordBox, $ ;field containing the x coordinates of the mouse in the draw_widget
               wYCoordBox:wYCoordBox, $	;field containing the y coordinates of the mouse in the draw_widget
               wIntensityValueBox:wIntensityValueBox, $ ;image intensity value
               wComment:wComment, $       ;comment line
               wImageNum:wImageNum, $  ;entry field to choose image
               wScroll:wScroll, $  ;entry field to choose image
               wLoadImage:wLoadImage, $  ;entry field to choose image
               ;
               ; auxiliar variables
               ;
	       image_mask:1, $
	       image_substract:1, $
               imageFormat:imageFormat, $  ;  image format (structure)
               no_block:no_block $  ;  no_block flag 
               }
   ;KLUDGE
   if info.min LE 0 then info.min=min([.1,info.max/100])

   ;print,size(*(info.pData))
   ;return
   ;activate window, set uvalue to store all data,display image...
      widget_control, wXplot2d, /NO_COPY, /REALIZE

      widget_control,wXplot2d,set_uvalue=info,/no_copy
      Widget_Control,/HourGlass
      tek_color
      xplot2d_display_image,wXplot2d

      xmanager, 'xplot2d', wXplot2d, GROUP_LEADER=group,no_block=no_block
                ;cleanup='Xplot2d_quit'
END
