
;=========================== copyright =========================================
;                       XAID (XAfs with IDl) package.                          ;
;     Copyright European Synchrotron Radiation Facility (1993-1996).           ;
; This software may be used, copied, or redistributed as long as it is not     ;
; sold and this copyright notice is reproduced on each copy made.              ;
; The software is provided as is without any express or implied warranties     ;
; whatsoever.                                                                  ;
; Other limitations apply as described in the file README.copyright.           ;
;                                                                              ;
; REFERENCE: "XAID: A package for XAFS data analysis based on IDL"             ;
;       Manuel Sanchez del Rio. To be published at the proceedings of          ;
;       the 9th International Conference on X-ray Absorption Fine Structure    ;
;       26th - 30th August 1996,  E S R F - Grenoble  France                   ;
;==============================================================================;
function integset,set,cdf,xrange=xrange
;+
; NAME:
;	INTEGSET
;
; PURPOSE:
;	This function calculates the integral of a set of data between
;	the limits defined in xrange.
;
; CATEGORY:
;	General utilities.
;
; CALLING SEQUENCE:
;
;	Result = INTEGSET(set)
;
; INPUTS:
;	set:	fltarr(2,npoints) set of data
;
; OPTIONAL INPUTS:
;	
; KEYWORD PARAMETERS:
;	XRANGE=[xmin,xmax] range of the integration
;
; OUTPUTS:
;	This function returns the value of the integral of the set.
;
; OPTIONAL OUTPUTS:
;	CDF: a set with the CDF function consisting in the integral 
;	from the starting point to the current abscissa.
;
; COMMON BLOCKS:
;
; SIDE EFFECTS:
;	None
;
; RESTRICTIONS:
;	None
;
; PROCEDURE:
;	The integral is the simple addition of all the coordinate values
;
; EXAMPLE:
;
;		inte = INTEGSET(set,xrange=[2.,12.]
;
; MODIFICATION HISTORY:
; 	Written by:	Manuel Sanchez del Rio. ESRF
;	may, 1993
;-
on_error,2
;
; define interval
;
if not(keyword_set(xrange)) then begin
  xrange = fltarr(2)
  xrange(0) = min(set(0,*))
  xrange(1) = max(set(0,*))
endif
;
; calculate the areas over individual intervals
;
npts = n_elements(set(0,*))
diff = shift(set(0,*),-1) - set(0,*)
val = 0.5*(shift(set(1,*),-1) + set(1,*))
tot = val*diff
tot_set = fltarr(2,npts) 
tot_set(1,*)=tot
tot_set(0,*)=set(0,*)  
;
; cut the result to the given interval; with this cut the effect of the
; last (ficticious) point is eliminated
;
cutset,tot_set,tot_set_cut,xrange=xrange
;
; calculate the acumulate integral (cdf)
;
npts2 = n_elements(tot_set_cut(0,*))
cdf = fltarr(2,npts2)
cdf(0,*) = tot_set_cut(0,*)
for i=0,npts2-1 do begin
  a=tot_set_cut(1,0:i)
  cdf(1,i)=total(a)
endfor
;
return,cdf(1,npts2-1)
end
