;+
;====================================  Xtc   ===================================
;
;   Xtc is an widget based graphical user interface to calculate on-axis
;   brilliance tuning curves for an ideal undulator insertion device using
;   the Bessel function approximation (regular planar device or helical device).
;   The calculation is performed by the program TC ("tuning curves") which is
;   executed from the XTC interface. The effects of the particle beam
;   emittance and beam energy spread are taken into account.
;
;
; DESCRIPTION OF THE CONTROLS IN THE MAIN WINDOW:
;
;  File:
;    Xtc input parameters: This option allows to save the current
;		parameters to a file for later loading. It also allows
;		to save the current parameters as defaults for being
;		used when the application is initialized. In the last
;		case, the file is named "application".xop (where 
;		"application " is the name of the current XOP
;		application) and is written in the directory pointed
;		by the XOP_DEFAULTS_DIR environment variable (which
;		must be set). The parameter file is ASCII and can be
;		read and edited with care.
;    TC file: This option allows to load/write the current parameter
;		and result files for further calculations. The used file
;		formats are those used by TC. Therefore, they
;		can be used to run TC without the interface or to
;		import TC files created outside the XTC application.
;  Quit: to exit from the program
;
; Set_Parameters:
;  Set Parameters: to define the parameters for the calculation.
;		The same result is obtained pressing the "Set Parameters"
;		button in the main XTC window. 
;               Please refer to the information under the HELP
;		button for a complete description of the parameters. After 
;		pressing the ACCEPT button, TC starts running.
;  Set Defaults: Sets the default parameters.
;
;  Show: Display results
;    Plot ...: Plots the different TC results.
;    View Results: Displays the TC result file, tc.out, in a scrollable 
;		window. The file may be printed from this window.
;
;  Help:   Shows the TC help and the XTC help (this text).
;
;
; COPYRIGHT:
;	xtc  belongs to XOP package and it is distributed within XOP.
;	PLEASE REFER TO THE XOP COPYRIGHT NOTICE BEFORE USING IT.
;
; CREDITS:
;	Published calculations made with XOP should refer:
;
;	  M. Sanchez del Rio and R. J. Dejus "XOP: Recent Developments"
;	  SPIE proceedings vol. 3448, pp.340-345, 1998.
;
; LAST MODIFICATION: msr/rjd/00-07-12
;
;
;-
; -----------------------------------------------------------------------
;	Author: Roger J. Dejus (dejus@aps.anl.gov), XFD/APS, May, 1996.
;	MODIFICATION HISTORY:
;
;        Date     | Name  | Description
; ----------------+-------+-----------------------------------------------------
; 09-MAY-1996     | RJD   | First version.
; ----------------+-------+-----------------------------------------------------
; 21-MAY-1996     | RJD   | First official release, v1.2.
; ----------------+-------+-----------------------------------------------------
; 08-SEPT-1997    | RJD   | Modified to run version 1.6 of the tuning curve
;                 |       | program. Current release is v1.3.
; ----------------+-------+-----------------------------------------------------
; 18-NOV-1997     | RJD   | Modified to run version 1.9 of the tuning curve
;                 |       | program. Changed FILTER keyword in PICKFILE so that
;                 |       | it is undefined for operating systems other than
;                 |       | UNIX. Added the keyword DIALOG_PARENT to
;                 |       | WIDGET_MESSAGE. Sets working directory if the GROUP
;                 |       | keyword is not set.
;                 |       | Current release is v1.4.
; ----------------+-------+-----------------------------------------------------
; 26-NOV-1998     | MSR   | Using the new (xop2.0) input file system.
;                 |       | Changed the menu look to adapt to xop "standards".
;                 |       | Added Xtc_Version function. Use of Xop_Spawn.
;                 |       | Backgroud task in a new window.
;                 |       | "Advanced" use of Xplot instead of XWindow.
;                 |       | Current release is v1.5Beta1.
; ----------------+-------+-----------------------------------------------------
; 26-NOV-1998     | MSR   | Version 1.5 for XOP 2.0
;-

FORWARD_FUNCTION sdep

;
;==============================================================================
;
FUNCTION Xtc_Version
RETURN,'1.5'
END ; Xtc_Version


;
;==============================================================================
;
FUNCTION xtc_read_dat, struct, FNAME=fname
if n_elements(fname) eq 0 then fname = 'tc.inp'

catch, error_status
if error_status ne 0 then begin
  print, !error_state.msg
  print, 'XTC_READ_DAT: Error reading TC input data file'
  return, error_status
endif

inp = XOP_DEFAULTS('xtc') ; define structure
openr, unit, fname, /get_lun
readf, unit, inp
free_lun, unit
struct = inp ; make assignment

return, error_status
END ; xtc_read_dat

;
;==============================================================================
;
FUNCTION xtc_read2d_plt, a, nd, nh, np, title, FNAME=fname
if n_elements(fname) eq 0 then fname = 'tc.out'

a = rascii(fname, npoints=nd, header=head, error_status=error_status)
if error_status ne 0 then begin
  print, 'XTC_READ2D_PLT: Error reading TC plot file'
  return, error_status
endif

; find number of harmonics and number of points/harmonic 
head = reform(head)
i    = strmatch1(' Nharm',head,6)
hharm= head(i)
nv   = strparse(hharm,':',hlist)
nh   = fix(hlist(1))	; number of harmonics
asize= size(a)
n    = fix(asize(2))	; total number of points
np   = n/nh		; number of points/harmonic
title= head(2)		; title

return, error_status
END ; xtc_read2d_plt

;
;==============================================================================
;
FUNCTION xtc_write_dat, struct, FNAME=fname
if n_elements(fname) eq 0 then fname = 'tc.inp'

catch, error_status
if error_status ne 0 then begin
  print, !error_state.msg
  print, 'XTC_WRITE_DAT: Error writing TC input data file'
  return, error_status
endif

;  Define text strings
text_machine   = '                       ring-energy(GeV) current(mA) sige'
text_beam      = '               sx(mm) sy(mm) sx1(mrad) sy1(mrad)'
text_undulator = '                               period(cm) N'
text_energy    = '                   emin emax ne'
text_harm      = '                       hmin hmax hstep'
text_parm      = '               helical method print_k neks'

openw, unit, fname, /get_lun

printf, unit, struct.title, format='(a)'
printf, unit, struct.energy, struct.cur, struct.sige, text_machine, format='(2f8.2,f8.5,a)'
printf, unit, struct.sigx, struct.sigy, struct.sigx1, struct.sigy1, text_beam, $
	      format='(2f8.3,2f8.4,a)'
printf, unit, struct.period, struct.np, text_undulator, $
	      format='(f8.2,i8,a)'
printf, unit, struct.emin, struct.emax, struct.n, text_energy, $
	      format='(2f10.1,i8,a)'
printf, unit, struct.ihmin, struct.ihmax, struct.ihstep, text_harm, $
	      format='(3i8,a)'
printf, unit, struct.ihel, struct.method, struct.ik, struct.neks, text_parm, $
	      format='(4i8,a)'
printf, unit, struct.run_mode_name, $
	      format='(a)'
free_lun, unit
END ; xtc_write_dat

;
;==============================================================================
;
PRO xtc_event, event

Catch, error_status
IF error_status NE 0 THEN BEGIN
   Message,/Info,'error caught: '+!error_state.msg
   itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
     'XTC_EVENT: error caught: '+!error_state.msg)
   IF N_Elements(info) NE 0 THEN Widget_Control, event.top, Set_UValue=info
   Catch, /Cancel
   On_Error,2
   GoTo,LBL900
ENDIF
;  Retrieve anonymous structure
WIDGET_CONTROL, event.top, GET_UVALUE=info
hc = info.hc ; pointer to control
hd = info.hd ; pointer to input data
hp = info.hp ; pointer to output results
;print,'xtc_event',hc,hd,hp
;help,/str,event
;  Get structures
; HANDLE_VALUE, hc, control   ; srio comments: , /NO_COPY
; HANDLE_VALUE, hd, data   ; srio comments: , /NO_COPY 
; HANDLE_VALUE, hp, plt   ; srio comments: , /NO_COPY
HANDLE_VALUE, hc, control, /NO_COPY
HANDLE_VALUE, hd, data, /NO_COPY 
HANDLE_VALUE, hp, plt, /NO_COPY

delimiter = control.delimiter
fname_res = control.fname_res
osversion = control.osversion
ds        = control.ds
ps        = control.ps

IF N_ELEMENTS(data) eq 0 THEN BEGIN
  l1d = 0b
ENDIF ELSE BEGIN
  IF STRMATCH1('dat1',TAG_NAMES(data)) ne -1 THEN l1d = 1b ELSE l1d = 0b
ENDELSE
IF N_ELEMENTS(plt) eq 0 THEN BEGIN
  l2p = 0b
ENDIF ELSE BEGIN
  IF STRMATCH1('plt2',TAG_NAMES(plt)) ne -1 THEN l2p = 1b ELSE l2p = 0b
ENDELSE
;print,'l1d l2p',l1d,l2p

;  Definitions
C_EV_KEV = 1.0e-3		; conversion eV -> keV
C_W_KW   = 1.0e-3		; conversion W  -> kW

; srio 
;IF (Where(StrUpCase(Tag_Names(event)) EQ 'VALUE'))(0) NE -1 THEN BEGIN
;  eVal = event.value
;ENDIF ELSE BEGIN
  uVal=''
  Widget_Control,event.id, Get_UValue=uVal
  eVal=uVal
;ENDELSE

nv = strparse(eVal, delimiter, elist)

;print,'nv elist',nv,' ',elist
;print, event.value
;print,'************'+eVal+'***********'

CASE eval OF


  'FILEINPUT': BEGIN
    action=''
    Widget_Control,event.id, Get_Value=action
    CASE action OF
      'Load from file...': BEGIN
        ;if sdep() EQ 'UNIX' then filter='*.xop' else filter=0
	filter='*.xop'
        str_par = Xop_Input_Load(Title=$
          'Select XTC input file...',$
          /NoConf,Filter=filter,Group=event.top)
        IF Type(str_par) EQ 8 THEN BEGIN
          tmp = data.dat1
          Copy_Structure,str_par,tmp,Group=event.top ; ,/OnlyFirstField
          data.dat1 = tmp
        ENDIF
      END
     'Save to file...': BEGIN
        str_par = data.dat1
        Xop_Input_Save,str_par,File='xtc.xop',$
          /Write, Group=event.top, Comment='; xop/xtc(v'+$
          xtc_version()+') input file on '+SysTime()
      END
     'Save as default': BEGIN
        str_par = data.dat1
        Xop_Input_Save,str_par,Group=event.top, $
          Default='xtc.xop',Comment='; xop/xtc(v'+$
          xtc_version()+') input file on '+SysTime()
      END
    ENDCASE
  END

  'File:Load  TC:Input      File': BEGIN
  CD, CURRENT=cwd & cwd = cwd +ds
  CASE osversion OF
    'UNIX':    filter = '*.inp'
    'WINDOWS': filter = 0
    'MACOS':   filter = 0 ; 'MACOS command here'
  ENDCASE
  fp = Dialog_PICKFILE(PATH=cwd, FILTER=filter, GET_PATH=path, GROUP=event.top)
  IF fp eq '' or fp eq cwd THEN GOTO, LBL900 ; cancel
  ff = FINDFILE(fp)
  fname = ff(0)
  IF fname eq '' THEN BEGIN
    stxt = STRMID(fp, STRLEN(path), STRLEN(fp) -STRLEN(path)) ; get filename
    junk = WIDGET_MESSAGE(/ERROR, DIALOG_PARENT=event.top, $
      ['',stxt,'File not found'])
    GOTO, LBL900
  ENDIF
  error_status = XTC_READ_DAT(dat1, FNAME=fname)
  IF error_status NE 0 THEN BEGIN
    stxt = STRMID(fname, STRLEN(path), STRLEN(fname) -STRLEN(path)) ; get filename
    junk = WIDGET_MESSAGE(/ERROR, DIALOG_PARENT=event.top, $
	['','XTC_EVENT: Error reading file: '+ stxt,'',$
	'This seems not a TC ','input file  '])
    GOTO, LBL900
  ENDIF
  data = {dat1:dat1}
  END
  'File:Load  TC:Output     File': BEGIN
  CD, CURRENT=cwd & cwd = cwd +ds
  CASE osversion OF
    'UNIX':    filter = '*.out'
    'WINDOWS': filter = 0
    'MACOS':   filter = 0 ; 'MACOS command here'
  ENDCASE
  fp = Dialog_PICKFILE(PATH=cwd, FILTER=filter, GET_PATH=path, GROUP=event.top)
  IF fp eq '' or fp eq cwd THEN GOTO, LBL900 ; cancel
  ff = FINDFILE(fp)
  fname = ff(0)
  IF fname eq '' THEN BEGIN
    stxt = STRMID(fp, STRLEN(path), STRLEN(fp) -STRLEN(path)) ; get filename
    junk = WIDGET_MESSAGE(/ERROR, DIALOG_PARENT=event.top, $
      ['',stxt,'File not found'])
    GOTO, LBL900
  ENDIF
  error_status = XTC_READ2D_PLT(a2, nd, nh, np, title, FNAME=fname)
  IF error_status NE 0 THEN BEGIN
    stxt = STRMID(fname, STRLEN(path), STRLEN(fname) -STRLEN(path)) ; get filename
    junk = WIDGET_MESSAGE(/ERROR, DIALOG_PARENT=event.top, $
        ['','XTC_EVENT: Error reading file: '+stxt, $
	'This seems not a TC ','output file  '])
    GOTO, LBL900
  ENDIF
  control = {delimiter:delimiter, fname_res:fname, osversion:osversion, $
	     ds:ds, ps:ps}
  nc = nd(0)
  plt2 = {a2:a2, nc:nc, nh:nh, np:np, title:title}
  plt  = {plt2:plt2}
  END
  'File:Load  TC:Default    File': BEGIN


  itmp = Dialog_Message(Dialog_Parent=event.top, $
    /Question,['This option initializes the',$
    'xtc parameters to their default values.',$
    'Then you must click Set_parameters to run the program.',$
    'Please confirm'],Title='xwiggler: set defaults')
  IF itmp EQ 'No' THEN GoTo,LBL900


  dat1 = XOP_DEFAULTS('xtc')
  data = {dat1:dat1}
  END
  'File:Write TC:Input  File': BEGIN
  CD, CURRENT=cwd & cwd = cwd +ds
  CASE osversion OF
    'UNIX':    filter = '*.inp'
    'WINDOWS': filter = 0
    'MACOS':   filter = 0 ; 'MACOS command here'
  ENDCASE
  fp = Dialog_PICKFILE(PATH=cwd, FILTER=filter, GET_PATH=path, $
    TITLE='Save File As', GROUP=event.top)
  IF fp eq '' or fp eq cwd THEN GOTO, LBL900 ; cancel
  stxt = 'tc.inp'		; check for existence of tc.dat
  ff = FINDFILE(path +stxt)
  fname = ff(0)
  IF fname eq '' THEN BEGIN ; file does not exist
    junk = Dialog_Message(/ERROR, DIALOG_PARENT=event.top, $
	['','Error copying file ' +stxt, $
	'File not found','Run TC first and try again!'])
    GOTO, LBL900
  ENDIF
  ff = FINDFILE(fp)
  fname = ff(0)
  IF fname ne '' THEN BEGIN ; file already exist
    stxt = STRMID(fp, STRLEN(path), STRLEN(fp) -STRLEN(path)) ; get filename
    junk = Dialog_Message(/QUESTION, DIALOG_PARENT=event.top, $
      ['',stxt,'Already exist; Overwrite File ?'])
    IF junk eq 'No' THEN GOTO, LBL900
  ENDIF
  CASE osversion OF
    'UNIX':    command = '/bin/cp ' +path +'tc.inp ' +fp
    'WINDOWS': command = 'copy '    +path +'tc.inp ' +fp
    'MACOS':   command = 'MACOS command here'
  ENDCASE
  message,/info,'Executing: '+command
  SPAWN, command
  END
  'File:Write TC:Output File': BEGIN
  CD, CURRENT=cwd & cwd = cwd +ds
  CASE osversion OF
    'UNIX':    filter = '*.out'
    'WINDOWS': filter = 0
    'MACOS':   filter = 0  ; 'MACOS command here'
  ENDCASE
  fp = Dialog_PICKFILE(PATH=cwd, FILTER=filter, GET_PATH=path, $
    TITLE='Save File As', GROUP=event.top)
  IF fp eq '' or fp eq cwd THEN GOTO, LBL900 ; cancel
  stxt = 'tc.out'		; check for existence of tc.out
  ff = FINDFILE(path +stxt)
  fname = ff(0)
  IF fname eq '' THEN BEGIN ; file does not exist
    junk = Dialog_Message(/ERROR, DIALOG_PARENT=event.top, $
	['','Error copying file ' +stxt, $
	'File not found','Run TC first and try again!'])
    GOTO, LBL900
  ENDIF
  ff = FINDFILE(fp)
  fname = ff(0)
  IF fname ne '' THEN BEGIN ; file already exist
    stxt = STRMID(fp, STRLEN(path), STRLEN(fp) -STRLEN(path)) ; get filename
    junk = Dialog_Message(/QUESTION, DIALOG_PARENT=event.top, $
      ['',stxt,'Already exist; Overwrite File ?'])
    IF junk eq 'No' THEN GOTO, LBL900
  ENDIF
  CASE osversion OF
    'UNIX':    command = '/bin/cp ' +path +'tc.out ' +fp
    'WINDOWS': command = 'copy '    +path +'tc.out ' +fp
    'MACOS':   command = 'MACOS command here'
  ENDCASE
  message,/info,'Executing: '+command
  SPAWN, command
  END
  'File:Display File': BEGIN
  CD, CURRENT=cwd & cwd = cwd +ds
  fp = Dialog_PICKFILE(PATH=cwd, GET_PATH=path, GROUP=event.top)
  IF fp eq '' or fp eq cwd THEN GOTO, LBL900 ; cancel
  XDISPLAYFILE1, fp
  END
  'File:Quit': BEGIN
  HANDLE_FREE, hc 		; Note free handle here
  WIDGET_CONTROL, event.top, /DESTROY
  ;CASE osversion OF
  ;  'UNIX':    command = '/bin/rm -f tc.inp tc.out' 
  ;  'WINDOWS': command = 'rm tc.inp tc.out'
  ;  'MACOS':   command = 'MACOS command here'
  ;ENDCASE
  ;message,/info,'Executing: '+command
  ;spawn,command
  Delete_Files,['tc.inp','tc.out'],Group=event.top
  END

  'Set & Run': BEGIN
  Widget_Control,/HourGlass
;  IF NOT l1d THEN BEGIN
;    ff = findfile('tc.inp')
;    fname = ff(0)
;    if fname eq '' THEN BEGIN
;      dat1 = XOP_DEFAULTS('xtc') ; not found, define new structure
;    ENDIF ELSE BEGIN ; use file tc.inp
;      error_status = XTC_READ_DAT(dat1)
;      IF error_status NE 0 THEN BEGIN
;	junk = Dialog_Message(/ERROR, DIALOG_PARENT=event.top, $
;	  ['','Error reading file', $
;	  'tc.inp','This seems not a TC ','input file  '])
;	GOTO, LBL900
;      ENDIF
;    ENDELSE
;    data = {dat1:dat1}
;  END

  action=''
  XTC_SET, data, ACTION=action, GROUP=event.top
  Widget_Control,/HourGlass
  IF action eq 'DONT' THEN GOTO, LBL900

;  Save tc.inp
  error_status = XTC_WRITE_DAT(data.dat1)
  IF error_status NE 0 THEN BEGIN
    junk = Dialog_Message(/ERROR, DIALOG_PARENT=event.top, $
      'Error writing file tc.inp')
    GOTO, LBL900
  ENDIF

;  Run TC
  IF data.dat1.run_mode_name eq 'foreground' THEN BEGIN ; foreground execution
    PRINT, ' Running TC in the foreground; PLEASE WAIT!'
    CASE osversion OF
      ;'UNIX':    command = 'source $XOP_RC ; tc' 
      'UNIX':    command = 'tc' 
      'WINDOWS': command = 'tc'
      'MACOS':   command = 'tc' ; 'MACOS command here'
    ENDCASE
    ;message,/info,'Executing: '+command
    ;SPAWN, command
    Xop_Spawn,command,CleanFiles=['tc.out']
    bell = STRING(7b)
    FOR i=0,1 DO BEGIN
      PRINT, bell
      WAIT, 0.2
    ENDFOR
  ENDIF ELSE BEGIN ; background execution
    CASE osversion OF
      'UNIX': BEGIN
	PRINT, ' Running TC in the background'
	command=''
	Xop_Spawn,'tc',onlyGetCommand=command,CleanFiles=['tc.out']
	;command = 'source $XOP_RC ; tc &'
	command = 'xterm -e '+command+' &'
	message,/info,'Executing: '+command
	SPAWN, command
      END
      'WINDOWS': BEGIN
	itmp = Dialog_Message(/INFO, DIALOG_PARENT=event.top, $
	  ['Background execution ','not implemented under Windows'])
      END
      'MACOS': BEGIN
	itmp = Dialog_Message(/INFO, DIALOG_PARENT=event.top, $
	  ['Background execution ','not implemented under MacOS'])
      END
    ENDCASE
  ENDELSE
  END

  'Help:TC': BEGIN
  hfile = Xop_GetEnv('XOP_HOME') +ds +'doc' +ds +'tc.txt'
  XDISPLAYFILE1, hfile
  END
  'Help:Xtc': Xhelp,'xtc',GROUP=event.top

ELSE: BEGIN ; Show
  CASE elist(1) OF 
    'Plot Results': BEGIN
    IF l2p THEN BEGIN ; display data saved in anonymous structure
      a2 = plt.plt2.a2 ; data
      nc = plt.plt2.nc ; # of columns
      nh = plt.plt2.nh ; # of harmonics
      np = plt.plt2.np ; # of points/harmonic
      title = plt.plt2.title 
    ENDIF ELSE BEGIN ; read data and save in anonymous structure
      error_status = XTC_READ2D_PLT(a2, nd, nh, np, title)
      IF error_status NE 0 THEN BEGIN
        stxt = 'tc.out'
        junk = Dialog_Message(/ERROR, DIALOG_PARENT=event.top, $
	  ['','XTC_EVENT: Error reading file: '+stxt,'',$
        'Please use "Set Parameters" button to set and run tc'] )
        GOTO, LBL900
      ENDIF
      nc = nd(0)
      plt2 = {a2:a2, nc:nc, nh:nh, np:np, title:title}
      plt  = {plt2:plt2}
    ENDELSE
    ptxt = ['',' data not found...', $
      ' Rerun TC and set "Save K-value/power?"']
    CASE elist(2) OF 
      'Xplot': BEGIN
      CASE elist(3) OF 
        'On-Axis Brilliance': BEGIN
	ap = a2(1:2,*)			; emittance shifted energy = second col
	ap(0,*) = ap(0,*)*C_EV_KEV	; emittance shifted energy = first col
	xmin = min(ap(0,*))
	ymax = max(ap(1,*))
	if nh gt 1 then begin
	  i = nh -2 
	  xmax = 0.75*max(ap(0,*))
	  ymin = min(ap(1,np*i:np*(i+1)-1))
	endif else begin			; only one harmonic
	  xmax = 1.00*max(ap(0,*))
	  ymin = min(ap(1,0:np-1))
	endelse
	xtitle = 'Energy (keV)'
	ytitle = 'Brilliance (ph/s/mrad^2/mm^2/0.1%bw)'
	xplot,ap,GROUP=event.top,xrange=[xmin,xmax],yrange=[ymin,ymax],/ylog, $
	  coltitles=[xtitle,ytitle], xtitle='-1',ytitle='-1',title=title
        END
        'All': BEGIN
	ap = fltarr(nc+1,np*nh)		; make room for energy shift -> use nc+1
	ap(0,*) = a2(0,*)*C_EV_KEV	; zero emittance energy (keV)
	ap(1,*) = a2(1,*)*C_EV_KEV	; emittance shifted energy (keV)
	ap(2,*) = ap(0,*)-ap(1,*)	; energy shift (keV)
	ap(3:nc,*) = a2(2:nc-1,*)
	if nc eq 6 then begin		; regular device
	  ap(5,*) = ap(5,*)*C_W_KW	; power (kW)
	  ap(6,*) = ap(6,*)*C_W_KW	; power density (kW/mrad^2)
	endif else if nc eq 7 then begin; helical device
	  ap(6,*) = ap(6,*)*C_W_KW	; power (kW)
	  ap(7,*) = ap(7,*)*C_W_KW	; power density (kW/mrad^2)
	endif
; for future use
;	xmin = min(ap(1,*))		; emittance shifted energy (keV)
;	ymax = max(ap(3,*))		; on-axis brilliance
;	if nh gt 1 then begin
;	  i = nh -2 
;	  xmax = 0.75*max(ap(1,*))
;	  ymin = min(ap(3,np*i:np*(i+1)-1))
;	endif else begin			; only one harmonic
;	  xmax = 1.00*max(ap(1,*))
;	  ymin = min(ap(3,0:np-1))
;	endelse
	coltitles = ['Zero Emittance Energy (keV)', $
	  'Energy (keV)', 'Energy Shift (keV)', $
	  'Brilliance (ph/s/mrad^2/mm^2/0.1%bw)']
	coltitles_extra = ['Ky', 'Ptot (kW)', 'Power Density (kW/mrad^2)']
	if nc eq 7 then coltitles_extra = ['Kx', coltitles_extra] ; helical
	if nc eq 6 or nc eq 7 then coltitles = [coltitles, coltitles_extra]
; for future use
;	xplot,ap,GROUP=event.top,xrange=[xmin,xmax],yrange=[ymin,ymax],/ylog, $
	xplot,ap,GROUP=event.top, $
	  coltitles=coltitles, xtitle='-1',ytitle='-1',title=title, $
	  xcol=2,ycol=4, WTitle='XTC results'
        END
      ENDCASE ; elist(3)
      END ; Xplot
      'Xwindow': BEGIN
      CASE elist(3) OF 
        'On-Axis Brilliance': BEGIN
	ap = a2(1:2,*)			; emittance shifted energy = second col
	ap(0,*) = ap(0,*)*C_EV_KEV	; emittance shifted energy = first col
	xtitle = 'Energy (keV)'
	ytitle = 'Brilliance (ph/s/mrad^2/mm^2/0.1%bw)'
	xmin = min(ap(0,*))
	ymax = max(ap(1,*))
	if nh gt 1 then begin
	  i = nh -2 
	  xmax = 0.75*max(ap(0,*))
	  ymin = min(ap(1,np*i:np*(i+1)-1))
	endif else begin			; only one harmonic
	  xmax = 1.00*max(ap(0,*))
	  ymin = min(ap(1,0:np-1))
	endelse
; ; prepare buffer for calling xwindow
; ; first line
; 	buffer = 'plotfile,data(*,0:' +strcompress(np-1,/rem) +'),' +$
; 		 'yrange=[' +strcompress(ymin,/rem) +',' +$
; 		 strcompress(ymax,/rem) +'],/ylog' +',' +$
; 		 'xrange=[' +strcompress(xmin,/rem) +',' +$
; 		 strcompress(xmax,/rem) +']' +',' +$
; 		 'xtitle=' +xtitle +',' +$
; 		 'ytitle=' +ytitle +',' +$
; 		 'title=' +"'" +title +"'"
; ;append second line
; 	buffer = [buffer,'for i=1,' +strcompress(nh-1,/rem) +$
; 		 ' do plotfile,data(*,' +strcompress(np,/rem) +'*i:' +$
; 		 strcompress(np,/rem) +'*(i+1)-1),kind=2']
; ;       print,buffer(0)
; ;       print,buffer(1)
; 	xwindow,buffer=buffer,data=ap,/edit
	p=0L
	Xplot,Parent=p,ap[*,0:np-1],/No_Block, XRange=[xmin,xmax],$
	  YRange=[ymin,ymax],XTitle=xtitle,YTitle=ytitle, Title=title, $
	  /YLog, Wtitle='XTC results ('+elist(3)+')'
	FOR i=1,nh-1 DO BEGIN
	  Xplot_SaveCurrent,p
	  Xplot_LoadFile,p,ap[*,np*i:np*(i+1)-1]
	ENDFOR


        END

        'Energy Shift': BEGIN
	ap = fltarr(2,np*nh)
	ap(0,*) = a2(0,*)*C_EV_KEV	; zero emittance energy (keV)
	ap(1,*) = (a2(0,*)-a2(1,*))*C_EV_KEV	; energy shift (keV)
	xtitle = 'Zero Emittance Energy (keV)'
	ytitle = 'Energy Shift (keV)'
	xmin = min(ap(0,*))
	xmax = max(ap(0,*))
	ymin = min(ap(1,*))
	ymax = max(ap(1,*))
;; prepare buffer for calling xwindow
;; first line
;	buffer = 'plotfile,data(*,0:' +strcompress(np-1,/rem) +'),' +$
;		 'yrange=[' +strcompress(ymin,/rem) +',' +$
;		 strcompress(ymax,/rem) +'],psym=-2' +',' +$
;		 'xrange=[' +strcompress(xmin,/rem) +',' +$
;		 strcompress(xmax,/rem) +']' +',' +$
;		 'xtitle=' +xtitle +',' +$
;		 'ytitle=' +ytitle +',' +$
;		 'title=' +"'" +title +"'"
;;append second line
;	buffer = [buffer,'for i=1,' +strcompress(nh-1,/rem) +$
;		 ' do plotfile,data(*,' +strcompress(np,/rem) +'*i:' +$
;		 strcompress(np,/rem) +'*(i+1)-1),psym=-2,kind=2']
;	xwindow,buffer=buffer,data=ap,/edit


	p=0L
	Xplot,Parent=p,ap[*,0:np-1],/No_Block, XRange=[xmin,xmax],$
	  YRange=[ymin,ymax],XTitle=xtitle,YTitle=ytitle, Title=title, $
	  Wtitle='XTC results ('+elist(3)+')'
	Xplot_Controls_Action,p,PSymbol=2
	FOR i=1,nh-1 DO BEGIN
	  Xplot_SaveCurrent,p
	  Xplot_LoadFile,p,ap[*,np*i:np*(i+1)-1]
	ENDFOR

	END

        'K-value': BEGIN
	if (nc ne 6) and (nc ne 7) then begin
	  junk = Dialog_Message(/INFO, DIALOG_PARENT=event.top, ptxt)
	  GOTO, LBL900
	endif
	ap = fltarr(2,np*nh)
	ap(0,*) = a2(0,*)*C_EV_KEV	; zero emittance energy (keV)
	ap(1,*) = a2(3,*)		; if helical Kx = Ky
	xtitle = 'Zero Emittance Energy (keV)'
	if nc eq 6 then ytitle = 'Ky' else ytitle = 'Kx, Ky'
	xmin = min(ap(0,*))
	ymin = min(ap(1,*))
	ymax = max(ap(1,*))
	if nh gt 1 then begin
	  i = nh -2 
	  xmax = 0.75*max(ap(0,*))
	endif else begin			; only one harmonic
	  xmax = 1.00*max(ap(0,*))
	endelse
;; prepare buffer for calling xwindow
;; first line
;	buffer = 'plotfile,data(*,0:' +strcompress(np-1,/rem) +'),' +$
;		 'yrange=[' +strcompress(ymin,/rem) +',' +$
;		 strcompress(ymax,/rem) +']' +',' +$
;		 'xrange=[' +strcompress(xmin,/rem) +',' +$
;		 strcompress(xmax,/rem) +']' +',' +$
;		 'xtitle=' +xtitle +',' +$
;		 'ytitle=' +ytitle +',' +$
;		 'title=' +"'" +title +"'"
;;append second line
;	buffer = [buffer,'for i=1,' +strcompress(nh-1,/rem) +$
;		 ' do plotfile,data(*,' +strcompress(np,/rem) +'*i:' +$
;		 strcompress(np,/rem) +'*(i+1)-1),kind=2']
;	xwindow,buffer=buffer,data=ap,/edit

	p=0L
	Xplot,Parent=p,ap[*,0:np-1],/No_Block, XRange=[xmin,xmax],$
	  YRange=[ymin,ymax],XTitle=xtitle,YTitle=ytitle, Title=title, $
	  Wtitle='XTC results ('+elist(3)+')'
	FOR i=1,nh-1 DO BEGIN
	  Xplot_SaveCurrent,p
	  Xplot_LoadFile,p,ap[*,np*i:np*(i+1)-1]
	ENDFOR

        END

        'Total Power': BEGIN
	if (nc ne 6) and (nc ne 7) then begin
	  junk = Dialog_Message(/INFO, DIALOG_PARENT=event.top, ptxt)
	  GOTO, LBL900
	endif
	ap = fltarr(2,np*nh)
	ap(0,*) = a2(0,*)*C_EV_KEV	; zero emittance energy (keV)
	if nc eq 6 then ap(1,*) = a2(4,*) else ap(1,*) = a2(5,*)
	ap(1,*) = ap(1,*)*C_W_KW	; convert W -> kW
	xtitle = 'Zero Emittance Energy (keV)'
	ytitle = 'Total Power (kW)'
	xmin = min(ap(0,*))
	ymin = min(ap(1,*))
	ymax = max(ap(1,*))
	if nh gt 1 then begin
	  i = nh -2 
	  xmax = 0.75*max(ap(0,*))
	endif else begin			; only one harmonic
	  xmax = 1.00*max(ap(0,*))
	endelse
;; prepare buffer for calling xwindow
;; first line
;	buffer = 'plotfile,data(*,0:' +strcompress(np-1,/rem) +'),' +$
;		 'yrange=[' +strcompress(ymin,/rem) +',' +$
;		 strcompress(ymax,/rem) +']' +',' +$
;		 'xrange=[' +strcompress(xmin,/rem) +',' +$
;		 strcompress(xmax,/rem) +']' +',' +$
;		 'xtitle=' +xtitle +',' +$
;		 'ytitle=' +ytitle +',' +$
;		 'title=' +"'" +title +"'"
;;append second line
;	buffer = [buffer,'for i=1,' +strcompress(nh-1,/rem) +$
;		 ' do plotfile,data(*,' +strcompress(np,/rem) +'*i:' +$
;		 strcompress(np,/rem) +'*(i+1)-1),kind=2']
;	xwindow,buffer=buffer,data=ap,/edit

	p=0L
	Xplot,Parent=p,ap[*,0:np-1],/No_Block, XRange=[xmin,xmax],$
	  YRange=[ymin,ymax],XTitle=xtitle,YTitle=ytitle, Title=title, $
	  Wtitle='XTC results ('+elist(3)+')'
	FOR i=1,nh-1 DO BEGIN
	  Xplot_SaveCurrent,p
	  Xplot_LoadFile,p,ap[*,np*i:np*(i+1)-1]
	ENDFOR
        END

        'Power Density': BEGIN
	if (nc ne 6) and (nc ne 7) then begin
	  junk = Dialog_Message(/INFO, DIALOG_PARENT=event.top, ptxt)
	  GOTO, LBL900
	endif
	ap = fltarr(2,np*nh)
	ap(0,*) = a2(0,*)*C_EV_KEV	; zero emittance energy (keV)
	if nc eq 6 then ap(1,*) = a2(5,*) else ap(1,*) = a2(6,*)
	ap(1,*) = ap(1,*)*C_W_KW	; convert W/mrad^2 -> kW/mrad^2
	xtitle = 'Zero Emittance Energy (keV)'
	ytitle = 'Power Density (kW/mrad^2)'
	xmin = min(ap(0,*))
	ymin = min(ap(1,*))
	ymax = max(ap(1,*))
	if nh gt 1 then begin
	  i = nh -2 
	  xmax = 0.75*max(ap(0,*))
	endif else begin			; only one harmonic
	  xmax = 1.00*max(ap(0,*))
	endelse
;; prepare buffer for calling xwindow
;; first line
;	buffer = 'plotfile,data(*,0:' +strcompress(np-1,/rem) +'),' +$
;		 'yrange=[' +strcompress(ymin,/rem) +',' +$
;		 strcompress(ymax,/rem) +']' +',' +$
;		 'xrange=[' +strcompress(xmin,/rem) +',' +$
;		 strcompress(xmax,/rem) +']' +',' +$
;		 'xtitle=' +xtitle +',' +$
;		 'ytitle=' +ytitle +',' +$
;		 'title=' +"'" +title +"'"
;;append second line
;	buffer = [buffer,'for i=1,' +strcompress(nh-1,/rem) +$
;		 ' do plotfile,data(*,' +strcompress(np,/rem) +'*i:' +$
;		 strcompress(np,/rem) +'*(i+1)-1),kind=2']
;	xwindow,buffer=buffer,data=ap,/edit

	p=0L
	Xplot,Parent=p,ap[*,0:np-1],/No_Block, XRange=[xmin,xmax],$
	  YRange=[ymin,ymax],XTitle=xtitle,YTitle=ytitle, Title=title, $
	  Wtitle='XTC results ('+elist(3)+')'
	FOR i=1,nh-1 DO BEGIN
	  Xplot_SaveCurrent,p
	  Xplot_LoadFile,p,ap[*,np*i:np*(i+1)-1]
	ENDFOR

        END
      ENDCASE ; elist(3)
      END ; Xwindow
    ENDCASE ; elist(2)
    END ; Plot Results

    'View  Results': BEGIN
    XDISPLAYFILE1, fname_res, GROUP=event.top
    END

  ENDCASE ; elist(1)

ENDELSE ; event.value
ENDCASE ; event.value

LBL900:
IF WIDGET_INFO(event.top, /VALID) THEN BEGIN $
  HANDLE_VALUE, hc, control, /SET, /NO_COPY
  HANDLE_VALUE, hd, data, /SET, /NO_COPY
;  save structure plt if not a new run
  IF eVal ne 'Set & Run' THEN HANDLE_VALUE, hp, plt, /SET, /NO_COPY
; WIDGET_CONTROL, event.top, SET_UVALUE=info; save syntax for future use
ENDIF
RETURN

END ; xtc_event
 
;
;==============================================================================
;
PRO xtc, GROUP=group, InputFile=inputFile

Forward_Function xop_defaults
Catch, error_status
IF error_status NE 0 THEN BEGIN
   Message,/Info,'error caught: '+!error_state.msg
   itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
     'XTC: error caught: '+!error_state.msg)
   Catch, /Cancel
   On_Error,2
   RETURN
ENDIF

;  Definitions and labels
;XTC_TITLE1 = 'Xtc 1.4'
XTC_TITLE1 = 'Xtc '+Xtc_version()
;XTC_LBL1   = 'TC: Tuning Curves'
XTC_LBL1   = 'TC'
XTC_LBL2   = 'Undulator Tuning Curves'

delimiter = ':'
osversion = sdep()	; system dependent operating system
ds = sdep(/ds)		; system dependent directory separator
ps = sdep(/ps)		; system dependent path separator

;  set working directory
; if not keyword_set(group) then xop_wd,/default

CD, CURRENT=cwd & cwd = cwd +ds
fname_res = cwd +'tc.out'

;  Structure for pull-down menu
junk = {CW_PDMENU_S, flags:0, name:''}

tcbase = WIDGET_BASE(COLUMN=1,TITLE=XTC_TITLE1,MBar=wMenuBar)

wFile = widget_button(wMenuBar,VALUE='File',/MENU)
  wtmp0 = widget_button(wFile,VALUE='Xtc input parameters', /Menu)
  wtmp = widget_button(wtmp0,VALUE='Load from file...',UValue='FILEINPUT')
  wtmp = widget_button(wtmp0,VALUE='Save to file...',UValue='FILEINPUT')
  wtmp = widget_button(wtmp0,VALUE='Save as default',UValue='FILEINPUT')

  wtmp0 = widget_button(wFile,VALUE='TC file', /Menu)
  wtmp = widget_button(wtmp0,VALUE='Load input file...',$
	UValue='File:Load  TC:Input      File')
  wtmp = widget_button(wtmp0,VALUE='Load output file...',$
	UValue='File:Load  TC:Output     File')
  wtmp = widget_button(wtmp0,VALUE='Write input file...',$
	UValue='File:Write TC:Input  File')
  wtmp = widget_button(wtmp0,VALUE='Write output file...',$
	UValue='File:Write TC:Output File')

; Not yet implemented
;  wtmp = widget_button(wFile,VALUE='Save files for XOP/Optics', $
;	UVALUE='OPTICSFILES')

  wtmp = widget_button(wFile,VALUE='Quit', UVALUE='File:Quit',/Separator)


wSetParameters = widget_button(wMenuBar,VALUE='Set_Parameters', /MENU)
  wtmp = widget_button(wSetParameters,VALUE='Set Parameters', $
	UVALUE='Set & Run')
  wtmp = widget_button(wSetParameters,VALUE='Set Defaults', $
	UVALUE='File:Load  TC:Default    File')

wResults = widget_button(wMenuBar,VALUE='Show',/MENU)
;  wtmp = widget_button(wResults,VALUE='Plot/Xplot/OnAxisBrilliance',$
;	UValue='Show:Plot Results:Xplot:On-Axis Brilliance')
  wtmp = widget_button(wResults,VALUE='Plot All',$
	UValue='Show:Plot Results:Xplot:All')
  wtmp = widget_button(wResults,VALUE='Plot OnAxis Brilliance',$
	UValue='Show:Plot Results:Xwindow:On-Axis Brilliance')
  wtmp = widget_button(wResults,VALUE='Plot Energy Shift',$
	UValue='Show:Plot Results:Xwindow:Energy Shift')
  wtmp = widget_button(wResults,VALUE='Plot K-value',$
	UValue='Show:Plot Results:Xwindow:K-value')
  wtmp = widget_button(wResults,VALUE='Plot Total Power',$
	UValue='Show:Plot Results:Xwindow:Total Power')
  wtmp = widget_button(wResults,VALUE='Plot Power Density',$
	UValue='Show:Plot Results:Xwindow:Power Density')
  wtmp = widget_button(wResults,VALUE='View results',$
	UValue='Show:View  Results')

wHelp = widget_button(wMenuBar,VALUE='Help', /Help)
  wtmp = widget_button(wHelp,VALUE='TC', $
	UValue='Help:TC')
  wtmp = widget_button(wHelp,VALUE='Xtc', $
	UValue='Help:Xtc')
;

wtmp = widget_button(tcbase,VALUE='Set Parameters', UVALUE='Set & Run')

;=======================================
;md = [ $
;  {CW_PDMENU_S, 1, 'File'}, $
;    {CW_PDMENU_S, 1, 'Load  TC'}, $
;      {CW_PDMENU_S, 0, 'Input      File'}, $
;      {CW_PDMENU_S, 0, 'Output     File'}, $
;      {CW_PDMENU_S, 2, 'Default    File'}, $
;    {CW_PDMENU_S, 1, 'Write TC'}, $
;      {CW_PDMENU_S, 0, 'Input  File'}, $
;      {CW_PDMENU_S, 2, 'Output File'}, $
;    {CW_PDMENU_S, 0, 'Display File'}, $
;    {CW_PDMENU_S, 2, 'Quit'}, $
;
;  {CW_PDMENU_S, 0, 'Set & Run'}, $
;
;  {CW_PDMENU_S, 1, 'Show'}, $
;    {CW_PDMENU_S, 1, 'Plot Results'}, $
;      {CW_PDMENU_S, 1, 'Xplot'}, $
;        {CW_PDMENU_S, 0, 'On-Axis Brilliance'}, $
;        {CW_PDMENU_S, 2, 'All'}, $
;      {CW_PDMENU_S, 3, 'Xwindow'}, $
;        {CW_PDMENU_S, 0, 'On-Axis Brilliance'}, $
;        {CW_PDMENU_S, 0, 'Energy Shift'}, $
;        {CW_PDMENU_S, 0, 'K-value'}, $
;        {CW_PDMENU_S, 0, 'Total Power'}, $
;        {CW_PDMENU_S, 2, 'Power Density'}, $
;    {CW_PDMENU_S, 2, 'View  Results'}, $
;
;  {CW_PDMENU_S, 3, 'Help'}, $
;    {CW_PDMENU_S, 0, 'TC'}, $
;    {CW_PDMENU_S, 2, 'Xtc'} ]
;
;pd = CW_PDMENU(tcbase, md, DELIMITER=delimiter, /RETURN_FULL_NAME)
;
;=======================================

CASE osversion OF
  'UNIX': BEGIN
    ;font='-b&h-lucida-bold-i-normal-sans-24-240-75-75-p-151-iso8859-1'
    font = '-adobe-helvetica-bold-o-normal--18-180-75-75-p-104-iso8859-1'
  END
  'WINDOWS': BEGIN
    font = 'VERDANA*BOLD*ITALIC*24'
  END
  'MACOS': BEGIN
    font = 'VERDANA*BOLD*ITALIC*24' ; 'MACOS command here'
  END
ENDCASE

lb = WIDGET_LABEL(tcbase, FONT=font, VALUE=XTC_LBL1)
lb = WIDGET_LABEL(tcbase, FONT=font, VALUE=XTC_LBL2)

control = {delimiter:delimiter, fname_res:fname_res, osversion:osversion, $
	   ds:ds, ps:ps}
hc = HANDLE_CREATE(VALUE=control)
hd = HANDLE_CREATE(hc)
hp = HANDLE_CREATE(hc)
xtc_str = Xop_Defaults('xtc')


IF KeyWord_Set(inputFile) THEN BEGIN
  str_par = Xop_Input_Load(InputFile=inputFile)
  IF Type(str_par) EQ 8 THEN BEGIN
    tmp = xtc_str
    Copy_Structure,str_par, tmp, Group=group
    xtc_str = tmp
  ENDIF
ENDIF


Handle_Value,hd,{dat1:xtc_str},/Set
info = {hc:hc, hd:hd, hp:hp}
;print,'main',hc,hd,hp,tcbase

WIDGET_CONTROL, tcbase, SET_UVALUE=info

WIDGET_CONTROL, tcbase, /REALIZE
XMANAGER, 'xtc', tcbase, GROUP_LEADER=group

END ; xtc
