
;=========================== copyright =========================================
;                       XAID (XAfs with IDl) package.                          ;
;     Copyright European Synchrotron Radiation Facility (1993-1996).           ;
; This software may be used, copied, or redistributed as long as it is not     ;
; sold and this copyright notice is reproduced on each copy made.              ;
; The software is provided as is without any express or implied warranties     ;
; whatsoever.                                                                  ;
; Other limitations apply as described in the file README.copyright.           ;
;                                                                              ;
; REFERENCE: "XAID: A package for XAFS data analysis based on IDL"             ;
;       Manuel Sanchez del Rio. To be published at the proceedings of          ;
;       the 9th International Conference on X-ray Absorption Fine Structure    ;
;       26th - 30th August 1996,  E S R F - Grenoble  France                   ;
;==============================================================================;
pro FASTBFTR,fourier,backftr,NPOINT=npoint,KRANGE=krange,RSTEP=rstep, $
  GROUP=group
;+
; NAME:
;	FASTBFTR
;
; PURPOSE:
;	This procedure calculates the Back Fast Fourier Transform of a set
;
; CATEGORY:
;	XAID xafs data analysis package. 
;
; CALLING SEQUENCE:
;
;	FASTBFTR, fourier, backftr
;
; INPUTS:
;       fourier:  a 4 col set with r,modulus,real and imaginary part
;               of a Fourier Transform of an Exafs spectum, as produced
;               by FTR or FASTFTR procedures
;       backftr: a 4-column set with the  conjugated variable k (col 0),
;       the real part (col 1), the modulus (col 2) and phase (col 3)
;       of the Back Fourier Transform
;
; OPTIONAL INPUTS:
;	
; KEYWORD PARAMETERS:
;	KRANGE=[kmin,kmax] : range of the conjugated variable for 
;		the transformation (default = [2,15])
;	NPOINT= number of points of the the fft calculation (default = 4096)
;	RSTEP = when this keyword is set then the fourier set is 
;		interpolated using the indicated value as step. Otherwise
;		the fourier set is not interpolated.
;	GROUP = the widget group leader Id. Used together with the 
;		Dialog_Parent keyword in Dialog_Message(s).
;
; OUTPUTS:
;       This procedure returns a 4-columns array (backftr) with
;       the conjugated variable (k) in column 0, the real part of the
;       BFT in col 1, the modulus in col 2 and the phase in col 3.
;
; OPTIONAL OUTPUTS:
;
; COMMON BLOCKS:
;
; SIDE EFFECTS:
;	None
;
; RESTRICTIONS:
;	None
;
; PROCEDURE:
;	Uses the IDL's FFT function
;
; EXAMPLE:
;               FASTBFTR,fourier,backftr,krange=[2,12]
;
; MODIFICATION HISTORY:
; 	Written by:	Manuel Sanchez del Rio. ESRF
;	March, 1993	
;	98-10-26 srio@esrf.fr uses Dialog_Message for error messages.
;-
;
on_error,2
;
if not(keyword_set(krange)) then begin
  kmin = 2.0
  kmax = 15.0
endif else begin
  kmin = float(krange(0))
  kmax = float(krange(1))
endelse
;
npoint = 4096
npt=n_elements(fourier(0,*))
fou=fltarr(4,npoint)
rmin = min(fourier(0,*))
rmax = max(fourier(0,*))
;
; fill "fou" set
;
;--- no interpolation
if not(keyword_set(rstep)) then begin
  nn=fix(npt/2)
  rstep = fourier(0,nn+1)-fourier(0,nn)
  rstep2 = fourier(0,nn+2)-fourier(0,nn+1)
  rdiff = abs (rstep - rstep2)
;  print,' back rstep = ',rstep
;  print,' rdiff = ',rdiff
  if (rdiff gt 1e-6) then begin
    message,/info, $
     'r griding is not regular; Use RSTEP keyword -> Abort'
    if sdep(/w) then itmp = Dialog_Message(/Error,$
     'r griding is not regular; Use RSTEP keyword -> Abort')
    return
  endif
  ptstart=fix(rmin/rstep)
;  print,' ptstart is ',ptstart
  fou(*,ptstart:ptstart+npt-1)=fourier
endif else begin
;
;--- interpolation
;  rstep=0.02
  fou(0,*)=fix(findgen(npoint))*rstep
  fou(1,*)=interpol(fourier(1,*),fourier(0,*),fou(0,*))
  fou(2,*)=interpol(fourier(2,*),fourier(0,*),fou(0,*))
  fou(3,*)=interpol(fourier(3,*),fourier(0,*),fou(0,*))
  for i=0,npoint-1 do begin
    if (fou(0,i) lt rmin) then fou(1:3,i)=0.0
    if (fou(0,i) gt rmax) then fou(1:3,i)=0.0
  endfor
endelse
;---
;
; call back fft
;
c=complex(fou(2,*),fou(3,*)*(-1.))   ;/coef
af=fft(c,1)
;
; create the array of the conjugated variable
;
kstep=!pi/npoint/rstep
kk=fix(findgen(npoint))*kstep
;print,' back kstep = ',kstep
;
; prepare the output array
;
coef=npoint*kstep/sqrt(!pi)*sqrt(2.) ; coefficienu used for direct fft
coef1=2./coef                        ; 2 because we are only 
				     ; considering one part of the ftr
afr= coef1*float(af)                 ; real part of back fft
afi= coef1*imaginary(af)          ; imaginary part of back fft
;
; cut the results to the selected interval in k (krange)
;
afr=afr(where((kk gt kmin)and(kk lt kmax)))
afi=afi(where((kk gt kmin)and(kk lt kmax)))
afk=kk(where((kk gt kmin)and(kk lt kmax)))
nptout=n_elements(afk)
;
; define the output set
;
backftr=fltarr(4,nptout)
backftr(0,*)=afk                  ; the conjugated variable (k [A^-1])
backftr(1,*)=afr                  ; the real part of backftr or atra
backftr(2,*)=sqrt(afr^2+afi^2)    ; the modulus of backftr
backftr(3,*)=atan(afi/afr)        ; the phase
;
; now correct the phase to make it continuous
;
for i=1,nptout-1 do begin
  if ( abs(backftr(3,i)-backftr(3,i-1)) gt 0.6*!pi) then begin
    for j=i,nptout-1 do backftr(3,j)=backftr(3,j)+!pi
  endif
endfor
;
return
end
