pro PLOT3,shadow_in,col1,col2,hz,hx,hy,XRANGE=xran,YRANGE=yran, $
NOLOST=p_lost,NBINSX=p_nbinsx, NBINSY=p_nbinsy,XYRANGE=xyran, $
REF=ref, HISTO=histo,TITLE=tottitle,NLEVELS=p_nlevels,_EXTRA=extra
;+
; NAME:
;	PLOT3
; PURPOSE:
;	Plot a 3-dimensional plot/histo of a Shadow file
; CATEGORY:
;	SHADOW's utilities.
; CALLING SEQUENCE:
;	plot3,shadow_in,col1,col2,keyword_parameters
; INPUTS:
;	shadow_in     an IDL structure with SHADOW data
;	col1    column of x-axis
;	col2    column of y-axis
; OPTIONAL OUTPUTS:
;       hz,hx,hy    named variables with the surface (hz) and two arrays
;                   with the x and y arrays (hx and hy)
;               ( this option permits you the reuse of the surface generated 
;               by PLOT3 for further manipulation, eg. IDL> surface,hz,hx,hy)
; KEYWORD PARAMETERS:
;	XRANGE = [ min_x,max_x ]
;	YRANGE = [ min_y,max_y ]
;	XYRANGE = [ min,max ] (sets XRANGE=YRANGE)
;	NOLOST = consider losses
;		0 all rays (default option)
;		1 exclude losses;
;	NBINSX = number of bins in x (def=30);
;	NBINSY = number of bins in y (def=30);
;       REF = include reflectivity
;               0 no (default)
;               1 yes (A**2 as weight=column 23)
;               n (21<n<34, n=column to weight (see PLOTXY for col description))
;
;       HISTO= lateral histogran setting
;               0 do not include histograms (default).
;               1 do plot histograms.
;       NLEVELS = number of levels for the contour plot (default=10).
;       TITLE   = a chain with the title
;
;	Any other keyword is passed to surface or/and countour procedures.
; OUTPUTS:
;	a graphic, and an IDL structure with the data (optional).
; COMMON BLOCKS:
;	None.
; SIDE EFFECTS:
;	None.
; RESTRICTIONS:
;	None.
; PROCEDURE:
;       Similar to Plotxy
; MODIFICATION HISTORY:
;	M. Sanchez del Rio. ESRF. Grenoble, Feb 1992
;	1-Sep-1992 Change input types.
;	98-06-17 srio@esrf.fr almost rewritten. Added _EXTRA kw.
;-

catch, error_status
 if error_status ne 0 then begin
   catch, /cancel
   message,/info,'Error caught: '+!err_string
   if sdep(/w) then itmp = Dialog_Message(/Error, $
	'PLOT3: Error caught: '+!err_string)
endif

if n_params() LT 3 then begin
  message,/info,'Using arrays...'
  case n_params() of
  2: begin
       arr1 =  shadow_in
       arr2 =  col1
     end
  1: begin
       arr1 =  shadow_in(0,*)
       arr2 =  shadow_in(1,*)
     end
  else: begin
     message,/info,'Error with inputs.'
     return
     end
  endcase
endif else begin
  arr1 = getshcol(shadow_in,col1,nolost=nolost)
  arr2 = getshcol(shadow_in,col2,nolost=nolost)
  int = arr1*0.0+1
  if keyword_set(ref) then begin
    if ref EQ 1 then ref = 23
    int = getshcol(shadow_in,ref)
  endif
endelse
;
; limits
;
off1 = 0.1*abs( max(arr1)-min(arr1) )
off2 = 0.1*abs( max(arr2)-min(arr2) )
if off1 lt 1e-10 then off1=0.5*max(arr1)
if off2 lt 1e-10 then off2=0.5*max(arr2)
if (not(keyword_set(xran)) and not(keyword_set(xyran)) ) then begin
  xrange =[min(arr1)-off1,max(arr1)+off1]
  xrange = round1( xrange,fix(alog(xrange(1)-xrange(0))/alog(10))-2)
endif 
if (not(keyword_set(yran)) and not(keyword_set(xyran)) ) then begin
  yrange =[min(arr2)-off2,max(arr2)+off2]
  yrange = round1( yrange,fix(alog(yrange(1)-yrange(0))/alog(10))-2)
endif
if keyword_set(xran) then xrange = xran
if keyword_set(yran) then yrange = yran
if keyword_set(xyran) then begin
  xrange = xyran
  yrange = xyran
endif

;
;	 histogram
;

; first the 3-D histo

if keyword_set(p_nbinsx) then nbinsx=p_nbinsx else nbinsx = 30
if keyword_set(p_nbinsy) then nbinsy=p_nbinsy else nbinsy = 30
binsizex =(xrange(1)-xrange(0))/float(nbinsx)
binsizey =(yrange(1)-yrange(0))/float(nbinsy)
hz = histogramw2(arr1,arr2,int,binsizex=binsizex,binsizey=binsizey, $
     xrange=xrange,yrange=yrange)
hx = fltarr(nbinsx)
hy = fltarr(nbinsy)
for i=0,nbinsx-1 do hx(i) = xrange(0)+ binsizex/2 + binsizex*i 
for i=0,nbinsy-1 do hy(i) = yrange(0)+ binsizey/2 + binsizey*i 

; second the proyections

if keyword_set(histo) then begin
  hlx = histogramw(arr1,int,binsize=binsizex, min=xrange(0),max=xrange(1))
  hly = histogramw(arr2,int,binsize=binsizey, min=yrange(0),max=yrange(1))
  glx = hlx & gly = hly
  for i=0,nbinsx-2,2 do begin
    glx(i) = (hlx(i) + hlx(i+1) )/2.0 
    glx(i+1) = glx(i)
  endfor
  for i=0,nbinsy-2,2 do begin
    gly(i) = (hly(i) + hly(i+1) )/2.0 
    gly(i+1) = gly(i)
  endfor
; now normalizations ans inclusion of the histograms in the surface
  gz = hz/max(hz)/1.2
  gz(*,nbinsy-1)=glx/max(hlx)/1.2
  gz(*,nbinsy-2)=glx/max(hlx)/1.2
  gz(nbinsx-1,*)=gly/max(hly)/1.2
  gz(nbinsx-2,*)=gly/max(hly)/1.2
endif else begin
  gz = hz
endelse
;
;      graphic
;
; prepare the titles
ztitle = ' Intensity [arb. units] '
xtitle=column_name(col1)
ytitle=column_name(col2)
title = ' '
if keyword_set(tottitle) then title=tottitle
; plots
t3d,/reset
surface,gz,hx,hy,/save,xrange=xrange,yrange=yrange, $
  xtitle=xtitle,ytitle=ytitle,ztitle=ztitle,title=title,_extra=extra
nlevels=10
if keyword_set(p_nlevels) then nlevels=p_nlevels
contour,hz,hx,hy,nlevels=nlevels,/t3d,/noerase,zvalue=1,/noclip,_extra=extra
;
;
;
return
end
