PRO MPP_MPlot,mpp,list,XCol=xCol,yCol=yCol,XRange=xRange,YRange=yRange,$
  Legend=iLegend,ShiftValue=shiftValue,_Extra=extra


;+
; 
; NAME: 
;	MPP_MPlot
;
; PURPOSE:
;	This procedure produces a multiple plot with the data stored
;	in the mpp handle
;
; CATEGORY:
;	MPP utilities to handle multiple array data
;
; CALLING SEQUENCE:
;	MPP_Plot,mpp,list
;
; INPUTS:
;     mpp: the MPP handle as created by MPP_New(). It can be a structure
;		or a pointer to a structure.
;     list: the indices of the arrays to be plotted
;
; OPTIONAL INPUTS:
;     INDEX: a string with the description of the array with the MPP 
;       indices to free. Examples: '[*]', '[1:2]'
;
; KEYWORD PARAMETERS:
;	XCol: the column to plot as abscissa
;	YCol: the ordinates column
;	XRange: a vector containing the X range ([xMin,xMax])
;	YRange: a vector containing the Y range ([yMin,yMax])
;	Legend: when this flag is set, a plot legend is created
;	ShiftValue: A factor to shift up the spectra 1,...N-1. 
;		The spectra are usually shifted up for better
;		visualization. The amount to shift each spectrum related 
;		to the one before is Abs(Max(y)-Min(y))*shiftValue, 
;		being y the ordinates of the first spectrum (index 0). 
;		Default = no shift
;	_Extra: any other keyword to be inherited by "plot"
;
;
; MODIFICATION HISTORY:
;	Initial version by M. Sanchez del Rio, February 2000.
;	00/03/20 srio@esrf.fr adds ShiftValue keyword.
;	00/04/11 srio@esrf.fr adds Extra keywords. Fix printing colors.
;
;-

Catch,error_status
IF error_status ne 0 THEN BEGIN
  Message,/Info,'error caught: '+!error_state.msg
  itmp=Dialog_Message(/Error, $
    'MPP_MPLOT: error caught: '+!error_state.msg)
  Catch, /Cancel
  RETURN
ENDIF

IF N_Elements(list) EQ 0 THEN list=LIndGen(MPP_Info(mpp,/N_Elements))
IF N_Elements(xCol) EQ 0 THEN xCol=1
IF N_Elements(yCol) EQ 0 THEN yCol=2
IF N_Elements(ilegend) EQ 0 THEN ilegend=1


IF N_Elements(xRange) NE 2 THEN BEGIN
  calculate_xrange = 1
ENDIF ELSE BEGIN
  calculate_xrange = 0
  IF xRange[0] EQ xRange[1] THEN  calculate_xrange = 1 
ENDELSE

IF N_Elements(yRange) NE 2 THEN BEGIN
  calculate_yrange = 1
ENDIF ELSE BEGIN
  calculate_yrange = 0
  IF yRange[0] EQ yRange[1] THEN  calculate_yrange = 1 
ENDELSE


ptrs = MPP_Value(mpp,List,/Pointer)
nn = N_Elements(ptrs)

IF calculate_xrange EQ 1 THEN BEGIN
  FOR i=0,N_Elements(ptrs)-1 DO $
    IF i EQ 0 THEN a0 = Reform((*ptrs[i])[xCol-1,*]) ELSE $
    a0 = [Temporary(a0),Reform((*ptrs[i])[xCol-1,*])]
  xrange=[Min(a0),Max(a0)]
ENDIF

IF calculate_yrange EQ 1 THEN BEGIN
  FOR i=0,N_Elements(ptrs)-1 DO $
    IF i EQ 0 THEN a1 = Reform((*ptrs[i])[yCol-1,*]) ELSE $
    a1 = [Temporary(a1),Reform((*ptrs[i])[yCol-1,*])]
  yrange=[Min(a1),Max(a1)]
ENDIF

IF N_Elements(shiftValue) NE 0 THEN BEGIN
  shiftValue = Float(shiftValue)
  tmp = (*ptrs[0])[yCol-1,*]
  yDelta = Abs(max(tmp)-min(tmp)) * shiftValue
  yRange[1]=yRange[1]+(yDelta*(N_Elements(ptrs)-1))
  yShift = FIndGen(N_Elements(ptrs)) * yDelta
ENDIF ELSE BEGIN
  yShift = Replicate(0.0, N_Elements(ptrs))
ENDELSE

plot,[0,1], XRange=xrange,YRange=yRange,/NoData,_Extra=extra
FOR i=0,N_Elements(ptrs)-1 DO $
   oplot,(*ptrs[i])[xCol-1,*],((*ptrs[i])[yCol-1,*])+yShift[i],Color=i+2

IF iLegend EQ 1 THEN $
  Legend,MPP_Value(mpp,List,/Title),/Fill, $
  PSym=Replicate(8,nn),Colors=SIndGen(nn)+2,Box=0

END
