


;+
; 
; NAME: 
;	MPP_LCFit
;
; PURPOSE:
;	Fits a data set with a linear combination of other data sets
;	Returns the fitted data set (optionally it also returns the
;	fitting parameters in the pars keyword).
;
;
; CATEGORY:
;	MPP utilities to handle multiple array data
;
; CALLING SEQUENCE:
;	out = MPP_LCFit,ptrMPP,iList
;
; INPUTS:
;     PTRMPP: the MPP handler (pointer) as created by MPP_New(). 
;     ILIST: the list of the arrays to be considered for the fit
;
; KEYWORD PARAMETERS:
;     DIALOG_PARENT: this keyword is passed to dialogs for centering
;		the widget with a parent widget.
;     PARS: Set this keyword to a named variable to receive the 
;		resulting parameters from the fit.
;
;
; EXAMPLES:
;
; MODIFICATION HISTORY:
;	Initial version by M. Sanchez del Rio, February 2000.
;	Still under development.
;
;-

;
;=========================================================================
;

PRO MPP_LCFit_Display, fcn, x, iter, FunctArgs=fcnargs, FMT=fmt, $
         Quiet=quiet, _Extra=iterargs

Forward_Function mpfit_enorm
; this routine is called by MPFit

Catch, error_status
IF error_status NE 0 THEN BEGIN
  Message,/Info,'error caught: '+!err_string
  itmp = Dialog_Message(/Error, $
    'MPP_LCFit_Display: error caught: '+!error_state.msg)
  Catch, /Cancel
  RETURN
ENDIF

IF Keyword_Set(quiet) THEN RETURN
fvec = Call_Function(fcn, x, _Extra=fcnargs)
fnorm = mpfit_enorm(fvec)

XDisplayfile1_Append, [String(iter, fnorm^2, $
  Format='("Iter ",I6,"   CHI-SQUARE = ",G20.8)'), $
  '           Timing [sec]:  '+$
	String(SysTime(1)-iterargs.t0,Format='(G6.2)')]
IF N_Elements(fmt) GT 0 THEN BEGIN
    XDisplayfile1_Append, String(p, format=fmt)
ENDIF ELSE BEGIN
    p = '  P('+StrTrim(LIndGen(N_Elements(x)),2)+') = ' $
      + StrTrim(String(x,Format='(G20.6)'),2) + '  '
    XDisplayfile1_Append, String('         '+p, Format='(A)')
ENDELSE

IF (SysTime(1)-iterargs.t0) GE iterargs.tmax THEN BEGIN
  itmp = Widget_Message(Dialog_Parent=iterargs.parent, $
	['Maximum time ('+StrCompress(Long(iterargs.tmax))+' sec) reached.',$
	 'Fitting process aborted.'])
  !err=-1
ENDIF
  
RETURN
END ; MPP_LCFit_Display

;
;=========================================================================
;

FUNCTION MPP_LCFit,ptrMPP,iList,Dialog_Parent=dialog_parent, PText=pText
COMMON MPP_LCFIT, lcFitStr

Catch, error_status
IF error_status NE 0 THEN BEGIN
   Message,/Info,'error caught: '+!err_string
   IF SDep(/w) THEN itmp = Dialog_Message(/Error,$
	'EXODUS_LCFIT: error caught: '+!err_string)
   Catch, /Cancel
   On_Error,2
   RETURN,'Error'
ENDIF

tmp=Vect2String(iList[1:N_Elements(iList)-1])
guess=Replicate(1.0,N_Elements(iList)-1)
guess = guess/Total(guess)
guess = vect2String(guess)


IF N_Elements(lcFitStr) EQ 0 THEN BEGIN
  tol = 1e-5
  maxIter=20
  tmax = 30.0
  lcFitStr = {target:iList[0], elem:tmp, guess:guess, $
    tol:tol, maxIter:maxIter, tMax:tMax}
ENDIF ELSE BEGIN
  lcFitStr.target = iList[0]
  lcFitStr.elem = tmp
  lcFitStr.guess = guess
ENDELSE

action = 0
XScrMenu,lcFitStr, FieldLen=50, Titles=[ $
  'Index [starting from zero] of target: ', $
  'Indices of LC components: ', 'Initial guess: ','Tolerances', $
  'Maximum number of iterations', 'Maximum running time [sec]'], $
  /NoType,/Interp, Dialog_Parent=dialog_Parent,Action=action, $
  WTitle='LC Fit pars'
  IF action EQ 'DONT' THEN RETURN,'Cancel'

data = MPP_Value(ptrMPP,lcFitStr.target)
x=Reform(data[0,*])
y=Reform(data[1,*])

a = 0
command = 'a = '+lcFitStr.guess
itmp =  Execute(command)

elem = 0
command = 'elem = '+lcFitStr.elem
itmp =  Execute(command)

IF N_Elements(a) NE N_Elements(elem) THEN BEGIN
  itmp = Dialog_Message(/Error,Dialog_Parent=dialog_Parent, $
   ['Number of elements in Guess are different than from components',$
    'Aborted'])
  RETURN,'Cancel'
ENDIF



tmpmpp = MPP_New()
FOR i=0,N_Elements(elem)-1 DO BEGIN
  itmp = elem[i]
  tmpdata = MPP_Value(ptrMPP,itmp)
  mpp_set,tmpmpp,Value=tmpdata,/Add
ENDFOR

;tmp = mpp_info(tmpmpp,/Verbose)
tmp = mpp_lcfunct(init=tmpmpp)

;a = Replicate(1.0D0,N_Elements(elem))
err = Replicate(1.0,N_Elements(x))

p=0

XDisplayfile1,Text=['',$
'Running MPFit (Levenberg-Marquardt [MINPACK-1]) on '+$
SysTime(),'',$
'Index [starting from zero] of target data: '+StrCompress(lcFitStr.target), $
'Indices of linear combination components: '+StrCompress(lcFitStr.elem), $
'Initial parameters (not normalized): '+Vect2String(a),$
'Tolerances: '+StrCompress(lcFitStr.tol,/Rem), $
'Maximum number of iterations: '+StrCompress(lcFitStr.maxIter,/Rem), $
'Limit time [sec]: '+StrCompress(lcFitStr.tmax,/Rem), $
; 'Fitting range: '+xrangetxt,$
; 'Fixed params (0=free,1=fix):'+Vect2String(parinfo(*).fixed),$
; 'Lower bounds (0=No,1=Yes):'+$
; Vect2String(parinfo(*).limited[0]),$
; 'Upper bounds (0=No,1=Yes):'+$
; Vect2String(parinfo(*).limited[1]),$
; 'Lower limits:'+Vect2String(parinfo(*).limits[0]),$
; 'Upper limits:'+Vect2String(parinfo(*).limits[1]),$
'',''],$
Group=dialog_parent, Dialog_Parent=dialog_Parent, $
Title='EXODUS: Non linear LC fitting results'

;p = MpFitFun('mpp_lcfunct',x,y,err,a,MaxIter=maxIter, $
;  XTol=1e-5,GTol=1e-5,FTol=1e-5,IterProc='mpp_lcfit_display')

IF Keyword_Set(dialog_Parent) THEN $
  iterArgs={t0:systime(1), tmax:lcFitStr.tmax, parent:dialog_Parent} $
  ELSE $
  iterArgs={t0:systime(1), tmax:lcFitStr.tmax}

pError=0
p = MpFitFun('mpp_lcfunct',x,y,err,a,MaxIter=maxIter, $
  XTol=tol,GTol=tol,FTol=tol,IterProc='mpp_lcfit_display',$
  Status=status,NPrint=1,IterArgs=iterArgs,PError=pError)

pp = p/total(p)

;r = mpfitexpr(equation, x, y, sigma, ParInfo=parInfo, $
;FTol=tol, XTol=tol, GTol=tol, $
;NPrint=state.defs.pref_nprint, MaxIter=itMax, $
;IterProc='xnlfit_mpfit_display',Status=status, $
;NIter=iter, Covar=covar, PError=pError,  $
;IterArgs={t0:systime(1), tmax:tmax, parent:event.top} )

CASE status OF
  0:  text='improper input parameters.'
  1:  text=['both actual and predicted relative reductions',$
  'in the sum of squares are at most FTOL(=TOL).']
  2:  text=['relative error between two consecutive iterates',$
  'is at most XTOL(=TOL)']
  3:  text=['both actual and predicted relative reductions',$
  'in the sum of squares are at most FTOL(=TOL). And, ',$
  'relative error between two consecutive iterates',$
  'is at most XTOL(=TOL)']
  4:  text=['the cosine of the angle between fvec and any ',$
  'column of the jacobian is at most GTOL(=TOL) in absolute',$
  'value.']
  5:  text='the maximum number of iterations has been reached'
  6:  text=['FTOL(=TOL) is too small. no further reduction in',$
  'the sum of squares is possible.']
  7:  text=['XTOL(=TOL) is too small. no further improvement ',$
  'in the approximate solution x is possible.']
  8:  text=['GTOL(=TOL) is too small. fvec is orthogonal to ',$
  'the columns of the jacobian to machine precision.']
  else: text='Unknown.'
ENDCASE



XDisplayFile1_Append,['','MPFit termination status: ',text,$
  '','Resulting params: '+Vect2String(p),$
  '','Resulting params (normalized): '+Vect2String(pp),$
  '','Task finished : '+SysTime(), $
;  '','Errors on resulting params: '+Vect2String(pError),$
  '', $
  '** Fitting result added to data set list **','']

;======================================================


pText = '#['+StrCompress(iList(lcFitStr.target),/Rem)+']=#'+$
        Vect2String(iList(elem))+'*'+vect2String(pp)


;p = MpFitFun('mpp_lcfunct',x,y,err,a,MaxIter=20, $
;  XTol=1e-5,GTol=1e-5,FTol=1e-5,IterProc='mpp_lcfit_display')
;
yFit = mpp_lcfunct(x,p)
out = Make_Set(x,yFit)
mpp_free,tmpmpp
RETURN,out
END ; mpp_lcFit.pro


