FUNCTION MPP_Info,mpp,N_Elements=nelem,Verbose=verbose


;+
; 
; NAME: 
;	MPP_Info
;
; PURPOSE:
;	Returns an array of strings with info on the MPP handle
;
; CATEGORY:
;	MPP utilities to handle multiple array data
;
; CALLING SEQUENCE:
;	MPP_Info,mpp 
;
; INPUTS:
;     mpp: the MPP handle as created by MPP_New(). It can be a structure
;		or a pointer to a structure.
;
; KEYWORD PARAMETERS:
;     N_Elements: when set, returns the number of elements (arrays) 
;	stored in the mpp handle
;     Verbose: if set, displays on the terminal the info
;
;
; EXAMPLES:
;	mpp = MPP_New()
;	mpp_set,mpp,Value=fltarr(3,3),/Add
;	tmp = mpp_info(mpp,/Verbose)
;	mpp_free,mpp  
;
; MODIFICATION HISTORY:
;	Initial version by M. Sanchez del Rio, February 2000.
;
;-


IF Type(mpp) EQ 10 THEN BEGIN
  IF Ptr_Valid(mpp) THEN mppstr=*mpp ELSE BEGIN
    Message,/Info,'Pointer not valid'
    Return,'MPP_INFO: Error: Pointer not valid' 
  ENDELSE
ENDIF ELSE BEGIN
  mppstr=mpp
ENDELSE

IF Keyword_Set(nelem) THEN BEGIN
  out = N_Elements(mppstr)
  GoTo,out
ENDIF

out = ''
txt = StrArr(N_Elements(mppstr)+1)
txt[0] = String('#',Format='(a4)') + $
      String('Flag',Format='(a6)') + $
      String('Type',Format='(a20)') + $
      String('Dim',Format='(a10)') + $
      ' Title'
FOR i=0,N_Elements(mppstr)-1 DO BEGIN
  title = (mppstr.title)[i]
  flag = (mppstr.flag)[i]
  pi = (mppstr.ptrData)[i] 
  IF Ptr_Valid( (mppstr.ptrData)[i] ) THEN BEGIN
    data = *( (mppstr[i]).ptrData )
    s = Size(data)
    IF s[0] GT 0 THEN dim = Vect2String(s[1:1+(s[0]-1)]) ELSE $
      dim=Vect2String(s[0]+1)
    td = Type(data,/Verbose)
  ENDIF ELSE BEGIN
    dim = '<None>'
    td = '<None>'
  ENDELSE
  txt[i+1] = String(i,Format='(I4)')+'  '+ $
        String(flag,Format='(I4)')+$
        String(td,Format='(a20)') + $
        String(dim,Format='(a10)') + $
        ' "'+title+'"'
ENDFOR
IF Keyword_Set(verbose) THEN BEGIN
  Print,'************************************************************'
  Print,'              MPP_INFO (Info on MultiPlotPointer)           '
  Print,''
  FOR i=0,N_Elements(txt)-1 DO Print,txt[i]
  Print,'************************************************************'
ENDIF

out=txt

out:
RETURN,out
END

